import bisect
import abc
from copy import deepcopy

import six
from awacs.lib import OrderedDict

from infra.awacs.proto import model_pb2
from awacs.model import components
from awacs.model.balancer.component_transports.diff import ComponentsDiff


CONTROLS_VOLUME_NAME = 'controls'
CONTROLS_VOLUME = {
    'name': CONTROLS_VOLUME_NAME,
    'type': 'ITS',
    'itsVolume': {
        'itsUrl': 'http://its.yandex-team.ru/v1',
        'periodSeconds': 5,
        'maxRetryPeriodSeconds': 10,
        'useSharedMemory': True,
    },
    'vaultSecretVolume': {
        'vaultSecret': {
            'secretVer': '',
            'secretId': '',
            'delegationToken': '',
            'secretName': '',
        },
    },
    'version': '',
    'secretVolume': {
        'keychainSecret': {
            'keychainId': '',
            'secretId': '',
            'secretRevisionId': '',
        },
        'secretName': '',
    },
    'templateVolume': {
        'template': [],
    },
}

EMPTY_HTTP_GET = {
    'path': '',
    'uriScheme': '',
    'httpHeaders': [],
    'host': '',
    'port': '',
}

EMPTY_TCP_SOCKET = {
    'host': '',
    'port': '',
}

EMPTY_EXEC_ACTIONS = {
    'command': [],
}

EMPTY_NOTIFY_ACTION = {
    'resourceRequest': {
        'request': [],
        'limit': [],
    },
    'handlers': [],
}

EMPTY_HANDLER = {
    'type': 'NONE',
    'execAction': EMPTY_EXEC_ACTIONS,
    'httpGet': EMPTY_HTTP_GET,
    'tcpSocket': EMPTY_TCP_SOCKET,
}


def get_exec_action_handler(command):
    rv = dict(EMPTY_HANDLER)
    rv['type'] = 'EXEC'
    rv['execAction'] = {
        'command': command,
    }
    return rv


NOTIFY_ACTION = {
    'resourceRequest': {
        'request': [],
        'limit': [],
    },
    'handlers': [
        get_exec_action_handler(['./awacslet', 'notify']),
    ]
}

INIT_CONTAINER = {
    'name': 'prepare',
    'command': ['./awacslet', 'prepare'],
}

AWACSLET_BALANCER_CONTAINER_NAME = 'balancer'
AWACSLET_BALANCER_CONTAINER_FIELDS = {
    'name': AWACSLET_BALANCER_CONTAINER_NAME,
    'command': ['./awacslet', 'start'],
    'reopenLogAction': {
        'handler': get_exec_action_handler(['./awacslet', 'reopenlog']),
    },
    'restartPolicy': {
        'periodBackoff': 2,
        'minPeriodSeconds': 5,
        'maxPeriodSeconds': 60,
        'periodJitterSeconds': 20,
    },
    'readinessProbe': {
        'periodBackoff': 2,
        'initialDelaySeconds': 5,
        'maxPeriodSeconds': 30,
        'minPeriodSeconds': 5,
        'handlers': [
            get_exec_action_handler(['./awacslet', 'status']),
        ],
    },
    'lifecycle': {
        'termBarrier': 'IGNORE',
        'preStop': get_exec_action_handler(['./awacslet', 'stop']),
        'stopGracePeriodSeconds': 90,
        'terminationGracePeriodSeconds': 30,
    },
}

AWACSLET_BALANCER_CONTAINER_FIELDS_0_0_3 = deepcopy(AWACSLET_BALANCER_CONTAINER_FIELDS)
AWACSLET_BALANCER_CONTAINER_FIELDS_0_0_3['command'] = ['./awacslet', 'start', 'balancer']
AWACSLET_BALANCER_CONTAINER_FIELDS_0_0_3['reopenLogAction']['handler'] = get_exec_action_handler(
    ['./awacslet', 'reopenlog', 'balancer'])

EMPTY_BALANCER_CONTAINER_FIELDS = {
    'command': [],
    'readinessProbe': {
        'handlers': [],
        'initialDelaySeconds': 5,
        'minPeriodSeconds': 5,
        'maxPeriodSeconds': 60,
        'periodBackoff': 2,
    },
    'restartPolicy': {
        'minPeriodSeconds': 1,
        'maxPeriodSeconds': 60,
        'periodBackoff': 2,
        'periodJitterSeconds': 20,
    },
    'lifecycle': {
        'preStop': EMPTY_HANDLER,
        'stopGracePeriodSeconds': 0,
        'terminationGracePeriodSeconds': 10,
        'termBarrier': 'IGNORE',
    },
    'reopenLogAction': {
        'handler': EMPTY_HANDLER,
    },
}

AWACSLET_PUSHCLIENT_CONTAINER_NAME = 'push-client'
AWACSLET_PUSHCLIENT_CONTAINER_FIELDS = dict(EMPTY_BALANCER_CONTAINER_FIELDS, **{
    'name': AWACSLET_PUSHCLIENT_CONTAINER_NAME,
    'command': ['./awacslet', 'start', 'push-client'],
    'reopenLogAction': {
        'handler': get_exec_action_handler(['./awacslet', 'reopenlog', 'push-client']),
    },
})


class AwacsletFiller(six.with_metaclass(abc.ABCMeta, object)):
    @staticmethod
    def find_controls_volume(volumes):
        for volume in volumes:
            if volume['name'] == CONTROLS_VOLUME_NAME:
                return volume

    @staticmethod
    def remove_controls_volume(volumes):
        for i, volume in enumerate(volumes):
            if volume['name'] == CONTROLS_VOLUME_NAME:
                volumes.pop(i)
                return True
        return False

    @staticmethod
    def find_container(containers, name):
        for container in containers:
            if container['name'] == name:
                return container

    @classmethod
    def clear(cls, instance_spec):
        prev_instance_spec = deepcopy(instance_spec)
        instance_spec['notifyAction'] = EMPTY_NOTIFY_ACTION
        instance_spec['initContainers'] = []

        cls.remove_controls_volume(instance_spec['volume'])
        instance_spec['containers'] = []

        return prev_instance_spec != instance_spec

    @classmethod
    @abc.abstractmethod
    def fill(cls, version, instance_spec):
        """
        :type version: semantic_version.Version
        :type instance_spec: dict
        """
        raise NotImplementedError


class AwacsletFiller_0_0_1(AwacsletFiller):
    @classmethod
    def fill(cls, version, instance_spec):
        """
        :type version: semantic_version.Version
        :type instance_spec: dict
        """
        prev_instance_spec = deepcopy(instance_spec)

        instance_spec['notifyAction'] = NOTIFY_ACTION
        instance_spec['initContainers'] = [INIT_CONTAINER]

        updated_volumes = deepcopy(instance_spec['volume'])

        controls_volume = cls.find_controls_volume(updated_volumes)
        if controls_volume is None:
            updated_volumes.append(deepcopy(CONTROLS_VOLUME))
        else:
            controls_volume.clear()
            controls_volume.update(deepcopy(CONTROLS_VOLUME))

        balancer_container = cls.find_container(instance_spec['containers'], AWACSLET_BALANCER_CONTAINER_NAME)
        if balancer_container is None:
            balancer_container = deepcopy(AWACSLET_BALANCER_CONTAINER_FIELDS)
        else:
            balancer_container.update(deepcopy(AWACSLET_BALANCER_CONTAINER_FIELDS))

        instance_spec['containers'] = [balancer_container]
        instance_spec['volume'] = updated_volumes
        return prev_instance_spec != instance_spec


class AwacsletFiller_0_0_3(AwacsletFiller):
    VERSION_0_0_6 = components.SemanticVersion.parse(u'0.0.6')

    @classmethod
    def fill(cls, version, instance_spec):
        """
        :type version: semantic_version.Version
        :type instance_spec: dict
        """
        prev_instance_spec = deepcopy(instance_spec)

        instance_spec['notifyAction'] = NOTIFY_ACTION
        instance_spec['initContainers'] = [INIT_CONTAINER]
        instance_spec['hostProvidedDaemons'] = []  # disable yasm agent https://st.yandex-team.ru/SWAT-6794
        if components.SemanticVersion(version) >= cls.VERSION_0_0_6:
            # https://st.yandex-team.ru/AWACS-942: [x] "Start skynet binded from host filesystem"
            instance_spec['hostProvidedDaemons'].append({
                'type': 'HOST_SKYNET',
            })

        updated_volumes = deepcopy(instance_spec['volume'])

        controls_volume = cls.find_controls_volume(updated_volumes)
        if controls_volume is None:
            updated_volumes.append(deepcopy(CONTROLS_VOLUME))
        else:
            controls_volume.clear()
            controls_volume.update(deepcopy(CONTROLS_VOLUME))

        updated_containers = []

        balancer_container = deepcopy(cls.find_container(instance_spec['containers'],
                                                         AWACSLET_BALANCER_CONTAINER_NAME))
        if balancer_container is None:
            balancer_container = deepcopy(AWACSLET_BALANCER_CONTAINER_FIELDS_0_0_3)
        else:
            balancer_container.update(deepcopy(AWACSLET_BALANCER_CONTAINER_FIELDS_0_0_3))
        updated_containers.append(balancer_container)

        if 'pushclient' in version.prerelease:
            # if version looks like 0.0.3-pushclient
            push_client_container = deepcopy(cls.find_container(instance_spec['containers'],
                                                                AWACSLET_PUSHCLIENT_CONTAINER_NAME))
            if push_client_container is None:
                push_client_container = deepcopy(AWACSLET_PUSHCLIENT_CONTAINER_FIELDS)
            else:
                push_client_container.update(deepcopy(AWACSLET_PUSHCLIENT_CONTAINER_FIELDS))
            updated_containers.append(push_client_container)

        instance_spec['containers'] = updated_containers
        instance_spec['volume'] = updated_volumes
        return prev_instance_spec != instance_spec


AWACSLET_FILLERS = {
    u'0.0.1': AwacsletFiller_0_0_1,
    u'0.0.3': AwacsletFiller_0_0_3,
}

awacslet_config = components.get_component_config(model_pb2.ComponentMeta.AWACSLET)
AWACSLET_FILLERS_BY_SORTED_VERSION = OrderedDict(
    sorted((awacslet_config.parse_version(v), f) for v, f in six.iteritems(AWACSLET_FILLERS))
)


def get_filler(version, fillers_by_sorted_version):
    """
    Find the filler for the closest version that is less than or equal to the requested version

    :type version: Version
    :type fillers_by_sorted_version: dict[Version, Type[AwacsletFiller]]
    :rtype: Type[AwacsletFiller]
    """
    assert fillers_by_sorted_version
    sorted_versions = tuple(fillers_by_sorted_version)
    if version in fillers_by_sorted_version:
        return fillers_by_sorted_version[version]
    v_index = min(max(bisect.bisect(sorted_versions, version) - 1, 0), len(sorted_versions) - 1)
    v = sorted_versions[v_index]
    return fillers_by_sorted_version[v]


def add_or_update(instance_spec, component_pb):
    """
    :type instance_spec: dict
    :type component_pb: model_pb2.Component
    """
    version = awacslet_config.parse_version(component_pb.meta.version)
    assert isinstance(version, components.SemanticVersion)
    filler = get_filler(version=version, fillers_by_sorted_version=AWACSLET_FILLERS_BY_SORTED_VERSION)
    rv = ComponentsDiff()
    if filler.fill(version.semver, instance_spec):
        rv.update(awacslet_config.type, component_pb.meta.version)
    return rv


def remove(instance_spec):
    """
    :type instance_spec: dict
    """
    rv = ComponentsDiff()
    if AwacsletFiller.clear(instance_spec):
        rv.remove(awacslet_config.type)
    return rv
