# coding: utf-8
from collections import namedtuple

import inject
import six

from awacs.lib.gutils import gevent_idle_iter
from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.model.cache import IAwacsCache, AwacsCache
from awacs.model.db import IMongoStorage, MongoStorage
from awacs.model.errors import NotFoundError
from awacs.model.zk import IZkStorage, ZkStorage
from awacs.model import objects
from awacs.wrappers.base import Holder, ValidationCtx
from infra.awacs.proto import model_pb2
from .errors import ConfigValidationError
from .generator import (
    get_yandex_config_pb,
    get_injected_full_upstream_ids,
    get_would_be_injected_full_backend_ids,
    get_would_be_injected_full_knob_ids,
    get_would_be_injected_full_cert_ids,
    get_would_be_injected_full_weight_section_ids,
    get_would_be_injected_full_internal_upstream_ids,
)
from .stateholder import BalancerStateHolder
from .vector import (
    BalancerVersion,
    UpstreamVersion,
    DomainVersion,
    BackendVersion,
    EndpointSetVersion,
    KnobVersion,
    CertVersion,
    find_revision_spec,
    maybe_find_revision_spec)


latest_versions_tuple = namedtuple('latest_versions_tuple', (
    'balancers',
    'upstreams',
    'domains',
    'backends',
    'endpoint_sets',
    'knobs',
    'certs',
    'weight_sections',
))

full_included_ids_tuple = namedtuple('full_included_ids_tuple', (
    'upstreams',
    'domains',
    'backends',
    'knobs',
    'certs',
    'weight_sections',
))

LARGE_VECTOR_INDIVIDUAL_OBJECTS_COUNT = 50
LARGE_VECTOR_TOTAL_OBJECTS_COUNT = 100


class InvalidBalancerConfig(Exception):
    def __init__(self, error):
        self.error = error
        super(InvalidBalancerConfig, self).__init__(error)


class NamespaceProxy(object):
    __slots__ = ('namespace_id',)

    cache = inject.attr(IAwacsCache)

    def __init__(self, namespace_id):
        self.namespace_id = namespace_id

    def get_latest_versions(self, curr_vector, is_cert_discoverable_func, is_backend_or_es_discoverable_func):
        """
        :param curr_vector:
        :type is_cert_discoverable_func: callable
        :type is_backend_or_es_discoverable_func: callable
        :rtype latest_versions_tuple
        """
        balancer_versions = {}
        for balancer_pb in self.cache.list_all_balancers(namespace_id=self.namespace_id):
            full_balancer_id = (balancer_pb.meta.namespace_id, balancer_pb.meta.id)
            balancer_versions[full_balancer_id] = BalancerVersion.from_pb(balancer_pb)

        upstream_versions = {}
        for upstream_pb in self.cache.list_all_upstreams(namespace_id=self.namespace_id):
            full_upstream_id = (upstream_pb.meta.namespace_id, upstream_pb.meta.id)
            upstream_versions[full_upstream_id] = UpstreamVersion.from_pb(upstream_pb)

        weight_section_versions = {}
        for weight_section_pb in objects.WeightSection.cache.list(namespace_id=self.namespace_id):
            full_weight_section_id = (weight_section_pb.meta.namespace_id, weight_section_pb.meta.id)
            weight_section_versions[full_weight_section_id] = objects.WeightSection.version.from_pb(weight_section_pb)

        domain_versions = {}
        for domain_pb in self.cache.list_all_domains(namespace_id=self.namespace_id):
            full_domain_id = (domain_pb.meta.namespace_id, domain_pb.meta.id)
            domain_versions[full_domain_id] = DomainVersion.from_pb(domain_pb)

        backend_versions = {}
        for backend_pb in self.cache.list_all_backends(namespace_id=self.namespace_id):
            full_backend_id = (backend_pb.meta.namespace_id, backend_pb.meta.id)
            backend_version = BackendVersion.from_pb(backend_pb)
            if (is_backend_or_es_discoverable_func is None or
                    is_backend_or_es_discoverable_func(backend_pb)):
                backend_versions[full_backend_id] = backend_version
            elif full_backend_id in curr_vector.backend_versions:
                backend_versions[full_backend_id] = curr_vector.backend_versions[full_backend_id]

        endpoint_set_versions = {}
        for endpoint_set_pb in self.cache.list_all_endpoint_sets(namespace_id=self.namespace_id):
            full_endpoint_set_id = (endpoint_set_pb.meta.namespace_id, endpoint_set_pb.meta.id)
            endpoint_set_version = EndpointSetVersion.from_pb(endpoint_set_pb)
            if (is_backend_or_es_discoverable_func is None or
                    is_backend_or_es_discoverable_func(endpoint_set_pb)):
                endpoint_set_versions[full_endpoint_set_id] = endpoint_set_version
            elif full_endpoint_set_id in curr_vector.endpoint_set_versions:
                endpoint_set_versions[full_endpoint_set_id] = curr_vector.endpoint_set_versions[full_endpoint_set_id]

        knob_versions = {}
        for knob_pb in self.cache.list_all_knobs(namespace_id=self.namespace_id):
            full_knob_id = (knob_pb.meta.namespace_id, knob_pb.meta.id)
            knob_versions[full_knob_id] = KnobVersion.from_pb(knob_pb)

        cert_versions = {}
        for cert_pb in self.cache.list_all_certs(namespace_id=self.namespace_id):
            full_cert_id = (cert_pb.meta.namespace_id, cert_pb.meta.id)
            cert_version = CertVersion.from_pb(cert_pb)
            if is_cert_discoverable_func is None or is_cert_discoverable_func(cert_pb):
                cert_versions[full_cert_id] = cert_version
            elif full_cert_id in curr_vector.cert_versions:
                cert_versions[full_cert_id] = curr_vector.cert_versions[full_cert_id]

        if curr_vector is not None:
            for full_backend_id, backend_version in six.iteritems(curr_vector.backend_versions):
                backend_versions[full_backend_id] = self.must_get_latest_backend_version(full_backend_id)

            for full_endpoint_set_id, endpoint_set_version in six.iteritems(curr_vector.endpoint_set_versions):
                endpoint_set_versions[full_endpoint_set_id] = self.must_get_latest_endpoint_set_version(
                    full_endpoint_set_id)

        return latest_versions_tuple(
            balancers=balancer_versions,
            upstreams=upstream_versions,
            domains=domain_versions,
            backends=backend_versions,
            endpoint_sets=endpoint_set_versions,
            knobs=knob_versions,
            certs=cert_versions,
            weight_sections=weight_section_versions,
        )

    def must_get_latest_balancer_version(self, full_balancer_id):
        """
        :type full_balancer_id: (six.text_type, six.text_type)
        :rtype: BalancerVersion
        """
        assert isinstance(full_balancer_id, tuple) and len(full_balancer_id) == 2  # by contract
        pb = self.cache.must_get_balancer(*full_balancer_id)
        return BalancerVersion.from_pb(pb)

    def must_get_latest_upstream_version(self, full_upstream_id):
        """
        :type full_upstream_id: (six.text_type, six.text_type)
        :rtype: UpstreamVersion
        """
        assert isinstance(full_upstream_id, tuple) and len(full_upstream_id) == 2  # by contract
        pb = self.cache.must_get_upstream(*full_upstream_id)
        return UpstreamVersion.from_pb(pb)

    def must_get_latest_weight_section_version(self, full_weight_section_id):
        """
        :type full_weight_section_id: (six.text_type, six.text_type)
        :rtype: objects.WeightSection.version
        """
        assert isinstance(full_weight_section_id, tuple) and len(full_weight_section_id) == 2  # by contract
        pb = objects.WeightSection.cache.must_get(*full_weight_section_id)
        return objects.WeightSection.version.from_pb(pb)

    def must_get_latest_domain_version(self, full_domain_id):
        """
        :type full_domain_id: (six.text_type, six.text_type)
        :rtype: DomainVersion
        """
        assert isinstance(full_domain_id, tuple) and len(full_domain_id) == 2  # by contract
        pb = self.cache.must_get_domain(*full_domain_id)
        return DomainVersion.from_pb(pb)

    def must_get_latest_backend_version(self, full_backend_id):
        """
        :type full_backend_id: (six.text_type, six.text_type)
        :rtype: BackendVersion
        """
        assert isinstance(full_backend_id, tuple) and len(full_backend_id) == 2  # by contract
        pb = self.cache.must_get_backend(*full_backend_id)
        return BackendVersion.from_pb(pb)

    def must_get_latest_endpoint_set_version(self, full_endpoint_set_id):
        """
        :type full_endpoint_set_id: (six.text_type, six.text_type)
        :rtype: EndpointSetVersion
        """
        assert isinstance(full_endpoint_set_id, tuple) and len(full_endpoint_set_id) == 2  # by contract
        pb = self.cache.must_get_endpoint_set(*full_endpoint_set_id)
        return EndpointSetVersion.from_pb(pb)

    def must_get_latest_knob_version(self, full_knob_id):
        """
        :type full_knob_id: (six.text_type, six.text_type)
        :rtype: BackendVersion
        """
        assert isinstance(full_knob_id, tuple) and len(full_knob_id) == 2  # by contract
        pb = self.cache.must_get_knob(*full_knob_id)
        return KnobVersion.from_pb(pb)

    def must_get_latest_cert_version(self, full_cert_id):
        """
        :type full_cert_id: (six.text_type, six.text_type)
        :rtype: BackendVersion
        """
        assert isinstance(full_cert_id, tuple) and len(full_cert_id) == 2  # by contract
        pb = self.cache.must_get_cert(*full_cert_id)
        return CertVersion.from_pb(pb)


class BalancerDiscoverer(object):
    zk = inject.attr(IZkStorage)  # type: ZkStorage
    cache = inject.attr(IAwacsCache)  # type: AwacsCache
    mongo_storage = inject.attr(IMongoStorage)  # type: MongoStorage

    def __init__(self, namespace_id, balancer_id, balancer_location):
        self._namespace_id = namespace_id
        self._balancer_id = balancer_id
        self._balancer_location = balancer_location.upper()

        self._last_seen_balancer_state_generation = None
        self._state_proxy = BalancerStateHolder(namespace_id, balancer_id)
        self._namespace = NamespaceProxy(namespace_id)

    @property
    def endpoint_set_latest_versions(self):
        return self._state_proxy._endpoint_set_latest_versions

    @property
    def endpoint_set_latest_valid_versions(self):
        return self._state_proxy._endpoint_set_latest_valid_versions

    @property
    def backend_latest_versions(self):
        return self._state_proxy._backend_latest_versions

    @property
    def backend_latest_valid_versions(self):
        return self._state_proxy._backend_latest_valid_versions

    def set_balancer_state_pb(self, balancer_state_pb, ctx):
        """
        :type balancer_state_pb: model_pb2.BalancerState
        :type ctx: context.OpCtx
        :rtype: bool
        """
        assert balancer_state_pb.namespace_id == self._namespace_id  # by contract
        assert balancer_state_pb.balancer_id == self._balancer_id  # by contract

        if (self._last_seen_balancer_state_generation is not None and
                self._last_seen_balancer_state_generation >= balancer_state_pb.generation):
            if self._last_seen_balancer_state_generation > balancer_state_pb.generation:
                ctx.log.warn('Received balancer state generation older than last seen: %d > %d',
                             self._last_seen_balancer_state_generation, balancer_state_pb.generation)
            return False

        self._last_seen_balancer_state_generation = balancer_state_pb.generation
        self._state_proxy.update(balancer_state_pb)
        return True

    def handle_balancer_state_update(self, balancer_state_pb, ctx):
        """
        :type balancer_state_pb: model_pb2.BalancerState
        :type ctx: context.OpCtx
        :rtype: bool
        """
        ctx = ctx.with_op(op_id='handle_balancer_state_update')
        if self.set_balancer_state_pb(balancer_state_pb, ctx):
            return self._process_balancer_state_update(ctx)

    def _balancer_version_to_spec(self, balancer_version):
        """
        :type balancer_version: BalancerVersion
        """
        assert balancer_version.balancer_id[0] == self._namespace_id
        return find_revision_spec(balancer_version)

    def _weight_section_versions_to_specs(self, weight_section_versions):
        """
        :type weight_section_versions: dict[(six.text_type, six.text_type), objects.WeightSection.version]
        """
        rv = {}
        for full_weight_section_id, weight_section_version in six.iteritems(weight_section_versions):
            assert full_weight_section_id[0] == self._namespace_id
            if weight_section_version.deleted:
                spec_pb = maybe_find_revision_spec(weight_section_version)
                if spec_pb is None:
                    continue
            else:
                spec_pb = find_revision_spec(weight_section_version)
            rv[full_weight_section_id] = spec_pb
        return rv

    def _upstream_versions_to_specs(self, upstream_versions):
        """
        :type upstream_versions: dict[(six.text_type, six.text_type), UpstreamVersion]
        """
        rv = {}
        for full_upstream_id, upstream_version in six.iteritems(upstream_versions):
            assert full_upstream_id[0] == self._namespace_id
            if upstream_version.deleted:
                spec_pb = maybe_find_revision_spec(upstream_version)
                if spec_pb is None:
                    continue
            else:
                spec_pb = find_revision_spec(upstream_version)
            rv[full_upstream_id] = spec_pb
        return rv

    def _domain_versions_to_specs(self, domain_versions):
        """
        :type domain_versions: dict[(six.text_type, six.text_type), DomainVersion]
        """
        rv = {}
        for full_domain_id, domain_version in six.iteritems(domain_versions):
            assert full_domain_id[0] == self._namespace_id
            if domain_version.deleted:
                spec_pb = maybe_find_revision_spec(domain_version)
                if spec_pb is None:
                    continue
            else:
                spec_pb = find_revision_spec(domain_version)
            rv[full_domain_id] = spec_pb
        return rv

    def _cert_versions_to_specs(self, cert_versions):
        """
        :type cert_versions: dict[(six.text_type, six.text_type), CertVersion]
        """
        rv = {}
        for full_cert_id, cert_version in six.iteritems(cert_versions):
            assert full_cert_id[0] == self._namespace_id
            if cert_version.deleted:
                spec_pb = maybe_find_revision_spec(cert_version)
                if spec_pb is None:
                    continue
            else:
                spec_pb = find_revision_spec(cert_version)
            rv[full_cert_id] = spec_pb
        return rv

    def _update_state(self,
                      versions_to_add=frozenset(),
                      full_upstream_ids_to_delete=frozenset(),
                      full_domain_ids_to_delete=frozenset(),
                      full_backend_ids_to_delete=frozenset(),
                      full_knob_ids_to_delete=frozenset(),
                      full_cert_ids_to_delete=frozenset(),
                      full_weight_section_ids_to_delete=frozenset(),
                      ):
        """
        :type versions_to_add: set[BalancerVersion | UpstreamVersion | DomainVersion | objects.WeightSection.version |
                                   BackendVersion | EndpointSetVersion | KnobVersion | CertVersion]
        :type full_upstream_ids_to_delete: set[(six.text_type, six.text_type)]
        :type full_domain_ids_to_delete: set[(six.text_type, six.text_type)]
        :type full_backend_ids_to_delete: set[(six.text_type, six.text_type)]
        :type full_knob_ids_to_delete: set[(six.text_type, six.text_type)]
        :type full_cert_ids_to_delete: set[(six.text_type, six.text_type)]
        :type full_weight_section_ids_to_delete: set[(six.text_type, six.text_type)]
        :rtype: bool
        """
        updated = False
        for balancer_state_pb in self.zk.update_balancer_state(self._namespace_id, self._balancer_id):
            h = L7BalancerStateHandler(balancer_state_pb)
            updated = False
            for version in versions_to_add:
                updated |= h.add_new_rev(version)
            for full_upstream_id in full_upstream_ids_to_delete:
                updated |= h.delete_upstream(full_upstream_id)
            for full_domain_id in full_domain_ids_to_delete:
                updated |= h.delete_domain(full_domain_id)
            for full_backend_id in full_backend_ids_to_delete:
                updated |= h.delete_backend(full_backend_id)
                updated |= h.delete_endpoint_set(full_backend_id)
            for full_knob_id in full_knob_ids_to_delete:
                updated |= h.delete_knob(full_knob_id)
            for full_cert_id in full_cert_ids_to_delete:
                updated |= h.delete_cert(full_cert_id)
            for full_weight_section_id in full_weight_section_ids_to_delete:
                updated |= h.delete_weight_section(full_weight_section_id)
            if not updated:
                break
        return updated

    def _get_included_object_ids(self, balancer_version, upstream_versions, domain_versions, cert_versions, ctx,
                                 weight_section_versions):
        """
        :type balancer_version: BalancerVersion
        :type upstream_versions: dict[(six.text_type, six.text_type), UpstreamVersion]
        :type domain_versions: dict[(six.text_type, six.text_type), DomainVersion]
        :type cert_versions: dict[(six.text_type, six.text_type), CertVersion]
        :param ctx
        :type weight_section_versions: dict[(six.text_type, six.text_type), objects.WeightSection.version]
        :rtype: full_included_ids_tuple
        :raises: InvalidBalancerConfig
        """
        balancer_spec_pb = self._balancer_version_to_spec(balancer_version)
        upstream_spec_pbs = self._upstream_versions_to_specs(upstream_versions)
        domain_spec_pbs = self._domain_versions_to_specs(domain_versions)
        cert_spec_pbs = self._cert_versions_to_specs(cert_versions)
        weight_section_spec_pbs = self._weight_section_versions_to_specs(weight_section_versions)

        available_full_upstream_ids = list(upstream_spec_pbs)
        try:
            balancer_config_pb = get_yandex_config_pb(balancer_spec_pb)
        except ConfigValidationError as e:
            raise InvalidBalancerConfig(e)

        validation_ctx = ValidationCtx.create_ctx_with_config_type_full(
            namespace_pb=self.cache.must_get_namespace(self._namespace_id),
            full_balancer_id=balancer_version.balancer_id,
            balancer_spec_pb=balancer_spec_pb,
            knob_spec_pbs={},
            knob_version_hints={},
            cert_spec_pbs=cert_spec_pbs,
            domain_spec_pbs=domain_spec_pbs,
            weight_section_spec_pbs=weight_section_spec_pbs,
            upstream_spec_pbs=upstream_spec_pbs,
        )
        balancer_config = Holder(balancer_config_pb)

        included_full_domain_ids = set()
        if balancer_config.is_l7_macro():
            if balancer_config.get_nested_module().includes_domains():
                included_full_domain_ids = set(domain_versions.keys())
            try:
                balancer_config.expand_immediate_contained_macro(ctx=validation_ctx)
            except Exception as e:
                ctx.log.exception('Failed to expand l7_macro: %s', e)

        included_full_upstream_ids = get_injected_full_upstream_ids(
            self._namespace_id, balancer_config, available_full_upstream_ids)
        included_full_backend_ids = get_would_be_injected_full_backend_ids(
            self._namespace_id, balancer_config)
        included_full_weight_section_ids = set()
        included_full_internal_upstream_ids = set()
        if validation_ctx.are_knobs_allowed():
            included_full_knob_ids = get_would_be_injected_full_knob_ids(
                self._namespace_id, balancer_config, ctx=validation_ctx)
        else:
            included_full_knob_ids = set()
        included_full_cert_ids = get_would_be_injected_full_cert_ids(
            self._namespace_id, balancer_config, ctx=validation_ctx)

        def process_upstream(full_upstream_id):
            if full_upstream_id in upstream_versions:
                upstream_spec_pb = upstream_spec_pbs[full_upstream_id]
                upstream_config = Holder(upstream_spec_pb.yandex_balancer.config)
                included_full_backend_ids.update(get_would_be_injected_full_backend_ids(
                    self._namespace_id, upstream_config))
                included_full_weight_section_ids.update(get_would_be_injected_full_weight_section_ids(
                    self._namespace_id, upstream_config
                ))
                included_full_internal_upstream_ids.update(get_would_be_injected_full_internal_upstream_ids(
                    self._namespace_id, upstream_config
                ))
                if validation_ctx.are_knobs_allowed():
                    included_full_knob_ids.update(get_would_be_injected_full_knob_ids(
                        self._namespace_id, upstream_config, ctx=validation_ctx))
                included_full_cert_ids.update(get_would_be_injected_full_cert_ids(
                    self._namespace_id, upstream_config, ctx=validation_ctx))

        for full_upstream_id in gevent_idle_iter(included_full_upstream_ids, 50):
            process_upstream(full_upstream_id)
        for full_upstream_id in gevent_idle_iter(included_full_internal_upstream_ids, 50):
            process_upstream(full_upstream_id)

        return full_included_ids_tuple(
            upstreams=included_full_upstream_ids | included_full_internal_upstream_ids,
            domains=included_full_domain_ids,
            backends=included_full_backend_ids,
            knobs=included_full_knob_ids,
            certs=included_full_cert_ids,
            weight_sections=included_full_weight_section_ids,
        )

    def is_cert_discoverable(self, cert_pb):
        """
        :type cert_pb: model_pb2.Certificate
        """
        if not cert_pb.meta.HasField('discoverability'):
            # for backward compatibility
            return True
        discoverability_pb = cert_pb.meta.discoverability
        per_location = {loc.upper(): cond_pb
                        for loc, cond_pb in six.iteritems(discoverability_pb.per_location.values)}

        is_discoverable = discoverability_pb.default.value
        if self._balancer_location in per_location:
            is_discoverable = per_location[self._balancer_location].value
        return is_discoverable

    def is_backend_or_es_discoverable(self, backend_or_es_pb):
        """
        :type backend_or_es_pb: model_pb2.Backend | model_pb2.EndpointSet
        """
        return not backend_or_es_pb.meta.is_system.value

    def process_namespace_update(self, ctx):
        """
        :returns: Whether the state has been updated
        :rtype: bool
        """
        ctx = ctx.with_op(op_id='process_namespace_update')
        ctx.log.info('Processing namespace change...')

        # See https://wiki.yandex-team.ru/awacs/development/rfcs/on-rolling-updates/
        # for details about is_XXX_discoverable logic.
        latest_versions = self._namespace.get_latest_versions(
            curr_vector=self._state_proxy.curr_vector,
            is_cert_discoverable_func=self.is_cert_discoverable,
            is_backend_or_es_discoverable_func=self.is_backend_or_es_discoverable)

        current_state_vector = self._state_proxy.curr_vector
        full_balancer_id = (self._namespace_id, self._balancer_id)
        if full_balancer_id not in latest_versions.balancers:
            ctx.log.info('Balancer id not in namespace balancer versions, skipping...')
            return False

        versions_to_add = set()

        namespace_balancer_version = latest_versions.balancers[full_balancer_id]
        if current_state_vector.balancer_version is None or namespace_balancer_version > current_state_vector.balancer_version:
            versions_to_add.add(namespace_balancer_version)
        elif namespace_balancer_version.version != current_state_vector.balancer_version.version:
            raise RuntimeError('namespace_balancer_version != current_state_vector.balancer_version ({} < {})'.format(
                namespace_balancer_version, current_state_vector.balancer_version))

        balancer_versions_to_try = set()  # type: set[BalancerVersion]
        if namespace_balancer_version is not None:
            balancer_versions_to_try.add(namespace_balancer_version)
        if current_state_vector.balancer_version is not None:
            balancer_versions_to_try.add(current_state_vector.balancer_version)
        valid_vector = self._state_proxy.valid_vector
        if valid_vector.balancer_version is not None:
            balancer_versions_to_try.add(valid_vector.balancer_version)

        all_included_full_domain_ids = set()
        all_included_full_upstream_ids = set()
        all_included_full_backend_ids = set()
        all_included_full_knob_ids = set()
        all_included_full_cert_ids = set()
        all_included_full_weight_section_ids = set()


        for balancer_version in sorted(balancer_versions_to_try, reverse=True):
            try:
                full_included_ids = self._get_included_object_ids(
                    balancer_version=balancer_version,
                    upstream_versions=latest_versions.upstreams,
                    domain_versions=latest_versions.domains,
                    cert_versions=latest_versions.certs,
                    weight_section_versions=latest_versions.weight_sections,
                    ctx=ctx,
                )
            except InvalidBalancerConfig as e:
                ctx.log.exception('InvalidBalancerConfig for current vector with balancer %s: %s',
                                  balancer_version, e.error.to_dict())
            else:
                all_included_full_upstream_ids.update(full_included_ids.upstreams)
                all_included_full_domain_ids.update(full_included_ids.domains)
                all_included_full_backend_ids.update(full_included_ids.backends)
                all_included_full_knob_ids.update(full_included_ids.knobs)
                all_included_full_cert_ids.update(full_included_ids.certs)
                all_included_full_weight_section_ids.update(full_included_ids.weight_sections)
        if (all_included_full_upstream_ids or
                all_included_full_domain_ids or
                all_included_full_backend_ids or
                all_included_full_knob_ids or
                all_included_full_weight_section_ids or
                all_included_full_cert_ids):
            existing_included_full_upstream_ids = all_included_full_upstream_ids & set(latest_versions.upstreams)
            for full_upstream_id in existing_included_full_upstream_ids:
                namespace_version = latest_versions.upstreams[full_upstream_id]
                if not namespace_version.deleted and full_upstream_id not in current_state_vector.upstream_versions:
                    versions_to_add.add(namespace_version)
                if (full_upstream_id in current_state_vector.upstream_versions and
                        namespace_version > current_state_vector.upstream_versions[full_upstream_id]):
                    versions_to_add.add(namespace_version)

            existing_included_full_domain_ids = all_included_full_domain_ids & set(latest_versions.domains)
            for full_domain_id in existing_included_full_domain_ids:
                namespace_version = latest_versions.domains[full_domain_id]
                if (not namespace_version.deleted
                        and not namespace_version.incomplete
                        and full_domain_id not in current_state_vector.domain_versions):
                    versions_to_add.add(namespace_version)
                if (full_domain_id in current_state_vector.domain_versions and
                        namespace_version > current_state_vector.domain_versions[full_domain_id]):
                    versions_to_add.add(namespace_version)

            for full_backend_id in all_included_full_backend_ids:
                try:
                    backend_pb = self.cache.must_get_backend(*full_backend_id)
                    namespace_version = BackendVersion.from_pb(backend_pb)
                except NotFoundError:
                    continue
                if not self.is_backend_or_es_discoverable(backend_pb):
                    continue
                if not namespace_version.deleted and full_backend_id not in current_state_vector.backend_versions:
                    versions_to_add.add(namespace_version)
                if (full_backend_id in current_state_vector.backend_versions and
                        namespace_version > current_state_vector.backend_versions[full_backend_id]):
                    versions_to_add.add(namespace_version)

            for full_endpoint_set_id in all_included_full_backend_ids:
                try:
                    # make sure endpoint set's backend still exists:
                    backend_namespace_version = self._namespace.must_get_latest_backend_version(full_endpoint_set_id)
                    # and only then retrieve current version:
                    endpoint_set_pb = self.cache.must_get_endpoint_set(*full_endpoint_set_id)
                    namespace_version = EndpointSetVersion.from_pb(endpoint_set_pb)
                except NotFoundError:
                    continue
                if not self.is_backend_or_es_discoverable(endpoint_set_pb):
                    continue
                if (not backend_namespace_version.deleted and
                        not namespace_version.deleted and
                        full_endpoint_set_id not in current_state_vector.endpoint_set_versions):
                    versions_to_add.add(namespace_version)
                if (full_endpoint_set_id in current_state_vector.endpoint_set_versions and
                        namespace_version > current_state_vector.endpoint_set_versions[full_endpoint_set_id]):
                    versions_to_add.add(namespace_version)

            for full_knob_id in all_included_full_knob_ids:
                try:
                    namespace_version = self._namespace.must_get_latest_knob_version(full_knob_id)
                except NotFoundError:
                    continue
                if not namespace_version.deleted and full_knob_id not in current_state_vector.knob_versions:
                    versions_to_add.add(namespace_version)
                if (full_knob_id in current_state_vector.knob_versions and
                        namespace_version > current_state_vector.knob_versions[full_knob_id]):
                    versions_to_add.add(namespace_version)

            for full_cert_id in all_included_full_cert_ids:
                try:
                    cert_pb = self.cache.must_get_cert(*full_cert_id)
                    namespace_version = CertVersion.from_pb(cert_pb)
                except NotFoundError:
                    continue
                if not self.is_cert_discoverable(cert_pb):
                    continue
                if (not namespace_version.deleted
                        and not namespace_version.incomplete
                        and full_cert_id not in current_state_vector.cert_versions):
                    versions_to_add.add(namespace_version)
                if (full_cert_id in current_state_vector.cert_versions and
                        namespace_version > current_state_vector.cert_versions[full_cert_id]):
                    versions_to_add.add(namespace_version)

            for full_weight_section_id in all_included_full_weight_section_ids:
                try:
                    namespace_version = self._namespace.must_get_latest_weight_section_version(full_weight_section_id)
                except NotFoundError:
                    continue
                if not namespace_version.deleted and full_weight_section_id not in current_state_vector.weight_section_versions:
                    versions_to_add.add(namespace_version)
                if (full_weight_section_id in current_state_vector.weight_section_versions and
                        namespace_version > current_state_vector.weight_section_versions[full_weight_section_id]):
                    versions_to_add.add(namespace_version)

        ctx.log.info('Examined balancer versions: %s', ', '.join([v.version for v in balancer_versions_to_try]))

        is_state_updated = False
        if versions_to_add:
            ctx.log.info('Versions to add: %s', versions_to_add)
            is_state_updated = self._update_state(versions_to_add=versions_to_add)
        return is_state_updated

    @staticmethod
    def _get_keys_w_statuses(m):
        rv = set()
        for k, v_pb in six.iteritems(m):
            if v_pb.statuses:
                rv.add(k)
        return rv

    def _process_balancer_state_update(self, ctx):
        """
        :returns: Whether the state has been updated
        :rtype: bool
        """
        ctx.log.info('Processing balancer state change...')

        active_vector = self._state_proxy.active_vector
        curr_vector = self._state_proxy.curr_vector
        valid_vector = self._state_proxy.valid_vector
        in_progress_vector = self._state_proxy.in_progress_vector

        full_domain_ids_to_delete_from_state = set()
        full_upstream_ids_to_delete_from_state = set()
        full_backend_ids_to_delete_from_state = set()
        full_knob_ids_to_delete_from_state = set()
        full_cert_ids_to_delete_from_state = set()
        full_weight_section_ids_to_delete_from_state = set()

        active_deleted_full_domain_ids = set()
        active_deleted_full_upstream_ids = set()
        active_deleted_full_backend_ids = set()
        active_deleted_full_knob_ids = set()
        active_deleted_full_cert_ids = set()
        active_deleted_full_weight_section_ids = set()

        included_domain_ids = set()
        included_upstream_ids = set()
        included_full_backend_ids = set()
        included_full_knob_ids = set()
        included_full_cert_ids = set()
        included_full_weight_section_ids = set()

        if active_vector.balancer_version:
            try:
                active_full_included_ids = self._get_included_object_ids(
                    balancer_version=active_vector.balancer_version,
                    upstream_versions=active_vector.upstream_versions,
                    domain_versions=active_vector.domain_versions,
                    cert_versions=active_vector.cert_versions,
                    weight_section_versions=active_vector.weight_section_versions,
                    ctx=ctx,
                )
            except InvalidBalancerConfig as e:
                ctx.log.exception('Unexpected InvalidBalancerConfig for active vector: %s', e)
            else:
                for active_full_upstream_id, upstream_version in six.iteritems(active_vector.upstream_versions):
                    if active_full_upstream_id in active_full_included_ids.upstreams:
                        if upstream_version.deleted:
                            active_deleted_full_upstream_ids.add(active_full_upstream_id)
                            full_upstream_ids_to_delete_from_state.add(active_full_upstream_id)
                        else:
                            included_upstream_ids.add(active_full_upstream_id)
                    else:
                        full_upstream_ids_to_delete_from_state.add(active_full_upstream_id)
                        if upstream_version.deleted:
                            active_deleted_full_upstream_ids.add(active_full_upstream_id)

                for active_full_domain_id, domain_version in six.iteritems(active_vector.domain_versions):
                    if active_full_domain_id in active_full_included_ids.domains:
                        if domain_version.deleted:
                            active_deleted_full_domain_ids.add(active_full_domain_id)
                            full_domain_ids_to_delete_from_state.add(active_full_domain_id)
                        else:
                            included_domain_ids.add(active_full_domain_id)
                    else:
                        full_domain_ids_to_delete_from_state.add(active_full_domain_id)
                        if domain_version.deleted:
                            active_deleted_full_domain_ids.add(active_full_domain_id)

                for active_full_backend_id, active_backend_version in six.iteritems(active_vector.backend_versions):
                    if active_full_backend_id in active_full_included_ids.backends:
                        if active_backend_version.deleted:
                            active_deleted_full_backend_ids.add(active_full_backend_id)
                            full_backend_ids_to_delete_from_state.add(active_full_backend_id)
                        else:
                            included_full_backend_ids.add(active_full_backend_id)
                    else:
                        full_backend_ids_to_delete_from_state.add(active_full_backend_id)
                        if active_backend_version.deleted:
                            active_deleted_full_backend_ids.add(active_full_backend_id)

                for active_full_knob_id, active_knob_version in six.iteritems(active_vector.knob_versions):
                    if active_full_knob_id in active_full_included_ids.knobs:
                        if active_knob_version.deleted:
                            active_deleted_full_knob_ids.add(active_full_knob_id)
                            full_knob_ids_to_delete_from_state.add(active_full_knob_id)
                        else:
                            included_full_knob_ids.add(active_full_knob_id)
                    else:
                        full_knob_ids_to_delete_from_state.add(active_full_knob_id)
                        if active_knob_version.deleted:
                            active_deleted_full_knob_ids.add(active_full_knob_id)

                for active_full_weight_section_id, active_weight_section_version in six.iteritems(active_vector.weight_section_versions):
                    if active_full_weight_section_id in active_full_included_ids.weight_sections:
                        if active_weight_section_version.deleted:
                            active_deleted_full_weight_section_ids.add(active_full_weight_section_id)
                            full_weight_section_ids_to_delete_from_state.add(active_full_weight_section_id)
                        else:
                            included_full_weight_section_ids.add(active_full_weight_section_id)
                    else:
                        full_weight_section_ids_to_delete_from_state.add(active_full_weight_section_id)
                        if active_weight_section_version.deleted:
                            active_deleted_full_weight_section_ids.add(active_full_weight_section_id)

                for active_full_cert_id, active_cert_version in six.iteritems(active_vector.cert_versions):
                    if active_full_cert_id in active_full_included_ids.certs:
                        if active_cert_version.deleted:
                            active_deleted_full_cert_ids.add(active_full_cert_id)
                            full_cert_ids_to_delete_from_state.add(active_full_cert_id)
                        else:
                            included_full_cert_ids.add(active_full_cert_id)
                    else:
                        full_cert_ids_to_delete_from_state.add(active_full_cert_id)
                        if active_cert_version.deleted:
                            active_deleted_full_cert_ids.add(active_full_cert_id)

            ctx.log.info('Processed active vector')
            ctx.log.info('Active deleted upstreams: %s', active_deleted_full_upstream_ids)
            ctx.log.info('Active deleted domains: %s', active_deleted_full_domain_ids)
            ctx.log.info('Active deleted backends: %s', active_deleted_full_backend_ids)
            ctx.log.info('Upstreams to delete from state: %s', full_upstream_ids_to_delete_from_state)
            ctx.log.info('Domains to delete from state: %s', full_domain_ids_to_delete_from_state)
            ctx.log.info('Backends to delete from state: %s', full_backend_ids_to_delete_from_state)
        else:
            ctx.log.info('Active vector has no balancer version, skipped')

        if valid_vector.balancer_version and valid_vector.greater_than(active_vector):
            ctx.log.info('Processing valid vector')
            try:
                valid_full_included_ids = self._get_included_object_ids(
                    balancer_version=valid_vector.balancer_version,
                    upstream_versions=valid_vector.upstream_versions,
                    domain_versions=valid_vector.domain_versions,
                    cert_versions=valid_vector.cert_versions,
                    weight_section_versions=valid_vector.weight_section_versions,
                    ctx=ctx,
                )
            except InvalidBalancerConfig as e:
                ctx.log.exception('Unexpected InvalidBalancerConfig for valid vector: %s', e)
            else:
                diff = active_vector.diff(valid_vector)
                for added_version in diff.added:
                    # added_version is present in valid vector but missing from active
                    if isinstance(added_version, UpstreamVersion):
                        full_upstream_id = added_version.upstream_id
                        if full_upstream_id in valid_full_included_ids.upstreams:
                            included_upstream_ids.add(full_upstream_id)
                        elif full_upstream_id not in included_upstream_ids:
                            full_upstream_ids_to_delete_from_state.add(full_upstream_id)
                    elif isinstance(added_version, DomainVersion):
                        full_domain_id = added_version.domain_id
                        if full_domain_id in valid_full_included_ids.domains:
                            included_domain_ids.add(full_domain_id)
                        elif full_domain_id not in included_domain_ids:
                            full_domain_ids_to_delete_from_state.add(full_domain_id)
                    elif isinstance(added_version, BackendVersion):
                        full_backend_id = added_version.backend_id
                        if full_backend_id in valid_full_included_ids.backends:
                            included_full_backend_ids.add(full_backend_id)
                        elif full_backend_id not in included_full_backend_ids:
                            full_backend_ids_to_delete_from_state.add(full_backend_id)
                    elif isinstance(added_version, KnobVersion):
                        full_knob_id = added_version.knob_id
                        if full_knob_id in valid_full_included_ids.knobs:
                            included_full_knob_ids.add(full_knob_id)
                        elif full_knob_id not in included_full_knob_ids:
                            full_knob_ids_to_delete_from_state.add(full_knob_id)
                    elif isinstance(added_version, objects.WeightSection.version):
                        full_weight_section_id = added_version.id
                        if full_weight_section_id in valid_full_included_ids.weight_sections:
                            included_full_weight_section_ids.add(full_weight_section_id)
                        elif full_weight_section_id not in included_full_weight_section_ids:
                            full_weight_section_ids_to_delete_from_state.add(full_weight_section_id)
                    elif isinstance(added_version, CertVersion):
                        full_cert_id = added_version.cert_id
                        if full_cert_id in valid_full_included_ids.certs:
                            included_full_cert_ids.add(full_cert_id)
                        elif full_cert_id not in included_full_cert_ids:
                            full_cert_ids_to_delete_from_state.add(full_cert_id)
                for _, to_version in diff.updated:
                    if isinstance(to_version, UpstreamVersion):
                        full_upstream_id = to_version.upstream_id
                        if full_upstream_id in valid_full_included_ids.upstreams:
                            included_upstream_ids.add(full_upstream_id)
                        elif full_upstream_id not in included_upstream_ids:
                            full_upstream_ids_to_delete_from_state.add(full_upstream_id)
                    elif isinstance(to_version, DomainVersion):
                        full_domain_id = to_version.domain_id
                        if full_domain_id in valid_full_included_ids.domains:
                            included_domain_ids.add(full_domain_id)
                        elif full_domain_id not in included_domain_ids:
                            full_domain_ids_to_delete_from_state.add(full_domain_id)
                    elif isinstance(to_version, BackendVersion):
                        full_backend_id = to_version.backend_id
                        if full_backend_id in valid_full_included_ids.backends:
                            included_full_backend_ids.add(full_backend_id)
                        elif full_backend_id not in included_full_backend_ids:
                            full_backend_ids_to_delete_from_state.add(full_backend_id)
                    elif isinstance(to_version, KnobVersion):
                        full_knob_id = to_version.knob_id
                        if full_knob_id in valid_full_included_ids.knobs:
                            included_full_knob_ids.add(full_knob_id)
                        elif full_knob_id not in included_full_knob_ids:
                            full_knob_ids_to_delete_from_state.add(full_knob_id)
                    elif isinstance(to_version, objects.WeightSection.version):
                        full_weight_section_id = to_version.id
                        if full_weight_section_id in valid_full_included_ids.weight_sections:
                            included_full_weight_section_ids.add(full_weight_section_id)
                        elif full_weight_section_id not in included_full_weight_section_ids:
                            full_weight_section_ids_to_delete_from_state.add(full_weight_section_id)
                    elif isinstance(to_version, CertVersion):
                        full_cert_id = to_version.cert_id
                        if full_cert_id in valid_full_included_ids.certs:
                            included_full_cert_ids.add(full_cert_id)
                        elif full_cert_id not in included_full_cert_ids:
                            full_cert_ids_to_delete_from_state.add(full_cert_id)
                for removed_version in diff.removed:
                    if isinstance(removed_version, UpstreamVersion):
                        full_upstream_id = removed_version.upstream_id
                        assert full_upstream_id not in valid_full_included_ids.upstreams
                        if full_upstream_id not in included_upstream_ids:
                            full_upstream_ids_to_delete_from_state.add(full_upstream_id)
                    elif isinstance(removed_version, DomainVersion):
                        full_domain_id = removed_version.domain_id
                        assert full_domain_id not in valid_full_included_ids.domains
                        if full_domain_id not in included_domain_ids:
                            full_domain_ids_to_delete_from_state.add(full_domain_id)
                    elif isinstance(removed_version, BackendVersion):
                        full_backend_id = removed_version.backend_id
                        assert full_backend_id not in valid_full_included_ids.backends
                        if full_backend_id not in included_full_backend_ids:
                            full_backend_ids_to_delete_from_state.add(full_backend_id)
                    elif isinstance(removed_version, KnobVersion):
                        full_knob_id = removed_version.knob_id
                        assert full_knob_id not in valid_full_included_ids.knobs
                        if full_knob_id not in included_full_knob_ids:
                            full_knob_ids_to_delete_from_state.add(full_knob_id)
                    elif isinstance(removed_version, objects.WeightSection.version):
                        full_weight_section_id = removed_version.id
                        assert full_weight_section_id not in valid_full_included_ids.weight_sections
                        if full_weight_section_id not in included_full_weight_section_ids:
                            full_weight_section_ids_to_delete_from_state.add(full_weight_section_id)
                    elif isinstance(removed_version, CertVersion):
                        full_cert_id = removed_version.cert_id
                        assert full_cert_id not in valid_full_included_ids.certs
                        if full_cert_id not in included_full_cert_ids:
                            full_cert_ids_to_delete_from_state.add(full_cert_id)
            ctx.log.info('Processed valid vector')
            ctx.log.info('Upstreams to delete from state: %s', full_upstream_ids_to_delete_from_state)
            ctx.log.info('Domains to delete from state: %s', full_domain_ids_to_delete_from_state)
            ctx.log.info('Backends to delete from state: %s', full_backend_ids_to_delete_from_state)
            ctx.log.info('Knobs to delete from state: %s', full_knob_ids_to_delete_from_state)
            ctx.log.info('Weight sections to delete from state: %s', full_weight_section_ids_to_delete_from_state)
            ctx.log.info('Certificates to delete from state: %s', full_cert_ids_to_delete_from_state)
        else:
            ctx.log.info('Valid vector has no balancer version or is older than active vector, skipped')

        if curr_vector.balancer_version and curr_vector.greater_than(valid_vector):
            ctx.log.info('Processing current vector')
            try:
                curr_full_included_ids = self._get_included_object_ids(
                    balancer_version=curr_vector.balancer_version,
                    upstream_versions=curr_vector.upstream_versions,
                    domain_versions=curr_vector.domain_versions,
                    cert_versions=curr_vector.cert_versions,
                    weight_section_versions=curr_vector.weight_section_versions,
                    ctx=ctx,
                )
            except InvalidBalancerConfig as e:
                ctx.log.exception('InvalidBalancerConfig for current vector: %s', e)
            else:
                diff = valid_vector.diff(curr_vector)
                for added_version in diff.added:
                    if isinstance(added_version, UpstreamVersion):
                        full_upstream_id = added_version.upstream_id
                        if full_upstream_id in curr_full_included_ids.upstreams:
                            included_upstream_ids.add(full_upstream_id)
                        elif full_upstream_id not in included_upstream_ids:
                            full_upstream_ids_to_delete_from_state.add(full_upstream_id)
                    elif isinstance(added_version, DomainVersion):
                        full_domain_id = added_version.domain_id
                        if full_domain_id in curr_full_included_ids.domains:
                            included_domain_ids.add(full_domain_id)
                        elif full_domain_id not in included_domain_ids:
                            full_domain_ids_to_delete_from_state.add(full_domain_id)
                    elif isinstance(added_version, BackendVersion):
                        full_backend_id = added_version.backend_id
                        if full_backend_id in curr_full_included_ids.backends:
                            included_full_backend_ids.add(full_backend_id)
                        elif full_backend_id not in included_full_backend_ids:
                            full_backend_ids_to_delete_from_state.add(full_backend_id)
                    elif isinstance(added_version, KnobVersion):
                        full_knob_id = added_version.knob_id
                        if full_knob_id in curr_full_included_ids.knobs:
                            included_full_knob_ids.add(full_knob_id)
                        elif full_knob_id not in included_full_knob_ids:
                            full_knob_ids_to_delete_from_state.add(full_knob_id)
                    elif isinstance(added_version, objects.WeightSection.version):
                        full_weight_section_id = added_version.id
                        if full_weight_section_id in curr_full_included_ids.weight_sections:
                            included_full_weight_section_ids.add(full_weight_section_id)
                        elif full_weight_section_id not in included_full_weight_section_ids:
                            full_weight_section_ids_to_delete_from_state.add(full_weight_section_id)
                    elif isinstance(added_version, CertVersion):
                        full_cert_id = added_version.cert_id
                        if full_cert_id in curr_full_included_ids.certs:
                            included_full_cert_ids.add(full_cert_id)
                        elif full_cert_id not in included_full_cert_ids:
                            full_cert_ids_to_delete_from_state.add(full_cert_id)
                for _, to_version in diff.updated:
                    if isinstance(to_version, UpstreamVersion):
                        full_upstream_id = to_version.upstream_id
                        if full_upstream_id in curr_full_included_ids.upstreams:
                            included_upstream_ids.add(full_upstream_id)
                        elif full_upstream_id not in included_upstream_ids:
                            full_upstream_ids_to_delete_from_state.add(full_upstream_id)
                    elif isinstance(to_version, DomainVersion):
                        full_domain_id = to_version.domain_id
                        if full_domain_id in curr_full_included_ids.domains:
                            included_domain_ids.add(full_domain_id)
                        elif full_domain_id not in included_domain_ids:
                            full_domain_ids_to_delete_from_state.add(full_domain_id)
                    elif isinstance(to_version, BackendVersion):
                        full_backend_id = to_version.backend_id
                        if full_backend_id in curr_full_included_ids.backends:
                            included_full_backend_ids.add(full_backend_id)
                        elif full_backend_id not in included_full_backend_ids:
                            full_backend_ids_to_delete_from_state.add(full_backend_id)
                    elif isinstance(to_version, KnobVersion):
                        full_knob_id = to_version.knob_id
                        if full_knob_id in curr_full_included_ids.knobs:
                            included_full_knob_ids.add(full_knob_id)
                        elif full_knob_id not in included_full_knob_ids:
                            full_knob_ids_to_delete_from_state.add(full_knob_id)
                    elif isinstance(to_version, objects.WeightSection.version):
                        full_weight_section_id = to_version.id
                        if full_weight_section_id in curr_full_included_ids.weight_sections:
                            included_full_weight_section_ids.add(full_weight_section_id)
                        elif full_weight_section_id not in included_full_weight_section_ids:
                            full_weight_section_ids_to_delete_from_state.add(full_weight_section_id)
                    elif isinstance(to_version, CertVersion):
                        full_cert_id = to_version.cert_id
                        if full_cert_id in curr_full_included_ids.certs:
                            included_full_cert_ids.add(full_cert_id)
                        elif full_cert_id not in included_full_cert_ids:
                            full_cert_ids_to_delete_from_state.add(full_cert_id)
                for removed_version in diff.removed:
                    if isinstance(removed_version, UpstreamVersion):
                        full_upstream_id = removed_version.upstream_id
                        assert full_upstream_id not in curr_full_included_ids.upstreams
                        if full_upstream_id not in included_upstream_ids:
                            full_upstream_ids_to_delete_from_state.add(full_upstream_id)
                    elif isinstance(removed_version, DomainVersion):
                        full_domain_id = removed_version.domain_id
                        assert full_domain_id not in curr_full_included_ids.domains
                        if full_domain_id not in included_domain_ids:
                            full_domain_ids_to_delete_from_state.add(full_domain_id)
                    elif isinstance(removed_version, BackendVersion):
                        full_backend_id = removed_version.backend_id
                        assert full_backend_id not in curr_full_included_ids.backends
                        if full_backend_id not in included_full_backend_ids:
                            full_backend_ids_to_delete_from_state.add(full_backend_id)
                    elif isinstance(removed_version, KnobVersion):
                        full_knob_id = removed_version.knob_id
                        assert full_knob_id not in included_full_knob_ids
                        if full_knob_id not in included_full_knob_ids:
                            full_knob_ids_to_delete_from_state.add(full_knob_id)
                    elif isinstance(removed_version, objects.WeightSection.version):
                        full_weight_section_id = removed_version.id
                        assert full_weight_section_id not in included_full_weight_section_ids
                        if full_weight_section_id not in included_full_weight_section_ids:
                            full_weight_section_ids_to_delete_from_state.add(full_weight_section_id)
                    elif isinstance(removed_version, CertVersion):
                        full_cert_id = removed_version.cert_id
                        assert full_cert_id not in included_full_cert_ids
                        if full_cert_id not in included_full_cert_ids:
                            full_cert_ids_to_delete_from_state.add(full_cert_id)
            ctx.log.info('Processed current vector')
        else:
            ctx.log.info('Current vector has no balancer version or is older than valid vector, skipped')

        vectors_to_spare = [in_progress_vector]
        if valid_vector != in_progress_vector and valid_vector != active_vector:
            # If valid vector is neither in progress nor activated, we don't want to touch it yet --
            # the transport might be processing it right now. Let's wait until it's activated.
            # See https://st.yandex-team.ru/AWACS-828#60b8d40fb2e6956bdbdf812f for details.
            vectors_to_spare.append(valid_vector)
        for vector_to_spare in vectors_to_spare:
            for full_upstream_id, upstream_version in six.iteritems(vector_to_spare.upstream_versions):
                if active_vector.upstream_versions.get(full_upstream_id) != upstream_version:
                    full_upstream_ids_to_delete_from_state.discard(full_upstream_id)
            for full_domain_id, domain_version in six.iteritems(vector_to_spare.domain_versions):
                if active_vector.domain_versions.get(full_domain_id) != domain_version:
                    full_domain_ids_to_delete_from_state.discard(full_domain_id)
            for full_backend_id, backend_version in six.iteritems(vector_to_spare.backend_versions):
                if active_vector.backend_versions.get(full_backend_id) != backend_version:
                    full_backend_ids_to_delete_from_state.discard(full_backend_id)
            for full_knob_id, knob_version in six.iteritems(vector_to_spare.knob_versions):
                if active_vector.knob_versions.get(full_knob_id) != knob_version:
                    full_knob_ids_to_delete_from_state.discard(full_knob_id)
            for full_weight_section_id, weight_section_version in six.iteritems(vector_to_spare.weight_section_versions):
                if active_vector.weight_section_versions.get(full_weight_section_id) != weight_section_version:
                    full_weight_section_ids_to_delete_from_state.discard(full_weight_section_id)
            for full_cert_id, cert_version in six.iteritems(vector_to_spare.cert_versions):
                if active_vector.cert_versions.get(full_cert_id) != cert_version:
                    full_cert_ids_to_delete_from_state.discard(full_cert_id)

        full_upstream_ids_to_delete_from_state -= included_upstream_ids
        full_domain_ids_to_delete_from_state -= included_domain_ids
        full_backend_ids_to_delete_from_state -= included_full_backend_ids
        full_knob_ids_to_delete_from_state -= included_full_knob_ids
        full_cert_ids_to_delete_from_state -= included_full_cert_ids
        full_weight_section_ids_to_delete_from_state -= included_full_weight_section_ids

        full_upstream_ids_to_delete_from_state.update(active_deleted_full_upstream_ids)
        full_domain_ids_to_delete_from_state.update(active_deleted_full_domain_ids)
        full_backend_ids_to_delete_from_state.update(active_deleted_full_backend_ids)
        full_knob_ids_to_delete_from_state.update(active_deleted_full_knob_ids)
        full_cert_ids_to_delete_from_state.update(active_deleted_full_cert_ids)
        full_weight_section_ids_to_delete_from_state.update(active_deleted_full_weight_section_ids)

        ctx.log.info('Processed all vectors')
        ctx.log.info('Final upstreams to delete from state: %s', full_upstream_ids_to_delete_from_state)
        ctx.log.info('Final domains to delete from state: %s', full_domain_ids_to_delete_from_state)
        ctx.log.info('Final backends to delete from state: %s', full_backend_ids_to_delete_from_state)
        ctx.log.info('Final knobs to delete from state: %s', full_knob_ids_to_delete_from_state)
        ctx.log.info('Final certificates to delete from state: %s', full_cert_ids_to_delete_from_state)
        ctx.log.info('Final weight sections to delete from state: %s', full_weight_section_ids_to_delete_from_state)
        ctx.log.info('Final active deleted domains: %s', active_deleted_full_domain_ids)
        ctx.log.info('Final active deleted upstreams: %s', active_deleted_full_upstream_ids)
        ctx.log.info('Final active deleted backends: %s', active_deleted_full_backend_ids)
        ctx.log.info('Final active deleted knobs: %s', active_deleted_full_knob_ids)
        ctx.log.info('Final active deleted certificates: %s', active_deleted_full_cert_ids)
        ctx.log.info('Final active deleted weight sections: %s', active_deleted_full_weight_section_ids)

        state_updated = False
        if (full_upstream_ids_to_delete_from_state or
                full_domain_ids_to_delete_from_state or
                full_backend_ids_to_delete_from_state or
                full_knob_ids_to_delete_from_state or
                full_weight_section_ids_to_delete_from_state or
                full_cert_ids_to_delete_from_state):
            state_updated = self._update_state(
                full_upstream_ids_to_delete=full_upstream_ids_to_delete_from_state,
                full_domain_ids_to_delete=full_domain_ids_to_delete_from_state,
                full_backend_ids_to_delete=full_backend_ids_to_delete_from_state,
                full_knob_ids_to_delete=full_knob_ids_to_delete_from_state,
                full_weight_section_ids_to_delete=full_weight_section_ids_to_delete_from_state,
                full_cert_ids_to_delete=full_cert_ids_to_delete_from_state)
        return state_updated
