# coding: utf-8
import functools
import logging
import random

import gevent
import gevent.pool
import inject
import six

from awacs import yamlparser, resolver
from awacs.lib import OrderedDict
from awacs.lib import ypclient, yp_service_discovery
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.strutils import flatten_full_id, flatten_full_id2, join_full_uids2
from awacs.lib.ypclient.factory import YpError
from awacs.model.balancer import errors
from awacs.model.util import clone_pb
from awacs.resolver import yp
from awacs.wrappers import l7macro
from awacs.wrappers.base import Chain, Holder, ValidationCtx, MacroBase
from awacs.wrappers.errors import (
    ValidationError, EndpointSetsDoNotExist, KnobDoesNotExist, CertDoesNotExist)
from awacs.wrappers.main import (Regexp, RegexpSection, L7UpstreamMacro,
                                 RegexpPath, RegexpPathSection,
                                 RegexpHost, RegexpHostSection,
                                 PrefixPathRouter, PrefixPathRouterSection,
                                 Ipdispatch, IpdispatchSection, InstanceMacro,
                                 AntirobotMacro, GeneratedProxyBackends, GeneratedProxyBackendsInstance, Balancer2)
from infra.awacs.proto import model_pb2, modules_pb2 as modules_proto, internals_pb2
from infra.swatlib import metrics
from sepelib.core import config as appconfig


METRICS_PATH = 'awacs', 'util'
registry = metrics.ROOT_REGISTRY.path(*METRICS_PATH)
s_y_t_ru_validate_l7_macro_hgram = registry.get_histogram('s-y-t-ru-validate-l7-macro')
s_y_t_ru_validate_hgram = registry.get_histogram('s-y-t-ru-validate')
s_y_t_ru_validate_shared_and_report_refs_hgram = registry.get_histogram('s-y-t-ru-validate-shared')
s_y_t_ru_expand_l7_macro_hgram = registry.get_histogram('s-y-t-ru-expand-l7-macro')
s_y_t_ru_expand_macro_hgram = registry.get_histogram('s-y-t-ru-expand-macro')


def inject_upstreams(namespace_id, balancer, upstreams, upstream_labels=None):
    """
    :type namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :type upstreams: dict[(six.text_type, six.text_type), awacs.wrappers.base.Holder]
    :type upstream_labels: dict[(six.text_type, six.text_type), dict[six.text_type, six.text_type]] | None
    """
    injected_upstream_ids = set()

    # branching module is a module with arbitrary number of sections,
    # e.g. regexp and ipdispatch modules, balancer2 module and so on.
    # we start with a chain from the balancer root module
    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=3):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            for branch in module.get_branches():
                chains.append(branch.walk_chain())

        assert module
        last_chain_module = module

        if last_chain_module.includes_upstreams():
            # here last_chain_module is branching and defines "include_upstreams" directive
            if isinstance(last_chain_module, Regexp):
                expected_root_module_cls = (RegexpSection, L7UpstreamMacro)
            elif isinstance(last_chain_module, RegexpPath):
                expected_root_module_cls = RegexpPathSection
            elif isinstance(last_chain_module, (Ipdispatch, InstanceMacro)):
                expected_root_module_cls = IpdispatchSection
            elif isinstance(last_chain_module, PrefixPathRouter):
                expected_root_module_cls = PrefixPathRouterSection
            elif isinstance(last_chain_module, RegexpHost):
                expected_root_module_cls = RegexpHostSection
            else:
                raise AssertionError()

            module_pb = last_chain_module.pb
            # we filter and order upstreams according to "include_upstreams" rules
            filtered_upstreams = last_chain_module.include_upstreams.filter_upstreams(
                namespace_id, upstreams, labels=upstream_labels)

            if isinstance(last_chain_module, Regexp):
                assert not module_pb.sections
                module_pb.sections.extend(module_pb.prepend_sections)
                module_pb.ClearField('prepend_sections')

            # note that upstreams is an ordered dict
            for full_upstream_id, upstream_holder in gevent_idle_iter(six.iteritems(filtered_upstreams), 100):
                assert full_upstream_id[0] == namespace_id
                upstream_id = full_upstream_id[1]
                if upstream_holder.is_empty():
                    raise errors.ConfigValidationError('can not attach upstream "{}" to balancer, '
                                                       'it has an empty config'.format(upstream_id))
                if upstream_holder.module:
                    root_module, root_chained_modules = upstream_holder.module, Chain()
                else:
                    # upstream holder can contain a chain, in that case we need to chop off its head
                    root_module, root_chained_modules = upstream_holder.chain.shift()
                if not isinstance(root_module, expected_root_module_cls):
                    if isinstance(expected_root_module_cls, type):
                        part = u'a {}'.format(expected_root_module_cls.__name__)
                    elif isinstance(expected_root_module_cls, (list, tuple)):
                        part = u'one of {}'.format(u', '.join(cls.__name__ for cls in expected_root_module_cls))
                    else:
                        # should never get here
                        part = u'an expected type'
                    raise errors.ConfigValidationError(u'can not attach upstream "{}" to balancer, '
                                                       u'it is not {}'.format(upstream_id, part))

                # if everything is good, add new key-value entry to the sections protobuf
                section_entry_pb = module_pb.sections.add()
                section_entry_pb.key = upstream_id
                if isinstance(root_module, L7UpstreamMacro):
                    regexp_section_pb = root_module.to_regexp_section_pb()
                    section_entry_pb.value.CopyFrom(regexp_section_pb)
                else:
                    # first attach the root module
                    section_entry_pb.value.CopyFrom(root_module.pb)
                    for chained_module in root_chained_modules.modules:
                        # if it has some chained modules, attach them as nested chain
                        section_entry_pb.value.nested.modules.append(chained_module.pb)

            module_pb.ClearField('include_upstreams')

            # update wrapper after we've changed protobuf structure
            last_chain_module.update_pb(module_pb)

            # update the set of injected upstream identifiers
            injected_upstream_ids.update(six.iterkeys(filtered_upstreams))

            gevent.idle()

    return injected_upstream_ids


def inject_backends_and_endpoint_sets(namespace_id, balancer, backend_spec_pbs, endpoint_set_spec_pbs,
                                      endpoint_set_revs_by_ids, backend_revs_by_ids,
                                      ipv6_only=True):
    """
    :type namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :type endpoint_set_revs_by_ids: dict[(six.text_type, six.text_type), EndpointSetVersion]
    :type backend_revs_by_ids: dict[(six.text_type, six.text_type), BackendVersion]
    :type backend_spec_pbs: dict[(six.text_type, six.text_type), awacs.proto.model_pb2.BackendSpec]
    :type endpoint_set_spec_pbs: dict[(six.text_type, six.text_type), awacs.proto.model_pb2.EndpointSetSpec]
    :type ipv6_only: bool
    """
    injected_backend_ids = set()
    injected_endpoint_set_ids = set()

    # branching module is a module with arbitrary number of sections,
    # e.g. regexp and ipdispatch modules, balancer2 module and so on.
    # we start with a chain from the balancer root module
    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=3):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            if module.includes_backends():
                assert isinstance(module, (GeneratedProxyBackends, AntirobotMacro))
                filtered_backend_spec_pbs, filtered_endpoint_set_spec_pbs = module.include_backends.filter(
                    namespace_id, backend_spec_pbs, endpoint_set_spec_pbs)
                for full_backend_id, backend_spec_pb in six.iteritems(filtered_backend_spec_pbs):
                    backend_namespace_id = full_backend_id[0]
                    if backend_namespace_id != namespace_id and not backend_spec_pb.is_global.value:
                        raise errors.ConfigValidationError(
                            'Backend "{}" is from another namespace and is not marked as global.'.format(
                                '/'.join(full_backend_id)),
                            cause=backend_revs_by_ids[full_backend_id])
                    if backend_spec_pb.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD:
                        for yp_endpoint_set_pb in backend_spec_pb.selector.yp_endpoint_sets:
                            module.pb.endpoint_sets.add(cluster=yp_endpoint_set_pb.cluster,
                                                        id=yp_endpoint_set_pb.endpoint_set_id)
                    else:
                        endpoint_set_spec_pb = filtered_endpoint_set_spec_pbs[full_backend_id]
                        if backend_namespace_id != namespace_id and not endpoint_set_spec_pb.is_global.value:
                            raise errors.ConfigValidationError(
                                'Endpoint set "{}" is from another namespace and is not marked as global.'.format(
                                    '/'.join(full_backend_id)),
                                cause=endpoint_set_revs_by_ids[full_backend_id])
                        for endpoint_set_instance_pb in endpoint_set_spec_pb.instances:
                            instance_pb = module.pb.instances.add()
                            instance_pb.host = endpoint_set_instance_pb.host
                            instance_pb.port = endpoint_set_instance_pb.port
                            instance_pb.cached_ip = resolver.get_resolved_ip(endpoint_set_instance_pb,
                                                                             ipv6_only=ipv6_only)
                            instance_pb.weight = endpoint_set_instance_pb.weight

                # remove "include_backends" directive from protobuf
                module.pb.ClearField('include_backends')

                # update wrapper after we've changed protobuf structure
                module.update_pb(module.pb)

                injected_backend_ids.update(filtered_backend_spec_pbs)
                injected_endpoint_set_ids.update(filtered_endpoint_set_spec_pbs)
                gevent.idle()

            if isinstance(module, Balancer2) and module.dynamic and not module.dynamic.pb.backends_name:
                assert module.generated_proxy_backends and module.generated_proxy_backends.include_backends
                backends_name = module.generated_proxy_backends.include_backends.get_dynamic_backends_name()
                module.dynamic.pb.backends_name = backends_name

            for branch in module.get_branches():
                chains.append(branch.walk_chain())
        assert module

    return injected_backend_ids, injected_endpoint_set_ids


def inject_knobs(namespace_id, balancer_id, balancer, knob_spec_pbs, ctx):
    """
    :type namespace_id: six.text_type
    :type balancer_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :type knob_spec_pbs: dict[(six.text_type, six.text_type), awacs.proto.model_pb2.KnobSpec]
    :type ctx: ValidationCtx
    """
    injected_knob_ids = set()
    if not ctx.are_knobs_enabled() and not ctx.are_knobs_allowed():
        return injected_knob_ids

    # branching module is a module with arbitrary number of sections,
    # e.g. regexp and ipdispatch modules, balancer2 module and so on.
    # we start with a chain from the balancer root module
    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=3):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        for module in chain:
            if module.includes_knobs(ctx=ctx):
                injected_knob_ids.update(
                    module.include_knobs(namespace_id, balancer_id, knob_spec_pbs, ctx=ctx))
                for composite_field in module.walk_composite_fields():
                    if composite_field.includes_knobs(ctx=ctx):
                        injected_knob_ids.update(
                            composite_field.include_knobs(namespace_id, balancer_id, knob_spec_pbs, ctx=ctx))
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
    return injected_knob_ids


def inject_certs(namespace_id, balancer, cert_spec_pbs, ctx):
    """
    :type namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :type cert_spec_pbs: dict[(six.text_type, six.text_type), awacs.proto.model_pb2.CertificateSpec]
    :type ctx: ValidationCtx
    """
    injected_cert_ids = set()
    if not cert_spec_pbs:
        return injected_cert_ids
    # branching module is a module with arbitrary number of sections,
    # e.g. regexp and ipdispatch modules, balancer2 module and so on.
    # we start with a chain from the balancer root module
    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=3):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        for module in chain:
            if module.includes_certs(ctx=ctx):
                injected_cert_ids.update(module.include_certs(namespace_id, cert_spec_pbs, ctx=ctx))
                for composite_field in module.walk_composite_fields():
                    if composite_field.includes_certs(ctx=ctx):
                        injected_cert_ids.update(composite_field.include_certs(namespace_id, cert_spec_pbs, ctx=ctx))
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
    return injected_cert_ids


if six.PY3:
    def cmp(x, y):
        if x < y:
            return -1
        elif x == y:
            return 0
        else:
            return 1


def instances_cmp(i_1, i_2):
    rv = cmp(i_1.host, i_2.host)
    if rv == 0:
        rv = cmp(i_1.port, i_2.port)
    if rv == 0:
        rv = cmp(i_1.weight, i_2.weight)
    if rv == 0:
        rv = cmp(i_1.ipv4_addr, i_2.ipv4_addr)
    if rv == 0:
        rv = cmp(i_1.ipv6_addr, i_2.ipv6_addr)
    return rv


def _shift_port(instance_pbs, offset):
    """
    :type instance_pbs: list[awacs.proto.internals_pb2.Instance]
    :type offset: int
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    rv = []
    for instance_pb in instance_pbs:
        updated_instance_pb = clone_pb(instance_pb)
        updated_instance_pb.port += offset
        rv.append(updated_instance_pb)
    return rv


def _override_port(instance_pbs, value):
    """
    :type instance_pbs: list[awacs.proto.internals_pb2.Instance]
    :type value: int
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    rv = []
    for instance_pb in instance_pbs:
        updated_instance_pb = clone_pb(instance_pb)
        updated_instance_pb.port = value
        rv.append(updated_instance_pb)
    return rv


def set_port(instance_pbs, port):
    """
    :type instance_pbs: list[awacs.proto.internals_pb2.Instance]
    :type port: awacs.proto.model_pb2.Port
    """
    if port.policy == port.KEEP:
        rv = instance_pbs
    elif port.policy == port.SHIFT:
        rv = _shift_port(instance_pbs, port.shift)
    elif port.policy == port.OVERRIDE:
        rv = _override_port(instance_pbs, port.override)
    else:
        raise RuntimeError('Unknown port policy: {}'.format(port.policy))
    return rv


def _override_weight(instance_pbs, value):
    """
    :type instance_pbs: list[awacs.proto.internals_pb2.Instance]
    :type value: float
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    rv = []
    for instance_pb in instance_pbs:
        updated_instance_pb = clone_pb(instance_pb)
        updated_instance_pb.weight = value
        rv.append(updated_instance_pb)
    return rv


def set_weight(instance_pbs, weight):
    """
    :type instance_pbs: list[awacs.proto.internals_pb2.Instance]
    :type weight: awacs.proto.model_pb2.Weight
    """
    if weight.policy == weight.KEEP:
        rv = instance_pbs
    elif weight.policy == weight.OVERRIDE:
        rv = _override_weight(instance_pbs, weight.override)
    else:
        raise RuntimeError('Unknown weight policy: {}'.format(weight.policy))
    return rv


def resolve_nanny_snapshot_pbs(snapshot_pbs, default_port=None, default_use_mtn=False):
    """
    :type snapshot_pbs: list[awacs.proto.modules_pb2.GeneratedProxyBackends.NannySnapshot |
                             awacs.proto.model_pb2.BackendSelector.NannySnapshot]
    :type default_port: awacs.proto.model_pb2.Port
    :type default_use_mtn: bool
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    nanny_client = inject.instance(resolver.INannyClient)
    instances_by_snapshots = []
    for snapshot_pb in snapshot_pbs:
        service_id = snapshot_pb.service_id
        snapshot_id = snapshot_pb.snapshot_id
        use_mtn = default_use_mtn
        if snapshot_pb.HasField('use_mtn'):
            use_mtn = snapshot_pb.use_mtn.value
        try:
            instances = nanny_client.list_nanny_snapshot_instances(service_id, snapshot_id, use_mtn=use_mtn)
        except resolver.NannyClientError as e:
            raise resolver.ResolvingError(
                'Failed to resolve snapshot {}:{} (use_mtn: {}): {}'.format(service_id, snapshot_id, use_mtn, e))
        else:
            if isinstance(snapshot_pb, model_pb2.BackendSelector.NannySnapshot):
                if snapshot_pb.port.policy != snapshot_pb.port.KEEP:
                    port = snapshot_pb.port
                else:
                    port = default_port
                if port:
                    instances = set_port(instances, port)
                if snapshot_pb.weight.policy != snapshot_pb.weight.KEEP:
                    instances = set_weight(instances, snapshot_pb.weight)
            elif isinstance(snapshot_pb, modules_proto.GeneratedProxyBackends.NannySnapshot):
                if snapshot_pb.HasField('port'):
                    instances = _override_port(instances, snapshot_pb.port.value)
                elif default_port:
                    instances = set_port(instances, default_port)
            else:
                raise AssertionError('Unexpected `snapshot_pb` type: {!r}'.format(type(snapshot_pb)))
            instances_by_snapshots.append(((service_id, snapshot_id), instances))
    merged_instances = resolver.merge_nanny_snapshot_instances(instances_by_snapshots)
    merged_instances.sort(key=functools.cmp_to_key(instances_cmp))
    return merged_instances


def resolve_nanny_snapshots(snapshots):
    """
    :type snapshots: list[awacs.wrappers.main.GeneratedProxyBackendsNannySnapshot]
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    return resolve_nanny_snapshot_pbs([s.pb for s in snapshots])


def resolve_gencfg_group_pbs(group_pbs, default_port=None, default_use_mtn=False):
    """
    :type group_pbs: list[awacs.proto.modules_pb2.GeneratedProxyBackends.GencfgGroup |
                          awacs.proto.model_pb2.BackendSelector.GencfgGroup]
    :type default_port: awacs.proto.model_pb2.Port
    :type default_use_mtn: bool
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    gencfg_client = inject.instance(resolver.IGencfgClient)
    instances_by_groups = []
    for group_pb in group_pbs:
        name = group_pb.name
        version = group_pb.version
        use_mtn = default_use_mtn
        if group_pb.HasField('use_mtn'):
            use_mtn = group_pb.use_mtn.value
        try:
            instances = gencfg_client.list_group_instances(name, version, use_mtn=use_mtn)
        except resolver.GencfgClientError as e:
            raise resolver.ResolvingError(
                'Failed to resolve gencfg group {}:{} (use_mtn: {}): {}'.format(name, version, use_mtn, e))
        else:
            if isinstance(group_pb, model_pb2.BackendSelector.GencfgGroup):
                if group_pb.port.policy != group_pb.port.KEEP:
                    port = group_pb.port
                else:
                    port = default_port
                if port:
                    instances = set_port(instances, port)
                if group_pb.weight.policy != group_pb.weight.KEEP:
                    instances = set_weight(instances, group_pb.weight)
            elif isinstance(group_pb, modules_proto.GeneratedProxyBackends.GencfgGroup):
                if group_pb.HasField('port'):
                    instances = _override_port(instances, group_pb.port.value)
                elif default_port:
                    instances = set_port(instances, default_port)
            else:
                raise AssertionError('Unexpected `group_pb` type: {!r}'.format(type(group_pb)))
            instances_by_groups.append(((name, version), instances))
    merged_instances = resolver.merge_gencfg_group_instances(instances_by_groups)
    merged_instances.sort(key=functools.cmp_to_key(instances_cmp))
    return merged_instances


def resolve_gencfg_groups(groups):
    """
    :type groups: list[awacs.wrappers.main.GeneratedProxyBackendsGencfgGroup]
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    return resolve_gencfg_group_pbs([group.pb for group in groups])


def resolve_yp_endpoint_set_pb(yp_client_factory, yp_endpoint_set_pb, default_port=None, reject_empty=False):
    """
    :type yp_client_factory: awacs.lib.ypclient.YpObjectServiceClientFactory
    :type yp_endpoint_set_pb: model_pb2.BackendSelector.YpEndpointSet
    :type default_port: awacs.proto.model_pb2.Port | None
    :type reject_empty: bool
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    cluster = yp_endpoint_set_pb.cluster
    yp_endpoint_set_id = yp_endpoint_set_pb.endpoint_set_id
    try:
        stub = yp_client_factory.get(cluster)
    except YpError:
        raise resolver.YpEndpointSetDoesNotExistError('Unknown YP cluster "{}"'.format(cluster))

    if not yp.does_endpoint_set_exist(stub, yp_endpoint_set_id):
        raise resolver.YpEndpointSetDoesNotExistError(
            u'YP endpoint set {}:{} does not exist'.format(cluster, yp_endpoint_set_id))
    try:
        instance_pbs = yp.list_endpoint_set_instances(stub, yp_endpoint_set_id)
    except Exception as e:
        raise resolver.ResolvingError(
            u'Failed to resolve YP endpoint set {}:{}: {}'.format(cluster, yp_endpoint_set_id, e))
    else:
        if yp_endpoint_set_pb.port.policy != yp_endpoint_set_pb.port.KEEP:
            port = yp_endpoint_set_pb.port
        else:
            port = default_port
        if port:
            instance_pbs = set_port(instance_pbs, port)
        if yp_endpoint_set_pb.weight.policy != yp_endpoint_set_pb.weight.KEEP:
            instance_pbs = set_weight(instance_pbs, yp_endpoint_set_pb.weight)

        if not instance_pbs and reject_empty:
            raise resolver.YpEndpointSetIsEmptyError(
                '{}:{} resolved to an empty list of endpoints'.format(cluster, yp_endpoint_set_id))

        return instance_pbs


class SdYpEndpointSetResolution(object):
    def __init__(self, yp_sd_timestamp, instance_pbs):
        """
        :type yp_sd_timestamp: int
        :type instance_pbs: list[internals_pb2.Instance]
        """
        self.yp_sd_timestamp = yp_sd_timestamp
        self.instance_pbs = instance_pbs


def resolve_yp_endpoint_set_pb_using_sd(yp_endpoint_set_pb, prev_yp_sd_timestamp,
                                        default_port=None,
                                        treat_not_exists_as_empty=False,
                                        reject_empty=False):
    """
    :type yp_endpoint_set_pb: model_pb2.BackendSelector.YpEndpointSet
    :type prev_yp_sd_timestamp: int
    :type default_port: awacs.proto.model_pb2.Port | None
    :type treat_not_exists_as_empty: bool
    :type reject_empty: bool
    :rtype: SdYpEndpointSetResolution
    """
    cluster = yp_endpoint_set_pb.cluster
    yp_endpoint_set_id = yp_endpoint_set_pb.endpoint_set_id

    req_id = yp_service_discovery.sd_resolver.generate_reqid()
    req_pb = internals_pb2.TReqResolveEndpoints(cluster_name=cluster, endpoint_set_id=yp_endpoint_set_id)
    try:
        resp_pb, instance_pbs = yp.list_endpoint_set_instances_sd(
            resolver=yp_service_discovery.IResolver.instance(),
            req_pb=req_pb,
            req_id=req_id)
    except Exception as e:
        message = u'Failed to resolve YP endpoint set {}:{}: {} (req id: {})'.format(
            cluster, yp_endpoint_set_id, e, req_id)
        status_message = u'Failed to resolve YP endpoint set {}:{}: {}'.format(
            cluster, yp_endpoint_set_id, e)
        raise resolver.ResolvingError(message, status_message=status_message)

    yp_sd_timestamp = resp_pb.timestamp
    if yp_sd_timestamp <= 0:
        message = u'Failed to resolve YP endpoint set {}:{}: timestamp is either missing or <= 0 (req id: {})'.format(
            cluster, yp_endpoint_set_id, req_id)
        status_message = u'Failed to resolve YP endpoint set {}:{}: timestamp is either missing or <= 0'.format(
            cluster, yp_endpoint_set_id)
        raise resolver.ResolvingError(message, status_message=status_message)
    else:
        if yp_sd_timestamp < prev_yp_sd_timestamp:
            if cluster == 'man' and random.uniform(0, 1) < float(appconfig.get_value('run.man_obsolete_sd_response_acceptance_prob', default=0)):
                pass
            else:
                message = (u'Received an obsolete response for YP endpoint set {}:{} '
                           u'(req id: {}, current ts: {}, received ts: {})').format(
                    cluster, yp_endpoint_set_id, req_id, prev_yp_sd_timestamp, yp_sd_timestamp)
                status_message = u'Received an obsolete response for YP endpoint set {}:{}'.format(
                    cluster, yp_endpoint_set_id)
                raise resolver.YpSdObsoleteResponseError(message, status_message=status_message)

    if resp_pb.resolve_status == internals_pb2.NOT_EXISTS:
        if treat_not_exists_as_empty:
            instance_pbs = []
        else:
            message = u'YP endpoint set {}:{} does not exist (req id: {})'.format(cluster, yp_endpoint_set_id, req_id)
            status_message = u'YP endpoint set {}:{} does not exist'.format(cluster, yp_endpoint_set_id)
            raise resolver.YpEndpointSetDoesNotExistError(message, status_message=status_message)

    if yp_endpoint_set_pb.port.policy != yp_endpoint_set_pb.port.KEEP:
        port = yp_endpoint_set_pb.port
    else:
        port = default_port
    if port:
        instance_pbs = set_port(instance_pbs, port)
    if yp_endpoint_set_pb.weight.policy != yp_endpoint_set_pb.weight.KEEP:
        instance_pbs = set_weight(instance_pbs, yp_endpoint_set_pb.weight)

    if not instance_pbs and reject_empty:
        message = u'{}:{} resolved to an empty list of endpoints (req id: {})'.format(
            cluster, yp_endpoint_set_id, req_id)
        status_message = u'{}:{} resolved to an empty list of endpoints'.format(cluster, yp_endpoint_set_id)
        raise resolver.YpEndpointSetIsEmptyError(message, status_message=status_message)

    return SdYpEndpointSetResolution(yp_sd_timestamp=yp_sd_timestamp,
                                     instance_pbs=instance_pbs)


def resolve_yp_endpoint_set_pbs(yp_endpoint_set_pbs, default_port=None, reject_empty=False):
    """
    :type yp_endpoint_set_pbs: list[model_pb2.BackendSelector.YpEndpointSet]
    :type default_port: awacs.proto.model_pb2.Port | None
    :type reject_empty: bool
    :rtype: list[awacs.proto.internals_pb2.Instance]
    """
    yp_client_factory = inject.instance(ypclient.IYpObjectServiceClientFactory)
    rv = []
    for yp_endpoint_set_pb in yp_endpoint_set_pbs:
        instances = resolve_yp_endpoint_set_pb(yp_client_factory, yp_endpoint_set_pb,
                                               default_port=default_port,
                                               reject_empty=reject_empty)
        rv.extend(instances)
    rv.sort(key=functools.cmp_to_key(instances_cmp))
    return rv


class Resolution(object):
    __slots__ = ('instance_pbs', 'yp_sd_timestamps')

    def __init__(self, instance_pbs, yp_sd_timestamps):
        """
        :type instance_pbs: list[internals_pb2.Instance]
        :type yp_sd_timestamps: dict[tuple[six.text_type, six.text_type], int]
        """
        self.instance_pbs = instance_pbs
        self.yp_sd_timestamps = yp_sd_timestamps


def resolve_yp_endpoint_set_pbs_using_sd(yp_endpoint_set_pbs, prev_yp_sd_timestamps,
                                         default_port=None,
                                         treat_not_exists_as_empty=False,  # https://st.yandex-team.ru/YP-2093
                                         reject_empty=False):
    """
    :type yp_endpoint_set_pbs: list[model_pb2.BackendSelector.YpEndpointSet]
    :type prev_yp_sd_timestamps: dict[tuple[six.text_type, six.text_type], int]
    :type default_port: awacs.proto.model_pb2.Port | None
    :type treat_not_exists_as_empty: bool
    :type reject_empty: bool
    :rtype: Resolution
    """
    instance_pbs = []
    yp_sd_timestamps = {}
    for yp_endpoint_set_pb in yp_endpoint_set_pbs:
        full_id = (yp_endpoint_set_pb.cluster, yp_endpoint_set_pb.endpoint_set_id)
        prev_yp_sd_timestamp = prev_yp_sd_timestamps.get(full_id, -1)
        resolution = resolve_yp_endpoint_set_pb_using_sd(
            yp_endpoint_set_pb,
            default_port=default_port,
            treat_not_exists_as_empty=treat_not_exists_as_empty,
            reject_empty=reject_empty,
            prev_yp_sd_timestamp=prev_yp_sd_timestamp)
        instance_pbs.extend(resolution.instance_pbs)
        assert full_id not in yp_sd_timestamps
        yp_sd_timestamps[full_id] = resolution.yp_sd_timestamp
    instance_pbs.sort(key=functools.cmp_to_key(instances_cmp))
    return Resolution(
        instance_pbs=instance_pbs,
        yp_sd_timestamps=yp_sd_timestamps
    )


def resolve_generated_proxy_backend_instances(backends, ipv6_only=True):
    """
    :type backends: awacs.wrappers.main.GeneratedProxyBackends | awacs.wrappers.main.AntirobotMacro
    :type ipv6_only: bool
    """
    if backends.includes_backends():
        return

    if backends.instances:
        for instance in backends.instances:
            instance_pb = instance.pb
            if not instance_pb.cached_ip:
                instance_pb.cached_ip = resolver.resolve_host(instance_pb.host, ipv6_only=ipv6_only)
                # instance.update_pb(instance_pb) not necessary -- nothing to update
    else:
        if backends.nanny_snapshots:
            instances = resolve_nanny_snapshots(backends.nanny_snapshots)
        elif backends.gencfg_groups:
            instances = resolve_gencfg_groups(backends.gencfg_groups)
        else:
            # For needs of "Antirobot Macro" due to possible absence of backends
            return
        backends_pb = backends.pb
        for instance in instances:
            instance_pb = backends_pb.instances.add()
            instance_pb.host = instance.host
            instance_pb.port = instance.port
            instance_pb.cached_ip = resolver.get_resolved_ip(instance, ipv6_only=ipv6_only)
            instance_pb.weight = instance.weight
            backends.instances.append(GeneratedProxyBackendsInstance(instance_pb))
        del backends_pb.nanny_snapshots[:]
        del backends.nanny_snapshots[:]
        del backends_pb.gencfg_groups[:]
        del backends.gencfg_groups[:]
        # backends.update_pb(backends_pb) not necessary -- nothing to update


def find_and_resolve_all_upstream_instances(module):
    """
    :type module: awacs.wrappers.base.Holder or awacs.wrappers.base.ModuleWrapperBase or awacs.wrappers.base.Chain
    """
    chains = [module.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=5):
        for module in chain:
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
            if isinstance(module, (GeneratedProxyBackends, AntirobotMacro)):
                resolve_generated_proxy_backend_instances(module)
                gevent.idle()


def map_backend_to_endpoint_set(backend_spec_pb):
    """
    Used in awacsctl2 compile.

    :type backend_spec_pb: awacs.proto.model_pb2.BackendSpec
    :rtype: awacs.proto.model_pb2.EndpointSetSpec
    """
    selector_pb = backend_spec_pb.selector
    instances = list()
    instances.extend(resolve_nanny_snapshot_pbs(selector_pb.nanny_snapshots))
    instances.extend(resolve_gencfg_group_pbs(selector_pb.gencfg_groups))
    instances.extend(resolve_yp_endpoint_set_pbs(selector_pb.yp_endpoint_sets))

    spec_pb = model_pb2.EndpointSetSpec()
    spec_pb.deleted = backend_spec_pb.deleted
    for instance_pb in instances:
        spec_pb.instances.add(
            host=instance_pb.host,
            port=instance_pb.port,
            ipv4_addr=instance_pb.ipv4_addr,
            ipv6_addr=instance_pb.ipv6_addr,
            weight=instance_pb.weight,
        )
    return spec_pb


def endpoint_set_as_list(es_spec_pb):
    result = list()
    for instance in es_spec_pb.instances:
        result.append(OrderedDict([
            ('host', instance.host),
            ('port', instance.port),
            ('weight', instance.weight),
            ('cached_ip', instance.ipv6_addr or instance.ipv4_addr),
        ]))
    return result


def generate_backends_json(backend_map, self_namespace_id=None):
    """
    Used in awacsctl2 compile.

    :type backend_map: dict[EndpointSetVersion, awacs.proto.model_pb2.EndpointSetSpec]
    :type self_namespace_id: six.text_type
    :rtype: dict
    """
    result = list()
    for es_version, endpoint_set in six.iteritems(backend_map):
        namespace_id, backend_id = es_version.endpoint_set_id
        if namespace_id == self_namespace_id:
            es_id = backend_id
        else:
            es_id = '{}/{}'.format(namespace_id, backend_id)
        result.append((es_id, endpoint_set_as_list(endpoint_set))),
    result.sort()
    return OrderedDict(result)


def get_yandex_config_pb(balancer_spec_pb):
    """
    :type balancer_spec_pb: awacs.proto.model_pb2.BalancerSpec
    :rtype: awacs.proto.modules_pb2.Holder
    """
    yandex_balancer_spec_pb = balancer_spec_pb.yandex_balancer
    if yandex_balancer_spec_pb.HasField('config'):
        balancer_pb = clone_pb(yandex_balancer_spec_pb.config)
    elif yandex_balancer_spec_pb.yaml:
        if yandex_balancer_spec_pb.template_engine == model_pb2.NONE:
            try:
                balancer_pb = yamlparser.parse(modules_proto.Holder, yandex_balancer_spec_pb.yaml)
            except yamlparser.Error as e:
                raise errors.ConfigValidationError(message='Failed to read balancer config', error=e)
        elif yandex_balancer_spec_pb.template_engine == model_pb2.DYNAMIC_JINJA2:
            raise errors.ConfigValidationError('Template engine DYNAMIC_JINJA2 is not supported anymore')
        else:
            template_engine_name = model_pb2.TemplateEngineType.Name(yandex_balancer_spec_pb.template_engine)
            raise errors.ConfigValidationError('Template engine {} is not supported yet'.format(template_engine_name))
    else:
        raise RuntimeError('Balancer spec has neither "config" nor "yaml" field specified')  # must never happen
    return balancer_pb


def validate_internal_balancer_compatibility(balancer_version, balancer_spec_pb, upstream_spec_pbs, domain_spec_pbs):
    """
    :type balancer_version: BalancerVersion
    :type balancer_spec_pb: awacs.proto.model_pb2.BalancerSpec
    :type upstream_spec_pbs: dict[UpstreamVersion, awacs.proto.model_pb2.UpstreamSpec]
    :type domain_spec_pbs: dict[DomainVersion, awacs.proto.model_pb2.DomainSpec]
    """
    expected_type = model_pb2.YANDEX_BALANCER

    if balancer_spec_pb.type != expected_type:
        type_str = model_pb2.BalancerType.Name(balancer_spec_pb.type)
        raise errors.ConfigValidationError('Balancer\'s balancer type "{}" is not supported yet'.format(type_str),
                                           cause=balancer_version)

    for upstream_version, upstream_spec_pb in six.iteritems(upstream_spec_pbs):
        if upstream_spec_pb.type != expected_type:
            type_str = model_pb2.BalancerType.Name(upstream_spec_pb.type)
            raise errors.ConfigValidationError(
                'Balancer type of upstream "{}" "{}" is not supported yet'.format(upstream_version.upstream_id,
                                                                                  type_str),
                cause=upstream_version)
        if upstream_spec_pb.type != balancer_spec_pb.type:
            raise errors.ConfigValidationError(
                'Balancer type of upstream "{}" does not '
                'match a balancer type of its balancer'.format(upstream_version.upstream_id),
                cause=upstream_version)
    for domain_version, domain_spec_pb in six.iteritems(domain_spec_pbs):
        if domain_spec_pb.type != expected_type:
            type_str = model_pb2.BalancerType.Name(domain_spec_pb.type)
            raise errors.ConfigValidationError(
                'Balancer type of domain "{}" "{}" is not supported yet'.format(domain_version.domain_id,
                                                                                type_str),
                cause=domain_version)
        if domain_spec_pb.type != balancer_spec_pb.type:
            raise errors.ConfigValidationError(
                'Balancer type of domain "{}" does not '
                'match a balancer type of its balancer'.format(domain_version.domain_id),
                cause=domain_version)


class ValidationResult(object):
    __slots__ = ('balancer', 'included_full_domain_ids', 'included_full_upstream_ids', 'included_full_backend_ids',
                 'included_full_endpoint_set_ids', 'included_full_knob_ids', 'included_full_cert_ids',
                 'included_full_weight_section_ids', 'rollback_count', 'validation_ctx')

    def __init__(self,
                 balancer,
                 included_full_domain_ids,
                 included_full_upstream_ids,
                 included_full_backend_ids,
                 included_full_endpoint_set_ids,
                 included_full_knob_ids,
                 included_full_cert_ids,
                 included_full_weight_section_ids,
                 rollback_count=0,
                 validation_ctx=None,
                 ):
        """
        :type balancer: Holder
        :type included_full_domain_ids: set[(str, str)]
        :type included_full_upstream_ids: set[(str, str)]
        :type included_full_backend_ids: set[(str, str)]
        :type included_full_endpoint_set_ids: set[(str, str)]
        :type included_full_knob_ids: set[(str, str)]
        :type included_full_cert_ids: set[(str, str)]
        :type included_full_weight_section_ids: set[(str, str)]
        :type rollback_count: int
        :type validation_ctx: Optional[ValidationCtx]
        """
        self.balancer = balancer
        self.included_full_domain_ids = included_full_domain_ids
        self.included_full_upstream_ids = included_full_upstream_ids
        self.included_full_backend_ids = included_full_backend_ids
        self.included_full_endpoint_set_ids = included_full_endpoint_set_ids
        self.included_full_knob_ids = included_full_knob_ids
        self.included_full_cert_ids = included_full_cert_ids
        self.included_full_weight_section_ids = included_full_weight_section_ids
        self.rollback_count = rollback_count
        self.validation_ctx = validation_ctx


def resolve_upstream_instances(log, upstreams):
    """
    :param log
    :type upstreams: dict[(awacs.model.balancer.vector.UpstreamVersion, Holder)]
    """
    log.debug('starting resolve_upstream_instances')
    upstreams_count = len(upstreams)
    for i, (upstream_version, upstream) in enumerate(six.iteritems(upstreams), start=1):
        try:
            find_and_resolve_all_upstream_instances(upstream)
        except resolver.ResolvingError as e:
            raise errors.ConfigValidationError(six.text_type(e), cause=upstream_version, error=e)
        if i % 100 == 0:
            log.debug('called resolve_upstream_instances for {} of {} upstreams'.format(i, upstreams_count))
            gevent.idle()
    log.debug('finished resolve_upstream_instances')


def find_cause_for_backend_does_not_exist_error(namespace_id, missing_full_backend_ids, available_backend_versions):
    """
    :type namespace_id: six.text_type
    :type missing_full_backend_ids: set[(six.text_type, six.text_type)]
    :type available_backend_versions: set[BackendVersion]
    :rtype: (six.text_type, BackendVersion | None)
    """
    available_backend_versions_by_ids = {}
    for version in available_backend_versions:
        available_backend_versions_by_ids[version.backend_id] = version

    for full_id in sorted(missing_full_backend_ids):
        if full_id not in available_backend_versions:
            continue
        version = available_backend_versions_by_ids[full_id]
        flat_id = flatten_full_id(namespace_id, full_id)
        if version.deleted:
            return 'Backend "{}" is deleted, but still in use.'.format(flat_id), version
        else:
            return 'Backend "{}" is not resolved yet.'.format(flat_id), version

    message = 'Some included backends are missing or not resolved yet: "{}"'.format(
        join_full_uids2(namespace_id, missing_full_backend_ids))
    return message, None


def get_validation_ctx(namespace_pb, balancer_version, balancer_spec_pb,
                       knob_version_to_spec_pbs, cert_version_to_spec_pbs, domain_version_to_spec_pbs,
                       weight_section_version_to_spec_pbs, upstream_version_to_spec_pbs,
                       namespace_id, threadpool):
    """
    :type namespace_pb: model_pb2.Namespace
    :type balancer_version: BalancerVersion
    :type balancer_spec_pb: model_pb2.BalancerSpec
    :type knob_version_to_spec_pbs: dict[KnobVersion, awacs.proto.model_pb2.KnobSpec]
    :type cert_version_to_spec_pbs: dict[CertVersion, awacs.proto.model_pb2.CertificateSpec]
    :type domain_version_to_spec_pbs: dict[DomainVersion, awacs.proto.model_pb2.DomainSpec]
    :type weight_section_version_to_spec_pbs: dict[objects.WeightSection.version, awacs.proto.model_pb2.WeightSectionSpec]
    :type upstream_version_to_spec_pbs dict[UpstreamVersion, awacs.proto.model_pb2.UpstreamSpec]
    :type namespace_id: six.text_type
    :type threadpool: gevent.threadpool.ThreadPool
    """
    knob_spec_pbs = {knob_version.knob_id: knob_spec_pb
                     for knob_version, knob_spec_pb in six.iteritems(knob_version_to_spec_pbs)}
    domain_spec_pbs = {domain_version.domain_id: domain_spec_pb
                       for domain_version, domain_spec_pb in six.iteritems(domain_version_to_spec_pbs)}
    knob_version_hints = {knob_version.knob_id: knob_version
                          for knob_version in six.iterkeys(knob_version_to_spec_pbs)}
    cert_spec_pbs = {cert_version.cert_id: cert_spec_pb
                     for cert_version, cert_spec_pb in six.iteritems(cert_version_to_spec_pbs)}
    weight_section_spec_pbs = {weight_section_version.id: weight_section_spec_pb
                               for weight_section_version, weight_section_spec_pb in
                               six.iteritems(weight_section_version_to_spec_pbs)}
    upstream_spec_pbs = {upstream_version.upstream_id: upstream_spec_pb
                         for upstream_version, upstream_spec_pb in six.iteritems(upstream_version_to_spec_pbs)}
    return ValidationCtx.create_ctx_with_config_type_full(
        namespace_pb=namespace_pb,
        full_balancer_id=balancer_version.balancer_id,
        balancer_spec_pb=balancer_spec_pb,
        knob_spec_pbs=knob_spec_pbs,
        knob_version_hints=knob_version_hints,
        cert_spec_pbs=cert_spec_pbs,
        domain_spec_pbs=domain_spec_pbs,
        weight_section_spec_pbs=weight_section_spec_pbs,
        upstream_spec_pbs=upstream_spec_pbs,
        disable_gevent_idle=threadpool is not None)


def run_heavy_operation(log, threadpool, timer, func, func_name, interval=None):
    if interval is None:
        interval = 3
    log.debug('%s started, threadpool: %s, %s', func_name, threadpool is not None, threadpool and len(threadpool) or 0)
    if threadpool is None:
        func()
    else:
        with timer.timer():
            res = threadpool.spawn(func)
            while not res.ready():
                log.debug('%s is not ready', func_name)
                gevent.sleep(interval)
            log.debug('%s is ready', func_name)
            res.get()
    log.debug('%s finished', func_name)
    gevent.idle()


def validate_config(namespace_pb, namespace_id, balancer_version, balancer_spec_pb,
                    upstream_spec_pbs, backend_spec_pbs, endpoint_set_spec_pbs,
                    knob_spec_pbs=None, cert_spec_pbs=None, domain_spec_pbs=None,
                    weight_section_spec_pbs=None, upstreams=None, threadpool=None,
                    _digest=False, ctx=None, threadpool_interval=None):
    """
    :type namespace_id: six.text_type
    :type balancer_version: BalancerVersion
    :type balancer_spec_pb: awacs.proto.model_pb2.BalancerSpec
    :type upstream_spec_pbs: dict[UpstreamVersion, awacs.proto.model_pb2.UpstreamSpec]
    :type backend_spec_pbs: dict[BackendVersion, awacs.proto.model_pb2.BackendSpec]
    :type endpoint_set_spec_pbs: dict[EndpointSetVersion, awacs.proto.model_pb2.EndpointSetSpec]
    :type knob_spec_pbs: dict[KnobVersion, awacs.proto.model_pb2.KnobSpec] | None
    :type cert_spec_pbs: dict[CertVersion, awacs.proto.model_pb2.CertificateSpec] | None
    :type domain_spec_pbs: dict[DomainVersion, awacs.proto.model_pb2.DomainSpec] | None
    :type weight_section_spec_pbs: dict[objects.WeightSection.version, awacs.proto.model_pb2.WeightSectionSpec] | None
    :param upstreams: already wrapped upstream_spec_pbs, optional
    :type upstreams: dict[UpstreamVersion, Holder]
    :type threadpool: gevent.threadpool.ThreadPool
    :type threadpool_interval: float
    :type _digest: bool
    :type ctx: context.OpCtx
    :rtype: ValidationResult
    """
    knob_spec_pbs = knob_spec_pbs or {}
    cert_spec_pbs = cert_spec_pbs or {}
    domain_spec_pbs = domain_spec_pbs or {}
    weight_section_spec_pbs = weight_section_spec_pbs or {}
    if ctx:
        log = ctx.with_op(op_id='validate_config').log
    else:
        log = logging.getLogger('validate_config("{}")'.format('/'.join(balancer_version.balancer_id)))

    validate_internal_balancer_compatibility(balancer_version, balancer_spec_pb, upstream_spec_pbs, domain_spec_pbs)

    balancer_pb = get_yandex_config_pb(balancer_spec_pb)
    gevent.idle()

    balancer = Holder(balancer_pb)
    gevent.idle()

    validation_ctx = get_validation_ctx(namespace_pb, balancer_version, balancer_spec_pb, knob_spec_pbs, cert_spec_pbs,
                                        domain_spec_pbs, weight_section_spec_pbs, upstream_spec_pbs,
                                        namespace_id, threadpool)

    all_injected_weight_section_ids = set()
    all_injected_domain_ids = set()
    all_injected_upstream_ids = set()
    is_balancer_l7_macro = balancer.is_l7_macro()
    if is_balancer_l7_macro:
        # validate top-level l7_macro
        try:
            run_heavy_operation(log=log, threadpool=threadpool, timer=s_y_t_ru_validate_l7_macro_hgram,
                                func=lambda: balancer.validate(ctx=validation_ctx),
                                func_name='balancer.validate() for l7_macro', interval=threadpool_interval)
        except ValidationError as e:
            raise errors.ConfigValidationError(six.text_type(e), error=e, cause=e.hint)

        if balancer.get_nested_module().includes_domains():
            all_injected_domain_ids = {v.domain_id for v in six.iterkeys(domain_spec_pbs)}
            validate_yandex_tld_trust_xffy(balancer.get_nested_module(), domain_spec_pbs)

        # now we can expand top-level macro into instance_macro to collect all upstreams and backends
        run_heavy_operation(log=log, threadpool=threadpool, timer=s_y_t_ru_expand_l7_macro_hgram,
                            func=lambda: balancer.expand_immediate_contained_macro(ctx=validation_ctx),
                            func_name='balancer.expand_immediate_contained_macro()', interval=threadpool_interval)
    if not upstreams:
        upstreams = {}
        should_wrap_upstreams = True
    else:
        should_wrap_upstreams = False

    are_all_upstreams_l7_upstream_macro = True
    upstream_labels = {}
    upstream_pbs_by_ids = {}
    weight_section_ids = {w.id for w in weight_section_spec_pbs}
    for upstream_version, upstream_spec_pb in gevent_idle_iter(six.iteritems(upstream_spec_pbs), 5):
        if should_wrap_upstreams:
            upstreams[upstream_version] = Holder(clone_pb(upstream_spec_pb.yandex_balancer.config))
        if upstreams[upstream_version].is_l7_upstream_macro():
            upstream = upstreams[upstream_version]
            for full_weight_section_id in upstream.get_nested_module().get_would_be_included_full_weight_section_ids(namespace_id):
                if full_weight_section_id not in weight_section_ids:
                    continue
                all_injected_weight_section_ids.add(full_weight_section_id)
            for full_upstream_id in upstream.get_nested_module().get_would_be_included_full_internal_upstream_ids(namespace_id):
                all_injected_upstream_ids.add(full_upstream_id)
        upstream_id = upstream_version.upstream_id
        upstream_labels[upstream_id] = dict(upstream_spec_pb.labels)
        upstream_pbs_by_ids[upstream_id] = upstreams[upstream_version]
        are_all_upstreams_l7_upstream_macro &= (upstream_spec_pb.yandex_balancer.mode ==
                                                upstream_spec_pb.yandex_balancer.EASY_MODE)

    resolve_upstream_instances(log, upstreams)
    log.debug('resolve_upstream_instances() finished')
    gevent.idle()

    endpoint_set_specs_by_ids = {}
    endpoint_set_revs_by_ids = {}
    for endpoint_set_version, endpoint_set_spec_pb in gevent_idle_iter(six.iteritems(endpoint_set_spec_pbs), 5):
        es_id = endpoint_set_version.endpoint_set_id
        endpoint_set_specs_by_ids[es_id] = clone_pb(endpoint_set_spec_pb)
        endpoint_set_revs_by_ids[es_id] = endpoint_set_version

    backend_specs_by_ids = {}
    backend_revs_by_ids = {}
    for backend_version, backend_spec_pb in gevent_idle_iter(six.iteritems(backend_spec_pbs), 5):
        backend_id = backend_version.backend_id
        backend_specs_by_ids[backend_id] = clone_pb(backend_spec_pb)
        backend_revs_by_ids[backend_id] = backend_version

    all_injected_backend_ids = set()
    all_injected_endpoint_set_ids = set()

    try:
        injected_upstream_ids = inject_upstreams(namespace_id, balancer,
                                                 upstreams=upstream_pbs_by_ids,
                                                 upstream_labels=upstream_labels)
        all_injected_upstream_ids.update(injected_upstream_ids)
        if is_balancer_l7_macro and not all_injected_upstream_ids and not all_injected_domain_ids:
            raise errors.ConfigValidationError(u'l7_macro can\'t find any upstreams to include',
                                               cause=balancer_version)
        log.debug('inject_upstreams() finished')
        gevent.idle()

        try:
            # collect backends and es that are directly included in config (not in underlying macroses)
            injected_backend_ids, injected_endpoint_set_ids = inject_backends_and_endpoint_sets(
                namespace_id=namespace_id,
                balancer=balancer,
                backend_spec_pbs=backend_specs_by_ids,
                endpoint_set_spec_pbs=endpoint_set_specs_by_ids,
                backend_revs_by_ids=backend_revs_by_ids,
                endpoint_set_revs_by_ids=endpoint_set_revs_by_ids)
        except EndpointSetsDoNotExist as e:
            available_backend_versions = set(backend_spec_pbs)
            message, cause = find_cause_for_backend_does_not_exist_error(
                namespace_id, e.full_ids, available_backend_versions)
            raise errors.ConfigValidationError(message, cause=cause)
        all_injected_endpoint_set_ids.update(injected_endpoint_set_ids)
        all_injected_backend_ids.update(injected_backend_ids)
        log.debug('inject_backends_and_endpoint_sets() finished')
        gevent.idle()

        # validate expanded instance_macro
        run_heavy_operation(log=log, threadpool=threadpool, timer=s_y_t_ru_validate_hgram,
                            func=lambda: balancer.validate(ctx=validation_ctx),
                            func_name='balancer.validate()', interval=threadpool_interval)
    except ValidationError as e:
        raise errors.ConfigValidationError(six.text_type(e), error=e, cause=e.hint)

    if _digest:
        # this flag is used by CLI tools in Arcadia
        return ValidationResult(balancer,
                                included_full_upstream_ids=all_injected_upstream_ids,
                                included_full_backend_ids=all_injected_backend_ids,
                                included_full_endpoint_set_ids=all_injected_endpoint_set_ids,
                                included_full_knob_ids=set(),
                                included_full_cert_ids=set(),
                                included_full_domain_ids=all_injected_domain_ids,
                                included_full_weight_section_ids=set(),
                                validation_ctx=validation_ctx,
                                )

    # now expand all macroses to collect the rest of backends and endpoint sets
    run_heavy_operation(log=log, threadpool=threadpool, timer=s_y_t_ru_expand_macro_hgram,
                        func=lambda: balancer.expand_macroses(ctx=validation_ctx),
                        func_name='balancer.expand_macroses()', interval=threadpool_interval)

    log.debug('find_and_resolve_all_upstream_instances(balancer) called')
    try:
        find_and_resolve_all_upstream_instances(balancer)
    except resolver.ResolvingError as e:
        raise errors.ConfigValidationError(six.text_type(e), error=e)
    log.debug('find_and_resolve_all_upstream_instances(balancer) finished')
    gevent.idle()

    try:
        # collect backends and endpoint sets that were implicitly included in all macroses
        injected_backend_ids, injected_endpoint_set_ids = inject_backends_and_endpoint_sets(
            namespace_id=namespace_id,
            balancer=balancer,
            backend_spec_pbs=backend_specs_by_ids,
            endpoint_set_spec_pbs=endpoint_set_specs_by_ids,
            backend_revs_by_ids=backend_revs_by_ids,
            endpoint_set_revs_by_ids=endpoint_set_revs_by_ids)
        all_injected_endpoint_set_ids.update(injected_endpoint_set_ids)
        all_injected_backend_ids.update(injected_backend_ids)
        log.debug('inject_backends_and_endpoint_sets() finished')
        gevent.idle()
    except EndpointSetsDoNotExist as e:
        available_backend_versions = set(backend_spec_pbs)
        message, cause = find_cause_for_backend_does_not_exist_error(
            namespace_id, e.full_ids, available_backend_versions)
        raise errors.ConfigValidationError(message, cause=cause)
    except ValidationError as e:
        raise errors.ConfigValidationError(six.text_type(e), error=e, cause=e.hint)

    knob_specs_by_ids = {
        knob_version.knob_id: clone_pb(knob_spec_pb)
        for knob_version, knob_spec_pb in six.iteritems(knob_spec_pbs)}
    cert_specs_by_ids = {
        cert_version.cert_id: clone_pb(cert_spec_pb)
        for cert_version, cert_spec_pb in six.iteritems(cert_spec_pbs)}
    _, balancer_id = balancer_version.balancer_id
    try:
        all_injected_knob_ids = inject_knobs(namespace_id, balancer_id, balancer, knob_specs_by_ids, ctx=validation_ctx)
        all_injected_cert_ids = inject_certs(namespace_id, balancer, cert_specs_by_ids, ctx=validation_ctx)
    except (KnobDoesNotExist, CertDoesNotExist, ValidationError) as e:
        raise errors.ConfigValidationError(six.text_type(e), error=e, cause=e.hint)

    if is_balancer_l7_macro and are_all_upstreams_l7_upstream_macro:
        log.debug('balancer.validate_shared_and_report_refs() skipped')
    else:
        try:
            run_heavy_operation(log=log, threadpool=threadpool, timer=s_y_t_ru_validate_shared_and_report_refs_hgram,
                                func=lambda: balancer.validate_shared_and_report_refs(validation_ctx=validation_ctx),
                                func_name='balancer.validate_shared_and_report_refs()', interval=threadpool_interval)
        except ValidationError as e:
            raise errors.ConfigValidationError(six.text_type(e), error=e, cause=e.hint)

    return ValidationResult(balancer,
                            included_full_upstream_ids=all_injected_upstream_ids,
                            included_full_backend_ids=all_injected_backend_ids,
                            included_full_endpoint_set_ids=all_injected_endpoint_set_ids,
                            included_full_knob_ids=all_injected_knob_ids,
                            included_full_cert_ids=all_injected_cert_ids,
                            included_full_domain_ids=all_injected_domain_ids,
                            included_full_weight_section_ids=all_injected_weight_section_ids,
                            validation_ctx=validation_ctx)


def validate_yandex_tld_trust_xffy(l7_macro_instance, domain_spec_pbs):
    """
    Check the value for `core.trust_x_forwarded_for_y` is valid for the current
    `l7_macro` version and domain configuration.
    :type l7_macro_instance: l7macro.L7Macro
    :type domain_spec_pbs: dict[DomainVersion, awacs.proto.model_pb2.DomainSpec]
    :rtype: DomainVersion
    """
    yandex_tld_version = get_yandex_tld_domain_version(domain_spec_pbs)
    is_behind_tld = bool(yandex_tld_version)
    l7_version = l7_macro_instance.get_version()
    trust_xffy = l7_macro_instance.core and l7_macro_instance.core.pb.trust_x_forwarded_for_y

    if (
            l7_version >= l7macro.VERSION_0_3_0
            and is_behind_tld
            and not trust_xffy
    ):
        raise errors.ConfigValidationError(
            'You must set the `l7_macro.core.trust_x_forwarded_for_y: true` '
            'because your service is behind yandex.tld (namespace has a YANDEX_TLD '
            'domain configured)',
            cause=yandex_tld_version)


def get_yandex_tld_domain_version(domain_spec_pbs):
    """
    Get the domain version containing a yandex.tdl domain
    :type domain_spec_pbs: dict[DomainVersion, awacs.proto.model_pb2.DomainSpec]
    :rtype: DomainVersion
    """
    for version, domain_spec_pb in six.iteritems(domain_spec_pbs):
        domain_type = domain_spec_pb.yandex_balancer.config.type
        if domain_type == model_pb2.DomainSpec.Config.YANDEX_TLD:
            return version
    return None


def get_injected_full_upstream_ids(namespace_id, balancer, full_upstream_ids):
    """
    :type namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :type full_upstream_ids: list[(six.text_type, six.text_type)]
    :rtype: set[(six.text_type, six.text_type)]
    """
    injected_full_upstream_ids = set()

    # branching module is a module with arbitrary number of sections,
    # e.g. regexp and ipdispatch modules, balancer2 module and so on.
    # we start with a chain from the balancer root module
    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=20):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            for branch in module.get_branches():
                chains.append(branch.walk_chain())

        assert module
        last_chain_module = module

        if last_chain_module.includes_upstreams():
            # here last_chain_module is branching and defines "include_upstreams" directive
            # we filter and order upstreams according to "include_upstreams" rules
            filtered_upstreams = last_chain_module.include_upstreams.get_included_upstream_ids(namespace_id,
                                                                                               full_upstream_ids)
            injected_full_upstream_ids.update(filtered_upstreams)

    return injected_full_upstream_ids


def get_would_be_injected_full_internal_upstream_ids(namespace_id, balancer):
    """
    :type namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :rtype: set[(six.text_type, six.text_type)]
    """
    rv = set()

    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=20):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            if module.would_include_internal_upstreams():
                rv.update(module.get_would_be_included_full_internal_upstream_ids(namespace_id))
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
        assert module

    return rv


def get_would_be_injected_full_backend_ids(current_namespace_id, balancer):
    """
    :type current_namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :rtype: set[(six.text_type, six.text_type)]
    """
    rv = set()

    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=20):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            if module.would_include_backends():
                included_full_backend_ids = module.get_would_be_included_full_backend_ids(current_namespace_id)
                rv.update(included_full_backend_ids)
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
        assert module

    return rv


def get_would_be_injected_full_weight_section_ids(current_namespace_id, balancer):
    """
    :type current_namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :rtype: set[(six.text_type, six.text_type)]
    """
    rv = set()

    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=20):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            if module.would_include_weight_sections():
                included_full_weight_section_ids = module.get_would_be_included_full_weight_section_ids(current_namespace_id)
                rv.update(included_full_weight_section_ids)
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
        assert module

    return rv


def get_would_be_injected_full_knob_ids(current_namespace_id, balancer, ctx):
    """
    :type current_namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :type ctx: ValidationCtx
    :rtype: set[(six.text_type, six.text_type)]
    """
    rv = set()

    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=20):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            rv.update(module.get_would_be_included_full_knob_ids(current_namespace_id, ctx=ctx))
            for w in module.walk_composite_fields():
                if w.would_include_knobs(ctx=ctx):
                    included_full_knob_ids = w.get_would_be_included_full_knob_ids(current_namespace_id, ctx=ctx)
                    rv.update(included_full_knob_ids)
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
        assert module

    return rv


def get_would_be_injected_full_cert_ids(current_namespace_id, balancer, ctx):
    """
    :type current_namespace_id: six.text_type
    :type balancer: awacs.wrappers.base.Holder
    :type ctx: ValidationCtx
    :rtype: set[(six.text_type, six.text_type)]
    """
    rv = set()
    chains = [balancer.walk_chain()]
    for chain in gevent_idle_iter(chains, idle_period=20):
        # follow the chain to the last module, visiting branches
        # (such as "geo" branch from geobase module, "checker" branch from antirobot,
        # or regexp sections)
        module = None
        for module in chain:
            rv.update(module.get_would_be_included_full_cert_ids(current_namespace_id, ctx=ctx))
            for w in module.walk_composite_fields():
                if w.includes_certs(ctx=ctx):
                    included_full_cert_ids = w.get_would_be_included_full_cert_ids(current_namespace_id, ctx=ctx)
                    rv.update(included_full_cert_ids)
            for branch in module.get_branches():
                chains.append(branch.walk_chain())
        assert module

    return rv


def noop():
    return


def get_included_full_backend_ids_from_holder(namespace_id, holder, ticker=noop):
    """
    :type namespace_id: six.text_type
    :type holder: Holder
    :type ticker: callable
    :rtype: set[(six.text_type, six.text_type)]
    """
    included_full_backend_ids = set()
    for module in holder.walk_chain(visit_branches=True):
        ticker()
        if module.includes_backends():
            included_full_backend_ids.update(module.include_backends.get_included_full_backend_ids(namespace_id))
        elif isinstance(module, MacroBase) and module.would_include_backends():
            included_full_backend_ids.update(module.get_would_be_included_full_backend_ids(namespace_id))
    return included_full_backend_ids


def get_included_full_backend_ids(upstream_pb, ticker=noop):
    """
    :type upstream_pb: model_pb2.Upstream
    :type ticker: callable
    :rtype: set[(six.text_type, six.text_type)]
    """
    namespace_id = upstream_pb.meta.namespace_id
    holder = Holder(upstream_pb.spec.yandex_balancer.config)
    return get_included_full_backend_ids_from_holder(namespace_id, holder, ticker=ticker)


def get_rev_index_pb(namespace_id, spec_pb, holder=None):
    """
    :type namespace_id: six.text_type
    :type spec_pb: model_pb2.BalancerSpec | model_pb2.UpstreamSpec
    :type holder: Holder
    :rtype model_pb2.RevisionGraphIndex
    """
    if holder is None:
        holder = Holder(spec_pb.yandex_balancer.config)

    included_full_backend_ids = get_included_full_backend_ids_from_holder(namespace_id, holder)

    ind_pb = model_pb2.RevisionGraphIndex()
    ind_pb.included_backend_ids.extend(flatten_full_id2(f_id) for f_id in included_full_backend_ids)
    return ind_pb
