# coding: utf-8
import random

import six
from infra.awacs.proto import model_pb2
from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import is_order_in_progress, has_actionable_spec, needs_removal
from awacs.model import events
from awacs.model.util import get_balancer_location, BALANCER_LOCATIONS
from awacs.model.balancer.ctl import BalancerCtl
from awacs.model.balancer.operations.ctl import BalancerOperationCtl
from awacs.model.balancer.order.order_ctl import BalancerOrderCtl
from awacs.model.balancer.removal.ctl import BalancerRemovalCtl


def get_allowed_balancer_matcher(allowed_locations):
    """
    :type allowed_locations: set[six.text_type]
    """
    allowed_locations = frozenset(allowed_locations)
    assert allowed_locations.issubset(set(BALANCER_LOCATIONS))

    def allowed_balancer_matcher(balancer_pb):
        return get_balancer_location(balancer_pb) in allowed_locations

    return allowed_balancer_matcher


def acquire_timeout_strategy():
    while 1:
        r = random.random()
        if r < 0.1:
            yield ctlmanager.HALF_A_MINUTE
        elif r < 0.2:
            yield random.randint(ctlmanager.ONE_MINUTE, ctlmanager.FIVE_MINUTES)
        elif r < 0.4:
            yield random.randint(ctlmanager.FIVE_MINUTES, ctlmanager.TEN_MINUTES)
        elif r < 0.6:
            yield random.randint(ctlmanager.TEN_MINUTES, 1.5 * ctlmanager.TEN_MINUTES)
        else:
            yield ctlmanager.TWELVE_HOURS


def standoff_strategy(cache, ctl_id):
    """
    :type cache: awacs.model.cache.AwacsCache
    :type ctl_id: (six.text_type, six.text_type)
    """
    namespace_id, balancer_id = ctl_id
    balancer_state_pb = cache.get_balancer_state(namespace_id=namespace_id, balancer_id=balancer_id)
    if balancer_state_pb and len(balancer_state_pb.upstreams) + len(balancer_state_pb.backends) < 100:
        # we'd like to move around only smaller balancers
        while 1:
            if random.random() > .6:
                yield random.randint(ctlmanager.FIVE_MINUTES, ctlmanager.TEN_MINUTES)
            else:
                yield ctlmanager.TWELVE_HOURS
    else:
        while 1:
            yield ctlmanager.TWELVE_HOURS


class BalancerCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None, allowed_balancer_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        :type allowed_balancer_matcher: callable | None
        """
        super(BalancerCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='balancer-ctl-manager',
            starting_events=(events.BalancerUpdate,),
            stopping_events=(events.BalancerRemove,),
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher
        self.allowed_balancer_matcher = allowed_balancer_matcher

    def _get_acquire_timeout_strategy(self, ctl_id):
        if six.PY3:
            return acquire_timeout_strategy().__next__
        else:
            return acquire_timeout_strategy().next

    def _get_standoff_strategy(self, ctl_id):
        if six.PY3:
            return standoff_strategy(self._cache, ctl_id).__next__
        else:
            return standoff_strategy(self._cache, ctl_id).next

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'balancer("{}:{}") ctl'.format(*ctl_id)

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _is_allowed_balancer(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: bool
        """
        return self.allowed_balancer_matcher is None or self.allowed_balancer_matcher(balancer_pb)

    def _is_starting_event(self, event):
        return (super(BalancerCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.namespace_id) and
                self._is_allowed_balancer(event.pb) and
                has_actionable_spec(event.pb) and
                not needs_removal(event.pb))

    def _is_stopping_event(self, event):
        if super(BalancerCtlManager, self)._is_stopping_event(event):
            return True
        return super(BalancerCtlManager, self)._is_starting_event(event) and needs_removal(event.pb)

    def _list_all_ctl_ids(self):
        """
        :rtype: (six.text_type, six.text_type)
        """
        namespace_pbs = self._cache.list_all_namespaces()
        for namespace_pb in namespace_pbs:
            namespace_id = namespace_pb.meta.id
            if not self._is_allowed_namespace(namespace_id):
                continue
            balancer_pbs = self._cache.list_all_balancers(namespace_id=namespace_id)
            for balancer_pb in balancer_pbs:
                if (self._is_allowed_balancer(balancer_pb)
                        and has_actionable_spec(balancer_pb)
                        and not needs_removal(balancer_pb)):
                    yield namespace_id, balancer_pb.meta.id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        namespace_id, balancer_id = ctl_id
        return BalancerCtl(namespace_id, balancer_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: (six.text_type, six.text_type)
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct balancer ctl id'.format(event.path))
        return ctl_id


class BalancerOrderCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None, allowed_balancer_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        :type allowed_balancer_matcher: callable | None
        """
        super(BalancerOrderCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='balancer-order-ctl-manager',
            starting_events=(events.BalancerUpdate,),
            stopping_events=(events.BalancerRemove,)
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher
        self.allowed_balancer_matcher = allowed_balancer_matcher

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _is_allowed_balancer(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: bool
        """
        return self.allowed_balancer_matcher is None or self.allowed_balancer_matcher(balancer_pb)

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'balancer-order("{}:{}") ctl'.format(*ctl_id)

    def _is_starting_event(self, event):
        return (super(BalancerOrderCtlManager, self)._is_starting_event(event)
                and self._is_allowed_namespace(event.pb.meta.namespace_id)
                and self._is_allowed_balancer(event.pb)
                and is_order_in_progress(event.pb)
                and not needs_removal(event.pb))

    def _is_stopping_event(self, event):
        if super(BalancerOrderCtlManager, self)._is_stopping_event(event):
            return True
        return (super(BalancerOrderCtlManager, self)._is_starting_event(event)
                and (not is_order_in_progress(event.pb) or needs_removal(event.pb)))

    def _list_all_ctl_ids(self):
        """
        :rtype: (six.text_type, six.text_type)
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if not self._is_allowed_namespace(namespace_id):
                continue
            for pb in self._cache.list_all_balancers(namespace_id=namespace_id):
                if (self._is_allowed_balancer(pb)
                        and is_order_in_progress(pb)
                        and not needs_removal(pb)):
                    yield namespace_id, pb.meta.id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        namespace_id, balancer_id = ctl_id
        return BalancerOrderCtl(namespace_id, balancer_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct balancer order ctl id'.format(event.path))
        return ctl_id


class BalancerRemovalCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None, allowed_balancer_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        :type allowed_balancer_matcher: callable | None
        """
        super(BalancerRemovalCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='balancer-removal-ctl-manager',
            starting_events=(events.BalancerUpdate,),
            stopping_events=(events.BalancerRemove,)
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher
        self.allowed_balancer_matcher = allowed_balancer_matcher

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _is_allowed_balancer(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: bool
        """
        return self.allowed_balancer_matcher is None or self.allowed_balancer_matcher(balancer_pb)

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'balancer-removal("{}:{}") ctl'.format(*ctl_id)

    def _is_starting_event(self, event):
        return (super(BalancerRemovalCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.namespace_id) and
                self._is_allowed_balancer(event.pb) and
                needs_removal(event.pb))

    def _list_all_ctl_ids(self):
        """
        :rtype: (six.text_type, six.text_type)
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if not self._is_allowed_namespace(namespace_id):
                continue
            for pb in self._cache.list_all_balancers(namespace_id=namespace_id):
                if self._is_allowed_balancer(pb) and needs_removal(pb):
                    yield namespace_id, pb.meta.id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        namespace_id, balancer_id = ctl_id
        return BalancerRemovalCtl(namespace_id, balancer_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct balancer removal ctl id'.format(event.path))
        return ctl_id


class BalancerOperationCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, member_id, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(BalancerOperationCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='balancer-op-ctl-manager',
            starting_events=(events.BalancerOperationUpdate,),
            stopping_events=(events.BalancerOperationRemove,),
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'balancer-op("{}:{}") ctl'.format(*ctl_id)

    def _is_starting_event(self, event):
        return (super(BalancerOperationCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.namespace_id) and
                (is_order_in_progress(event.pb) or needs_removal(event.pb)))

    def _list_all_ctl_ids(self):
        """
        :rtype: (six.text_type, six.text_type)
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if not self._is_allowed_namespace(namespace_id):
                continue
            for balancer_op_pb in self._cache.list_all_balancer_operations(namespace_id=namespace_id):
                if is_order_in_progress(balancer_op_pb):
                    yield namespace_id, balancer_op_pb.meta.id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        namespace_id, balancer_id = ctl_id
        return BalancerOperationCtl(namespace_id, balancer_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct balancer operation ctl id'.format(event.path))
        return ctl_id


class BalancerCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None, allowed_balancer_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        :type allowed_balancer_matcher: callable | None
        """
        self.allowed_balancer_matcher = allowed_balancer_matcher
        super(BalancerCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='balancer',
            subscribed_events=(events.BalancerUpdate, events.BalancerRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _is_allowed_balancer(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: bool
        """
        return self.allowed_balancer_matcher is None or self.allowed_balancer_matcher(balancer_pb)

    def _get_acquire_timeout_strategy(self, ctl_id):
        if six.PY3:
            return acquire_timeout_strategy().__next__
        else:
            return acquire_timeout_strategy().next

    def _get_standoff_strategy(self, ctl_id):
        if six.PY3:
            return standoff_strategy(self._cache, ctl_id).__next__
        else:
            return standoff_strategy(self._cache, ctl_id).next

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.BalancerRemove):
            return False
        return has_actionable_spec(event.pb) and self._is_allowed_balancer(event.pb) and not needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for balancer_pb in self._cache.list_all_balancers(namespace_id):
                if not self._is_allowed_balancer(balancer_pb):
                    continue
                yield events.BalancerUpdate(path=self._make_event_path_from_ctl_id(namespace_id, balancer_pb.meta.id),
                                            pb=balancer_pb)

    def _create_ctl(self, ctl_id):
        return BalancerCtl(*ctl_id)


class BalancerOrderCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None, allowed_balancer_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        :type allowed_balancer_matcher: callable | None
        """
        self.allowed_balancer_matcher = allowed_balancer_matcher
        super(BalancerOrderCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='balancer-order',
            subscribed_events=(events.BalancerUpdate, events.BalancerRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )

    def _is_allowed_balancer(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: bool
        """
        return self.allowed_balancer_matcher is None or self.allowed_balancer_matcher(balancer_pb)

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.BalancerRemove):
            return False
        return is_order_in_progress(event.pb) and self._is_allowed_balancer(event.pb) and not needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for balancer_pb in self._cache.list_all_balancers(namespace_id):
                if not self._is_allowed_balancer(balancer_pb):
                    continue
                yield events.BalancerUpdate(path=self._make_event_path_from_ctl_id(namespace_id, balancer_pb.meta.id),
                                            pb=balancer_pb)

    def _create_ctl(self, ctl_id):
        return BalancerOrderCtl(*ctl_id)


class BalancerRemovalCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None, allowed_balancer_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        :type allowed_balancer_matcher: callable | None
        """
        self.allowed_balancer_matcher = allowed_balancer_matcher
        super(BalancerRemovalCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='balancer-removal',
            subscribed_events=(events.BalancerUpdate, events.BalancerRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _is_allowed_balancer(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: bool
        """
        return self.allowed_balancer_matcher is None or self.allowed_balancer_matcher(balancer_pb)

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.BalancerRemove):
            return False
        return self._is_allowed_balancer(event.pb) and needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for balancer_pb in self._cache.list_all_balancers(namespace_id):
                if not self._is_allowed_balancer(balancer_pb):
                    continue
                yield events.BalancerUpdate(path=self._make_event_path_from_ctl_id(namespace_id, balancer_pb.meta.id),
                                            pb=balancer_pb)

    def _create_ctl(self, ctl_id):
        return BalancerRemovalCtl(*ctl_id)


class BalancerOperationCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(BalancerOperationCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='balancer-op',
            subscribed_events=(events.BalancerOperationUpdate, events.BalancerOperationRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.BalancerOperationRemove):
            return False
        return is_order_in_progress(event.pb) or needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for balancer_op_pb in self._cache.list_all_balancer_operations(namespace_id):
                yield events.BalancerOperationUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, balancer_op_pb.meta.id),
                    pb=balancer_op_pb)

    def _create_ctl(self, ctl_id):
        return BalancerOperationCtl(*ctl_id)
