import enum
import inject
from abc import ABCMeta

import six

from awacs.lib.ypliterpcclient import IYpLiteRpcClient, YpLiteRpcClient
from awacs.lib.order_processor.model import BaseProcessor, WithOrder
from awacs.model import dao, zk, util
from awacs.model.balancer.endpoints import endpoint_set_exists
from awacs.model.util import make_system_endpoint_set_id
from infra.awacs.proto import model_pb2
from yp_lite_ui_repo import endpoint_sets_api_pb2, endpoint_sets_pb2


class State(enum.Enum):
    STARTED = 1
    CREATING_SYSTEM_ENDPOINT_SET = 2
    CREATING_BACKEND = 3

    FINISHED = 10
    CANCELLING = 20
    CANCELLED = 30


def get_create_system_backend_processors():
    return (
        Started,
        CreatingSystemEndpointSet,
        CreatingBackend,
        Cancelling,
    )


class CreateSystemBackendOperation(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = 'CreateSystemBackendOperation'
    states = State

    def zk_update(self):
        return self.zk.update_balancer_operation(namespace_id=self.namespace_id,
                                                 balancer_id=self.id)

    def zk_get_balancer(self):
        return self.zk.must_get_balancer(namespace_id=self.namespace_id,
                                         balancer_id=self.id)

    def dao_update(self):
        return self.dao.update_balancer_operation(namespace_id=self.namespace_id,
                                                  balancer_id=self.id,
                                                  version=self.pb.meta.version,
                                                  comment='Saved balancer operation spec',
                                                  login=util.NANNY_ROBOT_LOGIN,
                                                  updated_spec_pb=self.pb.spec,
                                                  )


class CreateSystemBackendOperationOrderProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = ('operation',)

    def __init__(self, entity):
        super(CreateSystemBackendOperationOrderProcessor, self).__init__(entity)
        self.operation = self.entity  # type: CreateSystemBackendOperation


class Started(CreateSystemBackendOperationOrderProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.CREATING_SYSTEM_ENDPOINT_SET
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class CreatingSystemEndpointSet(CreateSystemBackendOperationOrderProcessor):
    __slots__ = ()
    state = State.CREATING_SYSTEM_ENDPOINT_SET
    next_state = State.CREATING_BACKEND
    cancelled_state = State.CANCELLING
    _yp_lite_rpc_client = inject.attr(IYpLiteRpcClient)  # type: YpLiteRpcClient

    def process(self, ctx):
        balancer_pb = self.entity.zk_get_balancer()
        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        endpoint_set_id = make_system_endpoint_set_id(nanny_service_id)
        cluster = balancer_pb.meta.location.yp_cluster

        if endpoint_set_exists(endpoint_set_id, cluster, self._yp_lite_rpc_client):
            # System endpoint set already exists, backend can be created
            return self.next_state

        req_pb = endpoint_sets_api_pb2.CreateEndpointSetRequest()
        req_pb.cluster = cluster
        req_pb.meta.id = endpoint_set_id
        req_pb.meta.service_id = nanny_service_id
        req_pb.meta.ownership = endpoint_sets_pb2.EndpointSetMeta.SYSTEM
        req_pb.spec.protocol = 'TCP'
        req_pb.spec.port = 80
        req_pb.spec.description = 'Created by awacs'
        self._yp_lite_rpc_client.create_endpoint_set(req_pb)
        return self.next_state


class CreatingBackend(CreateSystemBackendOperationOrderProcessor):
    __slots__ = ()
    state = State.CREATING_BACKEND
    next_state = State.FINISHED
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        meta_pb = model_pb2.BackendMeta(
            id=self.entity.id,
            namespace_id=self.entity.namespace_id
        )
        meta_pb.auth.type = meta_pb.auth.STAFF
        meta_pb.is_system.value = True
        meta_pb.is_system.author = util.NANNY_ROBOT_LOGIN
        meta_pb.is_system.mtime.GetCurrentTime()
        spec_pb = model_pb2.BackendSpec()
        spec_pb.selector.type = model_pb2.BackendSelector.BALANCERS
        spec_pb.selector.balancers.add(id=self.entity.id)
        self.entity.dao.create_backend_if_missing(meta_pb=meta_pb, spec_pb=spec_pb)

        self.operation.pb.spec.incomplete = False
        self.operation.dao_update()
        return self.next_state


class Cancelling(CreateSystemBackendOperationOrderProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    def process(self, ctx):
        self.operation.pb.spec.incomplete = False
        self.operation.dao_update()
        return self.next_state
