import inject
import six

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import needs_removal
from awacs.lib.order_processor.runner import StateRunner
from awacs.lib.strutils import quote_join_sorted
from awacs.model import events, cache, zk, dao
from awacs.model.balancer.operations.create_system_backend import (
    CreateSystemBackendOperation,
    get_create_system_backend_processors,
)
from awacs.model.balancer.operations.migrate_from_gencfg_to_yp_lite import (
    GencfgMigration,
    get_gencfg_migration_processors,
)
from infra.awacs.proto import model_pb2


class BalancerOperationCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    _runners = {
        u'migrate_from_gencfg_to_yp_lite': StateRunner(entity_class=GencfgMigration,
                                                       initial_state=GencfgMigration.states.STARTED,
                                                       final_state=GencfgMigration.states.FINISHED,
                                                       final_cancelled_state=GencfgMigration.states.CANCELLED,
                                                       processors=get_gencfg_migration_processors(),
                                                       processing_interval=PROCESSING_INTERVAL
                                                       ),
        u'create_system_backend': StateRunner(entity_class=CreateSystemBackendOperation,
                                              initial_state=CreateSystemBackendOperation.states.STARTED,
                                              final_state=CreateSystemBackendOperation.states.FINISHED,
                                              final_cancelled_state=CreateSystemBackendOperation.states.CANCELLED,
                                              processors=get_create_system_backend_processors(),
                                              processing_interval=PROCESSING_INTERVAL
                                              )
    }

    def __init__(self, namespace_id, domain_id):
        name = 'balancer-op-ctl("{}:{}")'.format(namespace_id, domain_id)
        super(BalancerOperationCtl, self).__init__(name)
        self._balancer_id = domain_id
        self._namespace_id = namespace_id
        self._pb = None  # type: model_pb2.BalancerOperation or None

    def _accept_event(self, event):
        return (isinstance(event, events.BalancerOperationUpdate) and
                event.pb.meta.id == self._balancer_id)

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process balancer operation on start: %s', e)
        self._cache.bind(self._callback)

    def _stop(self):
        self._cache.unbind(self._callback)

    def _process(self, ctx, event=None):
        """
        :type event: events.BalancerOperationUpdate
        :type ctx: context.OpCtx
        """
        cached_pb = self._cache.must_get_balancer_operation(self._namespace_id, self._balancer_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if needs_removal(self._pb):
            return self._self_delete(ctx)

        if not self._pb.HasField('order'):
            raise RuntimeError('Balancer operation has no "order" field')

        for field, runner in six.iteritems(self._runners):
            if self._pb.order.content.HasField(field):
                return runner.process(ctx, self._pb)
        raise RuntimeError(u'Domain operation unknown or not specified. Supported operations: "{}"'.format(
            quote_join_sorted(self._runners)))

    def _self_delete(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        ctx.log.info('started self deletion')
        self._dao.delete_balancer_operation(self._namespace_id, self._balancer_id)

    def _process_event(self, ctx, event):
        assert isinstance(event, events.BalancerOperationUpdate)
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
