import enum
import inject
from abc import ABCMeta

import six

from awacs.lib import nannyclient
from awacs.lib.order_processor.model import BaseProcessor, WithOrder, FeedbackMessage
from awacs.model import dao, zk, util
from infra.awacs.proto import model_pb2
from awacs.resolver.gencfg import client as gencfg_client
from infra.swatlib import metrics


class State(enum.Enum):
    STARTED = 1
    DETERMINING_ORTHOGONAL_TAGS = 2
    FILLING_REMAINING_TAGS = 3
    DETERMINING_VIRTUAL_SERVICES = 4
    CREATING_BALANCER = 5
    FINISHED = 10
    CANCELLING = 20
    CANCELLED = 30


def get_gencfg_migration_processors():
    return (
        Start,
        DeterminingOrthogonalTags,
        FillingRemainingTags,
        DeterminingVirtualServices,
        CreatingBalancer,
        Cancelling,
    )


class GencfgMigration(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = 'GencfgMigration'
    states = State

    def zk_update(self):
        return self.zk.update_balancer_operation(namespace_id=self.namespace_id,
                                                 balancer_id=self.id)

    def zk_get_balancer(self):
        return self.zk.must_get_balancer(namespace_id=self.namespace_id,
                                         balancer_id=self.id)

    def dao_update(self):
        return self.dao.update_balancer_operation(namespace_id=self.namespace_id,
                                                  balancer_id=self.id,
                                                  version=self.pb.meta.version,
                                                  comment='Saved balancer operation spec',
                                                  login=util.NANNY_ROBOT_LOGIN,
                                                  updated_spec_pb=self.pb.spec,
                                                  )

    def can_change_instance_tags(self):
        return self.pb.order.progress.state.id == State.FILLING_REMAINING_TAGS.name


class GencfgMigrationOrderProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = ('operation',)

    def __init__(self, entity):
        super(GencfgMigrationOrderProcessor, self).__init__(entity)
        self.operation = self.entity  # type: GencfgMigration


class Start(GencfgMigrationOrderProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.DETERMINING_ORTHOGONAL_TAGS
    cancelled_state = State.CANCELLING
    _nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient

    def process(self, ctx):
        old_balancer_pb = self.operation.zk_get_balancer()
        old_service_id = old_balancer_pb.spec.config_transport.nanny_static_file.service_id
        try:
            runtime_attrs = self._nanny_client.get_service_runtime_attrs(service_id=old_service_id)
        except nannyclient.NannyApiRequestException as e:
            if e.response.status_code == 400:
                ctx.log.exception('Got 400 BAD REQUEST while creating Nanny service: %s', e.response.json())
            raise
        engine_type = runtime_attrs['content']['engines']['engine_type']
        if engine_type == 'YP_LITE':
            raise AssertionError("Balancer's engine type can't be YP_LITE")
        gencfg_groups = runtime_attrs['content']['instances'].get('extended_gencfg_groups', {})
        self.operation.context['gencfg_groups'] = gencfg_groups.get('groups', ())
        self.operation.context['old_orthogonal_tags'] = gencfg_groups.get('orthogonal_tags', {})
        old_spec_pb = old_balancer_pb.spec
        if (old_spec_pb.yandex_balancer.mode == old_spec_pb.yandex_balancer.FULL_MODE and
                u'get_port_var' not in old_spec_pb.yandex_balancer.yaml):
            metrics.ROOT_REGISTRY.get_counter(u'dirty-yp-lite-migration-started').inc(1)
            ctx.log.warn(u'Dirty gencfg-powered balancer {}:{} has started migration to YP.lite'.format(
                old_balancer_pb.meta.namespace_id, old_balancer_pb.meta.id))
        return self.next_state


class DeterminingOrthogonalTags(GencfgMigrationOrderProcessor):
    state = State.DETERMINING_ORTHOGONAL_TAGS
    next_state = State.DETERMINING_VIRTUAL_SERVICES
    next_state_filling_tags = State.FILLING_REMAINING_TAGS
    cancelled_state = State.CANCELLING
    _gencfg_client = inject.attr(gencfg_client.IGencfgClient)  # type: gencfg_client.GencfgClient

    SYSTEM_PRJ_TAGS = ('cplb', 'cplb-dynamic', 'cplb-menace', 'cplb-saas', 'cplb-swat', 'quasar-iot',
                       'kernel-test-dev', 'kernel-test-main')

    def get_tag_values_from_list(self, tag_list, tag_name):
        prefix = 'a_{}_'.format(tag_name)
        rv = set()
        for tag in tag_list:
            if tag.startswith(prefix):
                tag_value = tag[len(prefix):]
                if tag_name == 'prj' and tag_value in self.SYSTEM_PRJ_TAGS:
                    continue
                rv.add(tag_value)
        return rv

    def process(self, ctx):
        old_orthogonal_tags = self.operation.context['old_orthogonal_tags']
        if 'prj' in old_orthogonal_tags and 'ctype' in old_orthogonal_tags:
            # Good case, use it
            self.operation.context['prj'] = old_orthogonal_tags['prj']
            self.operation.context['ctype'] = old_orthogonal_tags['ctype']
            return self.next_state
        local_tags = {
            'prj': set(),
            'ctype': set()
        }
        all_tags = {
            'prj': set(),
            'ctype': set()
        }
        for group in self.operation.context['gencfg_groups']:
            for instance in self._gencfg_client.do_list_group_instances_data(group['name'], group['release']):
                dc = (instance.get('dc') or instance['location']).upper()
                instance_tags = instance.get('tags', ())
                for tag_name in ('ctype', 'prj'):
                    all_tags[tag_name].update(self.get_tag_values_from_list(instance_tags, tag_name))
                    if dc == self.operation.pb.order.content.migrate_from_gencfg_to_yp_lite.allocation_request.location:
                        local_tags[tag_name].update(self.get_tag_values_from_list(instance_tags, tag_name))
        instance_tags_descriptions = {}
        instance_tags_candidates = {}
        for tag_name in ('ctype', 'prj'):
            if len(local_tags[tag_name]) == 1:
                self.operation.context[tag_name] = list(local_tags[tag_name])[0]
            elif len(all_tags[tag_name]) == 1:
                self.operation.context[tag_name] = list(all_tags[tag_name])[0]
            else:
                candidates = local_tags[tag_name] or all_tags[tag_name]
                instance_tags_descriptions[tag_name] = ('Multiple tags for {} are found: ("{}"), please choose which '
                                                        'one to use'.format(tag_name, '", "'.join(candidates)))
                instance_tags_candidates[tag_name] = candidates
        if instance_tags_descriptions:
            self.operation.context['instance_tags_descriptions'] = instance_tags_descriptions
            self.operation.context['instance_tags_candidates'] = instance_tags_candidates
            return self.next_state_filling_tags
        return self.next_state


class FillingRemainingTags(GencfgMigrationOrderProcessor):
    state = State.FILLING_REMAINING_TAGS
    next_state = State.DETERMINING_VIRTUAL_SERVICES
    cancelled_state = State.CANCELLING

    def _create_tags_error(self):
        f = model_pb2.BalancerOperationOrder.OrderFeedback
        return FeedbackMessage(
            pb_error_type=f.CANNOT_INFER_TAGS_ERROR,
            message=';'.join(sorted(self.operation.context['instance_tags_descriptions'].values())),
            content={'cannot_infer_tags_error': f.CannotInferTagsError(
                prj_candidates=self.operation.context['instance_tags_candidates'].get('prj', []),
                ctype_candidates=self.operation.context['instance_tags_candidates'].get('ctype', [])
            )})

    def process(self, ctx):
        fallback_pb = self.operation.pb.order.content.migrate_from_gencfg_to_yp_lite.fallback
        context = self.operation.context
        if 'ctype' not in context and fallback_pb.ctype:
            context['ctype'] = fallback_pb.ctype
        if 'prj' not in context and fallback_pb.prj:
            context['prj'] = fallback_pb.prj
        if 'ctype' not in context or 'prj' not in context:
            if 'instance_tags_candidates' in self.operation.context:
                return self._create_tags_error()
            else:
                return self.state  # backwards compat
        return self.next_state


class DeterminingVirtualServices(GencfgMigrationOrderProcessor):
    state = State.DETERMINING_VIRTUAL_SERVICES
    next_state = State.CREATING_BALANCER
    cancelled_state = State.CANCELLING

    _gencfg_client = inject.attr(gencfg_client.IGencfgClient)  # type: gencfg_client.GencfgClient

    def process(self, ctx):
        virtual_service_ids = set()
        for group in self.operation.context['gencfg_groups']:
            card = self._gencfg_client.get_group_card(group['name'], group['release'])
            virtual_service_ids.update(card.virtual_services)
        self.operation.context['virtual_service_ids'] = virtual_service_ids
        return self.next_state


class CreatingBalancer(GencfgMigrationOrderProcessor):
    state = State.CREATING_BALANCER
    next_state = State.FINISHED
    cancelled_state = State.CANCELLING

    def _make_meta_pb(self, old_balancer_pb):
        meta_pb = model_pb2.BalancerMeta()
        meta_pb.namespace_id = old_balancer_pb.meta.namespace_id
        meta_pb.auth.CopyFrom(old_balancer_pb.meta.auth)
        meta_pb.author = self.operation.pb.meta.author
        yp_cluster = self.operation.pb.order.content.migrate_from_gencfg_to_yp_lite.allocation_request.location.upper()
        meta_pb.id = self.operation.pb.order.content.migrate_from_gencfg_to_yp_lite.new_balancer_id
        meta_pb.location.type = meta_pb.location.YP_CLUSTER
        meta_pb.location.yp_cluster = yp_cluster
        return meta_pb

    def _make_order_content_pb(self, old_balancer_pb):
        self_order_pb = self.operation.pb.order.content.migrate_from_gencfg_to_yp_lite
        order_content_pb = model_pb2.BalancerOrder.Content()
        order_content_pb.allocation_request.CopyFrom(self_order_pb.allocation_request)
        order_content_pb.copy_spec_from_balancer_id = old_balancer_pb.meta.id
        order_content_pb.abc_service_id = self_order_pb.abc_service_id
        order_content_pb.activate_balancer = True
        order_content_pb.instance_tags.metaprj = 'yp'
        order_content_pb.instance_tags.ctype = self.operation.context['ctype']
        order_content_pb.instance_tags.prj = self.operation.context['prj']
        order_content_pb.allocation_request.virtual_service_ids.extend(
            sorted(self.operation.context['virtual_service_ids']))
        order_content_pb.do_not_create_user_endpoint_set = True
        return order_content_pb

    def process(self, ctx):
        old_balancer_pb = self.operation.zk_get_balancer()
        meta_pb = self._make_meta_pb(old_balancer_pb)
        order_content_pb = self._make_order_content_pb(old_balancer_pb)
        rev_index_pb = None
        for ind_pb in old_balancer_pb.meta.indices:
            if ind_pb.id == old_balancer_pb.meta.version:
                rev_index_pb = util.clone_pb(ind_pb)
                rev_index_pb.id = ''
                rev_index_pb.ClearField('ctime')
                break
        balancer_pb = self.operation.dao.create_balancer_if_missing(meta_pb, order_content_pb,
                                                                    rev_index_pb=rev_index_pb,
                                                                    login=self.operation.pb.meta.author)
        if not balancer_pb.spec.incomplete:
            self.operation.pb.spec.incomplete = False
            self.operation.dao_update()
            return self.next_state
        if balancer_pb.order.feedback.messages:
            error = balancer_pb.order.feedback.messages[0]
            if error.type == model_pb2.BalancerOrder.OrderFeedback.QUOTA_ERROR:
                raise RuntimeError(
                    "New balancer cannot be allocated because of insufficient quota: {}\n"
                    "To change the quota account, please go to the new balancer's page".format(error.text))
            else:
                raise RuntimeError('An error occurred while creating balancer:\n{}'.format(error.text))
        return self.state


class Cancelling(GencfgMigrationOrderProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    def process(self, ctx):
        self.operation.pb.spec.incomplete = False
        self.operation.dao_update()
        return self.next_state
