# coding: utf-8

import inject

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import is_spec_complete
from awacs.lib.order_processor.runner import StateRunner
from awacs.model import events, cache
from awacs.model.balancer.order.processors import get_balancer_order_processors, BalancerOrder
from awacs.model.zk import IZkStorage, ZkStorage
from infra.awacs.proto import model_pb2


class BalancerOrderCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(IZkStorage)  # type: ZkStorage

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    _state_runner = StateRunner(entity_class=BalancerOrder,
                                initial_state=BalancerOrder.states.START,
                                final_state=BalancerOrder.states.FINISH,
                                final_cancelled_state=BalancerOrder.states.CANCELLED,
                                processors=get_balancer_order_processors(),
                                processing_interval=PROCESSING_INTERVAL)

    def __init__(self, namespace_id, balancer_id):
        name = 'balancer-order-ctl("{}:{}")'.format(namespace_id, balancer_id)
        super(BalancerOrderCtl, self).__init__(name)
        self._balancer_id = balancer_id
        self._namespace_id = namespace_id
        self._pb = None  # type: model_pb2.Balancer or None

    def _accept_event(self, event):
        return (isinstance(event, events.BalancerUpdate) and
                event.pb.meta.namespace_id == self._namespace_id and
                event.pb.meta.id == self._balancer_id and
                not is_spec_complete(event.pb))

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process balancer order on start: %s', e)
        self._cache.bind(self._callback)

    def _stop(self):
        self._cache.unbind(self._callback)

    def _process(self, ctx, event=None):
        """
        :type event: events.BalancerUpdate
        :type ctx: context.OpCtx
        """
        cached_pb = self._cache.must_get_balancer(self._namespace_id, self._balancer_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if not self._pb.HasField('order'):
            raise RuntimeError('Balancer has no "order" field')

        return self._state_runner.process(ctx, self._pb)

    def _process_event(self, ctx, event):
        assert isinstance(event, events.BalancerUpdate)
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
