# coding: utf-8
import copy

import enum
import inject
import re
import six
from abc import abstractmethod, ABCMeta
from sepelib.core import config

import nanny_rpc_client
from awacs.lib import nannyrpcclient, nannyclient, staffclient
from awacs.lib.strutils import flatten_full_id2
from awacs.lib.nannyclient import NannyApiRequestException
from awacs.lib.order_processor.model import BaseProcessor, WithOrder, FeedbackMessage
from awacs.lib.ypliterpcclient import IYpLiteRpcClient, YpLiteRpcClient
from awacs.model import dao, zk, cache, util, components, external_clusters
from awacs.model.balancer.endpoints import endpoint_set_exists
from awacs.model.balancer.order.util import (
    make_nanny_service_id,
    make_easy_mode_balancer_yml,
    make_nanny_service_category,
    ONE_AND_HALF_GB,
    TEN_GB,
    ONE_GB,
    FIVE_GB,
    DEFAULT_SERVICE_OWNER_LOGINS,
    DEFAULT_SERVICE_OWNER_GROUP_IDS,
    DEFAULT_SERVICE_CONF_MANAGERS_GROUPS_IDS,
    DEFAULT_SERVICE_OPS_MANAGERS_GROUPS_IDS,
    DEFAULT_SERVICE_OBSERVERS_GROUPS_IDS,
    YP_LITE_TEMPLATE_SERVICE_ID,
    get_available_balancer_locations,
    make_easy_mode_with_domains_balancer_yml,
)
from awacs.model.namespace.util import is_dzen
from awacs.model.util import make_system_endpoint_set_id
from awacs.model.balancer.stateholder import BalancerStateHolder
from awacs.model.balancer.vector import BalancerVersion
from awacs.model.balancer.generator import get_included_full_backend_ids_from_holder
from awacs.model.validation import validate_and_parse_yaml_balancer_config
from infra.awacs.proto import model_pb2
from nanny_repo import repo_pb2
from infra.swatlib.auth import abc
from infra.swatlib.logutil import rndstr
from yp_lite_ui_repo import pod_sets_api_pb2, endpoint_sets_api_pb2, endpoint_sets_pb2


QUOTA_ERROR_RE = re.compile(r'Account "(?P<account>.+)" \(key abc:service:(?P<abc_id>\d+)\)(?P<details>.+) in segment')


class State(enum.Enum):
    START = 1
    ALLOCATING_YP_LITE_RESOURCES = 2
    GETTING_ABC_ROLE_STAFF_ID = 3
    CREATING_NANNY_SERVICE = 4
    SETTING_UP_CLEANUP_POLICY = 5
    SETTING_UP_REPLICATION_POLICY = 6
    CREATING_ENDPOINT_SET = 7
    CREATING_AWACS_BALANCER = 8
    FINISH = 9
    DEALLOCATING_YP_LITE_RESOURCES = 10
    CANCELLED = 11
    CREATING_USER_ENDPOINT_SET = 12
    VALIDATING_AWACS_BALANCER = 13
    ACTIVATING_AWACS_BALANCER = 14
    WAITING_FOR_APPROVAL_AFTER_ALLOCATION = 15
    WAITING_FOR_NEW_ABC_SERVICE_ID = 16
    REVOKING_APPROVAL_AFTER_CHANGING_ABC_SERVICE = 17
    CREATING_BALANCER_BACKEND = 18
    GETTING_VIRTUAL_SERVICE_IDS = 19
    COPYING_NANNY_SERVICE = 20
    UPDATING_COPIED_NANNY_SERVICE = 21


def get_balancer_order_processors():
    return (
        Start,
        GettingVirtualServiceIds,
        AllocatingYpLiteResources,
        WaitingForNewAbcServiceId,
        RevokingApprovalAfterChangingAbcService,
        WaitingForApprovalAfterAllocation,
        CopyingNannyService,
        UpdatingCopiedNannyService,
        GettingAbcRoleStaffId,
        CreatingNannyService,
        SettingUpCleanupPolicy,
        SettingUpReplicationPolicy,
        CreatingEndpointSet,
        CreatingUserEndpointSet,
        CreatingAwacsBalancer,
        CreatingBalancerBackend,
        ValidatingAwacsBalancer,
        ActivatingAwacsBalancer,
        DeallocatingYpLiteResources,
    )


def generate_instance_tags(namespace_id, order_instance_tags_pb=None):
    """
    :type namespace_id: six.text_type
    :type order_instance_tags_pb: Optional[model_pb2.BalancerOrder.Content.InstanceTags]
    :rtype: model_pb2.InstanceTags
    """
    return model_pb2.InstanceTags(
        itype=u'balancer',
        ctype=order_instance_tags_pb.ctype if order_instance_tags_pb and order_instance_tags_pb.ctype else u'prod',
        prj=order_instance_tags_pb.prj if order_instance_tags_pb and order_instance_tags_pb.prj else namespace_id
    )


class BalancerOrder(WithOrder):
    __slots__ = ('allocation_request_pb', 'location', 'cluster', 'nanny_service_id')

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = 'Balancer'
    states = State

    def __init__(self, pb):
        super(BalancerOrder, self).__init__(pb)
        self.allocation_request_pb = self.pb.order.content.allocation_request  # type: model_pb2.BalancerOrder.Content.LocationalYpLiteAllocationRequest
        self.location = self.allocation_request_pb.location.lower()
        if self.is_azure():
            self.cluster = external_clusters.AZURE_CLUSTERS_BY_NAME[self.location.upper()].yp_cluster
        else:
            self.cluster = self.location.upper()
        self.nanny_service_id = make_nanny_service_id(self.id, self.pb.meta.location.type)

    def zk_update(self):
        return self.zk.update_balancer(namespace_id=self.namespace_id,
                                       balancer_id=self.id)

    def dao_update(self, updated_spec_pb, rev_index_pbs):
        return self.dao.update_balancer(
            namespace_id=self.namespace_id,
            balancer_id=self.id,
            version=self.pb.meta.version,
            login=self.pb.meta.author,
            comment='Updated balancer spec',
            updated_spec_pb=updated_spec_pb,
            rev_index_pbs=rev_index_pbs,
        )

    def dao_unpause_balancer(self):
        comment = 'Unpaused balancer config updates'
        updated_transport_paused_pb = model_pb2.PausedCondition(
            value=False,
            author=util.NANNY_ROBOT_LOGIN,
            comment=comment,
        )
        updated_transport_paused_pb.mtime.GetCurrentTime()
        return self.dao.update_balancer(
            namespace_id=self.namespace_id,
            balancer_id=self.id,
            version=self.pb.meta.version,
            login=util.NANNY_ROBOT_LOGIN,
            comment=comment,
            updated_transport_paused_pb=updated_transport_paused_pb,
        )

    def is_easy_mode_with_domains(self):
        return self.pb.order.content.mode == model_pb2.BalancerOrder.Content.EASY_MODE_WITH_DOMAINS

    def is_azure(self):
        return self.pb.order.content.cloud_type == model_pb2.CT_AZURE

    def can_change_abc_group(self):
        return self.pb.order.progress.state.id == State.ALLOCATING_YP_LITE_RESOURCES.name


class BalancerOrderProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = ('balancer',)

    def __init__(self, entity):
        super(BalancerOrderProcessor, self).__init__(entity)
        self.balancer = self.entity  # type: BalancerOrder


class Start(BalancerOrderProcessor):
    __slots__ = ()
    state = State.START
    next_state = State.ALLOCATING_YP_LITE_RESOURCES
    next_state_get_virtual_service_ids = State.GETTING_VIRTUAL_SERVICE_IDS
    cancelled_state = State.DEALLOCATING_YP_LITE_RESOURCES

    def process(self, ctx):
        assert self.balancer.allocation_request_pb.type == self.balancer.allocation_request_pb.PRESET, \
            'Allocation request without preset is not supported'

        if self.balancer.is_azure():
            available_balancer_locations = [name.lower() for name in external_clusters.AZURE_CLUSTERS_BY_NAME]
        else:
            available_balancer_locations = get_available_balancer_locations()
        assert self.balancer.location in available_balancer_locations, \
            'Unknown location "{}"'.format(self.balancer.location)
        self.balancer.context['pre_allocation_id'] = '{}:{}'.format(rndstr(6), self.balancer.id)

        src_balancer_id = self.balancer.pb.order.content.copy_nanny_service.balancer_id
        if not src_balancer_id:
            return self.next_state
        balancer_pb = self.balancer.zk.must_get_balancer(self.balancer.namespace_id, src_balancer_id)
        src_nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        assert src_nanny_service_id, 'Balancer "{}" has no Nanny Service'.format(src_balancer_id)
        self.balancer.context['copy_nanny_service_id'] = src_nanny_service_id
        return self.next_state_get_virtual_service_ids


class GettingVirtualServiceIds(BalancerOrderProcessor):
    __slots__ = ()
    state = State.GETTING_VIRTUAL_SERVICE_IDS
    next_state = State.ALLOCATING_YP_LITE_RESOURCES
    cancelled_state = State.DEALLOCATING_YP_LITE_RESOURCES

    _yp_lite_rpc_client = inject.attr(IYpLiteRpcClient)  # type: YpLiteRpcClient
    _nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient

    def process(self, ctx):
        service_id = self.balancer.context['copy_nanny_service_id']
        src_runtime_attrs = self._nanny_client.get_service_runtime_attrs(service_id)['content']
        pods = src_runtime_attrs.get('instances', {}).get('yp_pod_ids', {}).get('pods')
        if not pods:
            return self.next_state
        req = pod_sets_api_pb2.GetPodRequest(pod_id=pods[0]['pod_id'], cluster=pods[0]['cluster'])
        pod_pb = self._yp_lite_rpc_client.get_pod(req).pod
        for ip6_address_request in pod_pb.spec.ip6_address_requests:
            if ip6_address_request.vlan_id == 'backbone':
                self.balancer.context['virtual_service_ids'] = ip6_address_request.virtual_service_ids
                break
        return self.next_state


class AllocatingYpLiteResources(BalancerOrderProcessor):
    __slots__ = ()
    state = State.ALLOCATING_YP_LITE_RESOURCES
    next_state = State.GETTING_ABC_ROLE_STAFF_ID
    next_state_copy_service = State.COPYING_NANNY_SERVICE
    next_state_approval = State.WAITING_FOR_APPROVAL_AFTER_ALLOCATION
    next_state_quota_error = State.WAITING_FOR_NEW_ABC_SERVICE_ID
    cancelled_state = State.DEALLOCATING_YP_LITE_RESOURCES

    _yp_lite_rpc_client = inject.attr(IYpLiteRpcClient)  # type: YpLiteRpcClient

    def _fill_allocation_request(self, pb):
        """
        :type pb: pod_sets_api_pb2.AllocationRequest
        """
        preset_pb = self.balancer.allocation_request_pb.preset
        pb.replicas = preset_pb.instances_count
        pb.snapshots_count = 5
        pb.root_fs_quota_megabytes = 512
        pb.work_dir_quota_megabytes = 512
        pb.root_volume_storage_class = 'hdd'
        # See https://st.yandex-team.ru/SWAT-7223 for details about HDD bandwidth guarantees and limits
        pb.root_bandwidth_guarantee_megabytes_per_sec = 5
        pb.root_bandwidth_limit_megabytes_per_sec = 5
        pb.network_macro = self.balancer.allocation_request_pb.network_macro
        if preset_pb.type == preset_pb.NANO:
            coeff = 1
        elif preset_pb.type == preset_pb.MICRO:
            coeff = 2
        elif preset_pb.type == preset_pb.SMALL:
            coeff = 4
        elif preset_pb.type == preset_pb.MEDIUM:
            coeff = 8
        else:
            raise AssertionError('Unsupported preset type {}'.format(preset_pb.type))

        if preset_pb.type == preset_pb.NANO:
            pb.memory_guarantee_megabytes = ONE_AND_HALF_GB  # https://st.yandex-team.ru/SWAT-6283#5ddd09a2a2b79e001c66a01c
            logs_volume_pb = pb.persistent_volumes.add(
                mount_point='/logs',
                disk_quota_megabytes=TEN_GB,
                storage_class='hdd')
        else:
            pb.memory_guarantee_megabytes = coeff * ONE_GB
            logs_volume_pb = pb.persistent_volumes.add(
                mount_point='/logs',
                disk_quota_megabytes=coeff * FIVE_GB,
                storage_class='hdd')

        if preset_pb.type == preset_pb.NANO:
            logs_volume_pb.bandwidth_guarantee_megabytes_per_sec = 10
            logs_volume_pb.bandwidth_limit_megabytes_per_sec = 25
        elif preset_pb.type == preset_pb.MICRO:
            coeff = 2
            logs_volume_pb.bandwidth_guarantee_megabytes_per_sec = 10
            logs_volume_pb.bandwidth_limit_megabytes_per_sec = 25
        elif preset_pb.type == preset_pb.SMALL:
            coeff = 4
            logs_volume_pb.bandwidth_guarantee_megabytes_per_sec = 15
            logs_volume_pb.bandwidth_limit_megabytes_per_sec = 35
        elif preset_pb.type == preset_pb.MEDIUM:
            coeff = 8
            logs_volume_pb.bandwidth_guarantee_megabytes_per_sec = 25
            logs_volume_pb.bandwidth_limit_megabytes_per_sec = 55
        else:
            raise AssertionError('Unsupported preset type {}'.format(preset_pb.type))

        # https://st.yandex-team.ru/SWAT-7395
        logs_volume_pb.bandwidth_guarantee_megabytes_per_sec -= 1
        logs_volume_pb.bandwidth_limit_megabytes_per_sec -= 1
        pb.persistent_volumes.add(
            mount_point='/awacs',
            disk_quota_megabytes=120,
            storage_class='hdd',
            bandwidth_guarantee_megabytes_per_sec=1,
            bandwidth_limit_megabytes_per_sec=1)

        pb.vcpu_guarantee = coeff * 500  # (coeff * 500) cpu
        pb.vcpu_limit = coeff * 500  # (coeff * 500) cpu
        # https://st.yandex-team.ru/SWAT-5980
        pb.sysctl_properties.add(name='net.ipv4.tcp_tw_reuse', value='1')
        pb.sysctl_properties.add(name='net.ipv4.tcp_retries2', value='8')

        if self.balancer.allocation_request_pb.virtual_service_ids:
            virtual_service_ids = self.balancer.allocation_request_pb.virtual_service_ids
        else:
            virtual_service_ids = self.balancer.context.get('virtual_service_ids')

        if virtual_service_ids:
            pb.virtual_service_ids.extend(virtual_service_ids)

        pb.labels.add(key='awacs_namespace_id', value=self.balancer.namespace_id)
        if self.balancer.pb.order.content.resource_group_label:
            pb.labels.add(key='azure_lb_rg', value=self.balancer.pb.order.content.resource_group_label)

    def _fill_antiaffinity_constraints(self, pb):
        """
        :type pb: pod_sets_api_pb2.AntiaffinityConstraints
        """
        instances_count = self.balancer.allocation_request_pb.preset.instances_count
        pb.node_max_pods = 1
        if config.get_value('run.balancer_order.disable_rack_max_pods', default=False):
            pb.rack_max_pods = 0
        else:
            if instances_count <= 100:
                pb.rack_max_pods = 1
            else:
                pb.rack_max_pods = 2

    def _fill_quota_settings(self, pb):
        """
        :type pb: pod_sets_api_pb2.ResourceQuotaSettings
        """
        pb.mode = pb.ABC_SERVICE
        pb.abc_service_id = self.balancer.pb.order.content.abc_service_id

    def _get_create_pod_set_req(self):
        req_pb = pod_sets_api_pb2.CreatePodSetRequest(
            allocation_mode=pod_sets_api_pb2.CreatePodSetRequest.PRE_ALLOCATION,
            service_id=self.balancer.nanny_service_id,
            cluster=self.balancer.cluster,
            pre_allocation_id=self.balancer.context['pre_allocation_id'],
        )
        self._fill_allocation_request(req_pb.allocation_request)
        self._fill_antiaffinity_constraints(req_pb.antiaffinity_constraints)
        self._fill_quota_settings(req_pb.quota_settings)
        return req_pb

    @staticmethod
    def _format_quota_error(error):
        match = QUOTA_ERROR_RE.search(error)
        details = None
        if match is not None:
            details = match.group('details')
            error = u'ABC service "{}" (id: {}){} in this location.'.format(
                match.group('account'), match.group('abc_id'), details)
        return error, details

    @staticmethod
    def _extract_pre_allocation_id_from_pod_set(pod_set):
        for attribute in pod_set.labels.attributes:
            if attribute.key == 'nanny_pre_allocation_id':
                return attribute.value

    def process(self, ctx):
        if self.balancer.context.get('pod_set_id') and self.balancer.context.get('pod_ids'):
            return self.next_state
        ctx.log.debug('Creating pod set %s:%s', self.balancer.location, self.balancer.nanny_service_id)
        req_pb = self._get_create_pod_set_req()
        existing_pod_set = None
        if self.balancer.is_azure():
            req_pb.node_segment_id = external_clusters.AZURE_CLUSTERS_BY_NAME[self.balancer.location.upper()].node_segment
        try:
            resp_pb = self._yp_lite_rpc_client.create_pod_set(req_pb)
        except nanny_rpc_client.exceptions.BadRequestError as e:
            # it's important to use u'' here as str(e) may fail with UnicodeEncodeError T_T
            ctx.log.info(u'Failed to create pod set, req_pb: %s, error: %s', req_pb, e)
            if six.PY2:
                error = e.message.encode('utf-8').decode('string-escape').decode('utf-8')
            else:
                try:
                    error = str(e).encode('latin-1', 'backslashreplace').decode('unicode-escape').encode('latin-1').decode('utf-8')
                except UnicodeEncodeError:
                    error = str(e)

            if 'Computing resource quota exceeded' in error:
                error, details = self._format_quota_error(error)
                self.balancer.context['quota_error_message'] = error
                self.balancer.context['quota_error_details'] = details
                f = model_pb2.BalancerOrder.OrderFeedback
                return FeedbackMessage(
                    pb_error_type=f.QUOTA_ERROR,
                    message=error,
                    content={'quota_error': f.QuotaError(details=details,
                                                         abc_service_id=self.balancer.pb.order.content.abc_service_id)})
            raise
        except nanny_rpc_client.exceptions.ConflictError as e:
            for conflict in e.conflicts:
                if conflict.object_type == "OT_POD_SET":
                    req_pb = pod_sets_api_pb2.GetPodSetRequest(
                        service_id=self.balancer.nanny_service_id,
                        cluster=self.balancer.cluster,
                    )
                    existing_pod_set = self._yp_lite_rpc_client.get_pod_set(req_pb).pod_set
                    pre_allocation_id = self._extract_pre_allocation_id_from_pod_set(existing_pod_set)
                    if pre_allocation_id != self.balancer.context['pre_allocation_id']:
                        return FeedbackMessage(
                            pb_error_type=model_pb2.BalancerOrder.OrderFeedback.POD_SET_EXISTS_ERROR,
                            message='Pod set "{}" already exists'.format(conflict.object_id)
                        )
            if not existing_pod_set:
                raise
        if existing_pod_set is not None:
            pod_ids = self._yp_lite_rpc_client.list_all_pod_ids(
                service_id=self.balancer.nanny_service_id,
                cluster=self.balancer.cluster,
            )
            self.balancer.context['pod_set_id'] = existing_pod_set.meta.id
            self.balancer.context['pod_ids'] = pod_ids
        else:
            self.balancer.context['pod_set_id'] = resp_pb.pod_set_id
            self.balancer.context['pod_ids'] = list(resp_pb.pod_ids)
            ctx.log.debug('Created pod set %s:%s', self.balancer.location, self.balancer.nanny_service_id)
        if self.balancer.pb.order.content.wait_for_approval_after_allocation:
            return self.next_state_approval
        if self.balancer.context.get('copy_nanny_service_id'):
            return self.next_state_copy_service
        return self.next_state


# deprecated
class WaitingForNewAbcServiceId(BalancerOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_NEW_ABC_SERVICE_ID
    next_state = State.ALLOCATING_YP_LITE_RESOURCES
    cancelled_state = State.DEALLOCATING_YP_LITE_RESOURCES

    def process(self, ctx):
        return self.next_state


# deprecated
class RevokingApprovalAfterChangingAbcService(BalancerOrderProcessor):
    __slots__ = ()
    state = State.REVOKING_APPROVAL_AFTER_CHANGING_ABC_SERVICE
    next_state = State.ALLOCATING_YP_LITE_RESOURCES
    cancelled_state = State.DEALLOCATING_YP_LITE_RESOURCES

    def process(self, ctx):
        return self.next_state


class WaitingForApprovalAfterAllocation(BalancerOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_APPROVAL_AFTER_ALLOCATION
    next_state = State.GETTING_ABC_ROLE_STAFF_ID
    cancelled_state = State.DEALLOCATING_YP_LITE_RESOURCES

    def process(self, ctx):
        if self.balancer.pb.order.approval.after_allocation:
            return self.next_state
        return self.state


class GettingAbcRoleStaffId(BalancerOrderProcessor):
    __slots__ = ()
    state = State.GETTING_ABC_ROLE_STAFF_ID
    next_state = State.CREATING_NANNY_SERVICE
    cancelled_state = State.DEALLOCATING_YP_LITE_RESOURCES

    _staff_client = inject.attr(staffclient.IStaffClient)  # type: staffclient.StaffClient
    _abc_client = inject.attr(abc.IAbcClient)  # type: abc.AbcClient

    @staticmethod
    def _choose_abc_group(abc_roles, abc_service_id):
        role_id_by_scope = {item["role_scope"]: item["id"] for item in abc_roles}
        return role_id_by_scope.get('administration') or role_id_by_scope.get('development', abc_service_id)

    def process(self, ctx):
        if self.balancer.context.get('abc_role_staff_id'):
            return self.next_state
        abc_service_id = self.balancer.pb.order.content.abc_service_id
        abc_roles = self._staff_client.get_abc_roles(abc_service_id)
        self.balancer.context['abc_role_staff_id'] = six.text_type(self._choose_abc_group(abc_roles, abc_service_id))
        return self.next_state


class CopyingNannyService(BalancerOrderProcessor):
    __slots__ = ()
    state = State.COPYING_NANNY_SERVICE
    next_state = State.UPDATING_COPIED_NANNY_SERVICE
    cancelled_state = None

    _nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient

    def process(self, ctx):
        if self.balancer.context.get('created_nanny_service_id'):
            return self.next_state
        self._nanny_client.copy_service(
            service_id=self.balancer.context['copy_nanny_service_id'],
            new_service_id=self.balancer.nanny_service_id,
            comment='Copied during new balancer order',
            category=make_nanny_service_category(namespace_id=self.balancer.namespace_id),
            description='awacs-powered balancer for {} in {}'.format(self.balancer.id, self.balancer.location.upper()),
            yp_cluster=self.balancer.cluster.upper()
        )
        self.balancer.context['created_nanny_service_id'] = self.balancer.nanny_service_id
        return self.next_state


class UpdatingCopiedNannyService(BalancerOrderProcessor):
    __slots__ = ()
    state = State.UPDATING_COPIED_NANNY_SERVICE
    next_state = State.CREATING_ENDPOINT_SET
    cancelled_state = None

    _nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient

    def _update_info_attrs(self, base_info_attrs):
        """
        :param dict base_info_attrs:
        :rtype: dict
        """
        info_attrs = copy.deepcopy(base_info_attrs)
        new_labels = []
        for label in info_attrs['content']['labels']:
            if label['key'] in ('dc', 'geo'):
                new_labels.append({'key': label['key'], 'value': self.balancer.cluster.lower()})
            elif label['key'] == 'pre_allocation_id':
                new_labels.append({'key': label['key'], 'value': self.balancer.context.get('pre_allocation_id', '')})
            else:
                new_labels.append(label)
        info_attrs['content']['labels'] = new_labels
        info_attrs['content']['tickets_integration'] = {'service_release_rules': []}

        juggler_settings = info_attrs['content']['monitoring_settings']['juggler_settings']['content']
        for juggler_host in juggler_settings.get('juggler_hosts', ()):
            juggler_host['name'] = self.balancer.nanny_service_id
        for active_check in juggler_settings.get('active_checks', ()):
            for check in active_check.get('checks', ()):
                check['juggler_host_name'] = self.balancer.nanny_service_id
            for check in active_check.get('passive_checks', ()):
                check['juggler_host_name'] = self.balancer.nanny_service_id

        return info_attrs

    def _update_runtime_attrs(self, base_runtime_attrs):
        """
        :param dict base_runtime_attrs:
        :rtype: dict
        """
        runtime_attrs = copy.deepcopy(base_runtime_attrs)
        pods = [{'cluster': self.balancer.cluster, 'pod_id': pod_id} for pod_id in self.balancer.context['pod_ids']]
        runtime_attrs['content']['instances']['chosen_type'] = 'YP_POD_IDS'
        runtime_attrs['content']['instances']['yp_pod_ids']['pods'] = pods
        return runtime_attrs

    def process(self, ctx):
        info_attrs = self._nanny_client.get_service_info_attrs(self.balancer.nanny_service_id)
        runtime_attrs = self._nanny_client.get_service_runtime_attrs(self.balancer.nanny_service_id)
        ctx.log.debug('Updating Nanny service %s', self.balancer.nanny_service_id)
        try:
            self._nanny_client.update_service(
                service_id=self.balancer.nanny_service_id,
                comment='Updated info labels and runtime instances',
                info_attrs=self._update_info_attrs(info_attrs),
                runtime_attrs=self._update_runtime_attrs(runtime_attrs),
            )
        except NannyApiRequestException as e:
            if e.response.status_code == 400:
                ctx.log.exception('Got 400 BAD REQUEST while updating Nanny service: %s', e.response.json())
            raise
        ctx.log.debug('Updated Nanny service %s', self.balancer.nanny_service_id)
        return self.next_state


class CreatingNannyService(BalancerOrderProcessor):
    __slots__ = ()
    state = State.CREATING_NANNY_SERVICE
    next_state = State.SETTING_UP_CLEANUP_POLICY
    cancelled_state = None

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient
    _nanny_rpc_client = inject.attr(nannyrpcclient.INannyRpcClient)  # type: nannyrpcclient.NannyRpcClient

    def _make_info_attrs(self, base_info_attrs, namespace_pb):
        """
        :param dict base_info_attrs:
        :rtype: dict
        """
        info_attrs = copy.deepcopy(base_info_attrs)
        info_attrs['category'] = make_nanny_service_category(namespace_id=self.balancer.namespace_id)
        info_attrs['abc_group'] = self.balancer.pb.order.content.abc_service_id
        info_attrs['desc'] = 'awacs-powered balancer for {} in {}'.format(self.balancer.id, self.balancer.cluster)
        info_attrs['yp_cluster'] = self.balancer.cluster.upper()
        instance_tags_pb = generate_instance_tags(self.balancer.namespace_id,
                                                  self.balancer.pb.order.content.instance_tags)
        info_attrs['labels'] = [
            {'key': 'geo', 'value': self.balancer.cluster.lower()},
            {'key': 'ctype', 'value': instance_tags_pb.ctype},
            {'key': 'itype', 'value': instance_tags_pb.itype},
            {'key': 'prj', 'value': instance_tags_pb.prj},
            {'key': 'dc', 'value': self.balancer.location},
            {'key': 'pre_allocation_id', 'value': self.balancer.context.get('pre_allocation_id', '')}
        ]

        info_attrs['tickets_integration'] = {'service_release_rules': []}

        alerting_enabled = namespace_pb.HasField('spec') and namespace_pb.spec.HasField('alerting')
        alerting_will_be_enabled = namespace_pb.order.content.WhichOneof('alerting') is not None

        spec_group_ids = [six.text_type(s) for s in namespace_pb.spec.alerting.juggler_raw_downtimers.staff_group_ids]
        order_group_id = namespace_pb.order.content.alerting_simple_settings.notify_staff_group_id
        if alerting_enabled and spec_group_ids:
            notification_groups = spec_group_ids
        elif alerting_will_be_enabled and order_group_id:
            notification_groups = [six.text_type(order_group_id)]
        else:
            notification_groups = [self.balancer.context['abc_role_staff_id']]

        if alerting_enabled or alerting_will_be_enabled:
            info_attrs['monitoring_settings']['deploy_monitoring']['is_enabled'] = False
            del info_attrs['monitoring_settings']['deploy_monitoring']['content']
        else:
            deploy_monitoring_settings = info_attrs['monitoring_settings']['deploy_monitoring']['content']
            deploy_monitoring_settings['alert_methods'] = ['telegram', 'email']
            deploy_monitoring_settings['responsible']['groups'] = notification_groups

        juggler_settings = info_attrs['monitoring_settings']['juggler_settings']['content']

        juggler_settings['juggler_tags'].append(self.balancer.namespace_id)
        juggler_settings['juggler_hosts'][0]['name'] = self.balancer.nanny_service_id

        passive_check_params = juggler_settings['active_checks'][0]['passive_checks']
        for item in passive_check_params:
            item['juggler_host_name'] = self.balancer.nanny_service_id
            for notification in item['notifications']:
                if notification['on_status_change']['method'] == ['email']:
                    notification['on_status_change']['users']['logins'] = []
                    notification['on_status_change']['users']['groups'] = notification_groups

        return info_attrs

    def _make_runtime_attrs(self, base_runtime_attrs):
        """
        :param dict base_runtime_attrs:
        :rtype: dict
        """
        runtime_attrs = copy.deepcopy(base_runtime_attrs)
        pods = [{'cluster': self.balancer.cluster, 'pod_id': pod_id} for pod_id in self.balancer.context['pod_ids']]
        runtime_attrs['instances']['chosen_type'] = 'YP_POD_IDS'
        order_instance_tags_pb = self.balancer.pb.order.content.instance_tags
        instance_tags_pb = generate_instance_tags(self.balancer.namespace_id,
                                                  order_instance_tags_pb=order_instance_tags_pb)
        runtime_attrs['instances']['yp_pod_ids'] = {
            'pods': pods,
            'orthogonal_tags': {
                'ctype': instance_tags_pb.ctype,
                'itype': 'balancer',
                'prj': instance_tags_pb.prj,
                'metaprj': order_instance_tags_pb.metaprj or 'unknown',  # ??? WTF metaprj is required
            },
        }
        return runtime_attrs

    def _make_auth_attrs(self, base_auth_attrs, namespace_pb):
        """
        :param dict base_auth_attrs:
        :rtype: dict
        """
        auth_attrs = copy.deepcopy(base_auth_attrs)
        owner_logins = list(set(self.balancer.pb.meta.auth.staff.owners.logins) |
                            set(namespace_pb.meta.auth.staff.owners.logins))
        owner_groups = [self.balancer.context['abc_role_staff_id']]
        owners = {
            'logins': DEFAULT_SERVICE_OWNER_LOGINS + owner_logins,
            'groups': DEFAULT_SERVICE_OWNER_GROUP_IDS + owner_groups,
        }
        auth_attrs['owners'] = owners
        auth_attrs['conf_managers']['groups'] = DEFAULT_SERVICE_CONF_MANAGERS_GROUPS_IDS + owner_groups
        auth_attrs['ops_managers']['groups'] = DEFAULT_SERVICE_OPS_MANAGERS_GROUPS_IDS + owner_groups
        auth_attrs['observers']['groups'] = DEFAULT_SERVICE_OBSERVERS_GROUPS_IDS
        return auth_attrs

    def _create_nanny_service(self, ctx, base_info_attrs, base_runtime_attrs, base_auth_attrs):
        """
        :type base_info_attrs: dict
        :type base_runtime_attrs: dict
        :type base_auth_attrs: dict
        """
        namespace_pb = self._cache.must_get_namespace(self.balancer.namespace_id)
        info_attrs = self._make_info_attrs(base_info_attrs, namespace_pb)
        runtime_attrs = self._make_runtime_attrs(base_runtime_attrs)
        auth_attrs = self._make_auth_attrs(base_auth_attrs, namespace_pb)
        try:
            self._nanny_client.create_service(
                service_id=self.balancer.nanny_service_id,
                comment='Created from L7 balancer template service "{}"'.format(YP_LITE_TEMPLATE_SERVICE_ID),
                info_attrs=info_attrs,
                runtime_attrs=runtime_attrs,
                auth_attrs=auth_attrs
            )
        except NannyApiRequestException as e:
            if e.response.status_code == 400:
                try:
                    info_attrs = self._nanny_client.get_service_info_attrs(self.balancer.nanny_service_id)
                except NannyApiRequestException:
                    pass
                else:
                    pre_allocation_id = self.balancer.context.get('pre_allocation_id')
                    for label in info_attrs['content'].get('labels', ()):
                        if label['key'] == 'pre_allocation_id' and label['value'] == pre_allocation_id:
                            # Service already was created, but we miss it from context
                            return
                ctx.log.exception('Got 400 BAD REQUEST while creating Nanny service: %s', e.response.json())
            raise

    def process(self, ctx):
        if self.balancer.context.get('created_nanny_service_id'):
            return self.next_state
        base_info_attrs = self._nanny_client.get_service_info_attrs(YP_LITE_TEMPLATE_SERVICE_ID)['content']
        base_runtime_attrs = self._nanny_client.get_service_runtime_attrs(YP_LITE_TEMPLATE_SERVICE_ID)['content']
        base_auth_attrs = self._nanny_client.get_service_auth_attrs(YP_LITE_TEMPLATE_SERVICE_ID)['content']
        ctx.log.debug('Creating Nanny service %s', self.balancer.nanny_service_id)
        self._create_nanny_service(
            ctx=ctx,
            base_info_attrs=base_info_attrs,
            base_runtime_attrs=base_runtime_attrs,
            base_auth_attrs=base_auth_attrs)
        ctx.log.debug('Created Nanny service %s', self.balancer.nanny_service_id)
        self.balancer.context['created_nanny_service_id'] = self.balancer.nanny_service_id
        return self.next_state


class SettingUpCleanupPolicy(BalancerOrderProcessor):
    __slots__ = ()
    state = State.SETTING_UP_CLEANUP_POLICY
    next_state = State.SETTING_UP_REPLICATION_POLICY
    cancelled_state = None

    _nanny_rpc_client = inject.attr(nannyrpcclient.INannyRpcClient)  # type: nannyrpcclient.NannyRpcClient

    def process(self, ctx):
        ctx.log.debug('Setting up a cleanup policy for %s', self.balancer.nanny_service_id)
        resp_pb = self._nanny_rpc_client.get_cleanup_policy(self.balancer.nanny_service_id)
        meta_pb = repo_pb2.CleanupPolicyMeta(
            id=self.balancer.nanny_service_id,
            version=resp_pb.policy.meta.version)
        spec_pb = repo_pb2.CleanupPolicySpec(
            type=repo_pb2.CleanupPolicySpec.SIMPLE_COUNT_LIMIT,
            simple_count_limit=repo_pb2.CleanupPolicySimpleCountLimit(snapshots_count=1, stalled_ttl='P7D'))
        self._nanny_rpc_client.update_cleanup_policy(meta_pb, spec_pb)
        ctx.log.debug('Set up a cleanup policy for %s', self.balancer.nanny_service_id)
        return self.next_state


class SettingUpReplicationPolicy(BalancerOrderProcessor):
    __slots__ = ()
    state = State.SETTING_UP_REPLICATION_POLICY
    next_state = State.CREATING_ENDPOINT_SET
    cancelled_state = None

    _nanny_rpc_client = inject.attr(nannyrpcclient.INannyRpcClient)  # type: nannyrpcclient.NannyRpcClient

    def process(self, ctx):
        ctx.log.debug('Setting up a replication policy for %s', self.balancer.nanny_service_id)
        resp_pb = self._nanny_rpc_client.get_replication_policy(self.balancer.nanny_service_id)
        meta_pb = repo_pb2.ReplicationPolicyMeta(
            id=self.balancer.nanny_service_id,
            version=resp_pb.policy.meta.version)
        spec_pb = repo_pb2.ReplicationPolicySpec(
            replication_method=repo_pb2.ReplicationPolicySpec.MOVE,
            involuntary_replication_choice=repo_pb2.ReplicationPolicySpec.ENABLED,
            maintenance_kind=repo_pb2.ReplicationPolicySpec.MK_ALL,
            disruption_budget_kind=repo_pb2.ReplicationPolicySpec.MIXED,
            max_tolerable_downtime_seconds=3600,
            max_unavailable=1,
            rate_limit=repo_pb2.RateLimit(
                delay_seconds=43200,  # 12 hours
            ),
        )
        self._nanny_rpc_client.update_replication_policy(meta_pb, spec_pb)
        ctx.log.debug('Set up a replication policy for %s', self.balancer.nanny_service_id)
        return self.next_state


class CreatingEndpointSetBase(six.with_metaclass(ABCMeta, BalancerOrderProcessor)):
    __slots__ = ()
    state = next_state = cancelled_state = NotImplemented

    _yp_lite_rpc_client = inject.attr(IYpLiteRpcClient)  # type: YpLiteRpcClient

    @abstractmethod
    def _get_endpoint_set_id(self):
        raise NotImplementedError

    @abstractmethod
    def _do_process(self, ctx):
        raise NotImplementedError

    def do_not_create(self):
        return False

    def _endpoint_set_exists(self):
        return endpoint_set_exists(
            endpoint_set_id=self._get_endpoint_set_id(),
            cluster=self.balancer.cluster,
            yp_lite_rpc_client=self._yp_lite_rpc_client,
        )

    def _create_endpoint_set(self, ownership):
        req_pb = endpoint_sets_api_pb2.CreateEndpointSetRequest()
        req_pb.cluster = self.balancer.cluster
        req_pb.meta.id = self._get_endpoint_set_id()
        req_pb.meta.service_id = self.balancer.nanny_service_id
        req_pb.meta.ownership = ownership
        req_pb.spec.protocol = 'TCP'
        req_pb.spec.port = 80
        req_pb.spec.description = 'Created by awacs'
        return self._yp_lite_rpc_client.create_endpoint_set(req_pb).endpoint_set.meta.id

    def process(self, ctx):
        if self.do_not_create():
            return self.next_state
        if self._endpoint_set_exists():
            return self.next_state

        self._do_process(ctx)
        return self.next_state


class CreatingEndpointSet(CreatingEndpointSetBase):
    state = State.CREATING_ENDPOINT_SET
    next_state = State.CREATING_USER_ENDPOINT_SET
    cancelled_state = None

    def _get_endpoint_set_id(self):
        return make_system_endpoint_set_id(self.balancer.nanny_service_id)

    def _do_process(self, ctx):
        ctx.log.debug('creating system endpoint set %s:%s', self.balancer.location, self._get_endpoint_set_id())
        self.balancer.context['system_endpoint_set_id'] = self._create_endpoint_set(
            ownership=endpoint_sets_pb2.EndpointSetMeta.SYSTEM)
        ctx.log.debug('created system endpoint set for %s', self.balancer.nanny_service_id)


class CreatingUserEndpointSet(CreatingEndpointSetBase):
    state = State.CREATING_USER_ENDPOINT_SET
    next_state = State.CREATING_AWACS_BALANCER
    cancelled_state = None

    def _get_endpoint_set_id(self):
        return self.balancer.nanny_service_id

    def do_not_create(self):
        return self.balancer.pb.order.content.do_not_create_user_endpoint_set

    def _do_process(self, ctx):
        ctx.log.debug('creating endpoint set %s:%s', self.balancer.location, self._get_endpoint_set_id())
        self.balancer.context['endpoint_set_id'] = self._create_endpoint_set(
            ownership=endpoint_sets_pb2.EndpointSetMeta.USER)
        ctx.log.debug('created endpoint set for %s', self.balancer.nanny_service_id)


class CreatingAwacsBalancer(BalancerOrderProcessor):
    __slots__ = ()
    state = State.CREATING_AWACS_BALANCER
    next_state = State.CREATING_BALANCER_BACKEND
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def _create_new_spec(self):
        spec_pb = model_pb2.BalancerSpec()
        spec_pb.config_transport.nanny_static_file.service_id = self.balancer.nanny_service_id
        spec_pb.yandex_balancer.template_engine = model_pb2.NONE

        content_pb = self.balancer.pb.order.content  # type: model_pb2.BalancerOrder.Content
        instance_tags_pb = generate_instance_tags(self.balancer.namespace_id,
                                                  order_instance_tags_pb=content_pb.instance_tags)
        spec_pb.config_transport.nanny_static_file.instance_tags.CopyFrom(instance_tags_pb)
        spec_pb.env_type = content_pb.env_type
        spec_pb.ctl_version = 6

        required_component_str_types = config.get_value('run.balancer_order.required_component_types', default=[])
        removed_component_str_types = config.get_value('run.balancer_order.removed_component_types', default=[])
        ignored_component_str_types = config.get_value('run.balancer_order.ignored_component_types', default=[])

        for component, component_pb in components.iter_balancer_components(spec_pb.components):
            if component.str_type in required_component_str_types:
                component_pb.state = component_pb.SET
                component_pb.version = component.must_get_default_version(self.cache, self.balancer.cluster)
            elif component.str_type in removed_component_str_types:
                component_pb.state = component_pb.REMOVED
                component_pb.version = ''
            elif component.str_type not in ignored_component_str_types:
                version = component.get_default_version(self.cache, self.balancer.cluster)
                if version:
                    component_pb.state = component_pb.SET
                    component_pb.version = version

        spec_pb.yandex_balancer.mode = spec_pb.yandex_balancer.EASY_MODE
        if self.balancer.is_easy_mode_with_domains():
            spec_pb.yandex_balancer.yaml = make_easy_mode_with_domains_balancer_yml(is_https=content_pb.cert_id)
        else:
            spec_pb.yandex_balancer.yaml = make_easy_mode_balancer_yml(content_pb.cert_id)
        return spec_pb

    def process(self, ctx):
        balancer_id_with_source_spec = self.balancer.pb.order.content.copy_spec_from_balancer_id
        if balancer_id_with_source_spec:
            ctx.log.debug('Copying balancer spec from balancer %s:%s',
                          self.balancer.namespace_id, balancer_id_with_source_spec)
            spec_pb = self.balancer.zk.must_get_balancer(namespace_id=self.balancer.namespace_id,
                                                         balancer_id=balancer_id_with_source_spec).spec
            spec_pb.config_transport.nanny_static_file.service_id = self.balancer.nanny_service_id
            if self.balancer.pb.order.content.HasField('instance_tags'):
                instance_tags_pb = self.balancer.pb.order.content.instance_tags
                if instance_tags_pb.ctype:
                    spec_pb.config_transport.nanny_static_file.instance_tags.ctype = instance_tags_pb.ctype
                if instance_tags_pb.prj:
                    spec_pb.config_transport.nanny_static_file.instance_tags.prj = instance_tags_pb.prj
        else:
            ctx.log.debug('Creating new balancer spec')
            spec_pb = self._create_new_spec()

        namespace_pb = self.balancer.zk.must_get_namespace(namespace_id=self.balancer.namespace_id)
        if is_dzen(namespace_pb):
            spec_pb.custom_service_settings.service = spec_pb.custom_service_settings.DZEN

        holder = validate_and_parse_yaml_balancer_config(
            spec_pb,
            full_balancer_id=(self.balancer.namespace_id, self.balancer.id),
            namespace_pb=self.cache.must_get_namespace(self.balancer.namespace_id))

        namespace_id = self.balancer.pb.meta.namespace_id
        included_full_backend_ids = get_included_full_backend_ids_from_holder(namespace_id, holder)
        rev_index_pb = model_pb2.RevisionGraphIndex()
        rev_index_pb.included_backend_ids.extend(flatten_full_id2(f_id) for f_id in included_full_backend_ids)

        self.balancer.dao_update(updated_spec_pb=spec_pb, rev_index_pbs=[rev_index_pb])
        return self.next_state


class CreatingBalancerBackend(BalancerOrderProcessor):
    __slots__ = ()
    state = State.CREATING_BALANCER_BACKEND
    next_state = State.VALIDATING_AWACS_BALANCER
    next_state_finish = State.FINISH
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        meta_pb = model_pb2.BackendMeta(
            id=self.balancer.id,
            namespace_id=self.balancer.namespace_id
        )
        meta_pb.auth.type = meta_pb.auth.STAFF
        meta_pb.is_system.value = True
        meta_pb.is_system.author = util.NANNY_ROBOT_LOGIN
        meta_pb.is_system.mtime.GetCurrentTime()
        spec_pb = model_pb2.BackendSpec()
        spec_pb.selector.type = model_pb2.BackendSelector.BALANCERS
        spec_pb.selector.balancers.add(id=self.balancer.id)
        self.balancer.dao.create_backend_if_missing(meta_pb=meta_pb, spec_pb=spec_pb)

        if self.balancer.pb.order.content.activate_balancer:
            return self.next_state
        else:
            return self.next_state_finish


class ValidatingAwacsBalancer(BalancerOrderProcessor):
    __slots__ = ()
    state = State.VALIDATING_AWACS_BALANCER
    next_state = State.ACTIVATING_AWACS_BALANCER
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def _is_balancer_valid(self):
        balancer_version = BalancerVersion.from_pb(self.balancer.pb)
        balancer_state_pb = self.cache.must_get_balancer_state(self.balancer.namespace_id, self.balancer.id)
        h = BalancerStateHolder(namespace_id=self.balancer.namespace_id,
                                balancer_id=self.balancer.id,
                                balancer_state_pb=balancer_state_pb)
        return h.valid_vector.balancer_version is not None and balancer_version <= h.valid_vector.balancer_version

    def process(self, ctx):
        if not self._is_balancer_valid():
            return self.state
        return self.next_state


class ActivatingAwacsBalancer(BalancerOrderProcessor):
    state = State.ACTIVATING_AWACS_BALANCER
    next_state = State.FINISH
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient

    def _get_snapshot_to_activate(self):
        balancer_state_pb = self.cache.must_get_balancer_state(self.balancer.namespace_id, self.balancer.id)
        statuses = balancer_state_pb.balancer.statuses
        if not statuses:
            return None
        snapshots = statuses[-1].in_progress.meta.nanny_static_file.snapshots
        if not snapshots:
            return None
        snapshot_pb = snapshots[-1]
        return snapshot_pb

    def process(self, ctx):
        self.balancer.dao_unpause_balancer()
        snapshot_pb = self._get_snapshot_to_activate()
        if snapshot_pb is None:
            ctx.log.info('No in-progress snapshots found')
            return self.state
        self.nanny_client.set_snapshot_state(
            service_id=snapshot_pb.service_id,
            snapshot_id=snapshot_pb.snapshot_id,
            state='ACTIVE',
            comment='Activating L7 balancer',
            recipe='common')
        return self.next_state


class DeallocatingYpLiteResources(BalancerOrderProcessor):
    __slots__ = ()
    state = State.DEALLOCATING_YP_LITE_RESOURCES
    next_state = State.CANCELLED
    cancelled_state = None

    _yp_lite_rpc_client = inject.attr(IYpLiteRpcClient)  # type: YpLiteRpcClient

    def _pod_set_exists(self):
        req_pb = pod_sets_api_pb2.GetPodSetRequest()
        req_pb.service_id = self.balancer.nanny_service_id
        req_pb.cluster = self.balancer.cluster
        try:
            self._yp_lite_rpc_client.get_pod_set(req_pb)
            return True
        except nanny_rpc_client.exceptions.NotFoundError:
            return False

    def _remove_pod_set(self, ctx):
        pre_allocation_id = self.balancer.context.get('pre_allocation_id')
        if not pre_allocation_id:
            ctx.log.info("No pre-allocation id in order, nothing to deallocate")
            return
        req_pb = pod_sets_api_pb2.RemovePodSetRequest(
            service_id=self.balancer.nanny_service_id,
            cluster=self.balancer.cluster,
            pre_allocation_id=pre_allocation_id
        )
        try:
            self._yp_lite_rpc_client.remove_pod_set(req_pb)
        except nanny_rpc_client.exceptions.BadRequestError:
            ctx.log.warn('failed to remove pod set %s:%s, pre_allocation_id %s',
                         self.balancer.location, req_pb.service_id, req_pb.pre_allocation_id, exc_info=True)
            raise
        except nanny_rpc_client.exceptions.NotFoundError:
            ctx.log.warn('pod set %s:%s, pre_allocation_id %s not found',
                         self.balancer.location, req_pb.service_id, req_pb.pre_allocation_id, exc_info=True)
            raise

    def process(self, ctx):
        if not self._pod_set_exists():
            ctx.log.info("Pod set doesn't exist")
            return self.next_state
        self._remove_pod_set(ctx)
        return self.next_state
