import six
from sepelib.core import config as appconfig

from infra.awacs.proto import model_pb2
from awacs.wrappers import l7macro


DEFAULT_AVAILABLE_BALANCER_LOCATIONS = ('man', 'sas', 'vla', 'myt', 'iva')


def get_available_balancer_locations():
    return appconfig.get_value('run.available_balancer_locations',
                               default=DEFAULT_AVAILABLE_BALANCER_LOCATIONS)


NANNY_SERVICE_CATEGORY_TEMPLATE = '/balancer/{namespace_id}'
NANNY_SERVICE_ID_TEMPLATE = 'rtc_balancer_{balancer_id}'
AZURE_NANNY_SERVICE_ID_TEMPLATE = 'azure_balancer_{balancer_id}'
YP_LITE_TEMPLATE_SERVICE_ID = 'rtc-yp-lite-balancer-template'
AWACS_BALANCER_ID_TEMPLATE = '{namespace_id}_{location}'

DEFAULT_SERVICE_OWNER_LOGINS = []  # Use "rlcb" abc-service instead
DEFAULT_SERVICE_OWNER_GROUP_IDS = ['75815']  # https://staff-api.yandex-team.ru/v3/groups?id=75815
DEFAULT_SERVICE_CONF_MANAGERS_GROUPS_IDS = ['33712']  # https://staff-api.yandex-team.ru/v3/groups?id=33712
DEFAULT_SERVICE_OPS_MANAGERS_GROUPS_IDS = ['33712']  # https://staff-api.yandex-team.ru/v3/groups?id=33712
DEFAULT_SERVICE_OBSERVERS_GROUPS_IDS = ['75815']  # https://staff-api.yandex-team.ru/v3/groups?id=75815

ONE_GB = (1 << 10)
FIVE_GB = 5 * ONE_GB
TEN_GB = 10 * ONE_GB
ONE_AND_HALF_GB = ONE_GB + (1 << 9)

BALANCER_YML = '''l7_macro:
  version: {}
  health_check_reply: {{}}
  announce_check_reply:
    url_re: '/ping'
  http: {{}}
'''.format(l7macro.LATEST_VERSION)

BALANCER_W_HTTPS_YML = '''l7_macro:
  version: {}
  health_check_reply: {{}}
  announce_check_reply:
    url_re: '/ping'
  http: {{}}
  https:
    certs:
      - id: '%s'
'''.format(l7macro.LATEST_VERSION)

BALANCER_WITH_DOMAINS_YML = '''l7_macro:
  version: {}
  health_check_reply: {{}}
  announce_check_reply:
    url_re: /ping
  http: {{}}
  include_domains: {{}}
'''.format(l7macro.LATEST_VERSION)

BALANCER_W_HTTPS_AND_DOMAINS_YML = '''l7_macro:
  version: {}
  health_check_reply: {{}}
  announce_check_reply:
    url_re: /ping
  http: {{}}
  https: {{}}
  include_domains: {{}}
'''.format(l7macro.LATEST_VERSION)


def make_nanny_service_id(balancer_id, location_type):
    """
    :type balancer_id: six.text_type
    :type location_type: model_pb2.BalancerMeta.Location.Type
    :rtype: six.text_type
    """
    if location_type == model_pb2.BalancerMeta.Location.AZURE_CLUSTER:
        return AZURE_NANNY_SERVICE_ID_TEMPLATE.format(balancer_id=balancer_id.replace('.', '_')).lower()
    return NANNY_SERVICE_ID_TEMPLATE.format(balancer_id=balancer_id.replace('.', '_')).lower()


def make_nanny_service_category(namespace_id):
    """
    :type namespace_id: six.text_type
    :rtype: six.text_type
    """
    return NANNY_SERVICE_CATEGORY_TEMPLATE.format(namespace_id=namespace_id)


def make_awacs_balancer_id(namespace_id, location):
    return AWACS_BALANCER_ID_TEMPLATE.format(namespace_id=namespace_id, location=location.lower())


def make_easy_mode_balancer_yml(cert_id):
    return BALANCER_W_HTTPS_YML % (cert_id,) if cert_id else BALANCER_YML


def make_easy_mode_with_domains_balancer_yml(is_https):
    return BALANCER_W_HTTPS_AND_DOMAINS_YML if is_https else BALANCER_WITH_DOMAINS_YML
