import inject

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import needs_removal
from awacs.lib.order_processor.runner import StateRunner
from awacs.model import events, cache
from awacs.model.balancer.removal.processors import BalancerRemoval


class BalancerRemovalCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    _state_runner = StateRunner(entity_class=BalancerRemoval,
                                initial_state=BalancerRemoval.states.STARTED,
                                final_state=BalancerRemoval.states.FINISHED,
                                final_cancelled_state=BalancerRemoval.states.CANCELLED,
                                processors=BalancerRemoval.get_processors(),
                                processing_interval=PROCESSING_INTERVAL)

    def __init__(self, namespace_id, balancer_id):
        name = u'balancer-removal-ctl("{}:{}")'.format(namespace_id, balancer_id)
        super(BalancerRemovalCtl, self).__init__(name)
        self._balancer_id = balancer_id
        self._namespace_id = namespace_id

    def _accept_event(self, event):
        return (isinstance(event, events.BalancerUpdate) and
                event.pb.meta.id == self._balancer_id)

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception(u'failed to process balancer removal on start: %s', e)
        self._cache.bind_on_specific_events(self._callback, [events.BalancerUpdate])

    def _stop(self):
        self._cache.unbind_from_specific_events(self._callback, [events.BalancerUpdate])

    def _process(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        pb = self._cache.must_get_balancer(self._namespace_id, self._balancer_id)
        if not needs_removal(pb):
            return
        return self._state_runner.process(ctx, pb)

    def _process_event(self, ctx, event):
        assert isinstance(event, events.BalancerUpdate)
        self._process(ctx)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
