import enum
import inject
import ujson
from datetime import datetime, timedelta
from yp_lite_ui_repo import pod_sets_api_pb2, endpoint_sets_api_pb2, endpoint_sets_pb2

import nanny_rpc_client
from awacs.lib import nannyclient
from awacs.lib.order_processor.model import BaseStandaloneProcessor, WithRemoval
from awacs.lib.ypliterpcclient import IYpLiteRpcClient, YpLiteRpcClient
from awacs.model import dao, zk, cache, external_clusters, util
from awacs.model.util import clone_pb
from infra.awacs.proto import model_pb2


@enum.unique
class State(enum.Enum):
    STARTED = 1
    REMOVING_BALANCER_BACKENDS = 2
    SHUTTING_DOWN_SERVICE = 30
    WAITING_FOR_TIMEOUT_OR_APPROVAL = 31
    REMOVING_SERVICE_SNAPSHOTS = 32
    REMOVING_SERVICE_FROM_DASHBOARDS = 33
    REMOVING_POD_SET = 34
    REMOVING_ENDPOINT_SETS = 35
    REMOVING_SERVICE = 36
    REMOVING_BALANCER = 40
    FINISHED = 50
    CANCELLING = 20
    RESTORING_BALANCER_BACKENDS = 21
    ACTIVATING_NANNY_SERVICE = 22
    CANCELLED = 60


class BalancerRemoval(WithRemoval):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'Balancer'
    states = State

    state_descriptions = {
        State.STARTED: u'Starting',
        State.REMOVING_BALANCER_BACKENDS: u'Removing system backends',
        State.SHUTTING_DOWN_SERVICE: u'Shutting down balancer service in Nanny',
        State.WAITING_FOR_TIMEOUT_OR_APPROVAL: u'Waiting before removing service',
        State.REMOVING_SERVICE_SNAPSHOTS: u'Removing service snapshots',
        State.REMOVING_SERVICE_FROM_DASHBOARDS: u'Cleaning up dashboards',
        State.REMOVING_POD_SET: u'Removing balancer pod set',
        State.REMOVING_ENDPOINT_SETS: u'Removing balancer endpoint sets',
        State.REMOVING_SERVICE: u'Removing balancer from Nanny',
        State.REMOVING_BALANCER: u'Removing balancer from Awacs',
        State.FINISHED: u'Finishing',
        State.CANCELLING: u'Cancelling removal',
        State.RESTORING_BALANCER_BACKENDS: u'Restoring system backends',
        State.ACTIVATING_NANNY_SERVICE: u'Activating balancer service in Nanny',
        State.CANCELLED: u'Finishing',
    }

    def zk_update(self):
        return self.zk.update_balancer(namespace_id=self.namespace_id,
                                       balancer_id=self.id)

    def self_remove(self):
        return self.dao.delete_balancer(
            namespace_id=self.namespace_id,
            balancer_id=self.id,
        )

    @staticmethod
    def get_processors():
        return (
            Start,
            RemovingBalancerBackends,
            ShuttingDownService,
            WaitingForTimeoutOrApproval,
            RemovingServiceSnapshots,
            RemovingServiceFromDashboards,
            RemovingPodSet,
            RemovingEndpointSets,
            RemovingService,
            RemovingBalancer,
            Cancelling,
            RestoringBalancerBackends,
            ActivatingNannyService,
        )

    @property
    def yp_cluster(self):
        if self.pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER:
            return self.pb.meta.location.yp_cluster
        if self.pb.meta.location.type == model_pb2.BalancerMeta.Location.AZURE_CLUSTER:
            return external_clusters.AZURE_CLUSTERS_BY_NAME[self.pb.meta.location.azure_cluster].yp_cluster
        else:
            # Balancer is gencfg-powered
            return None


class BalancerRemovalProcessor(BaseStandaloneProcessor):
    state = None
    next_state = None
    cancelled_state = None

    _nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _yp_lite_rpc_client = inject.attr(IYpLiteRpcClient)  # type: YpLiteRpcClient

    @classmethod
    def process(cls, ctx, balancer):
        """
        :type ctx: context.OpCtx
        :type balancer: BalancerRemoval
        :rtype: Result
        """
        raise NotImplementedError


class Start(BalancerRemovalProcessor):
    state = State.STARTED
    next_state = State.REMOVING_BALANCER_BACKENDS
    cancelled_state = State.CANCELLING

    @classmethod
    def process(cls, ctx, balancer):
        return balancer.make_result(cls.next_state)


class RemovingBalancerBackends(BalancerRemovalProcessor):
    state = State.REMOVING_BALANCER_BACKENDS
    next_state = State.REMOVING_BALANCER
    next_state_shutting_down_service = State.SHUTTING_DOWN_SERVICE
    cancelled_state = State.CANCELLING

    @classmethod
    def process(cls, ctx, balancer):
        backends = set()
        for backend_pb in cls._cache.list_all_backends(balancer.namespace_id):
            selector_pb = backend_pb.spec.selector
            if selector_pb.type == selector_pb.BALANCERS:
                for selector_balancer_pb in selector_pb.balancers:
                    if selector_balancer_pb.id == balancer.id:
                        backends.add(backend_pb.meta.id)
                        if not backend_pb.spec.deleted:
                            for b_pb in balancer.zk.update_backend(balancer.namespace_id, backend_pb.meta.id):
                                b_pb.spec.deleted = True
                        break
        if backends:
            backends = sorted(backends)
            return balancer.make_result(
                description=u'This balanced is still used in following backends: "{}"'.format(
                    u'", "'.join(backends)),
                content_pb=model_pb2.BalancerRemoval.RemovalFeedback.UsedInBackendsError(backend_ids=backends),
                next_state=cls.state,
                severity=model_pb2.FB_SEVERITY_ACTION_REQUIRED
            )
        if balancer.pb.removal.content.mode == balancer.pb.removal.content.AUTOMATIC:
            return balancer.make_result(cls.next_state_shutting_down_service)
        return balancer.make_result(cls.next_state)


class ShuttingDownService(BalancerRemovalProcessor):
    state = State.SHUTTING_DOWN_SERVICE
    next_state = State.WAITING_FOR_TIMEOUT_OR_APPROVAL
    next_state_incomplete = State.REMOVING_POD_SET
    cancelled_state = State.ACTIVATING_NANNY_SERVICE

    @classmethod
    def process(cls, ctx, balancer):
        if balancer.pb.spec.incomplete:
            # Service has not been created
            return balancer.make_result(cls.next_state_incomplete)
        cls._nanny_client.shutdown_service(service_id=balancer.pb.spec.config_transport.nanny_static_file.service_id,
                                           comment=u'Shutting down service to remove balancer in awacs')
        return balancer.make_result(cls.next_state)


class WaitingForTimeoutOrApproval(BalancerRemovalProcessor):
    state = State.WAITING_FOR_TIMEOUT_OR_APPROVAL
    next_state = State.REMOVING_SERVICE_SNAPSHOTS
    cancelled_state = State.ACTIVATING_NANNY_SERVICE

    AFTER_SHUTDOWN_TIMEOUT = timedelta(minutes=5)

    @classmethod
    def process(cls, ctx, balancer):
        if balancer.pb.removal.approval.after_service_shutdown.value:
            return balancer.make_result(cls.next_state)
        time_passed = datetime.utcnow() - balancer.pb.removal.progress.state.entered_at.ToDatetime()
        if time_passed < cls.AFTER_SHUTDOWN_TIMEOUT:
            return balancer.make_result(cls.state)
        return balancer.make_result(cls.next_state)


class RemovingServiceSnapshots(BalancerRemovalProcessor):
    state = State.REMOVING_SERVICE_SNAPSHOTS
    next_state = State.REMOVING_SERVICE_FROM_DASHBOARDS
    cancelled_state = State.ACTIVATING_NANNY_SERVICE

    @classmethod
    def process(cls, ctx, balancer):
        service_id = balancer.pb.spec.config_transport.nanny_static_file.service_id
        state = cls._nanny_client.get_service_state(service_id)
        active_snapshots = state[u'current_state'][u'content'][u'active_snapshots']
        if not active_snapshots:
            return balancer.make_result(cls.next_state)
        comment = u'Destroyed to remove balancer in awacs'
        for snapshot in active_snapshots:
            if snapshot[u'state'] not in (u'REMOVING', u'DESTROYING', u'DESTROYED'):
                cls._nanny_client.remove_service_snapshot(service_id, snapshot[u'snapshot_id'], comment)
        return balancer.make_result(cls.state)


class RemovingServiceFromDashboards(BalancerRemovalProcessor):
    state = State.REMOVING_SERVICE_FROM_DASHBOARDS
    next_state = State.REMOVING_POD_SET
    cancelled_state = None

    @classmethod
    def remove_service_from_dashboard(cls, service_id, dashboard_id):
        # Not the best method, but dashboards have no concurrency control and Nanny UI does it the same way
        dashboard = cls._nanny_client.get_dashboard(dashboard_id)
        content = dashboard[u'content']
        for group in content.get(u'groups', []):
            services = []
            for service in group.get(u'services', []):
                if service[u'service_id'] == service_id:
                    continue
                services.append(service)
            group[u'services'] = services
        comment = u'Removing service "{}" to remove balancer in awacs'.format(service_id)
        cls._nanny_client.update_dashboard(dashboard_id, content, comment)

    @classmethod
    def process(cls, ctx, balancer):
        service_id = balancer.pb.spec.config_transport.nanny_static_file.service_id
        dashboards = cls._nanny_client.list_service_dashboards(service_id)[u'value']
        dashboard_ids = [d[u'id'] for d in dashboards]
        for dashboard_id in dashboard_ids:
            cls.remove_service_from_dashboard(service_id, dashboard_id)
        return balancer.make_result(cls.next_state)


class RemovingPodSet(BalancerRemovalProcessor):
    state = State.REMOVING_POD_SET
    next_state = State.REMOVING_ENDPOINT_SETS
    next_state_gencfg_service = State.REMOVING_SERVICE
    cancelled_state = None

    @classmethod
    def _remove_pod_set(cls, ctx, req_pb):
        try:
            cls._yp_lite_rpc_client.remove_pod_set(req_pb)
        except nanny_rpc_client.exceptions.NotFoundError:
            # Somebody already removed pod set, it is ok
            pass
        except nanny_rpc_client.exceptions.BadRequestError:
            ctx.log.error(u'failed to remove pod set %s:%s', req_pb.service_id, req_pb.cluster)
            raise

    @classmethod
    def process(cls, ctx, balancer):
        yp_cluster = balancer.yp_cluster
        if yp_cluster is None:
            # Balancer is gencfg-powered, skipping...
            return balancer.make_result(cls.next_state_gencfg_service)

        if balancer.pb.spec.incomplete:
            # Service has not been created
            order_context = balancer.pb.order.progress.context
            pre_allocation_id = order_context.get(u'pre_allocation_id')
            pod_set_id = order_context.get(u'pod_set_id')
            if not pre_allocation_id or not pod_set_id:
                return balancer.make_result(cls.next_state)
            req_pb = pod_sets_api_pb2.RemovePodSetRequest(
                service_id=ujson.loads(pod_set_id),
                cluster=yp_cluster,
                pre_allocation_id=ujson.loads(pre_allocation_id)
            )
            cls._remove_pod_set(ctx, req_pb)
            return balancer.make_result(cls.next_state)

        service_id = balancer.pb.spec.config_transport.nanny_static_file.service_id
        req_pb = pod_sets_api_pb2.RemovePodSetRequest(
            service_id=service_id,
            cluster=yp_cluster
        )
        cls._remove_pod_set(ctx, req_pb)
        return balancer.make_result(cls.next_state)


class RemovingEndpointSets(BalancerRemovalProcessor):
    state = State.REMOVING_ENDPOINT_SETS
    next_state = State.REMOVING_SERVICE
    cancelled_state = None

    @classmethod
    def process(cls, ctx, balancer):
        if balancer.pb.spec.incomplete:
            # Service has not been created
            return balancer.make_result(cls.next_state)
        yp_cluster = balancer.yp_cluster
        if yp_cluster is None:
            # Balancer is gencfg-powered, skipping...
            return balancer.make_result(cls.next_state)

        service_id = balancer.pb.spec.config_transport.nanny_static_file.service_id
        req_pb = endpoint_sets_api_pb2.ListEndpointSetsRequest(
            service_id=service_id,
            cluster=yp_cluster
        )
        endpoint_sets_pb = cls._yp_lite_rpc_client.list_endpoint_sets(req_pb).endpoint_sets
        ids = []
        for es_pb in endpoint_sets_pb:
            if es_pb.meta.ownership == endpoint_sets_pb2.EndpointSetMeta.USER:
                ids.append(es_pb.meta.id)

        req_pb = endpoint_sets_api_pb2.RemoveEndpointSetsRequest(cluster=yp_cluster)
        req_pb.ids.extend(ids)
        try:
            cls._yp_lite_rpc_client.remove_endpoint_sets(req_pb)
        except nanny_rpc_client.exceptions.BadRequestError:
            ctx.log.error(u'failed to remove endpoints sets from %s:%s', service_id, yp_cluster)
            raise
        return balancer.make_result(cls.next_state)


class RemovingService(BalancerRemovalProcessor):
    state = State.REMOVING_SERVICE
    next_state = State.REMOVING_BALANCER
    cancelled_state = None

    @classmethod
    def process(cls, ctx, balancer):
        if balancer.pb.spec.incomplete:
            # Service has not been created
            return balancer.make_result(cls.next_state)
        cls._nanny_client.remove_service(balancer.pb.spec.config_transport.nanny_static_file.service_id)
        return balancer.make_result(cls.next_state)


class RemovingBalancer(BalancerRemovalProcessor):
    state = State.REMOVING_BALANCER
    next_state = State.FINISHED
    cancelled_state = None

    @classmethod
    def process(cls, ctx, balancer):
        balancer.self_remove()
        return balancer.make_result(cls.next_state)


class Cancelling(BalancerRemovalProcessor):
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    @classmethod
    def process(cls, ctx, balancer):
        if balancer.pb.spec.deleted:
            for b_pb in balancer.zk_update():
                b_pb.spec.deleted = False
        return balancer.make_result(cls.next_state)


class RestoringBalancerBackends(BalancerRemovalProcessor):
    state = State.RESTORING_BALANCER_BACKENDS
    next_state = State.CANCELLING
    cancelled_state = None

    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    @classmethod
    def process(cls, ctx, balancer):
        for backend_pb in cls._cache.list_all_backends(balancer.namespace_id):
            selector_pb = backend_pb.spec.selector
            if backend_pb.spec.deleted and selector_pb.type == selector_pb.BALANCERS:
                for selector_balancer_pb in selector_pb.balancers:
                    if selector_balancer_pb.id == balancer.id:
                        spec_pb = clone_pb(backend_pb.spec)
                        spec_pb.deleted = False
                        cls._dao.update_backend(balancer.namespace_id, backend_pb.meta.id,
                                                login=util.NANNY_ROBOT_LOGIN,
                                                comment=u'Restoring backend for balancer "{}"'.format(balancer.id),
                                                updated_spec_pb=spec_pb,
                                                version=backend_pb.meta.version,
                                                allow_restoration=True)
                        break
        if cls._cache.get_system_backend_for_balancer(balancer.namespace_id, balancer.id) is None:
            meta_pb = model_pb2.BackendMeta(
                id=balancer.id,
                namespace_id=balancer.namespace_id
            )
            meta_pb.auth.type = meta_pb.auth.STAFF
            meta_pb.is_system.value = True
            meta_pb.is_system.author = util.NANNY_ROBOT_LOGIN
            meta_pb.is_system.mtime.GetCurrentTime()
            spec_pb = model_pb2.BackendSpec()
            spec_pb.selector.type = model_pb2.BackendSelector.BALANCERS
            spec_pb.selector.balancers.add(id=balancer.id)
            balancer.dao.create_backend_if_missing(meta_pb=meta_pb, spec_pb=spec_pb)
        return balancer.make_result(cls.next_state)


class ActivatingNannyService(BalancerRemovalProcessor):
    state = State.ACTIVATING_NANNY_SERVICE
    next_state = State.RESTORING_BALANCER_BACKENDS
    cancelled_state = None

    @classmethod
    def process(cls, ctx, balancer):
        service_id = balancer.pb.spec.config_transport.nanny_static_file.service_id
        current_snapshot_id = cls._nanny_client.get_current_runtime_attrs_id(service_id)
        cls._nanny_client.set_snapshot_state(service_id, current_snapshot_id, u'ACTIVE',
                                             comment=u'Activating service to restore balancer in awacs')
        return balancer.make_result(cls.next_state)
