import six

from awacs.lib.strutils import flatten_full_id, to_full_id
from awacs.model import objects
from awacs.model.balancer.vector import (
    BalancerVersion,
    UpstreamVersion,
    DomainVersion,
    BackendVersion,
    EndpointSetVersion,
    KnobVersion,
    CertVersion,
)


class L7RevStatus(object):
    __slots__ = ('pb',)

    def __init__(self, pb):
        """
        :type pb: model_pb2.BalancerState.RevisionStatus
        """
        self.pb = pb

    def set_validated(self, status, message=None):
        """
        :type status: six.text_type
        :type message: six.text_type | None
        :rtype: bool
        """
        changed = False
        if self.pb.validated.status != status or (message is not None and self.pb.validated.message != message):
            self.pb.validated.status = status
            self.pb.validated.message = message
            self.pb.validated.last_transition_time.GetCurrentTime()
            changed = True
        return changed

    def modify_in_progress(self, updater):
        """
        :type updater: callable
        :rtype: bool
        """
        if not callable(updater):
            raise RuntimeError('"updater" is not callable')
        return updater(self.pb.in_progress)


class L7Entity(object):
    __slots__ = ('pb',)

    def __init__(self, pb):
        """
        :type pb: model_pb2.BalancerState.RevisionStatuses
        """
        self.pb = pb

    def set_active_rev(self, version):
        """
        :type version: union[BalancerVersion | UpstreamVersion | DomainVersion | BackendVersion | EndpointSetVersion |
                             KnobVersion | CertVersion | objects.WeightSection.version | six.text_type]
        :rtype: bool
        """
        if isinstance(version, (BalancerVersion,
                                UpstreamVersion,
                                DomainVersion,
                                BackendVersion,
                                EndpointSetVersion,
                                objects.WeightSection.version,
                                KnobVersion,
                                CertVersion,)):
            active_revision_id = version.version
        elif isinstance(version, six.string_types):
            active_revision_id = version
        else:
            raise RuntimeError('Unexpected "version" type: {}'.format(type(version)))
        changed = False
        for status_pb in self.pb.statuses:  # type: model_pb2.BalancerState.RevisionStatus
            if status_pb.revision_id == active_revision_id:
                if status_pb.active.status != 'True':
                    status_pb.active.status = 'True'
                    changed = True
            else:
                if status_pb.active.status != 'False':
                    status_pb.active.status = 'False'
                    changed = True
        return changed

    def omit_revs(self, matcher):
        """
        :type matcher: callable
        :rtype: bool
        """
        if not callable(matcher):
            raise RuntimeError('"matcher" is not callable')
        updated_status_pbs = []
        for status_pb in self.pb.statuses:  # type: model_pb2.BalancerState.RevisionStatus
            if matcher(status_pb):
                pass  # omit
            else:
                updated_status_pbs.append(status_pb)  # include
        if list(self.pb.statuses) != updated_status_pbs:
            del self.pb.statuses[:]
            self.pb.statuses.extend(updated_status_pbs)
            return True
        else:
            return False

    def add_new_rev(self, version):
        """
        :type version: Union[BalancerVersion | UpstreamVersion | DomainVersion | objects.WeightSection.version |
                             BackendVersion | EndpointSetVersion | KnobVersion | CertVersion]
        :rtype: bool
        """
        if not isinstance(version, (BalancerVersion,
                                    UpstreamVersion,
                                    DomainVersion,
                                    BackendVersion,
                                    EndpointSetVersion,
                                    objects.WeightSection.version,
                                    KnobVersion,
                                    CertVersion,)):
            raise RuntimeError('Unexpected version type: {}'.format(type(version)))

        if self.select_rev(version):
            return False

        status_pb = self.pb.statuses.add(
            revision_id=version.version,
            deleted=version.deleted,
        )
        if isinstance(version, CertVersion):
            status_pb.incomplete = version.incomplete
        status_pb.ctime.FromMicroseconds(version.ctime)
        status_pb.validated.status = 'Unknown'
        status_pb.validated.last_transition_time.GetCurrentTime()
        status_pb.in_progress.status = 'False'
        status_pb.active.status = 'False'
        return True

    def select_rev(self, version):
        """
        :type version: Union[BalancerVersion | UpstreamVersion | DomainVersion | objects.WeightSection.version |
                             BackendVersion | EndpointSetVersion | KnobVersion | CertVersion | six.text_type]
        :rtype: L7RevStatus | None
        """
        if isinstance(version, (BalancerVersion,
                                UpstreamVersion,
                                DomainVersion,
                                BackendVersion,
                                EndpointSetVersion,
                                objects.WeightSection.version,
                                KnobVersion,
                                CertVersion,)):
            revision_id = version.version
        elif isinstance(version, six.string_types):
            revision_id = version
        else:
            raise RuntimeError('Unexpected "version" type: {}'.format(type(version)))

        for status_pb in self.pb.statuses:
            if status_pb.revision_id == revision_id:
                return L7RevStatus(status_pb)
        return None

    def select_active_rev(self):
        """
        :rtype: L7RevStatus | None
        """
        for status_pb in self.pb.statuses:
            if status_pb.active.status == 'True':
                return L7RevStatus(status_pb)

    def list_revs(self):
        """
        :rtype: L7RevStatus | None
        """
        for status_pb in self.pb.statuses:
            yield L7RevStatus(status_pb)

    def list_in_progress_revs(self):
        """
        :rtype: L7RevStatus | None
        """
        for status_pb in self.pb.statuses:
            if status_pb.in_progress.status == 'True':
                yield L7RevStatus(status_pb)


class L7BalancerStateHandler(object):
    __slots__ = ('pb',)

    def __init__(self, pb):
        """
        :type pb: model_pb2.BalancerState
        """
        self.pb = pb

    def select(self, version, create=False):
        """
        :type create: bool
        :type version: Union[BalancerVersion | UpstreamVersion | DomainVersion | objects.WeightSection.version |
                             BackendVersion | EndpointSetVersion | KnobVersion | CertVersion]
        :rtype: L7Entity
        """
        if isinstance(version, BalancerVersion):
            assert version.balancer_id[0] == self.pb.namespace_id
            return self.select_balancer(create=create)
        elif isinstance(version, UpstreamVersion):
            return self.select_upstream(version.upstream_id, create=create)
        elif isinstance(version, DomainVersion):
            return self.select_domain(version.domain_id, create=create)
        elif isinstance(version, BackendVersion):
            return self.select_backend(version.backend_id, create=create)
        elif isinstance(version, EndpointSetVersion):
            return self.select_endpoint_set(version.endpoint_set_id, create=create)
        elif isinstance(version, objects.WeightSection.version):
            return self.select_weight_section(version.id, create=create)
        elif isinstance(version, KnobVersion):
            return self.select_knob(version.knob_id, create=create)
        elif isinstance(version, CertVersion):
            return self.select_cert(version.cert_id, create=create)
        else:
            raise RuntimeError('Unexpected "version" type: {}'.format(type(version)))

    def set_active_rev(self, version):
        """
        :type version: Union[BalancerVersion | UpstreamVersion | DomainVersion | objects.WeightSection.version |
                             BackendVersion | EndpointSetVersion | KnobVersion | CertVersion]
        :rtype: bool
        """
        return self.select(version).set_active_rev(version)

    def set_valid_rev(self, version, message=u''):
        """
        :type version: Union[BalancerVersion | UpstreamVersion | DomainVersion |
                             BackendVersion | EndpointSetVersion | KnobVersion | CertVersion]
        :typ message: six.text_type
        :rtype: bool
        """
        return self.select_rev(version).set_validated(status=u'True', message=message)

    def select_rev(self, version):
        """
        :type version: Union[BalancerVersion | UpstreamVersion | DomainVersion | objects.WeightSection.version |
                             BackendVersion | EndpointSetVersion | KnobVersion | CertVersion]
        :rtype: L7RevStatus | None
        """
        return self.select(version).select_rev(version)

    def add_new_rev(self, version):
        """
        :type version: Union[BalancerVersion | UpstreamVersion | DomainVersion | objects.WeightSection.version |
                             BackendVersion | EndpointSetVersion | KnobVersion | CertVersion | six.text_type]
        :rtype: bool
        """
        return self.select(version, create=True).add_new_rev(version)

    def select_balancer(self, create=False):
        """
        :type create: bool
        :return: L7Entity | None
        """
        if self.pb.HasField('balancer') or create:
            return L7Entity(self.pb.balancer)
        else:
            return None

    def select_upstream(self, full_upstream_id, create=False):
        """
        :type create: bool
        :param (six.text_type, six.text_type) full_upstream_id:
        :return: L7Entity | None
        """
        assert isinstance(full_upstream_id, tuple) and len(full_upstream_id) == 2
        flat_upstream_id = flatten_full_id(self.pb.namespace_id, full_upstream_id)
        assert '/' not in flat_upstream_id  # for now, until and if we allow using upstreams from other namespaces
        if flat_upstream_id in self.pb.upstreams or create:
            return L7Entity(self.pb.upstreams[flat_upstream_id])
        else:
            return None

    def iter_upstream_items(self):
        """
        :return: L7Entity | None
        """
        for flat_upstream_id, pb in six.iteritems(self.pb.upstreams):
            full_upstream_id = to_full_id(self.pb.namespace_id, flat_upstream_id)
            yield full_upstream_id, L7Entity(pb)

    def delete_upstream(self, full_upstream_id):
        """
        :param (six.text_type, six.text_type) full_upstream_id:
        :rtype: bool
        """
        assert isinstance(full_upstream_id, tuple) and len(full_upstream_id) == 2
        deleted = False
        flat_upstream_id = flatten_full_id(self.pb.namespace_id, full_upstream_id)
        assert '/' not in flat_upstream_id  # for now, until and if we allow using upstreams from other namespaces
        if flat_upstream_id in self.pb.upstreams:
            del self.pb.upstreams[flat_upstream_id]
            deleted = True
        return deleted

    def select_domain(self, full_domain_id, create=False):
        """
        :type create: bool
        :param (six.text_type, six.text_type) full_domain_id:
        :return: L7Entity | None
        """
        assert isinstance(full_domain_id, tuple) and len(full_domain_id) == 2
        flat_domain_id = flatten_full_id(self.pb.namespace_id, full_domain_id)
        assert '/' not in flat_domain_id  # for now, until and if we allow using domains from other namespaces
        if flat_domain_id in self.pb.domains or create:
            return L7Entity(self.pb.domains[flat_domain_id])
        else:
            return None

    def iter_domain_items(self):
        """
        :return: L7Entity | None
        """
        for flat_domain_id, pb in six.iteritems(self.pb.domains):
            full_domain_id = to_full_id(self.pb.namespace_id, flat_domain_id)
            yield full_domain_id, L7Entity(pb)

    def delete_domain(self, full_domain_id):
        """
        :param (six.text_type, six.text_type) full_domain_id:
        :rtype: bool
        """
        assert isinstance(full_domain_id, tuple) and len(full_domain_id) == 2
        deleted = False
        flat_domain_id = flatten_full_id(self.pb.namespace_id, full_domain_id)
        assert '/' not in flat_domain_id  # for now, until and if we allow using domains from other namespaces
        if flat_domain_id in self.pb.domains:
            del self.pb.domains[flat_domain_id]
            deleted = True
        return deleted

    def select_backend(self, full_backend_id, create=False):
        """
        :type create: bool
        :param (six.text_type, six.text_type) full_backend_id:
        :return: L7Entity | None
        """
        assert isinstance(full_backend_id, tuple) and len(full_backend_id) == 2
        flat_backend_id = flatten_full_id(self.pb.namespace_id, full_backend_id)
        if flat_backend_id in self.pb.backends or create:
            return L7Entity(self.pb.backends[flat_backend_id])
        else:
            return None

    def iter_backend_items(self):
        """
        :return: L7Entity | None
        """
        for flat_backend_id, pb in six.iteritems(self.pb.backends):
            full_backend_id = to_full_id(self.pb.namespace_id, flat_backend_id)
            yield full_backend_id, L7Entity(pb)

    def delete_backend(self, full_backend_id):
        """
        :param (six.text_type, six.text_type) full_backend_id:
        :rtype: bool
        """
        assert isinstance(full_backend_id, tuple) and len(full_backend_id) == 2
        deleted = False
        flat_backend_id = flatten_full_id(self.pb.namespace_id, full_backend_id)
        if flat_backend_id in self.pb.backends:
            del self.pb.backends[flat_backend_id]
            deleted = True
        return deleted

    def select_endpoint_set(self, full_endpoint_set_id, create=False):
        """
        :type create: bool
        :param (six.text_type, six.text_type) full_endpoint_set_id:
        :return: L7Entity | None
        """
        assert isinstance(full_endpoint_set_id, tuple) and len(full_endpoint_set_id) == 2
        flat_endpoint_set_id = flatten_full_id(self.pb.namespace_id, full_endpoint_set_id)
        if flat_endpoint_set_id in self.pb.endpoint_sets or create:
            return L7Entity(self.pb.endpoint_sets[flat_endpoint_set_id])
        else:
            return None

    def delete_endpoint_set(self, full_endpoint_set_id):
        """
        :param (six.text_type, six.text_type) full_endpoint_set_id:
        :rtype: bool
        """
        assert isinstance(full_endpoint_set_id, tuple) and len(full_endpoint_set_id) == 2
        deleted = False
        flat_endpoint_set_id = flatten_full_id(self.pb.namespace_id, full_endpoint_set_id)
        if flat_endpoint_set_id in self.pb.endpoint_sets:
            del self.pb.endpoint_sets[flat_endpoint_set_id]
            deleted = True
        return deleted

    def select_weight_section(self, full_weight_section_id, create=False):
        """
        :type create: bool
        :param (six.text_type, six.text_type) full_weight_section_id:
        :return: L7Entity | None
        """
        assert isinstance(full_weight_section_id, tuple) and len(full_weight_section_id) == 2
        flat_weight_section_id = flatten_full_id(self.pb.namespace_id, full_weight_section_id)
        if flat_weight_section_id in self.pb.weight_sections or create:
            return L7Entity(self.pb.weight_sections[flat_weight_section_id])
        else:
            return None

    def delete_weight_section(self, full_weight_section_id):
        """
        :param (six.text_type, six.text_type) full_weight_section_id:
        :rtype: bool
        """
        assert isinstance(full_weight_section_id, tuple) and len(full_weight_section_id) == 2
        deleted = False
        flat_weight_section_id = flatten_full_id(self.pb.namespace_id, full_weight_section_id)
        if flat_weight_section_id in self.pb.weight_sections:
            del self.pb.weight_sections[flat_weight_section_id]
            deleted = True
        return deleted

    def iter_knob_items(self):
        """
        :return: L7Entity | None
        """
        for flat_knob_id, pb in six.iteritems(self.pb.knobs):
            full_knob_id = to_full_id(self.pb.namespace_id, flat_knob_id)
            yield full_knob_id, L7Entity(pb)

    def select_knob(self, full_knob_id, create=False):
        """
        :type create: bool
        :param (six.text_type, six.text_type) full_knob_id:
        :return: L7Entity | None
        """
        assert isinstance(full_knob_id, tuple) and len(full_knob_id) == 2
        flat_knob_id = flatten_full_id(self.pb.namespace_id, full_knob_id)
        if flat_knob_id in self.pb.knobs or create:
            return L7Entity(self.pb.knobs[flat_knob_id])
        else:
            return None

    def delete_knob(self, full_knob_id):
        """
        :param (six.text_type, six.text_type) full_knob_id:
        :rtype: bool
        """
        assert isinstance(full_knob_id, tuple) and len(full_knob_id) == 2
        deleted = False
        flat_knob_id = flatten_full_id(self.pb.namespace_id, full_knob_id)
        if flat_knob_id in self.pb.knobs:
            del self.pb.knobs[flat_knob_id]
            deleted = True
        return deleted

    def iter_cert_items(self):
        """
        :return: L7Entity | None
        """
        for flat_cert_id, pb in six.iteritems(self.pb.certificates):
            full_cert_id = to_full_id(self.pb.namespace_id, flat_cert_id)
            yield full_cert_id, L7Entity(pb)

    def select_cert(self, full_cert_id, create=False):
        """
        :type create: bool
        :param (six.text_type, six.text_type) full_cert_id:
        :return: L7Entity | None
        """
        assert isinstance(full_cert_id, tuple) and len(full_cert_id) == 2
        flat_cert_id = flatten_full_id(self.pb.namespace_id, full_cert_id)
        if flat_cert_id in self.pb.certificates or create:
            return L7Entity(self.pb.certificates[flat_cert_id])
        else:
            return None

    def delete_cert(self, full_cert_id):
        """
        :param (six.text_type, six.text_type) full_cert_id:
        :rtype: bool
        """
        assert isinstance(full_cert_id, tuple) and len(full_cert_id) == 2
        deleted = False
        flat_cert_id = flatten_full_id(self.pb.namespace_id, full_cert_id)
        if flat_cert_id in self.pb.certificates:
            del self.pb.certificates[flat_cert_id]
            deleted = True
        return deleted
