# coding: utf-8
import collections
import logging

import gevent.lock
import gevent.pool
import inject
import itertools
import monotonic
import six
import time
from gevent.threadpool import ThreadPool

from awacs.lib import nannyclient, nannyrpcclient, context
from awacs.lib.context import OpCtx
from awacs.lib.ctlmanager import UNEXPECTED_EXCEPTIONS
from awacs.lib.eventsreporter import EventsReporter, Event
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.nannyclient import NannyApiRequestException
from awacs.lib.strutils import to_full_id
from awacs.model import cache, errors, zk, components, objects
from awacs.model.util import get_balancer_location
from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.model.balancer.stateholder import BalancerStateHolder
from awacs.model.balancer.transport_config_bundle import ConfigBundle
from awacs.model.balancer.vector import (
    BalancerVersion,
    DomainVersion,
    UpstreamVersion,
    BackendVersion,
    EndpointSetVersion,
    KnobVersion,
    CertVersion,
    get_human_readable_diff,
    find_revision_spec,
)
from infra.awacs.proto import model_pb2
from infra.swatlib import httpgridfsclient
from infra.swatlib import orly_client
from infra.swatlib.gevent import greenthread
from infra.swatlib.logutil import rndstr
from sepelib.core import config
from .generator import validate_config
from .registry import L7_CTL_REGISTRY
from ..util import is_large_balancer


class IdleIter(object):
    __slots__ = ('i',)

    def __init__(self):
        self.i = 0

    def __call__(self, seq):
        for item in seq:
            self.i += 1
            if self.i % 100 == 0:
                gevent.idle()
            yield item


MICROSECONDS_IN_SECOND = 1000000

EVENT_NANNY_ERROR = u'transport-nanny-error'
EVENT_UNEXPECTED_ERROR = u'transport-unexpected-error'


class BalancerTransport(greenthread.GreenThread):
    MAX_LOGGED_DATA_LENGTH = 10 * 1024

    DEFAULT_PROCESSING_INTERVAL = 5
    DEFAULT_POLLING_INTERVAL = 15
    DEFAULT_MAIN_LOOP_FREQ = 2.5

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient
    nanny_rpc_client = inject.attr(nannyrpcclient.INannyRpcClient)  # type: nannyrpcclient.NannyRpcClient
    gridfs_client = inject.attr(httpgridfsclient.IHttpGridfsClient)  # type: httpgridfsclient.HttpGridfsClient

    _orly_brake = orly_client.OrlyBrake(
        rule='awacs-push-l7-config',
        metrics_registry=L7_CTL_REGISTRY)

    _events = EventsReporter(events={
        Event(EVENT_NANNY_ERROR, log_level=logging.ERROR),
        Event(EVENT_UNEXPECTED_ERROR, log_level=logging.ERROR),
    }, metrics_registry=L7_CTL_REGISTRY)

    _from_created_to_applied_timer = L7_CTL_REGISTRY.get_histogram(
        'from-created-to-applied-delay',
        buckets=list(range(0, 240, 15)) + list(range(240, 600, 30)) + list(range(660, 3600, 180)))
    _from_validated_to_applied_timer = L7_CTL_REGISTRY.get_histogram(
        'from-validated-to-applied-delay',
        buckets=list(range(0, 240, 15)) + list(range(240, 600, 30)) + list(range(660, 3600, 180)))

    def __init__(self, namespace_id, balancer_id, log,
                 processing_interval=DEFAULT_PROCESSING_INTERVAL,
                 polling_interval=DEFAULT_POLLING_INTERVAL,
                 main_loop_freq=DEFAULT_MAIN_LOOP_FREQ):
        super(BalancerTransport, self).__init__()
        self.name = 'balancer-transport("{}:{}")'.format(namespace_id, balancer_id)
        self._log = log

        self._is_large = is_large_balancer(namespace_id, balancer_id)

        self.processing_interval = processing_interval
        self.polling_interval = polling_interval
        self.main_loop_freq = main_loop_freq

        self._namespace_id = namespace_id
        self._balancer_id = balancer_id

        self._balancer_state_pb = None
        self._last_seen_balancer_state_generation = None

        self._balancer_state_holder = BalancerStateHolder(
            namespace_id=self._namespace_id,
            balancer_id=self._balancer_id,
            balancer_state_pb=self._balancer_state_pb
        )
        self._reset_state()

        self._lock = gevent.lock.BoundedSemaphore()
        self._threadpool = None

    def _reset_state(self):
        self._versions_by_snapshot_full_ids = collections.defaultdict(set)
        self._snapshot_ctimes = {}

    def set_balancer_state_pb(self, pb, ctx):
        assert pb.namespace_id == self._namespace_id
        assert pb.balancer_id == self._balancer_id

        ctx = ctx.with_op(op_id='set_balancer_state_pb')
        ctx.log.debug('New balancer state generation: %d', pb.generation)

        self._balancer_state_pb = pb
        self._last_seen_balancer_state_generation = self._balancer_state_pb.generation
        self._balancer_state_holder.update(pb)
        self._reset_state()

        for rev_status_pb in self._balancer_state_pb.balancer.statuses:
            balancer_version = BalancerVersion.from_rev_status_pb((self._namespace_id, self._balancer_id),
                                                                  rev_status_pb)

            if rev_status_pb.in_progress.status == 'True':
                for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                    snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                    self._versions_by_snapshot_full_ids[snapshot_full_id].add(balancer_version)
                    if snapshot_full_id not in self._snapshot_ctimes:
                        self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

        for domain_id, domain_state_pb in six.iteritems(self._balancer_state_pb.domains):
            full_domain_id = to_full_id(self._namespace_id, domain_id)
            for rev_status_pb in domain_state_pb.statuses:
                domain_version = DomainVersion.from_rev_status_pb(full_domain_id, rev_status_pb)

                if rev_status_pb.in_progress.status == 'True':
                    for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                        snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                        self._versions_by_snapshot_full_ids[snapshot_full_id].add(domain_version)
                        if snapshot_full_id not in self._snapshot_ctimes:
                            self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

        for upstream_id, upstream_state_pb in six.iteritems(self._balancer_state_pb.upstreams):
            full_upstream_id = to_full_id(self._namespace_id, upstream_id)
            for rev_status_pb in upstream_state_pb.statuses:
                upstream_version = UpstreamVersion.from_rev_status_pb(full_upstream_id, rev_status_pb)

                if rev_status_pb.in_progress.status == 'True':
                    for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                        snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                        self._versions_by_snapshot_full_ids[snapshot_full_id].add(upstream_version)
                        if snapshot_full_id not in self._snapshot_ctimes:
                            self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

        for backend_id, backend_state_pb in six.iteritems(self._balancer_state_pb.backends):
            full_backend_id = to_full_id(self._namespace_id, backend_id)
            for rev_status_pb in backend_state_pb.statuses:
                backend_version = BackendVersion.from_rev_status_pb(full_backend_id, rev_status_pb)

                if rev_status_pb.in_progress.status == 'True':
                    for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                        snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                        self._versions_by_snapshot_full_ids[snapshot_full_id].add(backend_version)
                        if snapshot_full_id not in self._snapshot_ctimes:
                            self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

        for endpoint_set_id, endpoint_set_state_pb in six.iteritems(self._balancer_state_pb.endpoint_sets):
            full_endpoint_set_id = to_full_id(self._namespace_id, endpoint_set_id)
            for rev_status_pb in endpoint_set_state_pb.statuses:
                endpoint_set_version = EndpointSetVersion.from_rev_status_pb(full_endpoint_set_id, rev_status_pb)

                if rev_status_pb.in_progress.status == 'True':
                    for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                        snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                        self._versions_by_snapshot_full_ids[snapshot_full_id].add(endpoint_set_version)
                        if snapshot_full_id not in self._snapshot_ctimes:
                            self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

        for knob_id, knob_state_pb in six.iteritems(self._balancer_state_pb.knobs):
            full_knob_id = to_full_id(self._namespace_id, knob_id)
            for rev_status_pb in knob_state_pb.statuses:
                knob_version = KnobVersion.from_rev_status_pb(full_knob_id, rev_status_pb)

                if rev_status_pb.in_progress.status == 'True':
                    for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                        snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                        self._versions_by_snapshot_full_ids[snapshot_full_id].add(knob_version)
                        if snapshot_full_id not in self._snapshot_ctimes:
                            self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

        for cert_id, cert_state_pb in six.iteritems(self._balancer_state_pb.certificates):
            full_cert_id = to_full_id(self._namespace_id, cert_id)
            for rev_status_pb in cert_state_pb.statuses:
                cert_version = CertVersion.from_rev_status_pb(full_cert_id, rev_status_pb)

                if rev_status_pb.in_progress.status == 'True':
                    for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                        snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                        self._versions_by_snapshot_full_ids[snapshot_full_id].add(cert_version)
                        if snapshot_full_id not in self._snapshot_ctimes:
                            self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

        for weight_section_id, weight_section_state_pb in six.iteritems(self._balancer_state_pb.weight_sections):
            full_weight_section_id = to_full_id(self._namespace_id, weight_section_id)
            for rev_status_pb in weight_section_state_pb.statuses:
                weight_section_version = objects.WeightSection.version.from_rev_status_pb(full_weight_section_id, rev_status_pb)

                if rev_status_pb.in_progress.status == 'True':
                    for snapshot_pb in rev_status_pb.in_progress.meta.nanny_static_file.snapshots:
                        snapshot_full_id = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
                        self._versions_by_snapshot_full_ids[snapshot_full_id].add(weight_section_version)
                        if snapshot_full_id not in self._snapshot_ctimes:
                            self._snapshot_ctimes[snapshot_full_id] = snapshot_pb.ctime.ToMicroseconds()

    def _get_validated_config(self, ctx, vector, balancer_spec_pb, cert_spec_pbs):
        """
        :param: ctx
        :type vector: awacs.model.balancer.vector.Vector
        :rtype: ValidationResult
        """
        ctx.log.debug('Generating Lua-config from valid vector...')
        domain_spec_pbs = {}
        for domain_version in six.itervalues(vector.domain_versions):
            if domain_version.deleted:
                continue
            if domain_version.incomplete:
                continue
            domain_spec_pbs[domain_version] = find_revision_spec(domain_version)
        upstream_spec_pbs = {}
        for upstream_version in six.itervalues(vector.upstream_versions):
            if upstream_version.deleted:
                continue
            upstream_spec_pbs[upstream_version] = find_revision_spec(upstream_version)
        backend_spec_pbs = {}
        for backend_version in six.itervalues(vector.backend_versions):
            if backend_version.deleted:
                continue
            backend_spec_pbs[backend_version] = find_revision_spec(backend_version)
        endpoint_set_spec_pbs = {}
        for endpoint_set_version in six.itervalues(vector.endpoint_set_versions):
            if endpoint_set_version.deleted:
                continue
            endpoint_set_spec_pbs[endpoint_set_version] = find_revision_spec(endpoint_set_version)
        knob_spec_pbs = {}
        for knob_version in six.itervalues(vector.knob_versions):
            if knob_version.deleted:
                continue
            knob_spec_pbs[knob_version] = find_revision_spec(knob_version)
        weight_section_spec_pbs = {}
        for weight_section_version in six.itervalues(vector.weight_section_versions):
            if weight_section_version.deleted:
                continue
            weight_section_spec_pbs[weight_section_version] = find_revision_spec(weight_section_version)
        ctx.log.debug('Retrieved vector specs, constructing config')
        namespace_pb = cache.IAwacsCache.instance().must_get_namespace(self._namespace_id)  # type: model_pb2.Namespace
        valid_config = validate_config(namespace_pb, self._namespace_id, vector.balancer_version,
                                       balancer_spec_pb, upstream_spec_pbs, backend_spec_pbs, endpoint_set_spec_pbs,
                                       knob_spec_pbs=knob_spec_pbs, cert_spec_pbs=cert_spec_pbs,
                                       domain_spec_pbs=domain_spec_pbs, weight_section_spec_pbs=weight_section_spec_pbs,
                                       threadpool=self._threadpool,
                                       ctx=ctx)
        ctx.log.debug('Config is successfully constructed')
        return valid_config

    @staticmethod
    def _get_cert_spec_pbs_from_vector(vector):
        cert_spec_pbs = {}
        cert_spec_pbs_by_ids = {}
        for full_cert_id, cert_version in six.iteritems(vector.cert_versions):
            if cert_version.deleted:
                continue
            if cert_version.incomplete:
                continue
            cert = find_revision_spec(cert_version)
            cert_spec_pbs[cert_version] = cert
            cert_spec_pbs_by_ids[full_cert_id] = cert
        return cert_spec_pbs, cert_spec_pbs_by_ids

    def _generate_lua_config(self, ctx, service_id, to_vector, from_vector):
        """
        :type service_id: six.text_type
        :type to_vector: awacs.model.balancer.vector.Vector
        :type from_vector: awacs.model.balancer.vector.Vector
        :rtype: awacs.model.balancer.transport_config_bundle.ConfigBundle
        """
        if from_vector and from_vector.balancer_version:
            _, current_cert_spec_pbs = self._get_cert_spec_pbs_from_vector(from_vector)
        else:
            current_cert_spec_pbs = {}

        new_balancer_spec_pb = find_revision_spec(to_vector.balancer_version)
        new_cert_spec_pbs, new_cert_spec_pbs_by_ids = self._get_cert_spec_pbs_from_vector(to_vector)
        new_config = self._get_validated_config(ctx=ctx,
                                                vector=to_vector,
                                                balancer_spec_pb=new_balancer_spec_pb,
                                                cert_spec_pbs=new_cert_spec_pbs)
        included_new_cert_spec_pbs = {cert_id: new_cert_spec_pbs_by_ids[cert_id]
                                      for cert_id in new_config.included_full_cert_ids}
        gevent.idle()

        ctx.log.debug('Generating Config instance...')
        balancer_holder = new_config.balancer
        config = (balancer_holder.module or balancer_holder.chain).to_config(ctx=new_config.validation_ctx)
        ctx.log.debug('Config instance generated')
        gevent.idle()

        ctx.log.debug('Calling Config.to_top_level_lua...')
        if self._threadpool is not None:
            res = self._threadpool.spawn(config.to_top_level_lua, iter_=iter)
            while not res.ready():
                time.sleep(1)
            lua_config = res.get()
        else:
            lua_config = config.to_top_level_lua(iter_=IdleIter())
        ctx.log.debug('Config.to_top_level_lua finished')

        ctx.log.debug('Extracting component pbs...')
        component_pbs_to_set = {}
        components_to_remove = set()
        components_pb = new_balancer_spec_pb.components
        for component_config, component_pb in components.iter_balancer_components(components_pb):
            component_type = component_config.type
            if components.is_set(component_pb):
                component_pbs_to_set[component_type] = self.zk.must_get_component(component_type,
                                                                                  component_pb.version)
            elif components.is_removed(component_pb):
                components_to_remove.add(component_type)
        ctx.log.debug('Component pbs extracted')

        ctx.log.debug('Generating ConfigBundle...')
        nanny_static_file_pb = new_balancer_spec_pb.config_transport.nanny_static_file
        instance_tags_pb = nanny_static_file_pb.instance_tags if nanny_static_file_pb.HasField(
            'instance_tags') else None
        config_bundle = ConfigBundle(lua_config=lua_config,
                                     container_spec_pb=new_balancer_spec_pb.container_spec,
                                     current_cert_spec_pbs=current_cert_spec_pbs,
                                     new_cert_spec_pbs=included_new_cert_spec_pbs,
                                     component_pbs_to_set=component_pbs_to_set,
                                     components_to_remove=components_to_remove,
                                     ctl_version=new_balancer_spec_pb.ctl_version,
                                     instance_tags_pb=instance_tags_pb,
                                     service_id=service_id,
                                     custom_service_settings_pb=new_balancer_spec_pb.custom_service_settings)
        ctx.log.debug('ConfigBundle generated')
        return config_bundle

    def _get_comment(self, ctx, to_vector, from_vector, components_diff):
        """
        :type ctx: context.OpCtx
        :type to_vector: Vector
        :type from_vector: Vector
        :type components_diff: awacs.model.balancer.component_transports.diff.ComponentsDiff
        :rtype: str
        """
        rv = 'awacs: update configuration\n' + get_human_readable_diff(self._namespace_id, from_vector, to_vector,
                                                                       components_diff=components_diff)
        max_len = 500
        suffix = '\n\n' + ctx.id()
        if len(rv) + len(suffix) > max_len:
            rv = rv[:max_len - 3 - len(suffix)] + '...'
        rv += suffix  # add op_id to ease debugging
        return rv

    def _report_delays(self, ctx, to_vector, from_vector):
        now = time.time() * MICROSECONDS_IN_SECOND
        diff = from_vector.diff(to_vector)

        for version in itertools.chain(diff.added, diff.removed):
            from_created_to_applied_delay = (now - version.ctime) / MICROSECONDS_IN_SECOND
            self._from_created_to_applied_timer.observe(from_created_to_applied_delay)

            validated_at = self._balancer_state_holder.get_validated_last_transition_time_in_microseconds(version)
            if validated_at is None:
                ctx.log.debug(u'Failed to find %s in balancer state, how come?', version)
                continue
            from_validated_to_applied_delay = (now - validated_at) / MICROSECONDS_IN_SECOND
            self._from_validated_to_applied_timer.observe(from_validated_to_applied_delay)

    def _save_config_to_snapshot(self, ctx, service_id, to_vector, from_vector, config_bundle):
        balancer_spec_pb = find_revision_spec(to_vector.balancer_version)
        snapshot_priority = model_pb2.BalancerNannyStaticFileTransportSpec.SnapshotPriority.Name(
            balancer_spec_pb.config_transport.nanny_static_file.snapshot_priority)

        ctx.log.debug(u'Uploading Lua-config to gridfs...')
        gridfs_file = self.gridfs_client.put(custom_url=balancer_spec_pb.config_transport.nanny_static_file.gridfs_url,
                                             content=config_bundle.lua_config)
        ctx.log.debug(u'Saving Lua-config to Nanny...')
        service = self.nanny_client.get_service(service_id)
        info_attrs, runtime_attrs = service['info_attrs'], service['runtime_attrs']
        runtime_attrs_snapshot_id, runtime_attrs_content = runtime_attrs[u'_id'], runtime_attrs[u'content']

        balancer_pb = self.cache.must_get_balancer(self._namespace_id, self._balancer_id)
        location = get_balancer_location(balancer_pb)
        if config.get_value('run.unavailable_clusters.{}.skip_transport'.format(location.lower()), False):
            return runtime_attrs_snapshot_id, runtime_attrs[u'change_info'][u'ctime']

        info_attrs_snapshot_id, info_attrs_content = info_attrs[u'_id'], info_attrs[u'content']
        info_attrs_content, runtime_attrs_content = config_bundle.apply_to_service(
            info_attrs_content=info_attrs_content,
            runtime_attrs_content=runtime_attrs_content,
            gridfs_file=gridfs_file,
            ctx=ctx
        )
        comment = self._get_comment(ctx, to_vector, from_vector, config_bundle.components_diff)

        orly_op_id = to_vector.get_weak_hash_str()
        ctx.log.debug(u'Orly operation id: %s', orly_op_id)
        self._orly_brake.maybe_apply(op_id=orly_op_id, op_log=ctx.log, op_labels=[
            (u'namespace-id', self._namespace_id),
            (u'balancer-id', self._balancer_id),
        ])
        try:
            self.nanny_client.update_info_attrs_content(
                service_id=service_id,
                snapshot_id=info_attrs_snapshot_id,
                info_attrs_content=info_attrs_content,
                comment=comment
            )
        except NannyApiRequestException as e:
            self._events.report(EVENT_NANNY_ERROR, ctx=ctx)
            info_attrs_content = six.text_type(info_attrs_content)
            if len(info_attrs_content) > self.MAX_LOGGED_DATA_LENGTH:
                info_attrs_content = info_attrs_content[:self.MAX_LOGGED_DATA_LENGTH] + u'...'
            response_content = ''
            if e.response is not None:
                response_content = e.response.content
                if len(response_content) > self.MAX_LOGGED_DATA_LENGTH:
                    response_content = response_content[:self.MAX_LOGGED_DATA_LENGTH] + u'...'
            ctx.log.debug(u'info attrs: %s', info_attrs_content)
            ctx.log.debug(u'response: %s', response_content)
            ctx.log.warn(u'Failed to save service configuration to %s:%s:',
                         service_id, info_attrs_snapshot_id)
            raise
        except UNEXPECTED_EXCEPTIONS as e:
            self._events.report(EVENT_NANNY_ERROR, ctx=ctx)
            raise
        try:
            runtime_attrs_response = self.nanny_client.update_runtime_attrs_content(
                service_id=service_id,
                snapshot_id=runtime_attrs_snapshot_id,
                snapshot_priority=snapshot_priority,
                runtime_attrs_content=runtime_attrs_content,
                comment=comment
            )
        except NannyApiRequestException as e:
            self._events.report(EVENT_NANNY_ERROR, ctx=ctx)
            runtime_attrs_content = six.text_type(runtime_attrs_content)
            if len(runtime_attrs_content) > self.MAX_LOGGED_DATA_LENGTH:
                runtime_attrs_content = runtime_attrs_content[:self.MAX_LOGGED_DATA_LENGTH] + u'...'
            response_content = ''
            if e.response is not None:
                response_content = e.response.content
                if len(response_content) > self.MAX_LOGGED_DATA_LENGTH:
                    response_content = response_content[:self.MAX_LOGGED_DATA_LENGTH] + u'...'
            ctx.log.debug(u'runtime attrs: %s', runtime_attrs_content)
            ctx.log.debug(u'response: %s', response_content)
            ctx.log.warn(u'Failed to save service configuration to %s:%s:',
                         service_id, runtime_attrs_snapshot_id)
            raise
        except UNEXPECTED_EXCEPTIONS as e:
            self._events.report(EVENT_NANNY_ERROR, ctx=ctx)
            raise
        else:
            snapshot_id, snapshot_ctime = (runtime_attrs_response[u'_id'],
                                           runtime_attrs_response[u'change_info'][u'ctime'])
            ctx.log.debug(u'Saved runtime attrs to %s:%s', service_id, snapshot_id)
            return snapshot_id, snapshot_ctime

    def process(self, ctx):
        ctx = ctx.with_op(op_id='transport')
        valid_vector = self._balancer_state_holder.valid_vector
        in_progress_vector = self._balancer_state_holder.in_progress_vector
        active_vector = self._balancer_state_holder.active_vector

        if valid_vector == active_vector:
            return

        if valid_vector == in_progress_vector:
            return

        ctx.log.debug('Processing changes in balancer vectors')
        if not self._is_large:
            ctx.log.debug('valid vector: %s', valid_vector)
            ctx.log.debug('in-progress vector: %s', in_progress_vector)
            ctx.log.debug('active vector: %s', active_vector)

        balancer_spec_pb = find_revision_spec(valid_vector.balancer_version)
        service_id = balancer_spec_pb.config_transport.nanny_static_file.service_id

        if in_progress_vector.greater_than(active_vector):
            from_vector = in_progress_vector
        else:
            from_vector = active_vector
        config_bundle = self._generate_lua_config(ctx, service_id,
                                                  to_vector=valid_vector,
                                                  from_vector=from_vector)

        try:
            self._report_delays(ctx, to_vector=valid_vector, from_vector=from_vector)
        except Exception:
            ctx.log.warn('Failed to report delays')

        snapshot_id, snapshot_ctime = self._save_config_to_snapshot(ctx, service_id,
                                                                    to_vector=valid_vector,
                                                                    from_vector=from_vector,
                                                                    config_bundle=config_bundle)

        new_snapshot_id_pb = model_pb2.SnapshotId(service_id=service_id, snapshot_id=snapshot_id)
        new_snapshot_id_pb.ctime.FromMilliseconds(snapshot_ctime)

        def add_snapshot_and_set_in_progress(condition):
            snapshot_pbs = condition.meta.nanny_static_file.snapshots
            _updated = False
            if new_snapshot_id_pb not in snapshot_pbs:
                snapshot_pbs.add().CopyFrom(new_snapshot_id_pb)
                _updated = True
            if condition.status != 'True':
                condition.status = 'True'
                _updated = True
            return _updated

        ctx.log.debug('Setting "in progress" statuses')
        for balancer_state_pb in self.zk.update_balancer_state(self._namespace_id, self._balancer_id,
                                                               balancer_state_pb=self._balancer_state_pb):
            h = L7BalancerStateHandler(balancer_state_pb)
            updated = False
            updated |= h.select_rev(valid_vector.balancer_version).modify_in_progress(add_snapshot_and_set_in_progress)
            for domain_id, domain_version in six.iteritems(valid_vector.domain_versions):
                updated |= h.select_rev(domain_version).modify_in_progress(add_snapshot_and_set_in_progress)
            for upstream_id, upstream_version in six.iteritems(valid_vector.upstream_versions):
                upstream_entity = h.select(upstream_version)
                if upstream_entity is None:
                    if upstream_version.deleted:
                        # Let us be more tolerant to a quite benign race described in
                        # https://st.yandex-team.ru/AWACS-828#60b68f20619e3b3150f79b02
                        ctx.log.debug('Deleted upstream %s is not present in the state, skipping', upstream_version)
                        continue
                    else:
                        ctx.log.warn('Non-deleted upstream %s is not present in the state', upstream_version)
                        raise Exception('Upstream is missing from the balancer state')
                elif upstream_entity.select_rev(upstream_version) is None:
                    ctx.log.warn("Upstream revision %s is not present in the state", upstream_version)
                    raise Exception('Upstream revision is missing from the balancer state')
                updated |= h.select_rev(upstream_version).modify_in_progress(add_snapshot_and_set_in_progress)
            for full_backend_id, backend_version in six.iteritems(valid_vector.backend_versions):
                updated |= h.select_rev(backend_version).modify_in_progress(add_snapshot_and_set_in_progress)
            for full_endpoint_set_id, endpoint_set_version in six.iteritems(valid_vector.endpoint_set_versions):
                updated |= h.select_rev(endpoint_set_version).modify_in_progress(add_snapshot_and_set_in_progress)
            for full_knob_id, knob_version in six.iteritems(valid_vector.knob_versions):
                updated |= h.select_rev(knob_version).modify_in_progress(add_snapshot_and_set_in_progress)
            for full_cert_id, cert_version in six.iteritems(valid_vector.cert_versions):
                updated |= h.select_rev(cert_version).modify_in_progress(add_snapshot_and_set_in_progress)
            for full_weight_section_id, weight_section_version in six.iteritems(valid_vector.weight_section_versions):
                updated |= h.select_rev(weight_section_version).modify_in_progress(add_snapshot_and_set_in_progress)
            if not updated:
                break
        ctx.log.debug('Set "in progress" statuses')

    def handle_balancer_state_update(self, balancer_state_pb, ctx):
        """
        :type balancer_state_pb: model_pb2.BalancerState
        :type ctx: context.OpCtx
        """
        ctx = ctx.with_op(op_id='handle_balancer_state_update')
        with self._lock:
            assert balancer_state_pb.namespace_id == self._namespace_id  # by contract
            assert balancer_state_pb.balancer_id == self._balancer_id  # by contract
            if (self._last_seen_balancer_state_generation is not None and
                    self._last_seen_balancer_state_generation >= balancer_state_pb.generation):
                if self._last_seen_balancer_state_generation > balancer_state_pb.generation:
                    ctx.log.warn('Received balancer state generation older than last seen: %d > %d',
                                 self._last_seen_balancer_state_generation, balancer_state_pb.generation)
                return
            self.set_balancer_state_pb(balancer_state_pb, ctx)

    def _forget_snapshot(self, snapshot_full_id):
        """
        :type snapshot_full_id: (str, str)
        """
        if snapshot_full_id in self._versions_by_snapshot_full_ids:
            del self._versions_by_snapshot_full_ids[snapshot_full_id]
        if snapshot_full_id in self._snapshot_ctimes:
            del self._snapshot_ctimes[snapshot_full_id]

    def poll_snapshots(self, ctx):
        """
        :returns: whether balancer state has been updated
        :rtype: bool
        """
        ctx = ctx.with_op(op_id='poll_snapshots')
        snapshot_full_ids_from_old_to_new = list(self._versions_by_snapshot_full_ids)
        snapshot_full_ids_from_old_to_new.sort(key=self._snapshot_ctimes.get)

        by_services = collections.defaultdict(list)
        for snapshot_full_id in snapshot_full_ids_from_old_to_new:
            service_id, snapshot_id = snapshot_full_id
            versions = self._versions_by_snapshot_full_ids[snapshot_full_id]
            by_services[service_id].append((snapshot_id, versions))

        has_state_been_updated = False

        for service_id, service_snapshots_from_old_to_new in six.iteritems(by_services):
            processed_versions = set()
            activated_versions = set()
            activated_snapshots = []

            service_snapshots_from_new_to_old = reversed(service_snapshots_from_old_to_new)
            for snapshot_id, versions in service_snapshots_from_new_to_old:
                full_id_str = '{}:{}'.format(service_id, snapshot_id[:10])
                ctx.log.debug('Checking %s state changes history...', full_id_str)

                if self.nanny_rpc_client.has_snapshot_been_active(service_id, snapshot_id):
                    ctx.log.debug('Snapshot %s has been active at some point', full_id_str)
                    activated_snapshots.append((service_id, snapshot_id))
                    processed_versions.update(versions)
                    activated_versions.update(versions)
                    for older_snapshot_id, older_snapshot_versions in service_snapshots_from_new_to_old:
                        ctx.log.debug('Considering snapshot %s activated as '
                                      'it is older than %s', full_id_str, older_snapshot_id[:10])
                        processed_versions.update(older_snapshot_versions)
                        activated_snapshots.append((service_id, older_snapshot_id))
                    break
                else:
                    ctx.log.debug('Snapshot %s has never been active, '
                                  'continue checking older snapshots...', full_id_str)

            if not activated_snapshots:
                try:
                    current_snapshot_id, current_snapshot_ctime = \
                        self.nanny_client.get_current_runtime_attrs_id_and_ctime(service_id)
                except errors.NannyApiError as e:
                    ctx.log.warn('Failed to get current runtime attrs id for %s: %s', service_id, e)
                else:
                    current_snapshot_str = '{}:{}'.format(service_id, current_snapshot_id[:10])
                    ctx.log.debug(
                        'Checking current snapshot (%s) state changes history...', current_snapshot_str)

                    service_snapshots_from_new_to_old = reversed(service_snapshots_from_old_to_new)
                    latest_snapshot_id, latest_snapshot_versions = next(service_snapshots_from_new_to_old)
                    latest_snapshot_full_id = (service_id, latest_snapshot_id)

                    if current_snapshot_ctime * 1000 > self._snapshot_ctimes[latest_snapshot_full_id]:
                        ctx.log.debug('Current snapshot is newer than latest known')
                        if self.nanny_rpc_client.has_snapshot_been_active(service_id, current_snapshot_id):
                            ctx.log.debug(
                                'Current snapshot %s has been active at some point', current_snapshot_str)
                            activated_snapshots.append(latest_snapshot_full_id)
                            processed_versions.update(latest_snapshot_versions)
                            activated_versions.update(latest_snapshot_versions)
                            ctx.log.debug('Considering snapshot %s activated as '
                                          'it\'s older than current one', latest_snapshot_id[:10])

                            for older_snapshot_id, older_snapshot_versions in service_snapshots_from_new_to_old:
                                ctx.log.debug('Considering snapshot %s activated as '
                                              'it\'s older than current one', latest_snapshot_id[:10])
                                processed_versions.update(older_snapshot_versions)
                                activated_snapshots.append((service_id, older_snapshot_id))
                        else:
                            ctx.log.debug('Current snapshot %s has never been active', current_snapshot_str)

            def remove_activated_snapshots(condition):
                """
                :type condition: model_pb2.InProgressCondition
                """
                _updated = False
                snapshots = condition.meta.nanny_static_file.snapshots
                updated_snapshots = []
                for s in snapshots:
                    if (s.service_id, s.snapshot_id) not in activated_snapshots:
                        updated_snapshots.append(s)
                del snapshots[:]
                if len(snapshots) != len(updated_snapshots):
                    _updated |= True
                snapshots.extend(updated_snapshots)
                old_status = condition.status
                condition.status = 'True' if snapshots else 'False'
                _updated |= (old_status != condition.status)
                return _updated

            updated = False
            for attempt, balancer_state_pb in enumerate(self.zk.update_balancer_state(
                    self._namespace_id, self._balancer_id, balancer_state_pb=self._balancer_state_pb), start=1):
                ctx.log.debug('Maybe updating in-progress and active statuses, '
                              'attempt #%s, gen %s', attempt, balancer_state_pb.generation)
                h = L7BalancerStateHandler(balancer_state_pb)
                updated = False
                for version in gevent_idle_iter(processed_versions, idle_period=100):
                    updated |= h.select_rev(version).modify_in_progress(remove_activated_snapshots)
                for version in gevent_idle_iter(activated_versions, idle_period=100):
                    updated |= h.select(version).set_active_rev(version)
                if not updated:
                    break
            if updated:
                ctx.log.debug('In-progress and active statuses have been updated')
                for snapshot in activated_snapshots:
                    self._forget_snapshot(snapshot)
            else:
                ctx.log.debug('In-progress and active statuses have NOT been updated')

            has_state_been_updated |= updated

        return has_state_been_updated

    def cleanup(self, ctx):
        """
        :returns: whether balancer state has been updated
        :rtype: bool
        """
        ctx = ctx.with_op(op_id='transport_cleanup')
        active_vector = self._balancer_state_holder.active_vector
        updated = False
        for attempt, balancer_state_pb in enumerate(self.zk.update_balancer_state(
                self._namespace_id, self._balancer_id, balancer_state_pb=self._balancer_state_pb), start=1):
            h = L7BalancerStateHandler(balancer_state_pb)
            updated = False
            if active_vector.balancer_version:
                def matcher(status_pb):
                    return status_pb.ctime.ToMicroseconds() < active_vector.balancer_version.ctime

                updated |= h.select_balancer().omit_revs(matcher)

            for full_domain_id, domain_active_version in six.iteritems(active_vector.domain_versions):
                domain_h = h.select_domain(full_domain_id)
                if domain_h:
                    def matcher(status_pb):
                        return status_pb.ctime.ToMicroseconds() < domain_active_version.ctime

                    updated |= domain_h.omit_revs(matcher)

            for full_upstream_id, upstream_active_version in six.iteritems(active_vector.upstream_versions):
                upstream_h = h.select_upstream(full_upstream_id)
                if upstream_h:
                    def matcher(status_pb):
                        return status_pb.ctime.ToMicroseconds() < upstream_active_version.ctime

                    updated |= upstream_h.omit_revs(matcher)

            for full_backend_id, backend_active_version in six.iteritems(active_vector.backend_versions):
                backend_h = h.select_backend(full_backend_id)
                if backend_h:
                    def matcher(status_pb):
                        return status_pb.ctime.ToMicroseconds() < backend_active_version.ctime

                    updated |= backend_h.omit_revs(matcher)

            for full_endpoint_set_id, endpoint_set_active_version in six.iteritems(active_vector.endpoint_set_versions):
                endpoint_set_h = h.select_endpoint_set(full_endpoint_set_id)
                if endpoint_set_h:
                    def matcher(status_pb):
                        return status_pb.ctime.ToMicroseconds() < endpoint_set_active_version.ctime

                    updated |= endpoint_set_h.omit_revs(matcher)

            for full_knob_id, knob_active_version in six.iteritems(active_vector.knob_versions):
                knob_h = h.select_knob(full_knob_id)
                if knob_h:
                    def matcher(status_pb):
                        return status_pb.ctime.ToMicroseconds() < knob_active_version.ctime

                    updated |= knob_h.omit_revs(matcher)

            for full_cert_id, cert_active_version in six.iteritems(active_vector.cert_versions):
                cert_h = h.select_cert(full_cert_id)
                if cert_h:
                    def matcher(status_pb):
                        return status_pb.ctime.ToMicroseconds() < cert_active_version.ctime

                    updated |= cert_h.omit_revs(matcher)

            for full_weight_section_id, weight_section_active_version in six.iteritems(active_vector.weight_section_versions):
                weight_section_h = h.select_weight_section(full_weight_section_id)
                if weight_section_h:
                    def matcher(status_pb):
                        return status_pb.ctime.ToMicroseconds() < weight_section_active_version.ctime

                    updated |= weight_section_h.omit_revs(matcher)

            if not updated:
                break

            ctx.log.debug('Deleting old revisions from balancer state, '
                          'attempt #%s, gen %s', attempt, balancer_state_pb.generation)

        if updated:
            ctx.log.debug('Old revisions have been deleted from state')

        if active_vector.balancer_version is not None:
            for balancer_pb in self.zk.update_balancer(self._namespace_id, self._balancer_id):
                if not self._delete_old_indices(balancer_pb, active_vector.balancer_version.ctime):
                    break

        for (namespace_id, upstream_id), upstream_active_version in six.iteritems(active_vector.upstream_versions):
            assert namespace_id == self._namespace_id
            for upstream_pb in self.zk.update_upstream(namespace_id, upstream_id):
                if not self._delete_old_indices(upstream_pb, upstream_active_version.ctime):
                    break

        return updated

    @staticmethod
    def _delete_old_indices(entity_pb, active_ctime):
        """
        :type entity_pb: model_pb2.Upstream | model_pb2.Balancer
        :type active_ctime: long
        :param active_ctime in microseconds
        :returns True if we need to update spec in zk, False otherwise
        :rtype bool
        """
        updated_index_pbs = []
        for ind_pb in entity_pb.meta.indices:
            if ind_pb.ctime.ToMicroseconds() >= active_ctime:
                updated_index_pbs.append(ind_pb)
        need_update = False
        if list(entity_pb.meta.indices) != updated_index_pbs:
            del entity_pb.meta.indices[:]
            entity_pb.meta.indices.extend(updated_index_pbs)
            need_update = True
        return need_update

    def _run(self):
        last_processing_time = monotonic.monotonic() - self.processing_interval
        last_polling_time = monotonic.monotonic() - self.polling_interval
        root_ctx = context.BackgroundCtx()
        ctx, cancel = root_ctx.with_cancel()
        while 1:
            gevent.sleep(self.main_loop_freq)
            balancer_pb = self.cache.must_get_balancer(self._namespace_id, self._balancer_id)
            op_ctx = ctx.with_op(op_id=rndstr(), log=self._log)

            if monotonic.monotonic() - last_polling_time >= self.polling_interval:
                with self._lock:
                    if not balancer_pb.meta.transport_paused.value:
                        self.poll_snapshots(op_ctx)
                        self.cleanup(op_ctx)
                    last_polling_time = monotonic.monotonic()

            if monotonic.monotonic() - last_processing_time >= self.processing_interval:
                with self._lock:
                    if not balancer_pb.meta.transport_paused.value:
                        self.process(op_ctx)
                        self.cleanup(op_ctx)
                    last_processing_time = monotonic.monotonic()

    def run(self):
        ctx = OpCtx(op_id=rndstr(), log=self._log)
        while 1:
            self._log.info(u'Running %s', self.name)
            try:
                if self._is_large:
                    self._threadpool = ThreadPool(1)
                else:
                    self._threadpool = None
                self._run()
            except UNEXPECTED_EXCEPTIONS as e:
                if not isinstance(e, orly_client.OrlyBrakeApplied):
                    self._events.report(EVENT_UNEXPECTED_ERROR, ctx=ctx)
                self._log.exception(u'Unexpected exception while running %s: %s',
                                    self.name, six.text_type(e))
                gevent.sleep(15)
            else:
                break
            finally:
                if self._threadpool is not None:
                    self._threadpool.kill()
