import collections
import copy
import json

import inject
import semantic_version
import six
from google.protobuf.json_format import MessageToJson, MessageToDict

from awacs.lib import certs, ya_vault
from awacs.lib.strutils import flatten_full_id2, join_full_uids
from awacs.model import components
from awacs.model.balancer.component_transports import (
    awacslet,
    instancectl,
    sandbox_file,
    url_file,
    layer,
    diff,
    static_file
)
from awacs.model.util import clone_pb
from infra.awacs.proto import model_pb2


INSTANCECTL_MIN_VERSION_YAV_SECRETS_STR = '1.172'
INSTANCECTL_MIN_VERSION_YAV_SECRETS = semantic_version.Version(INSTANCECTL_MIN_VERSION_YAV_SECRETS_STR, partial=True)

NANNY_SECRET_TYPE = 'SECRET'
YAV_SECRET_TYPE = 'VAULT_SECRET'
ISS_HOOK_STOP_MAX_EXECUTION_TIME = 100


class SecretVolume(object):
    __slots__ = ('name', 'secret_name', 'storage_key', 'content', 'volume_type', 'delegation_token')

    def __init__(self, name, secret_name, storage_key, content, volume_type, delegation_token=None):
        """
        :type name: six.text_type
        :type secret_name: six.text_type
        :type storage_key: Union[tuple[six.text_type, six.text_type, six.text_type], six.text_type]
        :type content: dict
        :type volume_type: six.text_type
        :type delegation_token: Optional[six.text_type]
        """
        self.name = name
        self.secret_name = secret_name
        self.storage_key = storage_key
        self.delegation_token = delegation_token
        self.content = content
        self.volume_type = volume_type

    @classmethod
    def from_nanny_volume(cls, volume_type, volume):
        """
        :type volume_type: six.text_type
        :type volume: dict
        :type: Optional[Union[tuple[six.text_type, six.text_type, six.text_type], six.text_type]]
        """
        if volume_type == NANNY_SECRET_TYPE:
            secret = volume.get('secretVolume', {}).get('keychainSecret')
            if not secret:
                raise ValueError('Empty secret in volume: {}'.format(volume))
            k, s, r = secret.get('keychainId'), secret.get('secretId'), secret.get('secretRevisionId')
            if not (k and s and r):
                raise ValueError('Incomplete secret in volume: {}'.format(volume))
            return cls(name=volume['name'],
                       secret_name=volume['secretVolume'].get('secretName'),  # optional field in Nanny
                       storage_key=(k, s, r),
                       content=volume,
                       volume_type=volume_type)
        elif volume_type == YAV_SECRET_TYPE:
            secret = volume.get('vaultSecretVolume', {}).get('vaultSecret')
            if not secret:
                raise ValueError('Empty secret in volume: {}'.format(volume))
            s, v = secret.get('secretId'), secret.get('secretVer')
            if not (s and v):
                raise ValueError('Incomplete secret in volume: {}'.format(volume))
            return cls(name=volume['name'],
                       secret_name=secret.get('secretName'),  # optional field in Nanny
                       storage_key=s,
                       delegation_token=secret.get('delegationToken'),  # optional field in Nanny
                       content=volume,
                       volume_type=volume_type)
        raise ValueError('Unknown secret volume type {}'.format(volume_type))

    def update(self, volume):
        """
        When instance spec is edited in UI, volumes section is filled with empty placeholders for all possible sections.
        To preserve zero-diff in instance spec when importing an existing cert, we modify the volume inplace to preserve
        these placeholders.

        :param volume: dict
        :return:
        """
        new_volume_type = volume['type']
        self.content['type'] = new_volume_type
        self.content['name'] = volume['name']
        if new_volume_type == NANNY_SECRET_TYPE:
            self.content['secretVolume'] = volume['secretVolume']
            if self.volume_type != new_volume_type:
                self.content.pop('vaultSecretVolume', None)
        elif new_volume_type == YAV_SECRET_TYPE:
            self.content['vaultSecretVolume'] = volume['vaultSecretVolume']
            if self.volume_type != new_volume_type:
                self.content.pop('secretVolume', None)
        else:
            raise AssertionError()


class ConfigBundle(object):
    __slots__ = ('lua_config', 'container_spec_pb', 'current_cert_spec_pbs', 'new_cert_spec_pbs', 'service_id',
                 'component_pbs_to_set', 'components_to_remove', '_runtime_attrs_content', '_info_attrs_content',
                 '_gridfs_file', 'ctl_version', 'components_diff', 'instance_tags_pb', 'manage_layers',
                 'custom_service_settings_pb')

    JUGGLER_CHECKS_BUNDLE_EXTRACT_PATH = '{{JUGGLER_CHECKS_PATH}}/{juggler_local_file_path}'
    CONFIG_NAME = 'config.lua'
    CONTAINER_SPEC_FILE_NAME = 'awacs-balancer-container-spec.pb.json'

    _yav_client = inject.attr(ya_vault.IYaVaultClient)  # type: ya_vault.YaVaultClient

    def __init__(self, lua_config, container_spec_pb, current_cert_spec_pbs, new_cert_spec_pbs, service_id,
                 component_pbs_to_set, components_to_remove, ctl_version, instance_tags_pb, custom_service_settings_pb):
        """
        :type lua_config: six.text_type
        :type container_spec_pb: model_pb2.BalancerContainerSpec
        :type current_cert_spec_pbs: dict[tuple(six.text_type, six.text_type), model_pb2.CertificateSpec]
        :type new_cert_spec_pbs: dict[tuple(six.text_type, six.text_type), model_pb2.CertificateSpec]
        :type service_id: six.text_type
        :type component_pbs_to_set: dict[model_pb2.ComponentMeta.Type, model_pb2.Component]
        :type components_to_remove: set[model_pb2.ComponentMeta.Type]
        :type ctl_version: int
        :type instance_tags_pb: Optional[model_pb2.InstanceTags]
        :type custom_service_settings_pb: model_pb2.BalancerSpec.CustomServiceSettings
        """
        self.lua_config = lua_config
        self.container_spec_pb = container_spec_pb
        self.current_cert_spec_pbs = current_cert_spec_pbs
        self.new_cert_spec_pbs = new_cert_spec_pbs
        self.service_id = service_id
        self.component_pbs_to_set = component_pbs_to_set
        self.components_to_remove = components_to_remove
        self._runtime_attrs_content = self._info_attrs_content = self._gridfs_file = None
        self.ctl_version = ctl_version
        self.components_diff = None
        self.instance_tags_pb = instance_tags_pb
        self.custom_service_settings_pb = custom_service_settings_pb
        self.manage_layers = any(layer_type in self.component_pbs_to_set or layer_type in self.components_to_remove
                                 for layer_type in components.LAYERS_SEQUENCE)

    @staticmethod
    def _fill_container_spec_requirements(pb):
        """
        :type pb: model_pb2.BalancerContainerSpec
        """
        if pb.outbound_tunnels or pb.virtual_ips or pb.inbound_tunnels:
            # for now we require shawshank to set up out outbound tunnels
            pb.requirements.add(name='shawshank')

    def _add_container_spec_to_static_files(self):
        if self.container_spec_pb == model_pb2.BalancerContainerSpec():
            # container spec is empty, let's remove `container_spec_file_name`
            static_files = self._runtime_attrs_content['resources']['static_files']
            static_files = [f for f in static_files if f['local_path'] != self.CONTAINER_SPEC_FILE_NAME]
            self._runtime_attrs_content['resources']['static_files'] = static_files
            return self._runtime_attrs_content

        container_spec_pb = clone_pb(self.container_spec_pb)
        self._fill_container_spec_requirements(container_spec_pb)
        if six.PY3:
            data = MessageToDict(container_spec_pb)
            content = json.dumps(data, indent=2, sort_keys=True, separators=(', ', ': '))
        else:
            content = MessageToJson(container_spec_pb, sort_keys=True)

        found = False
        for resource in self._runtime_attrs_content['resources']['static_files']:
            if resource['local_path'] == self.CONTAINER_SPEC_FILE_NAME:
                resource['content'] = content
                found = True
        if not found:
            self._runtime_attrs_content['resources']['static_files'].append({
                'local_path': self.CONTAINER_SPEC_FILE_NAME,
                'content': content,
            })

        return self._runtime_attrs_content

    def _add_lua_config_to_static_files(self):
        found = False
        updated = False
        for resource in self._runtime_attrs_content['resources']['url_files']:
            if resource['local_path'] == self.CONFIG_NAME:
                if resource['url'] not in (self._gridfs_file.url, self._gridfs_file.checksum_url):
                    resource['url'] = self._gridfs_file.checksum_url
                    updated = True
                found = True
                break
        if not found:
            self._runtime_attrs_content['resources']['url_files'].append({
                'local_path': self.CONFIG_NAME,
                'url': self._gridfs_file.checksum_url,
            })
            updated = True
        if updated:
            # remove static files with config_name
            self._runtime_attrs_content['resources']['static_files'] = [
                f for f in self._runtime_attrs_content['resources']['static_files']
                if f['local_path'] != self.CONFIG_NAME
            ]
        return self._runtime_attrs_content

    def _fill_instance_tags(self):
        if self._runtime_attrs_content['engines']['engine_type'] == 'YP_LITE':
            self._runtime_attrs_content['instances']['yp_pod_ids']['orthogonal_tags'].update({
                'ctype': self.instance_tags_pb.ctype,
                'itype': self.instance_tags_pb.itype,
                'prj': self.instance_tags_pb.prj
            })
        else:
            self._runtime_attrs_content['instances']['extended_gencfg_groups']['orthogonal_tags'].update({
                'ctype': self.instance_tags_pb.ctype,
                'itype': self.instance_tags_pb.itype,
                'prj': self.instance_tags_pb.prj
            })

    @staticmethod
    def _remove_url_file_if_exists(component_config, resources, component_type, local_path):
        """
        :type component_config: components.ComponentConfig
        :type resources: dict
        :type component_type: model_pb2.ComponentMeta.Type
        :type local_path: six.text_type
        :rtype: diff.ComponentsDiff
        """
        components_diff = diff.ComponentsDiff()
        if components.SourceType.URL_FILE in component_config.possible_source_types:
            components_diff += url_file.remove(resources=resources,
                                               component_type=component_type,
                                               local_path=local_path)
        return components_diff

    @staticmethod
    def _remove_sandbox_file_if_exists(component_config, resources, component_type, local_path):
        """
        :type component_config: components.ComponentConfig
        :type resources: dict
        :type component_type: model_pb2.ComponentMeta.Type
        :type local_path: six.text_type
        :rtype: diff.ComponentsDiff
        """
        components_diff = diff.ComponentsDiff()
        if components.SourceType.SANDBOX_RESOURCE in component_config.possible_source_types:
            components_diff += sandbox_file.remove(resources=resources,
                                                   component_type=component_type,
                                                   local_path=local_path)
        return components_diff

    @staticmethod
    def _remove_static_file_if_exists(component_config, resources, component_type, local_path):
        """
        :type component_config: components.ComponentConfig
        :type resources: dict
        :type component_type: model_pb2.ComponentMeta.Type
        :type local_path: six.text_type
        :rtype: diff.ComponentsDiff
        """
        components_diff = diff.ComponentsDiff()
        components_diff += static_file.remove(resources=resources,
                                              component_type=component_type,
                                              local_path=local_path)
        return components_diff

    def _update_component_files(self, resources):
        components_diff = diff.ComponentsDiff()
        for component_config in components.COMPONENT_CONFIGS:
            if component_config.sandbox_resource is None or component_config.local_file_path is None:
                continue
            extract_path = ''
            component_type = component_config.type
            local_path = component_config.local_file_path
            if component_type == model_pb2.ComponentMeta.JUGGLER_CHECKS_BUNDLE:
                extract_path = self.JUGGLER_CHECKS_BUNDLE_EXTRACT_PATH.format(juggler_local_file_path=local_path)
            if component_type in self.component_pbs_to_set:
                component_pb = self.component_pbs_to_set[component_type]
                if component_pb.spec.source.HasField('sandbox_resource'):
                    components_diff += self._remove_url_file_if_exists(component_config, resources, component_type,
                                                                       local_path)
                    components_diff += self._remove_static_file_if_exists(component_config, resources, component_type,
                                                                          local_path)
                    components_diff += sandbox_file.add_or_update(resources=resources,
                                                                  component_pb=component_pb,
                                                                  local_path=local_path,
                                                                  extract_path=extract_path)

                elif component_pb.spec.source.HasField('url_file'):
                    components_diff += self._remove_sandbox_file_if_exists(component_config, resources, component_type,
                                                                           local_path)
                    components_diff += self._remove_static_file_if_exists(component_config, resources, component_type,
                                                                          local_path)
                    components_diff += url_file.add_or_update(resources=resources,
                                                              component_pb=component_pb,
                                                              local_path=local_path,
                                                              extract_path=extract_path)
                else:
                    raise RuntimeError('Unknown component source type')
            elif component_type in self.components_to_remove:
                components_diff += self._remove_sandbox_file_if_exists(component_config, resources, component_type,
                                                                       local_path)
                components_diff += self._remove_url_file_if_exists(component_config, resources, component_type,
                                                                   local_path)

        return components_diff

    def _process_components(self):
        """
        :rtype: awacs.model.balancer.component_transports.diff.ComponentsDiff
        """
        instance_spec = self._runtime_attrs_content['instance_spec']
        resources = self._runtime_attrs_content['resources']
        components_diff = diff.ComponentsDiff()

        components_diff += self._update_component_files(resources)

        instancectl_pb = self.component_pbs_to_set.get(model_pb2.ComponentMeta.INSTANCECTL)
        if instancectl_pb is not None:
            components_diff += instancectl.update(instance_spec, instancectl_pb)

        if model_pb2.ComponentMeta.AWACSLET in self.component_pbs_to_set:
            awacslet_pb = self.component_pbs_to_set[model_pb2.ComponentMeta.AWACSLET]
            components_diff += awacslet.add_or_update(instance_spec, awacslet_pb)
        elif model_pb2.ComponentMeta.AWACSLET in self.components_to_remove:
            components_diff += awacslet.remove(instance_spec)

        if self.manage_layers:
            prev_instance_spec = copy.deepcopy(instance_spec)
            instance_spec['layersConfig']['layer'] = []
            for layer_type in components.LAYERS_SEQUENCE:
                old_layer = layer.find_layer(prev_instance_spec['layersConfig']['layer'], layer_type)
                if layer_type in self.component_pbs_to_set:
                    layer_pb = self.component_pbs_to_set[layer_type]
                    new_layer = layer.make_layer(layer_pb)
                    instance_spec['layersConfig']['layer'].append(new_layer)
                    if old_layer is None:
                        components_diff.add(layer_type, layer_pb.meta.version)
                    elif old_layer != new_layer:
                        components_diff.update(layer_type, layer_pb.meta.version)
                elif old_layer is not None:
                    if layer_type in self.components_to_remove:
                        components_diff.remove(layer_type)
                    else:
                        instance_spec['layersConfig']['layer'].append(old_layer)
        return components_diff

    def _get_yav_secret_with_latest_certs_tarball(self, ctx, full_cert_id, secret_id):
        """
        :type full_cert_id: tuple[six.text_type, six.text_type]
        :type secret_id: six.text_type
        :type: six.text_type
        """
        cert_id = full_cert_id[1]
        flat_cert_id = flatten_full_id2(full_cert_id)
        ctx.log.debug('cert %s: getting tarball from secret %s', flat_cert_id, secret_id)
        cert_secret = self._yav_client.get_version(version=secret_id)
        secret_ver, cert_secret = cert_secret['version'], cert_secret['value']
        cert_sn = certs.decimal_to_padded_hexadecimal(self.new_cert_spec_pbs[full_cert_id].fields.serial_number)
        public_key, private_key, cert_tarball = certs.extract_certs_from_yav_secret(log=ctx.log,
                                                                                    flat_cert_id=flat_cert_id,
                                                                                    serial_number=cert_sn,
                                                                                    cert_secret=cert_secret)
        # vault_client.create_diff_version creates new version even if the content matches, so we check ourselves
        tarball_needs_update, reason = certs.tarball_needs_update(
            cert_tarball=cert_tarball,
            private_key=private_key,
            public_key=public_key,
            cert_id=cert_id,
        )
        if tarball_needs_update:
            cert_tarball = certs.pack_certs(public_key=public_key, private_key=private_key, cert_id=cert_id)
            secret_ver = self._yav_client.add_certs_to_secret(secret_ver, cert_tarball, comment=reason)
            ctx.log.debug('cert %s: tarball was updated: %s', flat_cert_id, reason)
        else:
            ctx.log.debug('cert %s: tarball is up to date', flat_cert_id)
        return secret_ver

    @staticmethod
    def _get_secret_volumes(volumes):
        """
        :type volumes: list[dict]
        :rtype: dict[six.text_type, SecretVolume]
        """
        rv = {}
        for volume in volumes:
            volume_type = volume.get('type')
            if volume_type in ('SECRET', 'VAULT_SECRET'):
                v = SecretVolume.from_nanny_volume(volume_type, volume)
                if v.storage_key in rv:
                    raise RuntimeError('Multiple volumes with identical storage key: {}'.format(v.storage_key))
                rv[v.storage_key] = v
        return rv

    @staticmethod
    def _get_cert_storage_key(full_cert_id, storage_pb):
        """
        :type full_cert_id: tuple[six.text_type, six.text_type]
        :type storage_pb: model_pb2.CertificateSpec.Storage
        :rtype: collections.Hashable
        """
        if storage_pb.type == model_pb2.CertificateSpec.Storage.NANNY_VAULT:
            return (storage_pb.nanny_vault_secret.keychain_id,
                    storage_pb.nanny_vault_secret.secret_id,
                    storage_pb.nanny_vault_secret.secret_revision_id)
        if storage_pb.type == model_pb2.CertificateSpec.Storage.YA_VAULT:
            return storage_pb.ya_vault_secret.secret_id
        raise RuntimeError('cert {}: Unknown cert storage type {}'.format(
            flatten_full_id2(full_cert_id),
            model_pb2.CertificateSpec.Storage.StorageType.Name(storage_pb.type)
        ))

    def _get_cert_secret_storage_keys(self, cert_spec_pbs):
        """
        :type cert_spec_pbs: dict[[six.text_type, six.text_type], model_pb2.CertificateSpec]
        :rtype dict
        """
        rv = {}
        for full_cert_id, cert_spec_pb in six.iteritems(cert_spec_pbs):
            rv[full_cert_id] = self._get_cert_storage_key(full_cert_id, cert_spec_pb.storage)
        return rv

    def _add_cert_volumes(self, ctx):
        instance_spec = self._runtime_attrs_content['instance_spec']
        if self.new_cert_spec_pbs:
            self._validate_instancectl_version_for_certs(instance_spec)
        volumes = instance_spec.get('volume')
        if not volumes:
            volumes = instance_spec['volume'] = []

        ctx.log.debug('Collecting secret volumes from instance spec')
        secret_volumes = self._get_secret_volumes(volumes)
        secret_volumes_by_name = {s.secret_name: s for s in six.itervalues(secret_volumes)}
        ctx.log.debug('Calculating storage keys for currently active certs')
        current_certs_storage_keys = self._get_cert_secret_storage_keys(self.current_cert_spec_pbs)
        ctx.log.debug('Calculating storage keys for new certs')
        new_certs_storage_keys = self._get_cert_secret_storage_keys(self.new_cert_spec_pbs)

        current_full_cert_ids = set(self.current_cert_spec_pbs)
        new_full_cert_ids = set(self.new_cert_spec_pbs)

        full_cert_ids_to_remove = current_full_cert_ids - new_full_cert_ids
        ctx.log.debug('Current cert ids to remove from nanny service: "%s"', join_full_uids(full_cert_ids_to_remove))
        full_cert_ids_to_add = new_full_cert_ids - current_full_cert_ids
        ctx.log.debug('New cert ids to add to nanny service: "%s"', join_full_uids(full_cert_ids_to_add))
        full_cert_ids_to_modify = current_full_cert_ids & new_full_cert_ids
        ctx.log.debug('Cert ids to maybe modify in nanny service: "%s"', join_full_uids(full_cert_ids_to_modify))

        for full_cert_id in (full_cert_ids_to_modify | full_cert_ids_to_add):
            # for every new and maybe-modified cert, look for existing volume and reuse it if possible
            current_cert_storage_key = current_certs_storage_keys.get(full_cert_id)
            new_cert_storage_key = new_certs_storage_keys[full_cert_id]
            existing_volume = (secret_volumes.get(new_cert_storage_key) or secret_volumes.get(current_cert_storage_key)
                               or secret_volumes_by_name.get(self._make_secret_volume_name(full_cert_id)))
            new_cert_spec_pb = self.new_cert_spec_pbs[full_cert_id]
            if existing_volume:
                new_volume = self._get_secret_volume(ctx=ctx,
                                                     full_cert_id=full_cert_id,
                                                     storage_pb=new_cert_spec_pb.storage,
                                                     existing_volume=existing_volume)
                if new_volume == existing_volume.content:
                    ctx.log.debug('cert %s: nanny volume is up to date', flatten_full_id2(full_cert_id))
                else:
                    old_content = copy.deepcopy(existing_volume.content)
                    existing_volume.update(new_volume)
                    if existing_volume.content == old_content:
                        ctx.log.debug('cert %s: nanny volume is up to date', flatten_full_id2(full_cert_id))
                    else:
                        ctx.log.debug('cert %s: modified existing nanny volume', flatten_full_id2(full_cert_id))
            else:
                new_volume = self._get_secret_volume(ctx=ctx,
                                                     full_cert_id=full_cert_id,
                                                     storage_pb=new_cert_spec_pb.storage,
                                                     existing_volume=None)
                ctx.log.debug('cert %s: created new nanny volume', flatten_full_id2(full_cert_id))
                volumes.append(new_volume)
        for full_cert_id in full_cert_ids_to_remove:
            # for every removed cert, try to remove the corresponding volume
            cert_storage_key = current_certs_storage_keys[full_cert_id]
            volume_to_remove = secret_volumes.get(cert_storage_key)
            if volume_to_remove:
                ctx.log.debug('cert %s: removed nanny volume', flatten_full_id2(full_cert_id))
                volumes.remove(volume_to_remove.content)
            else:
                ctx.log.debug("cert %s: wanted to remove nanny volume, but it's already deleted",
                              flatten_full_id2(full_cert_id))

    def _validate_instancectl_version_for_certs(self, instance_spec):
        """
        :param instance_spec: dict
        :raises: RuntimeError
        """
        ictl_version = instance_spec.get('instancectl', {}).get('version')
        ictl_version = semantic_version.Version(ictl_version, partial=True) if ictl_version else None
        if not ictl_version or ictl_version < INSTANCECTL_MIN_VERSION_YAV_SECRETS:
            raise RuntimeError('Instancectl version in service "{}" must be {}+ to support certificates'
                               .format(self.service_id, INSTANCECTL_MIN_VERSION_YAV_SECRETS_STR))

    @staticmethod
    def _make_secret_volume_name(full_cert_id):
        """
        :type full_cert_id: tuple[six.text_type, six.text_type]
        :rtype six.text_type
        """
        return 'secrets_{}'.format(certs.normalize_cert_id(full_cert_id[1]))

    def _get_secret_volume(self, ctx, full_cert_id, storage_pb, existing_volume):
        """
        :type full_cert_id: tuple[six.text_type, six.text_type]
        :type storage_pb: model_pb2.CertificateSpec.Storage
        :type existing_volume: Optional[SecretVolume]
        :rtype six.text_type
        """
        if storage_pb.type == model_pb2.CertificateSpec.Storage.NANNY_VAULT:
            return self._get_nanny_vault_secret_volume(ctx=ctx,
                                                       full_cert_id=full_cert_id,
                                                       existing_volume=existing_volume,
                                                       storage_pb=storage_pb)
        elif storage_pb.type == model_pb2.CertificateSpec.Storage.YA_VAULT:
            return self._get_yav_secret_volume(ctx=ctx,
                                               full_cert_id=full_cert_id,
                                               existing_volume=existing_volume,
                                               storage_pb=storage_pb)
        else:
            raise RuntimeError('cert {}: Unknown cert storage type {}'.format(
                flatten_full_id2(full_cert_id),
                model_pb2.CertificateSpec.Storage.StorageType.Name(storage_pb.type))
            )

    def _get_nanny_vault_secret_volume(self, ctx, full_cert_id, storage_pb, existing_volume=None):
        """
        :type full_cert_id: tuple[six.text_type, six.text_type]
        :type existing_volume: Optional[SecretVolume]
        :type storage_pb: model_pb2.CertificateSpec.Storage
        :rtype Optional[dict]
        """
        flat_cert_id = flatten_full_id2(full_cert_id)
        volume_name = secret_name = self._make_secret_volume_name(full_cert_id)
        if existing_volume:
            if existing_volume.storage_key == self._get_cert_storage_key(full_cert_id, storage_pb):
                ctx.log.debug('cert %s: storage type has not changed, preserving volume name', flat_cert_id)
                volume_name = existing_volume.name
                secret_name = existing_volume.secret_name
            else:
                ctx.log.debug('cert %s: storage contents have changed, updating volume name', flat_cert_id)
        secret_pb = storage_pb.nanny_vault_secret
        return {
            'name': volume_name,
            'type': 'SECRET',
            'secretVolume': {
                'secretName': secret_name,
                'keychainSecret': {
                    'keychainId': secret_pb.keychain_id,
                    'secretId': secret_pb.secret_id,
                    'secretRevisionId': secret_pb.secret_revision_id,
                },
            },
        }

    def _get_yav_secret_volume(self, ctx, full_cert_id, storage_pb, existing_volume=None):
        """
        :type full_cert_id: tuple[six.text_type, six.text_type]
        :type existing_volume: Optional[SecretVolume]
        :type storage_pb: model_pb2.CertificateSpec.Storage
        :rtype Optional[dict]
        """
        flat_cert_id = flatten_full_id2(full_cert_id)
        secret_id = storage_pb.ya_vault_secret.secret_id

        secret_ver = self._get_yav_secret_with_latest_certs_tarball(ctx, full_cert_id, secret_id)
        volume_name = secret_name = self._make_secret_volume_name(full_cert_id)

        delegation_token = None
        if existing_volume:
            if existing_volume.storage_key == self._get_cert_storage_key(full_cert_id, storage_pb):
                ctx.log.debug('cert %s: secret_id has not changed, preserving volume name', flat_cert_id)
                volume_name = existing_volume.name
                secret_name = existing_volume.secret_name
            else:
                ctx.log.debug('cert %s: storage contents have changed, updating volume name', flat_cert_id)
            old_token = existing_volume.delegation_token
            if old_token and self._check_delegation_token(ctx, flat_cert_id, secret_id, old_token):
                ctx.log.debug('cert %s: delegation token is up to date', flat_cert_id)
                delegation_token = old_token
        if not delegation_token:
            delegation_token = self._get_new_delegation_token(ctx, flat_cert_id, secret_id)
        return {
            'name': volume_name,
            'type': 'VAULT_SECRET',
            'vaultSecretVolume': {
                'vaultSecret': {
                    'secretName': secret_name,
                    'secretId': secret_id,
                    'secretVer': secret_ver,
                    'delegationToken': delegation_token,
                },
            },
        }

    def _check_delegation_token(self, ctx, flat_cert_id, secret_id, delegation_token):
        """
        :type flat_cert_id: six.text_type
        :type secret_id: six.text_type
        :type delegation_token: Optional[six.text_type]
        :rtype bool
        """
        ctx.log.debug('cert %s: checking delegation token', flat_cert_id)
        try:
            return self._yav_client.check_token(secret_id=secret_id,
                                                service_id=self.service_id,
                                                token=delegation_token)
        except Exception as e:
            ctx.log.exception('cert %s: delegation token validation failed: %s', flat_cert_id, e)
            return False

    def _get_new_delegation_token(self, ctx, flat_cert_id, secret_id):
        """
        :type flat_cert_id: six.text_type
        :type secret_id: six.text_type
        :rtype six.text_type
        """
        delegation_token = self._yav_client.get_token(secret_id=secret_id, service_id=self.service_id)
        ctx.log.debug('cert %s: acquired new delegation token', flat_cert_id)
        return delegation_token

    def _fill_info_attrs_content(self):
        sandbox_file_paths = []
        for component_type in self.component_pbs_to_set:
            component_config = components.get_component_config(component_type)
            if component_config.make_nanny_resource_readonly:
                sandbox_file_paths.append(component_config.local_file_path)
        for component_type in self.components_to_remove:
            component_config = components.get_component_config(component_type)
            if component_config.make_nanny_resource_readonly:
                sandbox_file_paths.append(component_config.local_file_path)
        if self.ctl_version > 5:
            sandbox_file_paths.append(self.CONFIG_NAME)
            if self.container_spec_pb != model_pb2.BalancerContainerSpec():
                sandbox_file_paths.append(self.CONTAINER_SPEC_FILE_NAME)
        if 'awacs_managed_settings' not in self._info_attrs_content:
            self._info_attrs_content['awacs_managed_settings'] = {}
        self._info_attrs_content['awacs_managed_settings']['components'] = {
            'instancectl': model_pb2.ComponentMeta.INSTANCECTL in self.component_pbs_to_set,
            'layers': self.manage_layers,
            'sandbox_files': sandbox_file_paths,
            'instance_spec': model_pb2.ComponentMeta.AWACSLET in self.component_pbs_to_set
        }

    def apply_to_service(self, info_attrs_content, runtime_attrs_content, gridfs_file, ctx):
        """
        :type ctx: context.OpCtx
        :type runtime_attrs_content: nanny.model.docs.service.RuntimeAttrsContent
        :type info_attrs_content: nanny.model.docs.service.InfoAttrsContent
        :type gridfs_file: httpgridfsclient.HttpGridfsFile
        :rtype: (nanny.model.docs.service.InfoAttrsContent, nanny.model.docs.service.RuntimeAttrsContent)
        """
        ctx = ctx.with_op(op_id='config_bundle')
        self._runtime_attrs_content = runtime_attrs_content
        self._info_attrs_content = info_attrs_content
        self._gridfs_file = gridfs_file
        self._add_lua_config_to_static_files()
        # We should process component before adding cert volumes, because instancectl version can change
        self.components_diff = self._process_components()
        if self.instance_tags_pb is not None:
            self._fill_instance_tags()
        self._add_cert_volumes(ctx)
        self._add_container_spec_to_static_files()
        if self.custom_service_settings_pb.service == self.custom_service_settings_pb.DZEN:
            network_properties = self._runtime_attrs_content['instance_spec'].setdefault('network_properties', {})
            network_properties['resolv_conf'] = 'USE_NAT64_LOCAL'
        self._fill_info_attrs_content()
        if self.ctl_version >= 5:
            # https://st.yandex-team.ru/AWACS-756
            iss_hook_stop = (self._runtime_attrs_content['instances']
                             .setdefault('iss_settings', {})
                             .setdefault('hooks_time_limits', {})
                             .setdefault('iss_hook_stop', {}))
            iss_hook_stop['max_execution_time'] = ISS_HOOK_STOP_MAX_EXECUTION_TIME
        return self._info_attrs_content, self._runtime_attrs_content
