# coding: utf-8
import logging
from collections import namedtuple

import inject
import monotonic
import six
import ujson
from sepelib.core import config as appconfig

from awacs.lib.eventsreporter import EventsReporter, Event
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.strutils import flatten_full_id2
from awacs.model import zk, cache, objects
from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.model.db import find_endpoint_set_revision2
from awacs.model.util import clone_pb, is_large_balancer
from awacs.wrappers.base import Holder, DEFAULT_CTX
from awacs.wrappers.errors import ValidationError
from awacs.wrappers.main import IncludeUpstreams
from infra.awacs.proto import model_pb2
from . import errors
from .generator import validate_config, ValidationResult, get_would_be_injected_full_backend_ids, \
    get_would_be_injected_full_cert_ids
from .modes import CommonServicesBalancerModeValidator, L7FastBalancerModeValidator
from .registry import L7_CTL_REGISTRY
from .stateholder import BalancerStateHolder
from .vector import (
    Vector,
    BalancerVersion,
    UpstreamVersion,
    BackendVersion,
    EndpointSetVersion,
    KnobVersion,
    CertVersion,
    DomainVersion,
    get_human_readable_diff,
    find_revision_spec)


spec_pbs_tuple = namedtuple('spec_pbs_tuple', (
    'namespace',
    'balancer',
    'upstreams',
    'upstream_ids',
    'domains',
    'backends',
    'backend_ids',
    'endpoint_sets',
    'knobs',
    'certs',
    'weight_sections',
    'cert_ids',
))

EVENT_L7_CONFIG_VALIDATION_TOO_LONG = u'l7-config-validation-too-long'


class BalancerValidator(object):
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _infinite_loop_counter = L7_CTL_REGISTRY.get_counter(u'validator-infinite-loop')
    _rollback_validation_timer = L7_CTL_REGISTRY.get_histogram('rollback-validation-timer')
    _rollback_gauge = L7_CTL_REGISTRY.get_gauge(u'rollback-count')
    _config_validation_timer = L7_CTL_REGISTRY.get_histogram('config-validation-timer')
    _events = EventsReporter(events={
        Event(EVENT_L7_CONFIG_VALIDATION_TOO_LONG, log_level=logging.WARN),
    }, metrics_registry=L7_CTL_REGISTRY)

    def __init__(self, namespace_id, balancer_id, threadpool=None):
        self._namespace_id = namespace_id
        self._balancer_id = balancer_id

        self._is_large = is_large_balancer(namespace_id, balancer_id)

        # protobufs
        self._balancer_state_pb = None
        self._last_seen_balancer_state_generation = None

        self._common_services_balancer_mode_validator = CommonServicesBalancerModeValidator(namespace_id, balancer_id)
        self._l7_fast_balancer_mode_validator = L7FastBalancerModeValidator(namespace_id, balancer_id)

        self._state_proxy = BalancerStateHolder(namespace_id, balancer_id)
        self._curr_vector = None
        self._threadpool = threadpool

    def set_balancer_state_pb(self, pb):
        assert isinstance(pb, model_pb2.BalancerState)
        assert pb.namespace_id == self._namespace_id, '{} != {}'.format(pb.namespace_id, self._namespace_id)
        assert pb.balancer_id == self._balancer_id, '{} != {}'.format(pb.balancer_id, self._balancer_id)
        self._balancer_state_pb = pb
        self._last_seen_balancer_state_generation = self._balancer_state_pb.generation
        self._state_proxy.update(pb)
        self._curr_vector = self._state_proxy.curr_vector

    def _collect_upstreams(self, spec_pbs):
        upstreams = {}
        upstream_labels = {}
        for upstream_version, upstream_spec_pb in gevent_idle_iter(six.iteritems(spec_pbs.upstreams), idle_period=5):
            upstream_config = Holder(clone_pb(upstream_spec_pb.yandex_balancer.config))
            included_full_backend_ids = get_would_be_injected_full_backend_ids(self._namespace_id, upstream_config)
            missing_backends = included_full_backend_ids.difference(spec_pbs.backend_ids)
            if missing_backends:
                raise errors.ConfigValidationError(
                    u'Upstream "{}": some of the included backends are missing: "{}"'.format(
                        upstream_version.upstream_id[1],
                        '", "'.join(flatten_full_id2(full_id) for full_id in sorted(missing_backends))),
                    cause=upstream_version)
            upstreams[upstream_version] = upstream_config
            upstream_labels[upstream_version.upstream_id] = dict(upstream_spec_pb.labels)
        return upstreams, upstream_labels

    def _check_domain_certs_and_upstreams(self, spec_pbs, l7_macro, upstream_labels):
        balancer_has_https_enabled = bool(l7_macro.https)
        for domain_version, domain_spec_pb in gevent_idle_iter(six.iteritems(spec_pbs.domains), idle_period=5):
            config_pb = domain_spec_pb.yandex_balancer.config
            if balancer_has_https_enabled and config_pb.protocol != config_pb.HTTP_ONLY:
                included_cert_ids = set()
                for cert_id in (config_pb.cert.id, config_pb.secondary_cert.id):
                    if cert_id:
                        included_cert_ids.add((self._namespace_id, cert_id))
                missing_certs = included_cert_ids.difference(spec_pbs.cert_ids)
                if missing_certs:
                    raise errors.ConfigValidationError(
                        u'Domain "{}": some of the included certs are missing: "{}"'.format(
                            domain_version.domain_id[1],
                            '", "'.join(full_cert_id[1] for full_cert_id in sorted(included_cert_ids))),
                        cause=domain_version)
            try:
                IncludeUpstreams(config_pb.include_upstreams).filter_upstreams(
                    self._namespace_id,
                    upstreams={uid: None for uid in spec_pbs.upstream_ids},
                    labels=upstream_labels)
            except ValidationError as e:
                raise errors.ConfigValidationError(u'Domain "{}": {}'.format(
                    domain_version.domain_id[1], e.message), cause=domain_version)

    def _check_certs(self, spec_pbs, balancer_version, balancer_config):
        included_cert_ids = get_would_be_injected_full_cert_ids(self._namespace_id, balancer_config, DEFAULT_CTX)
        missing_certs = included_cert_ids.difference(spec_pbs.cert_ids)
        if missing_certs:
            raise errors.ConfigValidationError(
                u'Some of the included certs are missing: "{}"'.format(
                    '", "'.join(full_cert_id[1] for full_cert_id in sorted(included_cert_ids))),
                cause=balancer_version)

    def _validate_vector(self, ctx, vector, valid_vector):
        """
        :param Vector vector: vector to validate
        :param Vector valid_vector: valid vector
        :return: Holder
        :raises: ConfigValidationError
        """
        spec_pbs = self._vector_to_specs(vector)
        upstreams, upstream_labels = self._collect_upstreams(spec_pbs)
        balancer_config = Holder(spec_pbs.balancer.yandex_balancer.config)
        l7_macro = balancer_config.get_nested_module() if balancer_config.is_l7_macro() else None
        if l7_macro and l7_macro.includes_domains():
            self._check_domain_certs_and_upstreams(spec_pbs, l7_macro, upstream_labels)
        else:
            self._check_certs(spec_pbs, vector.balancer_version, balancer_config)

        self._common_services_balancer_mode_validator.validate(
            valid_vector, vector.balancer_version, spec_pbs.balancer, spec_pbs.upstreams, upstreams)
        self._l7_fast_balancer_mode_validator.validate(
            vector.balancer_version, spec_pbs.balancer, spec_pbs.upstreams)

        with self._config_validation_timer.timer():
            start_time = monotonic.monotonic()
            result = validate_config(
                namespace_pb=spec_pbs.namespace,
                namespace_id=self._namespace_id,
                balancer_version=vector.balancer_version,
                balancer_spec_pb=spec_pbs.balancer,
                upstream_spec_pbs=spec_pbs.upstreams,
                backend_spec_pbs=spec_pbs.backends,
                endpoint_set_spec_pbs=spec_pbs.endpoint_sets,
                knob_spec_pbs=spec_pbs.knobs,
                cert_spec_pbs=spec_pbs.certs,
                domain_spec_pbs=spec_pbs.domains,
                weight_section_spec_pbs=spec_pbs.weight_sections,
                upstreams=upstreams,
                threadpool=self._threadpool,
                ctx=ctx)
            validation_time = monotonic.monotonic() - start_time
            if validation_time >= appconfig.get_value('run.l7_config_validation_too_long_in_seconds', 180):
                self._events.report(EVENT_L7_CONFIG_VALIDATION_TOO_LONG, ctx=ctx)
        return result

    def _vector_to_specs(self, vector):
        """
        :type vector: Vector
        :rtype: spec_pbs_tuple
        """
        balancer_spec_pb = find_revision_spec(vector.balancer_version)
        upstream_spec_pbs = {}
        upstream_ids = set()
        backend_spec_pbs = {}
        backend_ids = set()
        endpoint_set_spec_pbs = {}
        endpoint_set_rev_pbs = {}
        knob_spec_pbs = {}
        cert_spec_pbs = {}
        cert_ids = set()
        domain_spec_pbs = {}
        weight_section_spec_pbs = {}

        for full_domain_id, domain_version in six.iteritems(vector.domain_versions):
            if domain_version.deleted:
                continue
            if domain_version.incomplete:
                continue
            assert full_domain_id[0] == self._namespace_id
            domain_spec_pbs[domain_version] = find_revision_spec(domain_version)

        for full_upstream_id, upstream_version in six.iteritems(vector.upstream_versions):
            if upstream_version.deleted:
                continue
            assert full_upstream_id[0] == self._namespace_id
            upstream_spec_pbs[upstream_version] = find_revision_spec(upstream_version)
            upstream_ids.add(full_upstream_id)

        for full_backend_id, backend_version in six.iteritems(vector.backend_versions):
            if backend_version.deleted:
                continue
            backend_spec_pbs[backend_version] = find_revision_spec(backend_version)
            backend_ids.add(full_backend_id)

        for full_endpoint_set_id, endpoint_set_version in six.iteritems(vector.endpoint_set_versions):
            if endpoint_set_version.deleted:
                continue
            if full_endpoint_set_id not in vector.backend_versions:
                continue
            backend_version = vector.backend_versions[full_endpoint_set_id]
            if backend_version.deleted:
                continue
            endpoint_set_rev_pb = find_endpoint_set_revision2(endpoint_set_version)
            endpoint_set_rev_pbs[endpoint_set_version] = endpoint_set_rev_pb
            endpoint_set_spec_pbs[endpoint_set_version] = endpoint_set_rev_pb.spec

        for full_backend_id, backend_version in six.iteritems(vector.backend_versions):
            if backend_version.deleted:
                continue
            if backend_spec_pbs[backend_version].selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD:
                continue
            flattened_id = flatten_full_id2(full_backend_id)
            if full_backend_id not in vector.endpoint_set_versions:
                raise errors.ConfigValidationError(
                    'Backend "{}" is not resolved yet. '
                    'In some cases resolution can take up to a couple of minutes.'.format(flattened_id),
                    cause=backend_version)
            endpoint_set_version = vector.endpoint_set_versions[full_backend_id]
            endpoint_set_rev_pb = endpoint_set_rev_pbs[endpoint_set_version]
            if backend_version.version not in endpoint_set_rev_pb.meta.backend_versions:
                if backend_version.ctime > endpoint_set_version.ctime:
                    raise errors.ConfigValidationError(
                        'Backend "{}" is not resolved yet. '
                        'In some cases resolution can take up to a couple of minutes.'.format(flattened_id),
                        cause=backend_version)
                else:
                    raise errors.ConfigValidationError(
                        'Endpoint set "{}" is not compatible with its backend.'.format(flattened_id),
                        cause=max(backend_version, endpoint_set_version))

        for full_knob_id, knob_version in six.iteritems(vector.knob_versions):
            if knob_version.deleted:
                continue
            knob_spec_pbs[knob_version] = find_revision_spec(knob_version)

        for full_cert_id, cert_version in six.iteritems(vector.cert_versions):
            if cert_version.deleted:
                continue
            if cert_version.incomplete:
                continue
            cert_spec_pbs[cert_version] = find_revision_spec(cert_version)
            cert_ids.add(full_cert_id)

        for full_weight_section_id, weight_section_version in six.iteritems(vector.weight_section_versions):
            if weight_section_version.deleted:
                continue
            if weight_section_version.incomplete:
                continue
            weight_section_spec_pbs[weight_section_version] = find_revision_spec(weight_section_version)

        return spec_pbs_tuple(
            namespace=self._cache.must_get_namespace(self._namespace_id),
            balancer=balancer_spec_pb,
            domains=domain_spec_pbs,
            upstreams=upstream_spec_pbs,
            upstream_ids=upstream_ids,
            backends=backend_spec_pbs,
            backend_ids=backend_ids,
            endpoint_sets=endpoint_set_spec_pbs,
            knobs=knob_spec_pbs,
            certs=cert_spec_pbs,
            weight_sections=weight_section_spec_pbs,
            cert_ids=cert_ids,
        )

    def _report_rollback_count(self, ctx, rollback_count):
        ctx.log.debug('rollback_count=%s', rollback_count)
        self._rollback_gauge.set(rollback_count)

    def _get_version_to_rollback(self, ctx, vector, ignore_priority=False):
        """
        :param ignore_priority: used to rollback every single version from the vector, so sorting isn't needed
        """
        diff_balancer_version = None
        if vector.balancer_version:
            validated_pb = vector.get_validated_pb(vector.balancer_version, vector.balancer_version.balancer_id)
            if validated_pb and validated_pb.status == u'False':
                return vector.balancer_version
            valid_balancer_version = self._state_proxy.valid_vector.balancer_version
            if vector.balancer_version != valid_balancer_version:
                # try not to rollback balancer if it wasn't valid even once
                # i'm not sure it's correct, leaving it as is for now to minimize tests diff
                if valid_balancer_version is not None:
                    return vector.balancer_version
                diff_balancer_version = vector.balancer_version  # remember it as a last resort
        if ignore_priority and diff_balancer_version:
            return diff_balancer_version

        # the ordering of these fields is intentional, it's chosen to provide the most useful error messages to the user
        for version_type in (u'domain_versions', u'upstream_versions', u'backend_versions',
                             u'endpoint_set_versions', u'knob_versions', u'cert_versions', u'weight_section_versions'):
            first_diff_version = None
            first_invalid_version, first_invalid_version_mtime = None, None
            for full_id, version in six.iteritems(vector.must_get_version_dict(version_type)):
                validated_pb = vector.get_validated_pb(version, full_id)
                if validated_pb and validated_pb.status == u'False':
                    mtime = validated_pb.last_transition_time.ToMicroseconds()
                    if first_invalid_version is None or first_invalid_version_mtime < mtime:
                        first_invalid_version = version
                        first_invalid_version_mtime = mtime
                else:
                    valid_version = self._state_proxy.valid_vector.get_version_item(version_type, full_id)
                    if version != valid_version:
                        if isinstance(version, EndpointSetVersion) and valid_version is None:
                            # don't roll back ES version if there is no valid version of it
                            continue
                        else:
                            if first_diff_version is None or first_diff_version.ctime < version.ctime:
                                first_diff_version = version
                if ignore_priority:
                    if first_invalid_version is not None:
                        return first_invalid_version
                    if first_diff_version is not None:
                        return first_diff_version

            # prefer invalid version, but a mismatching one also works
            if first_invalid_version is not None:
                return first_invalid_version
            elif first_diff_version is not None:
                return first_diff_version
            # otherwise, try the next objects group

        # if we didn't find anything invalid or different, attempt to return balancer version as the last resort
        if diff_balancer_version is not None:
            return diff_balancer_version

        return None

    def _rollback_version(self, vector, version_to_rollback):
        valid_vector = self._state_proxy.valid_vector
        if isinstance(version_to_rollback, BalancerVersion):
            return vector.replace_balancer_version(valid_vector.balancer_version)
        elif isinstance(version_to_rollback, DomainVersion):
            full_domain_id = version_to_rollback.domain_id
            if full_domain_id in valid_vector.domain_versions:
                return vector.replace_domain_version(full_domain_id,
                                                     valid_vector.domain_versions[full_domain_id])
            else:
                return vector.remove_domain_version(full_domain_id)
        elif isinstance(version_to_rollback, UpstreamVersion):
            full_upstream_id = version_to_rollback.upstream_id
            if full_upstream_id in valid_vector.upstream_versions:
                return vector.replace_upstream_version(full_upstream_id,
                                                       valid_vector.upstream_versions[full_upstream_id])
            else:
                return vector.remove_upstream_version(full_upstream_id)
        elif isinstance(version_to_rollback, BackendVersion):
            full_backend_id = version_to_rollback.backend_id
            if full_backend_id in valid_vector.backend_versions:
                return vector.replace_backend_version(full_backend_id, valid_vector.backend_versions[full_backend_id])
            else:
                rv = vector.remove_backend_version(full_backend_id)
                if full_backend_id in rv.endpoint_set_versions:
                    return rv.remove_endpoint_set_version(full_backend_id)
                else:
                    return rv
        elif isinstance(version_to_rollback, EndpointSetVersion):
            endpoint_set_id = version_to_rollback.endpoint_set_id
            if endpoint_set_id in valid_vector.endpoint_set_versions:
                return vector.replace_endpoint_set_version(endpoint_set_id,
                                                           valid_vector.endpoint_set_versions[endpoint_set_id])
            else:
                rv = vector.remove_endpoint_set_version(endpoint_set_id)
                if endpoint_set_id in rv.backend_versions:
                    return rv.remove_backend_version(endpoint_set_id)
                else:
                    return rv
        elif isinstance(version_to_rollback, KnobVersion):
            full_knob_id = version_to_rollback.knob_id
            if full_knob_id in valid_vector.knob_versions:
                return vector.replace_knob_version(full_knob_id, valid_vector.knob_versions[full_knob_id])
            else:
                return vector.remove_knob_version(full_knob_id)
        elif isinstance(version_to_rollback, CertVersion):
            full_cert_id = version_to_rollback.cert_id
            if full_cert_id in valid_vector.cert_versions:
                return vector.replace_cert_version(full_cert_id, valid_vector.cert_versions[full_cert_id])
            else:
                return vector.remove_cert_version(full_cert_id)
        elif isinstance(version_to_rollback, objects.WeightSection.version):
            full_weight_section_id = version_to_rollback.id
            if full_weight_section_id in valid_vector.weight_section_versions:
                return vector.replace_weight_section_version(full_weight_section_id, valid_vector.weight_section_versions[full_weight_section_id])
            else:
                return vector.remove_weight_section_version(full_weight_section_id)
        else:
            raise AssertionError('Unsupported version type: {}'.format(type(version_to_rollback)))

    def _mark_version_as(self, status, version, message):
        """
        :param status:
        :param version:
        :type message: six.text_type
        :return:
        """
        for balancer_state_pb in self._zk.update_balancer_state(self._namespace_id, self._balancer_id,
                                                                balancer_state_pb=self._balancer_state_pb):
            h = L7BalancerStateHandler(balancer_state_pb)
            updated = h.select_rev(version).set_validated(status=status, message=message)
            if not updated:
                break

    def _mark_vector_as_valid(self, vector, message):
        for balancer_state_pb in self._zk.update_balancer_state(self._namespace_id, self._balancer_id,
                                                                balancer_state_pb=self._balancer_state_pb):
            updated = False
            h = L7BalancerStateHandler(balancer_state_pb)
            for version in vector:
                updated |= h.select_rev(version).set_validated(status='True', message=message)
            if not updated:
                break

    def validate(self, ctx):
        with self._rollback_validation_timer.timer():
            result = self._validate(ctx)
        self._report_rollback_count(ctx, result.rollback_count)
        return result

    def _validate(self, ctx):
        # the algorithm behind this code is described on wiki:
        # https://wiki.yandex-team.ru/users/romanovich/balancers-2.0-implementation/
        root_ctx = ctx
        ctx = ctx.with_op(op_id='validate')

        valid_v = self._state_proxy.valid_vector
        curr_v = self._state_proxy.curr_vector
        if curr_v != valid_v:
            ctx.log.debug('Validating changes')
            if self._is_large:
                diff = get_human_readable_diff(self._namespace_id, valid_v, curr_v, show_revision_ids=True)
                for line in diff.split('\n'):
                    ctx.log.debug(line)
            else:
                ctx.log.debug('valid vector: %s', valid_v)
                ctx.log.debug('current vector: %s', curr_v)

        result = ValidationResult(None, set(), set(), set(), set(), set(), set(), set())
        rollback_count = 0
        while 1:
            if curr_v == valid_v:
                ctx.log.debug('Current vector is already valid, nothing to validate...')
                result.rollback_count = rollback_count
                return result

            if not curr_v.balancer_version:
                ctx.log.debug('Current vector does not contain balancer version, nothing to validate...')
                break

            if not curr_v.needs_validation():
                ctx.log.debug('Nothing to validate in current vector')
                break

            curr_v.omit_unneeded_endpoint_sets()
            try:
                if self._is_large:
                    ctx.log.debug('Validating current vector...')
                else:
                    ctx.log.debug('Validating current vector (%s)...', curr_v)
                result = self._validate_vector(ctx=root_ctx, vector=curr_v, valid_vector=valid_v)
            except errors.ConfigValidationError as e:
                ctx.log.debug('Current vector is not valid: %s', e)
                # if current vector is not valid, we must find a change that caused an error
                tryout_curr_vector = curr_v
                # remember the first error
                error = e
                # and start looking
                while 1:
                    ctx.log.debug('Start looking for a change that caused vector to be invalid...')
                    # undo the latest change
                    version_to_rollback = None
                    if error.cause:
                        if appconfig.get_value('run.l7_validator_allow_infinite_error', False):
                            version_to_rollback = error.cause
                        else:
                            valid_version = self._state_proxy.valid_vector.get_version_item_by_version(error.cause)
                            if valid_version == error.cause:
                                ctx.log.warning('Rollback hint matches valid version, ignoring it: %s', error.cause)
                            else:
                                ctx.log.debug('Error caused by %s', error.cause)
                                version_to_rollback = error.cause
                    if not version_to_rollback:
                        ctx.log.debug('Choosing rollback version by heuristic')
                        version_to_rollback = self._get_version_to_rollback(ctx, tryout_curr_vector)
                    ctx.log.debug('Version to rollback: %s', version_to_rollback)
                    if not version_to_rollback:
                        curr_v = tryout_curr_vector
                        break
                    prev_tryout_curr_vector, tryout_curr_vector = (
                        tryout_curr_vector, self._rollback_version(tryout_curr_vector, version_to_rollback)
                    )
                    rollback_count += 1
                    if prev_tryout_curr_vector == tryout_curr_vector:
                        self._infinite_loop_counter.inc(1)
                        raise RuntimeError('Infinite loop detected: rolling back {} from {}:{} results in the same '
                                           'vector'.format(version_to_rollback, self._namespace_id, self._balancer_id))
                    if self._is_large:
                        ctx.log.debug('Rolled back %s, validating tryout vector', version_to_rollback)
                    else:
                        ctx.log.debug('Rolled back %s, validating tryout vector %s',
                                      version_to_rollback, tryout_curr_vector)

                    tryout_curr_vector.omit_unneeded_endpoint_sets()
                    if not tryout_curr_vector.needs_validation():
                        # it means that we just rolled back last non-deleted invalid upstream version,
                        # let's consider w valid to stop here and rollback this version from the `curr_v`
                        pass
                    else:
                        try:
                            self._validate_vector(ctx=root_ctx, vector=tryout_curr_vector, valid_vector=valid_v)
                        except errors.ConfigValidationError as e:
                            # if the vector is still invalid, remember the error and continue undoing
                            error = e
                            ctx.log.debug('Tryout vector is not valid: %s', e)
                            continue

                    ctx.log.debug('Tryout vector is valid, mark %s as invalid and '
                                  'roll it back from current vector', version_to_rollback)
                    # the vector is valid, therefore we can tell that `version_to_rollback` caused the error
                    # we mark change as invalid
                    ctx.log.debug('Marking version as invalid: %s (%s)', version_to_rollback, error.message)
                    self._mark_version_as('False', version_to_rollback, ujson.dumps(error.to_dict(), indent=2))
                    # and remove it from the current vector
                    curr_v = self._rollback_version(curr_v, version_to_rollback)
                    if self._is_large:
                        ctx.log.debug('New current vector')
                    else:
                        ctx.log.debug('New current vector: %s', curr_v)
                    break
            else:
                break

        if not curr_v.needs_validation():
            version_to_rollback = self._get_version_to_rollback(ctx, curr_v, ignore_priority=True)
            ctx.log.debug('Marking vector as pending, version to rollback: %s', version_to_rollback)
            while version_to_rollback:
                if isinstance(version_to_rollback, BalancerVersion):
                    message = 'Can not validate balancer spec due to absence of valid upstreams and backends'
                elif isinstance(version_to_rollback, DomainVersion):
                    message = 'Can not remove last domain from balancer'
                elif isinstance(version_to_rollback, UpstreamVersion):
                    message = 'Can not remove last upstream from balancer'
                elif isinstance(version_to_rollback, BackendVersion):
                    message = 'Can not remove last backend from balancer'
                elif isinstance(version_to_rollback, EndpointSetVersion):
                    message = 'Can not remove last endpoint set from balancer'
                elif isinstance(version_to_rollback, KnobVersion):
                    message = 'Can not remove last knob from balancer'
                elif isinstance(version_to_rollback, CertVersion):
                    message = 'Can not remove last certificate from balancer'
                elif isinstance(version_to_rollback, objects.WeightSection.version):
                    message = 'Can not remove last weight section from balancer'
                else:
                    raise RuntimeError('Unknown version to rollback: {}'.format(version_to_rollback))
                ctx.log.debug('Marking version as pending: %s (%s)', version_to_rollback, message)
                self._mark_version_as('Pending', version_to_rollback, ujson.dumps({'message': message}))
                curr_v = self._rollback_version(curr_v, version_to_rollback)
                version_to_rollback = self._get_version_to_rollback(ctx, curr_v, ignore_priority=True)
        else:
            assert result is not None
            if self._is_large:
                ctx.log.debug('New valid vector')
            else:
                ctx.log.debug('New valid vector: %s', curr_v)
            ctx.log.debug('Setting validated to true for balancer, its domains, '
                          'upstreams, backends, endpoint sets, knobs, certs, and weight sections')
            for full_domain_id, domain_version in list(six.iteritems(curr_v.domain_versions)):
                if domain_version.deleted:
                    continue
                if domain_version.incomplete:
                    continue
                if full_domain_id not in result.included_full_domain_ids:
                    curr_v.domain_versions.pop(full_domain_id, None)
            for full_upstream_id, upstream_version in list(six.iteritems(curr_v.upstream_versions)):
                if upstream_version.deleted:
                    continue
                if full_upstream_id not in result.included_full_upstream_ids:
                    curr_v.upstream_versions.pop(full_upstream_id, None)
            for full_backend_id, backend_version in list(six.iteritems(curr_v.backend_versions)):
                if backend_version.deleted:
                    continue
                if full_backend_id not in result.included_full_backend_ids:
                    curr_v.backend_versions.pop(full_backend_id, None)
            for full_endpoint_set_id, endpoint_set_version in list(six.iteritems(curr_v.endpoint_set_versions)):
                if endpoint_set_version.deleted:
                    continue
                if full_endpoint_set_id not in result.included_full_backend_ids:
                    curr_v.endpoint_set_versions.pop(full_endpoint_set_id, None)
            for full_knob_id, knob_version in list(six.iteritems(curr_v.knob_versions)):
                if knob_version.deleted:
                    continue
                if full_knob_id not in result.included_full_knob_ids:
                    curr_v.knob_versions.pop(full_knob_id, None)
            for full_cert_id, cert_version in list(six.iteritems(curr_v.cert_versions)):
                if cert_version.deleted:
                    continue
                if cert_version.incomplete:
                    continue
                if full_cert_id not in result.included_full_cert_ids:
                    curr_v.cert_versions.pop(full_cert_id, None)
            for full_weight_section_id, weight_section_version in list(six.iteritems(curr_v.weight_section_versions)):
                if weight_section_version.deleted:
                    continue
                if full_weight_section_id not in result.included_full_weight_section_ids:
                    curr_v.weight_section_versions.pop(full_weight_section_id, None)
            if self._is_large:
                ctx.log.debug('Filtered valid vector')
            else:
                ctx.log.debug('Filtered valid vector: %s', curr_v)
            self._mark_vector_as_valid(curr_v, message='')
            ctx.log.debug('Set validated to true for balancer, its domains, '
                          'upstreams, backends, endpoint sets, knobs, certs, and weight sections')
        result.rollback_count = rollback_count
        return result

    def handle_balancer_state_update(self, balancer_state_pb, ctx):
        """
        :type balancer_state_pb: model_pb2.BalancerState
        :type ctx: context.OpCtx
        """
        pb = balancer_state_pb
        if balancer_state_pb.namespace_id != self._namespace_id:
            return

        if (self._last_seen_balancer_state_generation is not None and
                self._last_seen_balancer_state_generation >= pb.generation):
            if self._last_seen_balancer_state_generation > balancer_state_pb.generation:
                ctx = ctx.with_op(op_id='validator_state_update')
                ctx.log.warn('Received balancer state generation older than last seen: %d > %d',
                             self._last_seen_balancer_state_generation, pb.generation)
            return

        curr_v = self._state_proxy.curr_vector
        self.set_balancer_state_pb(pb)
        new_curr_v = self._state_proxy.curr_vector
        if curr_v != new_curr_v:
            self.validate(ctx)
