# coding: utf-8
import collections

import enum
import itertools
import six

from awacs.lib.order_processor.model import needs_removal
from awacs.lib.strutils import flatten_full_id2, flatten_full_id
from awacs.model import util, db, errors, objects
from awacs.model.balancer.component_transports.diff import ComponentVersion
from infra.awacs.proto import model_pb2


class VersionName(enum.Enum):
    DOMAIN = 'DomainVersion'
    UPSTREAM = 'UpstreamVersion'
    BACKEND = 'BackendVersion'
    ENDPOINT_SET = 'EndpointSetVersion'
    BALANCER = 'BalancerVersion'
    KNOB = 'KnobVersion'
    CERT = 'CertVersion'
    WEIGHT_SECTION = 'WeightSectionVersion'


def _join_as_bytes(full_id, version):
    """
    :type full_id: (six.text_type, six.text_type)
    :rtype: six.binary_type
    """
    return u'{}:{}:{}'.format(full_id[0], full_id[1], version).encode('utf-8')


def copy_dict_of_dicts(d):
    return {key: dict(values) for key, values in six.iteritems(d)}


class Vector(object):
    __slots__ = ('balancer_version', 'upstream_versions', 'backend_versions', 'endpoint_set_versions', 'knob_versions',
                 'cert_versions', 'domain_versions', 'weight_section_versions', '_validated_pbs')

    def __init__(self,
                 balancer_version,
                 upstream_versions,
                 domain_versions,
                 backend_versions,
                 endpoint_set_versions,
                 knob_versions,
                 cert_versions,
                 weight_section_versions,
                 validated_pbs=None):
        """
        :type balancer_version: BalancerVersion | None
        :type upstream_versions: dict[(six.text_type, six.text_type), UpstreamVersion]
        :type backend_versions: dict[(six.text_type, six.text_type), BackendVersion]
        :type endpoint_set_versions: dict[(six.text_type, six.text_type), EndpointSetVersion]
        :type knob_versions: dict[(six.text_type, six.text_type), KnobVersion]
        :type cert_versions: dict[(six.text_type, six.text_type), CertVersion]
        :type domain_versions: dict[(six.text_type, six.text_type), DomainVersion]
        :type weight_section_versions: dict[(six.text_type, six.text_type), objects.WeightSection.version]
        :type validated_pbs: dict[six.text_type, dict[(six.text_type, six.text_type), model_pb2.Condition]]
        """
        self.balancer_version = balancer_version
        self.upstream_versions = upstream_versions
        self.domain_versions = domain_versions
        self.backend_versions = backend_versions
        self.endpoint_set_versions = endpoint_set_versions
        self.knob_versions = knob_versions
        self.cert_versions = cert_versions
        self.weight_section_versions = weight_section_versions
        self._validated_pbs = validated_pbs or {}

    @classmethod
    def empty(cls):
        return cls(balancer_version=None,
                   upstream_versions={},
                   domain_versions={},
                   backend_versions={},
                   endpoint_set_versions={},
                   knob_versions={},
                   cert_versions={},
                   weight_section_versions={})

    def __repr__(self):
        return '{}(upstreams={}, domains={}, backends={}, endpoint_sets={}, knobs={}, certs={}, weight_sections={})'.format(
            util.version_to_str(self.balancer_version),
            util.versions_dict_to_str(self.upstream_versions),
            util.versions_dict_to_str(self.domain_versions),
            util.versions_dict_to_str(self.backend_versions),
            util.versions_dict_to_str(self.endpoint_set_versions),
            util.versions_dict_to_str(self.knob_versions),
            util.versions_dict_to_str(self.cert_versions),
            util.versions_dict_to_str(self.weight_section_versions),
        )

    def __iter__(self):
        if self.balancer_version:
            yield self.balancer_version
        for version in itertools.chain(
                six.itervalues(self.upstream_versions),
                six.itervalues(self.domain_versions),
                six.itervalues(self.backend_versions),
                six.itervalues(self.endpoint_set_versions),
                six.itervalues(self.knob_versions),
                six.itervalues(self.cert_versions),
                six.itervalues(self.weight_section_versions),
        ):
            yield version

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return NotImplemented
        return (
                self.balancer_version == other.balancer_version and
                self.upstream_versions == other.upstream_versions and
                self.domain_versions == other.domain_versions and
                self.backend_versions == other.backend_versions and
                self.endpoint_set_versions == other.endpoint_set_versions and
                self.knob_versions == other.knob_versions and
                self.cert_versions == other.cert_versions and
                self.weight_section_versions == other.weight_section_versions
        )

    def __ne__(self, other):
        return not self.__eq__(other)

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        h = 0
        if self.balancer_version:
            h = util.crc32(self.balancer_version.get_weak_hash(), h)
        for backend_id, backend_version in sorted(six.iteritems(self.backend_versions)):
            h = util.crc32(backend_version.get_weak_hash(), h)
        for upstream_id, upstream_version in sorted(six.iteritems(self.upstream_versions)):
            h = util.crc32(upstream_version.get_weak_hash(), h)
        for domain_id, domain_version in sorted(six.iteritems(self.domain_versions)):
            h = util.crc32(domain_version.get_weak_hash(), h)
        for endpoint_set_id, endpoint_set_version in sorted(six.iteritems(self.endpoint_set_versions)):
            h = util.crc32(endpoint_set_version.get_weak_hash(), h)
        for knob_id, knob_version in sorted(six.iteritems(self.knob_versions)):
            h = util.crc32(knob_version.get_weak_hash(), h)
        for cert_id, cert_version in sorted(six.iteritems(self.cert_versions)):
            h = util.crc32(cert_version.get_weak_hash(), h)
        for weight_section_id, weight_section_version in sorted(six.iteritems(self.weight_section_versions)):
            h = util.crc32(weight_section_version.get_weak_hash(), h)

        return util.int_to_hex_bytes(h)

    def get_weak_hash_str(self):
        """
        :rtype: six.text_type
        """
        return self.get_weak_hash().decode('utf-8')

    def is_empty(self):
        return (
            not self.balancer_version and
            not self.upstream_versions and
            not self.domain_versions and
            not self.backend_versions and
            not self.endpoint_set_versions and
            not self.knob_versions and
            not self.cert_versions and
            not self.weight_section_versions
        )

    def greater_than(self, other):
        return self != other and (
            (
                (self.balancer_version is None and other.balancer_version is None) or
                (self.balancer_version is not None and other.balancer_version is None) or
                (self.balancer_version is not None and
                 other.balancer_version is not None and
                 self.balancer_version >= other.balancer_version)
            ) and
            all((self.upstream_versions[upstream_id] >= other.upstream_versions[upstream_id]
                 for upstream_id in self.upstream_versions
                 if upstream_id in other.upstream_versions)) and
            all((self.domain_versions[domain_id] >= other.domain_versions[domain_id]
                 for domain_id in self.domain_versions
                 if domain_id in other.domain_versions)) and
            all((self.backend_versions[backend_id] >= other.backend_versions[backend_id]
                 for backend_id in self.backend_versions
                 if backend_id in other.backend_versions)) and
            all((self.endpoint_set_versions[endpoint_set_id] >= other.endpoint_set_versions[endpoint_set_id]
                 for endpoint_set_id in self.endpoint_set_versions
                 if endpoint_set_id in other.endpoint_set_versions)) and
            all((self.knob_versions[knob_id] >= other.knob_versions[knob_id]
                 for knob_id in self.knob_versions
                 if knob_id in other.knob_versions)) and
            all((self.cert_versions[cert_id] >= other.cert_versions[cert_id]
                 for cert_id in self.cert_versions
                 if cert_id in other.cert_versions)) and
            all((self.weight_section_versions[weight_section_id] >= other.weight_section_versions[weight_section_id]
                 for weight_section_id in self.weight_section_versions
                 if weight_section_id in other.weight_section_versions))
        )

    def diff(self, to):
        """
        :type to: Vector
        :rtype: util.Diff
        """
        updated = set()
        added = set()
        removed = set()
        if self.balancer_version != to.balancer_version:
            if self.balancer_version is None:
                added.add(to.balancer_version)
            elif to.balancer_version is None:
                removed.add(self.balancer_version)
            else:
                updated.add((self.balancer_version, to.balancer_version))

        for upstream_id, to_upstream_version in six.iteritems(to.upstream_versions):
            if upstream_id in self.upstream_versions:
                from_upstream_version = self.upstream_versions[upstream_id]
                if from_upstream_version != to_upstream_version:
                    updated.add((from_upstream_version, to_upstream_version))
            else:
                added.add(to_upstream_version)
        for upstream_id, from_upstream_version in six.iteritems(self.upstream_versions):
            if upstream_id not in to.upstream_versions:
                removed.add(from_upstream_version)

        for domain_id, to_domain_version in six.iteritems(to.domain_versions):
            if domain_id in self.domain_versions:
                from_domain_version = self.domain_versions[domain_id]
                if from_domain_version != to_domain_version:
                    updated.add((from_domain_version, to_domain_version))
            else:
                added.add(to_domain_version)
        for domain_id, from_domain_version in six.iteritems(self.domain_versions):
            if domain_id not in to.domain_versions:
                removed.add(from_domain_version)

        for backend_id, to_backend_version in six.iteritems(to.backend_versions):
            if backend_id in self.backend_versions:
                from_backend_version = self.backend_versions[backend_id]
                if from_backend_version != to_backend_version:
                    updated.add((from_backend_version, to_backend_version))
            else:
                added.add(to_backend_version)
        for backend_id, from_backend_version in six.iteritems(self.backend_versions):
            if backend_id not in to.backend_versions:
                removed.add(from_backend_version)

        for endpoint_set_id, to_endpoint_set_version in six.iteritems(to.endpoint_set_versions):
            if endpoint_set_id in self.endpoint_set_versions:
                from_endpoint_set_version = self.endpoint_set_versions[endpoint_set_id]
                if from_endpoint_set_version != to_endpoint_set_version:
                    updated.add((from_endpoint_set_version, to_endpoint_set_version))
            else:
                added.add(to_endpoint_set_version)
        for endpoint_set_id, from_endpoint_set_version in six.iteritems(self.endpoint_set_versions):
            if endpoint_set_id not in to.endpoint_set_versions:
                removed.add(from_endpoint_set_version)

        for knob_id, to_knob_version in six.iteritems(to.knob_versions):
            if knob_id in self.knob_versions:
                from_knob_version = self.knob_versions[knob_id]
                if from_knob_version != to_knob_version:
                    updated.add((from_knob_version, to_knob_version))
            else:
                added.add(to_knob_version)
        for knob_id, from_knob_version in six.iteritems(self.knob_versions):
            if knob_id not in to.knob_versions:
                removed.add(from_knob_version)

        for cert_id, to_cert_version in six.iteritems(to.cert_versions):
            if cert_id in self.cert_versions:
                from_cert_version = self.cert_versions[cert_id]
                if from_cert_version != to_cert_version:
                    updated.add((from_cert_version, to_cert_version))
            else:
                added.add(to_cert_version)
        for cert_id, from_cert_version in six.iteritems(self.cert_versions):
            if cert_id not in to.cert_versions:
                removed.add(from_cert_version)

        for weight_section_id, to_weight_section_version in six.iteritems(to.weight_section_versions):
            if weight_section_id in self.weight_section_versions:
                from_weight_section_version = self.weight_section_versions[weight_section_id]
                if from_weight_section_version != to_weight_section_version:
                    updated.add((from_weight_section_version, to_weight_section_version))
            else:
                added.add(to_weight_section_version)
        for weight_section_id, from_weight_section_version in six.iteritems(self.weight_section_versions):
            if weight_section_id not in to.weight_section_versions:
                removed.add(from_weight_section_version)

        return util.Diff(updated=updated, added=added, removed=removed)

    def replace_balancer_version(self, balancer_version):
        """
        :type balancer_version: BalancerVersion | None
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.balancer_version = balancer_version
        new_vector._validated_pbs[VersionName.BALANCER].pop(self.balancer_version.balancer_id, None)
        return new_vector

    def replace_upstream_version(self, full_upstream_id, upstream_version):
        """
        :type full_upstream_id: (six.text_type, six.text_type)
        :type upstream_version: UpstreamVersion
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.upstream_versions[full_upstream_id] = upstream_version
        new_vector._validated_pbs[VersionName.UPSTREAM].pop(full_upstream_id, None)
        return new_vector

    def remove_upstream_version(self, full_upstream_id):
        """
        :type full_upstream_id: (six.text_type, six.text_type)
        :rtype: Vector
        """
        assert isinstance(full_upstream_id, tuple) and len(full_upstream_id) == 2
        new_vector = self.clone()
        del new_vector.upstream_versions[full_upstream_id]
        new_vector._validated_pbs[VersionName.UPSTREAM].pop(full_upstream_id, None)
        return new_vector

    def replace_domain_version(self, full_domain_id, domain_version):
        """
        :type full_domain_id: (six.text_type, six.text_type)
        :type domain_version: DomainVersion
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.domain_versions[full_domain_id] = domain_version
        new_vector._validated_pbs[VersionName.DOMAIN].pop(full_domain_id, None)
        return new_vector

    def remove_domain_version(self, full_domain_id):
        """
        :type full_domain_id: (six.text_type, six.text_type)
        :rtype: Vector
        """
        assert isinstance(full_domain_id, tuple) and len(full_domain_id) == 2
        new_vector = self.clone()
        del new_vector.domain_versions[full_domain_id]
        new_vector._validated_pbs[VersionName.DOMAIN].pop(full_domain_id, None)
        return new_vector

    def replace_backend_version(self, full_backend_id, backend_version):
        """
        :type full_backend_id: (six.text_type, six.text_type)
        :type backend_version: BackendVersion
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.backend_versions[full_backend_id] = backend_version
        new_vector._validated_pbs[VersionName.BACKEND].pop(full_backend_id, None)
        return new_vector

    def remove_backend_version(self, full_backend_id):
        """
        :type full_backend_id: (six.text_type, six.text_type)
        :rtype: Vector
        """
        new_vector = self.clone()
        del new_vector.backend_versions[full_backend_id]
        new_vector._validated_pbs[VersionName.BACKEND].pop(full_backend_id, None)
        return new_vector

    def replace_endpoint_set_version(self, full_endpoint_set_id, endpoint_set_version):
        """
        :type full_endpoint_set_id: (six.text_type, six.text_type)
        :type endpoint_set_version: EndpointSetVersion
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.endpoint_set_versions[full_endpoint_set_id] = endpoint_set_version
        new_vector._validated_pbs[VersionName.ENDPOINT_SET].pop(full_endpoint_set_id, None)
        return new_vector

    def remove_endpoint_set_version(self, full_endpoint_set_id):
        """
        :type full_endpoint_set_id: (six.text_type, six.text_type)
        :rtype: Vector
        """
        new_vector = self.clone()
        del new_vector.endpoint_set_versions[full_endpoint_set_id]
        new_vector._validated_pbs[VersionName.ENDPOINT_SET].pop(full_endpoint_set_id, None)
        return new_vector

    def replace_knob_version(self, full_knob_id, knob_version):
        """
        :type full_knob_id: (six.text_type, six.text_type)
        :type knob_version: KnobVersion
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.knob_versions[full_knob_id] = knob_version
        new_vector._validated_pbs[VersionName.KNOB].pop(full_knob_id, None)
        return new_vector

    def remove_knob_version(self, full_knob_id):
        """
        :type full_knob_id: (six.text_type, six.text_type)
        :rtype: Vector
        """
        new_vector = self.clone()
        del new_vector.knob_versions[full_knob_id]
        new_vector._validated_pbs[VersionName.KNOB].pop(full_knob_id, None)
        return new_vector

    def replace_cert_version(self, full_cert_id, cert_version):
        """
        :type full_cert_id: (six.text_type, six.text_type)
        :type cert_version: CertVersion
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.cert_versions[full_cert_id] = cert_version
        new_vector._validated_pbs[VersionName.CERT].pop(full_cert_id, None)
        return new_vector

    def remove_cert_version(self, full_cert_id):
        """
        :type full_cert_id: (six.text_type, six.text_type)
        :rtype: Vector
        """
        new_vector = self.clone()
        del new_vector.cert_versions[full_cert_id]
        new_vector._validated_pbs[VersionName.CERT].pop(full_cert_id, None)
        return new_vector

    def replace_weight_section_version(self, full_weight_section_id, weight_section_version):
        """
        :type full_weight_section_id: (six.text_type, six.text_type)
        :type weight_section_version: objects.WeightSection.version
        :rtype: Vector
        """
        new_vector = self.clone()
        new_vector.weight_section_versions[full_weight_section_id] = weight_section_version
        new_vector._validated_pbs[VersionName.WEIGHT_SECTION].pop(full_weight_section_id, None)
        return new_vector

    def remove_weight_section_version(self, full_weight_section_id):
        """
        :type full_weight_section_id: (six.text_type, six.text_type)
        :rtype: Vector
        """
        new_vector = self.clone()
        del new_vector.weight_section_versions[full_weight_section_id]
        new_vector._validated_pbs[VersionName.WEIGHT_SECTION].pop(full_weight_section_id, None)
        return new_vector

    def clone(self):
        return Vector(self.balancer_version,
                      dict(self.upstream_versions),
                      dict(self.domain_versions),
                      dict(self.backend_versions),
                      dict(self.endpoint_set_versions),
                      dict(self.knob_versions),
                      dict(self.cert_versions),
                      dict(self.weight_section_versions),
                      copy_dict_of_dicts(self._validated_pbs))

    def omit_unneeded_endpoint_sets(self):
        for endpoint_set_id in list(self.endpoint_set_versions):
            if endpoint_set_id not in self.backend_versions:
                del self.endpoint_set_versions[endpoint_set_id]
                self._validated_pbs[VersionName.ENDPOINT_SET].pop(endpoint_set_id, None)

    def needs_validation(self):
        if not self.balancer_version:
            return False
        for domain_version in six.itervalues(self.domain_versions):
            if not (domain_version.deleted or domain_version.incomplete):
                return True
        for upstream_version in six.itervalues(self.upstream_versions):
            if not upstream_version.deleted:
                return True
        for backend_version in six.itervalues(self.backend_versions):
            if not backend_version.deleted:
                return True
        for knob_version in six.itervalues(self.knob_versions):
            if not knob_version.deleted:
                return True
        for cert_version in six.itervalues(self.cert_versions):
            if not (cert_version.deleted or cert_version.incomplete):
                return True
        for weight_section_version in six.itervalues(self.weight_section_versions):
            if not (weight_section_version.deleted or weight_section_version.incomplete):
                return True
        return False

    def must_get_version_dict(self, version_field_name):
        return getattr(self, version_field_name)

    def get_version_item(self, version_field_name, version_id):
        if version_field_name == 'balancer_version':
            return self.balancer_version
        return self.must_get_version_dict(version_field_name).get(version_id)

    def get_version_item_by_version(self, version):
        version_class = type(version)
        if version_class is BalancerVersion:
            return self.balancer_version
        full_id = getattr(version, version_class.id_field_name)
        return self.must_get_version_dict(version_class.vector_field_name).get(full_id)

    def get_validated_pb(self, version, full_id):
        return self._validated_pbs[VersionName(version.__class__.__name__)].get(full_id)


class BalancerVersion(collections.namedtuple('BalancerVersion', ['ctime', 'balancer_id', 'version'])):
    deleted = False
    id_field_name = 'balancer_id'
    vector_field_name = 'balancer_version'

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        return _join_as_bytes(self.balancer_id, self.version)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Balancer
        :rtype: BalancerVersion
        """
        return cls(pb.meta.mtime.ToMicroseconds(), (pb.meta.namespace_id, pb.meta.id), pb.meta.version)

    @classmethod
    def from_rev_status_pb(cls, full_balancer_id, pb):
        """
        :type full_balancer_id: (six.text_type, six.text_type)
        :type pb: awacs.proto.model_pb2.BalancerState.RevisionStatus
        :rtype: BalancerVersion
        """
        assert isinstance(full_balancer_id, tuple) and len(full_balancer_id) == 2
        return cls(pb.ctime.ToMicroseconds(), full_balancer_id, pb.revision_id)

    def __repr__(self):
        return 'Bal(id={}, v={}, ctime={})'.format(self.balancer_id, self.version[:8], self.ctime)


class UpstreamVersion(collections.namedtuple('UpstreamVersion',
                                             ('ctime', 'upstream_id', 'version', 'deleted'))):
    id_field_name = 'upstream_id'
    vector_field_name = 'upstream_versions'

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        return _join_as_bytes(self.upstream_id, self.version)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Upstream
        :rtype: UpstreamVersion
        """
        return cls(pb.meta.mtime.ToMicroseconds(), (pb.meta.namespace_id, pb.meta.id), pb.meta.version, pb.spec.deleted)

    @classmethod
    def from_rev_status_pb(cls, full_upstream_id, pb):
        """
        :type full_upstream_id: (six.text_type, six.text_type)
        :type pb: awacs.proto.model_pb2.BalancerState.RevisionStatus
        :rtype: UpstreamVersion
        """
        assert isinstance(full_upstream_id, tuple) and len(full_upstream_id) == 2
        return cls(pb.ctime.ToMicroseconds(), full_upstream_id, pb.revision_id, pb.deleted)

    def __repr__(self):
        return 'Up(id={}, v={}, ctime={}, d={})'.format(
            self.upstream_id, self.version[:8], self.ctime, int(self.deleted))


class DomainVersion(collections.namedtuple('DomainVersion',
                                           ('ctime', 'domain_id', 'version', 'deleted', 'incomplete'))):
    id_field_name = 'domain_id'
    vector_field_name = 'domain_versions'

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        return _join_as_bytes(self.domain_id, self.version)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Domain
        :rtype: DomainVersion
        """
        return cls(pb.meta.mtime.ToMicroseconds(),
                   (pb.meta.namespace_id, pb.meta.id),
                   pb.meta.version,
                   pb.spec.deleted,
                   pb.spec.incomplete)

    @classmethod
    def from_rev_status_pb(cls, full_domain_id, pb):
        """
        :type full_domain_id: (six.text_type, six.text_type)
        :type pb: awacs.proto.model_pb2.BalancerState.RevisionStatus
        :rtype: DomainVersion
        """
        assert isinstance(full_domain_id, tuple) and len(full_domain_id) == 2
        return cls(pb.ctime.ToMicroseconds(), full_domain_id, pb.revision_id, pb.deleted, pb.incomplete)

    def __repr__(self):
        return 'Domain(id={}, v={}, ctime={}, d={}, incomplete={})'.format(
            self.domain_id, self.version[:8], self.ctime, int(self.deleted), int(self.incomplete))


class BackendVersion(collections.namedtuple('BackendVersion',
                                            ('ctime', 'backend_id', 'version', 'deleted'))):
    id_field_name = 'backend_id'
    vector_field_name = 'backend_versions'

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        return _join_as_bytes(self.backend_id, self.version)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Backend
        :rtype: BackendVersion
        """
        return cls(pb.meta.mtime.ToMicroseconds(), (pb.meta.namespace_id, pb.meta.id), pb.meta.version, pb.spec.deleted)

    @classmethod
    def from_rev_status_pb(cls, full_backend_id, pb):
        """
        :type full_backend_id: (six.text_type, six.text_type)
        :type pb: awacs.proto.model_pb2.BalancerState.RevisionStatus
        :rtype: BackendVersion
        """
        assert isinstance(full_backend_id, tuple) and len(full_backend_id) == 2
        return cls(pb.ctime.ToMicroseconds(), full_backend_id, pb.revision_id, pb.deleted)

    def short_ver(self):
        return self.version[:8]

    def __repr__(self):
        return 'Back(id={}, v={}, ctime={}, d={})'.format(
            flatten_full_id2(self.backend_id), self.short_ver(), self.ctime, int(self.deleted))


class EndpointSetVersion(collections.namedtuple('EndpointSetVersion',
                                                ('ctime', 'endpoint_set_id', 'version', 'deleted'))):
    id_field_name = 'endpoint_set_id'
    vector_field_name = 'endpoint_set_versions'

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        return _join_as_bytes(self.endpoint_set_id, self.version)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.EndpointSet
        :rtype: EndpointSetVersion
        """
        return cls(pb.meta.mtime.ToMicroseconds(), (pb.meta.namespace_id, pb.meta.id), pb.meta.version, pb.spec.deleted)

    @classmethod
    def from_rev_status_pb(cls, full_endpoint_set_id, pb):
        """
        :type full_endpoint_set_id: (six.text_type, six.text_type)
        :type pb: awacs.proto.model_pb2.BalancerState.RevisionStatus
        :rtype: EndpointSetVersion
        """
        assert isinstance(full_endpoint_set_id, tuple) and len(full_endpoint_set_id) == 2
        return cls(pb.ctime.ToMicroseconds(), full_endpoint_set_id, pb.revision_id, pb.deleted)

    def __repr__(self):
        return 'ESet(id={}, v={}, ctime={}, d={})'.format(
            flatten_full_id2(self.endpoint_set_id), self.version[:8], self.ctime, int(self.deleted))


class KnobVersion(collections.namedtuple('KnobVersion',
                                         ('ctime', 'knob_id', 'version', 'deleted'))):
    id_field_name = 'knob_id'
    vector_field_name = 'knob_versions'

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        return _join_as_bytes(self.knob_id, self.version)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Knob
        :rtype: KnobVersion
        """
        return cls(pb.meta.mtime.ToMicroseconds(), (pb.meta.namespace_id, pb.meta.id), pb.meta.version, pb.spec.deleted)

    @classmethod
    def from_rev_status_pb(cls, full_knob_id, pb):
        """
        :type full_knob_id: (six.text_type, six.text_type)
        :type pb: awacs.proto.model_pb2.BalancerState.RevisionStatus
        :rtype: KnobVersion
        """
        assert isinstance(full_knob_id, tuple) and len(full_knob_id) == 2
        return cls(pb.ctime.ToMicroseconds(), full_knob_id, pb.revision_id, pb.deleted)

    def __repr__(self):
        return 'Knob(id={}, v={}, ctime={}, d={})'.format(
            flatten_full_id2(self.knob_id), self.version[:8], self.ctime, int(self.deleted))


class CertVersion(collections.namedtuple('CertVersion',
                                         ('ctime', 'cert_id', 'version', 'deleted', 'incomplete'))):
    id_field_name = 'cert_id'
    vector_field_name = 'cert_versions'

    def get_weak_hash(self):
        """
        :rtype: six.binary_type
        """
        return _join_as_bytes(self.cert_id, self.version)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Certificate
        :rtype: CertVersion
        """
        return cls(pb.meta.mtime.ToMicroseconds(),
                   (pb.meta.namespace_id, pb.meta.id),
                   pb.meta.version,
                   needs_removal(pb),
                   pb.spec.incomplete)

    @classmethod
    def from_rev_status_pb(cls, full_cert_id, pb):
        """
        :type full_cert_id: (six.text_type, six.text_type)
        :type pb: awacs.proto.model_pb2.BalancerState.RevisionStatus
        :rtype: CertVersion
        """
        assert isinstance(full_cert_id, tuple) and len(full_cert_id) == 2
        return cls(pb.ctime.ToMicroseconds(), full_cert_id, pb.revision_id, pb.deleted, pb.incomplete)

    def __repr__(self):
        return 'Certificate(id={}, v={}, ctime={}, d={}, incomplete={})'.format(
            flatten_full_id2(self.cert_id), self.version[:8], self.ctime, int(self.deleted), int(self.incomplete))


def _change_key(change):
    if isinstance(change, ComponentVersion):
        return -1, change.type, change.version
    if isinstance(change, tuple) and len(change) == 2:
        _, to_version = change
    else:
        to_version = change
    if isinstance(to_version, BalancerVersion):
        return 0, to_version.balancer_id
    elif isinstance(to_version, DomainVersion):
        return 1, to_version.domain_id
    elif isinstance(to_version, UpstreamVersion):
        return 2, to_version.upstream_id
    elif isinstance(to_version, BackendVersion):
        return 3, to_version.backend_id
    elif isinstance(to_version, EndpointSetVersion):
        return 4, to_version.endpoint_set_id
    elif isinstance(to_version, KnobVersion):
        return 5, to_version.knob_id
    elif isinstance(to_version, CertVersion):
        return 6, to_version.cert_id
    elif isinstance(to_version, objects.WeightSection.version):
        return 7, to_version.id
    else:
        raise AssertionError('Unexpected change {}'.format(type(change)))


def _version_to_str(version):
    rv = version.version[:8]
    if version.deleted:
        rv = 'del@' + rv
    return rv


class SectionType(enum.Enum):
    NONE = 0
    ADDED = 1
    UPDATED = 2
    REMOVED = 3


def get_human_readable_diff(namespace_id, from_vector, to_vector, show_revision_ids=False, components_diff=None):
    """
    Used both for logging and for constructing commit messages for Nanny snapshots.

    :type namespace_id: str
    :type from_vector: Vector
    :type to_vector: Vector
    :type show_revision_ids: bool
    :type components_diff: awacs.model.balancer.component_transports.diff.ComponentsDiff
    :rtype: str
    """
    diff = from_vector.diff(to_vector)

    removed = set(diff.removed)
    updated = set()
    for from_version, to_version in diff.updated:
        if to_version.deleted:
            removed.add(to_version)
        else:
            updated.add((from_version, to_version))

    added = diff.added
    if components_diff is not None:
        updated.update(components_diff.updated)
        removed.update(components_diff.removed)
        added.update(components_diff.added)

    sections = (
        (SectionType.UPDATED, 'Updated:', updated),
        (SectionType.ADDED, 'Added:', added),
        (SectionType.REMOVED, 'Removed:', removed),
    )

    parts = []
    for section_type, section_name, section_changes in sections:
        if not section_changes:
            continue
        parts.append(section_name)
        for change in sorted(section_changes, key=_change_key):
            annotation = ''
            if isinstance(change, ComponentVersion):
                to_version = change
            elif isinstance(change, tuple) and len(change) == 2:
                from_version, to_version = change
                if show_revision_ids:
                    annotation = ' ({} -> {})'.format(_version_to_str(from_version), _version_to_str(to_version))
            else:
                to_version = change
                if show_revision_ids:
                    annotation = ' ({})'.format(_version_to_str(to_version))
            if isinstance(to_version, BalancerVersion):
                parts.append(' * balancer{}'.format(annotation))
            elif isinstance(to_version, DomainVersion):
                parts.append(' * domain "{}"{}'.format(
                    flatten_full_id(namespace_id, to_version.domain_id), annotation))
            elif isinstance(to_version, UpstreamVersion):
                parts.append(' * upstream "{}"{}'.format(
                    flatten_full_id(namespace_id, to_version.upstream_id), annotation))
            elif isinstance(to_version, BackendVersion):
                parts.append(' * backend "{}"{}'.format(
                    flatten_full_id(namespace_id, to_version.backend_id), annotation))
            elif isinstance(to_version, EndpointSetVersion):
                parts.append(' * endpoint set "{}"{}'.format(
                    flatten_full_id(namespace_id, to_version.endpoint_set_id), annotation))
            elif isinstance(to_version, KnobVersion):
                parts.append(' * knob "{}"{}'.format(
                    flatten_full_id(namespace_id, to_version.knob_id), annotation))
            elif isinstance(to_version, CertVersion):
                parts.append(' * cert "{}"{}'.format(
                    flatten_full_id(namespace_id, to_version.cert_id), annotation))
            elif isinstance(to_version, objects.WeightSection.version):
                parts.append(' * weight section "{}"{}'.format(
                    flatten_full_id(namespace_id, to_version.id), annotation))
            elif isinstance(to_version, ComponentVersion):
                part = ' * component "{}"'.format(model_pb2.ComponentMeta.Type.Name(to_version.type))
                if to_version.version is not None:
                    if section_type == SectionType.UPDATED:
                        part += ' to version "{}"'.format(to_version.version)
                    else:
                        part += ', version "{}"'.format(to_version.version)
                parts.append(part)
            else:
                raise RuntimeError('Unknown version type {}'.format(type(to_version)))
    return '\n'.join(parts)


def maybe_find_revision_spec(version):
    """
    :type version: (BalancerVersion | DomainVersion | UpstreamVersion | BackendVersion |
                    EndpointSetVersion | KnobVersion | CertVersion | objects.WeightSection.version)
    :rtype: Optional[model_pb2.*Spec]
    """
    try:
        return find_revision_spec(version)
    except errors.NotFoundError:
        return None


def find_revision_spec(version):
    """
    :type version: (BalancerVersion | DomainVersion | UpstreamVersion | BackendVersion |
                    EndpointSetVersion | KnobVersion | CertVersion | objects.WeightSection.version)
    :raises: awacs.model.errors.NotFoundError
    :rtype: model_pb2.*Spec
    """
    if isinstance(version, BalancerVersion):
        return db.find_balancer_revision_spec(version)
    elif isinstance(version, DomainVersion):
        return db.find_domain_revision_spec(version)
    elif isinstance(version, UpstreamVersion):
        return db.find_upstream_revision_spec(version)
    elif isinstance(version, BackendVersion):
        return db.find_backend_revision_spec(version)
    elif isinstance(version, EndpointSetVersion):
        return db.find_endpoint_set_revision_spec(version)
    elif isinstance(version, KnobVersion):
        return db.find_knob_revision_spec(version)
    elif isinstance(version, CertVersion):
        return db.find_cert_revision_spec(version)
    elif isinstance(version, objects.WeightSection.version):
        return objects.WeightSection.find_rev_spec(version)
    else:
        raise ValueError('Unknown version type: {}'.format(type(version)))
