import collections
import copy
import logging
import os.path
import sys
import threading

import cachetools
import enum
import gevent.event
import gevent.pool
import gevent.queue
import inject
import six
import time
from boltons import cacheutils, funcutils, namedutils
from datetime import timedelta, datetime
from sepelib.core import config as appconfig
from six.moves import map
from infra.swatlib.stacksampler.stacksampler import Sampler

from awacs.lib import zookeeper_client, pagination, gutils
from awacs.lib.models.classes import get_model_by_zk_prefix
from awacs.lib.order_processor.model import is_order_in_progress, is_order_cancelled
from awacs.lib.strutils import to_full_id
from awacs.model import storage_modern, util, errors, events, components, objects
from awacs.wrappers.l7macro import VALID_VERSIONS as VALID_L7MACRO_VERSIONS
from infra.awacs.proto import model_pb2
from infra.swatlib import metrics
from infra.swatlib.gevent import greenthread
from infra.swatlib.gevent.singletonparty import node_name_to_member_id
from infra.swatlib.logutil import get_op_log
from infra.swatlib.zk import treecache


fqdn_info_tuple = namedutils.namedtuple('domain_info_tuple', ('entity_type', 'namespace_id', 'domain_id'))
dns_record_fqdn_info_tuple = namedutils.namedtuple('dns_record_fqdn_info_tuple', ('entity_type', 'namespace_id', 'id'))

L3Config = namedutils.namedtuple('L3ConfigDigest', ['known_since', 'ctime', 'service_id', 'config_id'])
L7Config = namedutils.namedtuple('L7ConfigDigest', ['known_since', 'ctime', 'service_id', 'snapshot_id'])
NamespaceState = namedutils.namedtuple('NamespaceStateDigest', ['known_since', 'ctime', 'state'])
AlertingState = namedutils.namedtuple('AlertingStateDigest',
                                      ['last_sync_attempt_time', 'last_successful_sync_attempt_time'])
L3State = namedutils.namedtuple('L3StateDigest', ['known_since', 'ctime', 'state'])
Paused = namedutils.namedtuple('Paused', ['paused_at', 'namespace_id', 'id_'])
CertIndiscoverability = namedutils.namedtuple('CertIndiscoverability', ['since', 'namespace_id', 'id_'])

FqdnRemovalResult = namedutils.namedlist('FqdnRemovalResult',
                                         ['last_level_part', 'higher_level_part', 'removed', 'message'])


class IAwacsCache(object):
    @classmethod
    def instance(cls):
        """
        :rtype: AwacsCache
        """
        return inject.instance(cls)


def identity(v): return v


def requires(reqs):
    def wrapper(f):
        f.is_first_call = True

        @funcutils.wraps(f)
        def inner(self, *args, **kwargs):
            if f.is_first_call:
                for req in reqs:
                    if not self._structure.get_child(req):
                        raise RuntimeError('method {} requires watching {} zk node'.format(f.__name__, req))
                f.is_first_call = False
            return f(self, *args, **kwargs)

        return inner

    return wrapper


def kazoo_lock_key(c):
    """
    This comparator is copy-pasted from kazoo's Lock recipe.
    To be able to tell who's holding the lock, we need to pretend and
    sort nodes exactly as kazoo does.
    """
    for name in ('__lock__', '__rlock__'):
        idx = c.find(name)
        if idx != -1:
            return c[idx + len(name):]
    # Sort unknown node names eg. "lease_holder" last.
    return '~'


class AwacsCache(greenthread.GreenThread):
    METRICS_PATH = ('awacs', 'cache')

    STANDARD_IN_PROGRESS_TIME = timedelta(hours=1)  # everything that takes longer is considered stuck
    STANDARD_L3_IN_PROGRESS_TIME = timedelta(minutes=30)  # everything that takes longer is considered stuck
    # we want to let some things to "settle" in the cache before using them for alerting purposes
    # (see SWAT-6611 for slightly more detail):
    SETTLE_TIME = timedelta(minutes=10)
    STANDARD_PAUSED_TIME = timedelta(days=1)  # everything that paused longer is considered harmful
    STUCK_ALERTING_THRESHOLD = timedelta(hours=8)  # everything that takes longer is considered stuck
    # everything that is not discoverable by default longer than the threshold is considered "abused":
    CERT_INDISCOVERABILITY_THRESHOLD = timedelta(hours=24)
    CERT_EXPIRATION_THRESHOLD = timedelta(hours=72)

    class L3BalancersQueryTarget(enum.Enum):
        L3MGR_SERVICE_ID_IN = 1

    class BalancersQueryTarget(enum.Enum):
        NANNY_SERVICE_ID_IN = 1
        L7_MACRO_VERSION_IN = 2
        ID_IN = 3
        COMPONENT_IN = 4
        CANONICAL_ID_IN = 5

    class BalancerOpsQueryTarget(enum.Enum):
        ID_IN = 1

    class UpstreamsSortTarget(enum.Enum):
        ID = 1
        ORDER_LABEL = 2
        MTIME = 3

    class DomainsSortTarget(enum.Enum):
        ID = 1
        MTIME = 2

    class UpstreamsQueryTarget(enum.Enum):
        ID_IN = 1
        ID_REGEXP = 2
        VALIDATED_STATUS_IN = 3
        IN_PROGRESS_STATUS_IN = 4
        ACTIVE_STATUS_IN = 5
        DELETED = 6
        TYPE_IN = 7

    class DomainsQueryTarget(enum.Enum):
        ID_IN = 1
        ID_REGEXP = 2
        VALIDATED_STATUS_IN = 3
        IN_PROGRESS_STATUS_IN = 4
        ACTIVE_STATUS_IN = 5

    class BackendsSortTarget(enum.Enum):
        ID = 1
        MTIME = 2

    class BackendsQueryTarget(enum.Enum):
        ID_IN = 1
        ID_REGEXP = 2
        VALIDATED_STATUS_IN = 3
        IN_PROGRESS_STATUS_IN = 4
        ACTIVE_STATUS_IN = 5
        NANNY_SERVICE_ID_IN = 6
        GENCFG_GROUP_NAME_IN = 7
        YP_ENDPOINT_SET_FULL_ID_IN = 8
        ONLY_SYSTEM = 9
        EXCLUDE_SYSTEM = 10

    class KnobsSortTarget(enum.Enum):
        ID = 1
        MTIME = 2

    class CertsSortTarget(enum.Enum):
        ID = 1
        MTIME = 2

    class KnobsQueryTarget(enum.Enum):
        ID_IN = 1
        ID_REGEXP = 2
        VALIDATED_STATUS_IN = 3
        IN_PROGRESS_STATUS_IN = 4
        ACTIVE_STATUS_IN = 5
        MODE = 6

    class CertsQueryTarget(enum.Enum):
        ID_IN = 1
        ID_REGEXP = 2
        VALIDATED_STATUS_IN = 3
        IN_PROGRESS_STATUS_IN = 4
        ACTIVE_STATUS_IN = 5

    class CertRenewalsQueryTarget(enum.Enum):
        ID_REGEXP = 1
        INCOMPLETE = 2
        PAUSED = 3
        VALIDITY_NOT_BEFORE_GTE = 4
        VALIDITY_NOT_BEFORE_LTE = 5

    class CertRenewalsSortTarget(enum.Enum):
        ID = 1
        TARGET_CERT_VALIDITY_NOT_AFTER = 2

    class NamespacesQueryTarget(enum.Enum):
        ID_IN = 1
        CATEGORY_IN = 2
        ABC_SERVICE_ID_IN = 3
        LAYOUT_TYPE_IN = 4

    class ComponentQueryTarget(enum.Enum):
        TYPE_IN = 1
        STATUS_IN = 2

    class ComponentsSortTarget(enum.Enum):
        TYPE = 1
        VERSION = 2
        CTIME = 3
        STATUS = 4

    class InProgressL7ConfigsQueryTarget(enum.Enum):
        STUCK_ONLY = 1
        DOWNTIMED = 2

    class InProgressL3ConfigsQueryTarget(enum.Enum):
        STUCK_ONLY = 1

    DEFAULT_NAMESPACES_LIMIT = 3000
    MAX_NAMESPACES_LIMIT = 3000
    DEFAULT_BALANCERS_LIMIT = 1000
    MAX_BALANCERS_LIMIT = 2000
    DEFAULT_UPSTREAMS_LIMIT = 1000
    MAX_UPSTREAMS_LIMIT = 2000
    DEFAULT_BACKENDS_LIMIT = 1000
    MAX_BACKENDS_LIMIT = 2000
    DEFAULT_DOMAINS_LIMIT = 1000
    MAX_DOMAINS_LIMIT = 2000
    DEFAULT_COMPONENTS_LIMIT = 500
    MAX_COMPONENTS_LIMIT = 1000

    LARGE_NS_INDIVIDUAL_OBJECT_DEFAULT_LIMIT = 50
    LARGE_NS_TOTAL_OBJECT_DEFAULT_LIMIT = 100

    @staticmethod
    def normalise_namespace_name(namespace_id):
        return namespace_id.replace('.', '-').replace('_', '-')

    @staticmethod
    def normalise_prj_tag(prj):
        return prj.replace('.', '_')

    def _must_be_ignored(self, path, upd_pb, meta_field=u'meta'):
        """
        :type path: six.text_type
        :param upd_pb:
        :type meta_field: Optional[six.text_type]
        :rtype: bool
        """
        typename = type(upd_pb).__name__
        meta_pb = getattr(upd_pb, meta_field) if meta_field else upd_pb
        upd_generation = meta_pb.generation
        upd_ctime = meta_pb.ctime.ToNanoseconds()

        processed_gen_and_ctime = self._processed_objects[typename].get(path)
        if processed_gen_and_ctime is not None:
            generation, ctime = processed_gen_and_ctime
            # we need to check ctime here because of rare error with object recreation
            # see https://st.yandex-team.ru/AWACS-285
            must_be_ignored = generation >= upd_generation and ctime >= upd_ctime
        else:
            must_be_ignored = False

        if must_be_ignored:
            self._ignored_updates_counter.inc(1)
        else:
            self._processed_objects[typename][path] = (upd_generation, upd_ctime)

        return must_be_ignored

    def __init__(self, zk_client, path='/', structure=None,
                 enable_extended_signals=False, proxy_counters_cache_ttl=30):
        """
        :type zk_client: zookeeper_client.ZookeeperClient
        """
        super(AwacsCache, self).__init__()
        self._path = path.strip('/')
        self._path_w_trailing_slash = self._path + '/'
        self._zk_client = zk_client

        self._log = logging.getLogger('cache')
        self._callbacks_lock = threading.Lock()
        self._filtered_callbacks_lock = threading.Lock()
        self._structure = structure or storage_modern.construct_full_zk_structure()
        self._tree_cache = treecache.TreeCache(
            client=self._zk_client.client,
            path=self._path,
            structure=structure)

        registry = metrics.ROOT_REGISTRY.path(*self.METRICS_PATH)
        buckets = (
            .0001, .0005, .001, .0025, .005, 0.0075,
            .01, .02, .03, .04, .05, .06, .07, .08, .09,
            .1, .15, .20, .25, .5, .75,
            1.0, 1.25, 1.5, 1.75, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5, 5.0, 7.5, 10.0,
            15.0, 20.0, 30.0, 50.0, 100.0
        )
        self._run_callbacks_calls_counter = registry.get_counter('run-callbacks-calls')
        self._run_callbacks_calls_timer = registry.get_histogram('run-callbacks-timer', buckets=buckets)
        self._run_callbacks_failures_counter = registry.get_counter('run-callbacks-failures')
        self._callback_calls_counter = registry.get_counter('callback-calls')
        self._ignored_updates_counter = registry.get_counter('ignored-updates-counter')

        self._run_filtered_callbacks_calls_counter = registry.get_counter('run-filtered-callbacks-calls')
        self._run_filtered_callbacks_calls_timer = registry.get_histogram('run-filtered-callbacks-timer',
                                                                          buckets=buckets)
        self._run_filtered_callbacks_failures_counter = registry.get_counter('run-filtered-callbacks-failures')
        self._filtered_callback_calls_counter = registry.get_counter('filtered-callback-calls')

        self._run_subscribers_counter = registry.get_counter('run-subscribers-calls')
        self._run_subscribers_timer = registry.get_histogram('run-subscribers-timer', buckets=buckets)
        self._run_subscribers_failure_counter = registry.get_counter('run-subscribers-failures')
        self._run_subscribers_callback_counter = registry.get_counter('run-subscribers-callbacks')

        self._large_ns_individual_limit = appconfig.get_value('run.large_namespace_individual_objects_limit',
                                                              self.LARGE_NS_INDIVIDUAL_OBJECT_DEFAULT_LIMIT)
        self._large_ns_total_limit = appconfig.get_value('run.large_namespace_total_objects_limit',
                                                         self.LARGE_NS_TOTAL_OBJECT_DEFAULT_LIMIT)
        self._marked_large_namespaces = set(appconfig.get_value('run.large_namespace_ids', []))

        if enable_extended_signals:
            self._large_unmarked_namespaces_counter_proxy = registry.get_proxy(
                'large-unmarked-namespaces-counter', 'axxx', self._count_large_unmarked_namespaces)
            self._stuck_balancers_counter_proxy = registry.get_proxy(
                'stuck-balancers-counter', 'axxx', self._count_stuck_balancers)
            self._stuck_l3_balancers_counter_proxy = registry.get_proxy(
                'stuck-l3-balancers-counter', 'axxx', self._count_stuck_l3_balancers)
            self._in_progress_balancers_counter_proxy = registry.get_proxy(
                'in-progress-balancers-counter', 'axxx', self._count_in_progress_balancers)
            self._in_progress_l3_balancers_counter_proxy = registry.get_proxy(
                'in-progress-l3-balancers-counter', 'axxx', self._count_in_progress_l3_balancers)
            self._stuck_being_created_namespaces_counter_proxy = registry.get_proxy(
                'being-created-namespaces-counter', 'axxx', self._count_being_created_namespaces)
            self._stuck_namespaces_counter_proxy = registry.get_proxy(
                'stuck-namespaces-counter', 'axxx', self._count_stuck_being_created_namespaces)
            self._stuck_alerting_counter_proxy = registry.get_proxy(
                'stuck-alerting-counter', 'axxx', self._count_stuck_alerting_namespaces)
            self._being_created_l3_balancers_counter_proxy = registry.get_proxy(
                'being-created-l3-balancers-counter', 'axxx', self._count_being_created_l3_balancers)
            self._stuck_being_created_l3_balancers_counter_proxy = registry.get_proxy(
                'stuck-being-created-l3-balancers-counter', 'axxx', self._count_stuck_being_created_l3_balancers)
            self._paused_too_long_balancers_counter_proxy = registry.get_proxy(
                'paused-too-long-balancers-counter', 'axxx', self._count_paused_too_long_balancers)
            self._paused_too_long_l3_balancers_counter_proxy = registry.get_proxy(
                'paused-too-long-l3-balancers-counter', 'axxx', self._count_paused_too_long_l3_balancers)
            self._indiscoverable_too_long_certs_counter_proxy = registry.get_proxy(
                'indiscoverable-too-long-certs-counter', 'axxx', self._count_indiscoverable_too_long_certs)
            self._expired_soon_certs_counter_proxy = registry.get_proxy(
                'expires-soon-certs-counter', 'axxx', self._count_expired_soon_certs)
            self._namespaces_with_enabled_its_proxy = registry.get_proxy(
                'namespaces-with-enabled-its-counter', 'axxx', self.count_namespaces_with_enabled_its)
            self._proxy_counters_cache = cachetools.TTLCache(maxsize=100, ttl=proxy_counters_cache_ttl)

        self._stopped = gevent.event.Event()
        self._namespaces = {}
        self._namespace_aspects_sets = {}
        self._namespace_aspects_set_generations = {}
        self._components = {}
        self._component_default_versions = collections.defaultdict(lambda: collections.defaultdict(set))
        self._balancer_paths_by_component_version = collections.defaultdict(lambda: collections.defaultdict(set))
        self._knobs = {}
        self._balancers = {}
        self._balancer_states = {}
        self._balancer_aspects_sets = {}
        self._dns_records = {}
        self._dns_record_states = {}
        self._dns_record_operations = {}
        self._name_servers = {}
        self._l3_balancers = {}
        self._l3_balancer_states = {}
        self._upstreams = {}
        self._domains = {}
        self._domain_operations = {}  # type: dict[six.text_type, model_pb2.DomainOperation]  # noqa
        self._balancer_operations = {}
        self._backends = {}
        self._endpoint_sets = {}
        self._certs = {}
        self._cert_renewals = {}

        self._processed_objects = collections.defaultdict(dict)

        # Callbacks:
        self._callbacks = set()
        self._filtered_callbacks = collections.defaultdict(set)
        self._update_subscribers = collections.defaultdict(set)
        self._removal_subscribers = collections.defaultdict(set)

        # Indices:
        self._namespace_normalised_names = set()
        self._namespace_id_by_normalised_prj = {}
        self._namespaces_with_enabled_its = set()
        self._namespace_paths_by_layout_type = collections.defaultdict(set)

        self._component_ctimes = {}
        self._component_statuses = {}

        self._knob_paths_by_namespace_path = collections.defaultdict(set)
        self._knob_mtimes = {}

        self._cert_paths_by_namespace_path = collections.defaultdict(set)
        self._cert_mtimes = {}
        self._cert_not_afters = {}

        self._cert_renewal_paths_by_namespace_path = collections.defaultdict(set)

        self._dns_record_op_paths_by_namespace_id = collections.defaultdict(set)
        self._dns_record_op_paths_by_dns_record_id = collections.defaultdict(set)

        self._balancer_paths_by_namespace_path = collections.defaultdict(set)
        self._balancer_paths_by_balancer_id = collections.defaultdict(set)
        self._balancer_paths_by_canonical_balancer_id = collections.defaultdict(set)
        self._balancer_paths_by_l7_macro_version = collections.defaultdict(set)
        for v in VALID_L7MACRO_VERSIONS:
            self._balancer_paths_by_l7_macro_version[six.text_type(v)] = set()
        self._balancer_paths_by_l7_macro_version['none'] = set()
        self._balancer_paths_by_nanny_service_id = collections.defaultdict(set)
        self._balancer_state_paths_by_namespace_path = collections.defaultdict(set)

        self._dns_record_paths_by_namespace_id = collections.defaultdict(set)
        self._dns_record_paths_by_name_server_id = collections.defaultdict(set)
        self._name_server_paths_by_namespace_path = collections.defaultdict(set)

        self._l3_balancer_paths_by_namespace_path = collections.defaultdict(set)
        self._l3_balancer_state_paths_by_namespace_path = collections.defaultdict(set)
        self._dns_records_paths_by_l3_balancer_full_id = collections.defaultdict(set)

        self._upstream_paths_by_namespace_path = collections.defaultdict(set)
        self._upstream_order_labels = {}
        self._upstream_mtimes = {}
        self._upstream_paths_by_type = collections.defaultdict(set)
        self._deleted_upstream_paths = set()
        self._easy_mode_upstream_ids_by_backend_id = collections.defaultdict(set)
        self._backend_ids_by_easy_mode_upstream_id = collections.defaultdict(set)

        self._domain_paths_by_namespace_path = collections.defaultdict(set)
        self._domain_mtimes = {}

        self._domain_operation_paths_by_namespace_path = collections.defaultdict(set)

        self._balancer_operation_paths_by_namespace_path = collections.defaultdict(set)

        self._backend_paths_by_namespace_path = collections.defaultdict(set)
        self._backend_paths_by_nanny_service_id = collections.defaultdict(set)
        self._backend_paths_by_gencfg_group_name = collections.defaultdict(set)
        self._backend_paths_by_yp_endpoint_set_full_id = collections.defaultdict(set)
        self._backend_paths_by_full_balancer_id = collections.defaultdict(set)
        self._full_balancer_id_by_yp_endpoint_set_full_id = {}
        self._backend_paths_by_backend_id = collections.defaultdict(set)
        self._backend_paths_system = set()
        self._backend_paths_not_system = set()
        self._backend_mtimes = {}

        self._endpoint_set_paths_by_namespace_path = collections.defaultdict(set)
        self._endpoint_set_mtimes = {}

        # all FQDNs used in domains and domain ops
        self._domain_fqdns = collections.defaultdict(dict)  # type: dict[six.text_type, dict[six.text_type, set[fqdn_info_tuple]]]  # noqa

        # all FQDNs used in dns records
        self._dns_record_fqdns = collections.defaultdict(dict)  # type: dict[six.text_type, dict[six.text_type, dns_record_fqdn_info_tuple]]  # noqa
        # we use _namespace_id_by_dns_record_wildcard and _dns_records_fqdns_coverage to check
        # 2) if fqdn is covered by some wildcard from another namespace
        # 1) if some wildcard cover some fqdns/another wildcard from another namespace
        self._namespace_id_by_dns_record_wildcard = dict()  # dict[six.text_type, six.text_type]
        # DNS Records with fqdns:
        # awacs.nanny.yandex.net (from ns_id_1), nanny.yandex.net (from ns_id_2), sandbox.yandex.net (from ns_id_3) ->
        # -> namespaces_covered_fqdns_count_by_dns_record_potential_wildcard = {"nanny.yandex.net": {"ns_id_1": 1},
        #  "yandex.net": {"ns_id_1": 1, "ns_id_2": 1, "ns_id_3": 1}, "net": {"ns_id_1": 1, "ns_id_2": 1, "ns_id_3": 1}}
        # Note: this index implies only dns record fqdns (not awacs domains fqdns)
        self.namespaces_covered_fqdns_count_by_dns_record_potential_wildcard = collections.defaultdict(
            lambda: collections.defaultdict(int))  # dict[six.text_type, dict[six.text_type, int]]

        # Mappings that tell which balancer states include which backends/certs and vice versa
        # type: dict[(set, set), (set, set)]  # noqa
        self._full_backend_ids_to_full_balancer_id_sets = collections.defaultdict(set)
        self._full_balancer_ids_to_full_backend_id_sets = collections.defaultdict(set)

        self._full_backend_ids_to_full_l3_balancer_id_sets = collections.defaultdict(set)
        self._full_l3_balancer_ids_to_full_backend_id_sets = collections.defaultdict(set)

        self._full_backend_ids_to_full_dns_record_id_sets = collections.defaultdict(set)
        self._full_dns_record_ids_to_full_backend_id_sets = collections.defaultdict(set)

        self._full_cert_ids_to_full_balancer_id_sets = collections.defaultdict(set)
        self._full_balancer_ids_to_full_cert_id_sets = collections.defaultdict(set)

        self._full_domain_ids_to_full_balancer_id_sets = collections.defaultdict(set)
        self._full_balancer_ids_to_full_domain_id_sets = collections.defaultdict(set)

        # Indices to ease finding stuck configurations
        self._balancer_to_in_progress_configs = {}  # type: dict[(six.text_type, six.text_type), set[L7Config]]  # noqa
        self._l3_balancer_to_in_progress_configs = {}  # type: dict[(six.text_type, six.text_type), set[L3Config]]  # noqa
        self._namespace_being_created_states = {}  # type: dict[six.text_type, NamespaceState]  # noqa
        self._namespaces_with_stuck_alerting_states = {}  # type: dict[six.text_type, AlertingState]  # noqa
        self._l3_balancer_being_created_states = {}  # type: dict[(six.text_type, six.text_type), L3State]  # noqa
        self._namespace_objects_count = collections.defaultdict(lambda: {'domains': 0, 'upstreams': 0, 'backends': 0})
        self._large_namespaces = set()

    def reset_proxy_counters_cache(self):
        if self._proxy_counters_cache is None:
            return
        self._proxy_counters_cache.expire(time=time.time() + sys.maxsize)
        assert len(self._proxy_counters_cache) == 0

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_in_progress_balancers(self):
        return len(self.get_balancer_in_progress_configs())

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_in_progress_l3_balancers(self):
        return len(self.get_l3_balancer_in_progress_configs())

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_stuck_balancers(self):
        query = {self.InProgressL7ConfigsQueryTarget.STUCK_ONLY: True, self.InProgressL7ConfigsQueryTarget.DOWNTIMED: False}
        return len(self.get_balancer_in_progress_configs(query))

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_stuck_l3_balancers(self):
        query = {self.InProgressL3ConfigsQueryTarget.STUCK_ONLY: True}
        return len(self.get_l3_balancer_in_progress_configs(query))

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_being_created_namespaces(self):
        return len(self.list_namespaces_being_created())

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_stuck_being_created_namespaces(self):
        return len(self.list_namespaces_being_created(stuck_only=True))

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_being_created_l3_balancers(self):
        return len(self.list_l3_balancers_being_created())

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_stuck_being_created_l3_balancers(self):
        return len(self.list_l3_balancers_being_created(stuck_only=True))

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_paused_too_long_balancers(self):
        return len(self.list_paused_balancers(paused_too_long_only=True))

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_paused_too_long_l3_balancers(self):
        return len(self.list_paused_l3_balancers(paused_too_long_only=True))

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_indiscoverable_too_long_certs(self):
        return len(self.list_indiscoverable_certs(indiscoverable_too_long_only=True))

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_expired_soon_certs(self):
        now = datetime.utcnow()
        rv = 0
        for pb in self.list_all_certs():
            if pb.spec.incomplete:
                continue
            expired_at = pb.spec.fields.validity.not_after.ToDatetime()
            if (expired_at - now) < self.CERT_EXPIRATION_THRESHOLD:
                rv += 1
        return rv

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def count_namespaces_with_enabled_its(self):
        return len(self._namespaces_with_enabled_its)

    def _update_large_namespaces(self, namespace_id):
        domains = self._namespace_objects_count[namespace_id]['domains']
        upstreams = self._namespace_objects_count[namespace_id]['upstreams']
        backends = self._namespace_objects_count[namespace_id]['backends']
        if ((domains + upstreams + backends) > self._large_ns_total_limit
                or any(obj_count > self._large_ns_individual_limit for obj_count in (domains, upstreams, backends))):
            self._large_namespaces.add(namespace_id)
        else:
            self._large_namespaces.discard(namespace_id)

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_large_unmarked_namespaces(self):
        for ns_id in self._namespace_objects_count:
            self._update_large_namespaces(ns_id)
        return len(self._large_namespaces - self._marked_large_namespaces)

    @cacheutils.cachedmethod(lambda self: self._proxy_counters_cache)
    def _count_stuck_alerting_namespaces(self):
        return len(self.list_namespaces_with_stuck_alerting())

    def _list_paused(self, pbs, paused_too_long_only=False):
        """
        :type pbs: Iterable[model_pb2.Balancer | model_pb2.L3Balancer]
        :type paused_too_long_only: bool
        :rtype: list[Paused]
        """
        now = datetime.utcnow()
        rv = []
        for pb in pbs:
            if pb.meta.transport_paused.value:
                paused_at = pb.meta.transport_paused.mtime.ToDatetime()
                is_paused_too_long = now - paused_at > self.STANDARD_PAUSED_TIME
                if not paused_too_long_only or is_paused_too_long:
                    rv.append(Paused(paused_at, pb.meta.namespace_id, pb.meta.id))
        return rv

    @requires([objects.L7BalancerDescriptor.zk_prefix])
    def list_paused_balancers(self, paused_too_long_only=False):
        """
        :type paused_too_long_only: bool
        :rtype: list[Paused]
        """
        return self._list_paused(self.list_all_balancers(), paused_too_long_only=paused_too_long_only)

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def list_paused_l3_balancers(self, paused_too_long_only=False):
        """
        :type paused_too_long_only: bool
        :rtype: list[Paused]
        """
        return self._list_paused(self.list_all_l3_balancers(), paused_too_long_only=paused_too_long_only)

    def _list_certs_indiscoverable_by_default(self, cert_pbs, indiscoverable_too_long_only=False):
        """
        :type cert_pbs: Iterable[model_pb2.Certificate]
        :type indiscoverable_too_long_only: bool
        :rtype: list[CertIndiscoverability]
        """
        now = datetime.utcnow()
        rv = []
        for pb in cert_pbs:
            if pb.meta.HasField('discoverability') and not pb.meta.discoverability.default.value:
                since = pb.meta.discoverability.default.mtime.ToDatetime()
                is_indiscoverable_too_long = (now - since) > self.CERT_INDISCOVERABILITY_THRESHOLD
                if indiscoverable_too_long_only and not is_indiscoverable_too_long:
                    continue
                rv.append(CertIndiscoverability(since, pb.meta.namespace_id, pb.meta.id))
        return rv

    @requires([objects.CertDescriptor.zk_prefix])
    def list_indiscoverable_certs(self, indiscoverable_too_long_only=False):
        """
        :type indiscoverable_too_long_only: bool
        :rtype: list[CertIndiscoverability]
        """
        return self._list_certs_indiscoverable_by_default(self.list_all_certs(),
                                                          indiscoverable_too_long_only=indiscoverable_too_long_only)

    @requires([objects.NamespaceDescriptor.zk_prefix,
               objects.DomainDescriptor.zk_prefix,
               objects.UpstreamDescriptor.zk_prefix,
               objects.BackendDescriptor.zk_prefix])
    def list_large_namespaces(self):
        """
        :rtype: six.text_type, dict[six.text_type, int]
        """
        for ns_id in self._namespace_objects_count:
            self._update_large_namespaces(ns_id)
        for ns_id in self._large_namespaces:
            yield ns_id, self._namespace_objects_count[ns_id]

    @requires([objects.L7BalancerDescriptor.zk_prefix])
    def count_balancer_usage_by_l7_macro_version(self):
        return {version: len(balancers)
                for version, balancers in six.iteritems(self._balancer_paths_by_l7_macro_version)}

    @requires([objects.ComponentDescriptor.zk_prefix])
    def list_component_default_versions(self, component_type):
        return copy.deepcopy(self._component_default_versions[component_type])

    @requires([objects.L7BalancerDescriptor.zk_prefix, objects.ComponentDescriptor.zk_prefix])
    def count_balancer_usage_by_component_version(self, component_type):
        return {version: len(items)
                for version, items in six.iteritems(self._balancer_paths_by_component_version[component_type])}

    @requires([objects.ComponentDescriptor.zk_prefix])
    def get_component_default_version(self, component_type, cluster):
        versions = self._component_default_versions[component_type][cluster]
        if not versions:
            return None
        return sorted(versions)[0]

    @requires([objects.ComponentDescriptor.zk_prefix])
    def must_get_component_default_version(self, component_type, cluster):
        version = self.get_component_default_version(component_type, cluster)
        if version is None:
            raise errors.InternalError('No default version for {} component found'
                                       .format(model_pb2.ComponentMeta.Type.Name(component_type)))
        return version

    def _is_stuck_and_settled_configs(self, configs, now):
        """
        :type configs: set[L3Config | L7Config]
        :type now: datetime.datetime
        :rtype: bool
        """
        oldest_config = min(configs)
        is_settled = now - oldest_config.known_since > self.SETTLE_TIME
        is_stuck = now - oldest_config.ctime > self.STANDARD_IN_PROGRESS_TIME
        return is_settled and is_stuck

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def get_downtime_flag(self, namespace_id, balancer_id):
        """
        :type namespace_id: six.text_type
        :param balancer_id: L7Balancer id
        :type balancer_id: six.text_type
        :rtype: model_pb2.TimedBoolCondition
        """
        return self.get_balancer(namespace_id, balancer_id).meta.flags.downtime_stuck_activation_check

    def _is_downtimed(self, now, namespace_id, balancer_id):
        """
        :type now: datetime.datetime
        :type namespace_id: six.text_type
        :param balancer_id: L7Balancer id
        :type balancer_id: six.text_type
        :rtype: bool
        """
        downtime_flag_pb = self.get_downtime_flag(namespace_id, balancer_id)
        return downtime_flag_pb.value and now <= downtime_flag_pb.not_after.ToDatetime()

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def get_balancer_in_progress_configs(self, query=None):
        """
        :type query: dict[FIXME]
        :rtype: dict[(six.text_type, six.text_type), L7Config]
        """
        if not query:
            return dict(self._balancer_to_in_progress_configs)

        rv = {}
        now = datetime.utcnow()
        for full_id, configs in six.iteritems(self._balancer_to_in_progress_configs):
            stuck_only = query.get(self.InProgressL7ConfigsQueryTarget.STUCK_ONLY, None)
            if stuck_only and not self._is_stuck_and_settled_configs(configs, now):
                continue
            downtimed = query.get(self.InProgressL7ConfigsQueryTarget.DOWNTIMED, None)
            if downtimed is not None:
                if self._is_downtimed(now, *full_id):
                    if not downtimed:
                        continue
                else:
                    if downtimed:
                        continue
            rv[full_id] = configs
        return rv

    @requires([objects.L3BalancerStateDescriptor.zk_prefix])
    def get_l3_balancer_in_progress_configs(self, query=None):
        """
        :type query: dict[FIXME]
        :rtype: dict[(six.text_type, six.text_type), L3Config]
        """
        if not query:
            return dict(self._l3_balancer_to_in_progress_configs)

        rv = {}
        now = datetime.utcnow()
        for full_id, configs in six.iteritems(self._l3_balancer_to_in_progress_configs):
            stuck_only = query.get(self.InProgressL3ConfigsQueryTarget.STUCK_ONLY, None)
            if stuck_only and not self._is_stuck_and_settled_configs(configs, now):
                continue
            rv[full_id] = configs
        return rv

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def list_namespaces_being_created(self, stuck_only=False):
        """
        :type stuck_only: bool
        :rtype: dict[six.text_type, NamespaceState]
        """
        now = datetime.utcnow()
        rv = {}
        if stuck_only:
            for id_, state in six.iteritems(self._namespace_being_created_states):
                is_settled = now - state.known_since > self.SETTLE_TIME
                is_stuck = now - state.ctime > self.STANDARD_IN_PROGRESS_TIME
                if is_settled and is_stuck:
                    rv[id_] = state
        else:
            rv = dict(self._namespace_being_created_states)
        return rv

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def list_namespaces_with_stuck_alerting(self):
        """
        :rtype: dict[six.text_type, AlertingState]
        """
        return dict(self._namespaces_with_stuck_alerting_states)

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def list_l3_balancers_being_created(self, stuck_only=False):
        """
        :type stuck_only: bool
        :rtype: dict[six.text_type, L3State]
        """
        now = datetime.utcnow()
        rv = {}
        if stuck_only:
            for id_, state in six.iteritems(self._l3_balancer_being_created_states):
                is_settled = now - state.known_since > self.SETTLE_TIME
                is_stuck = now - state.ctime > self.STANDARD_IN_PROGRESS_TIME
                if is_settled and is_stuck:
                    rv[id_] = state
        else:
            rv = dict(self._l3_balancer_being_created_states)
        return rv

    @requires([storage_modern.PARTIES_NODE_ZK_PREFIX])
    def list_party_member_ids(self, party_id):
        party_path = '/' + os.path.join(storage_modern.PARTIES_NODE_ZK_PREFIX, party_id) + '/'
        node_names = self._tree_cache.get_children(party_path)
        return list(map(node_name_to_member_id, node_names))

    @requires([storage_modern.EXCLUSIVE_SERVICES_NODE_ZK_PREFIX])
    def list_exclusive_services(self):
        rv = {}

        service_names = self._tree_cache.get_children(storage_modern.EXCLUSIVE_SERVICES_NODE_ZK_PREFIX)
        for service_name in gutils.gevent_idle_iter(service_names, idle_period=200):
            service_path = os.path.join(storage_modern.EXCLUSIVE_SERVICES_NODE_ZK_PREFIX, service_name)
            lock_node_names = sorted(self._tree_cache.get_children(service_path), key=kazoo_lock_key)
            contenders = []
            for lock_node_name in lock_node_names:
                lock_node = self._tree_cache.get_data(os.path.join(service_path, lock_node_name))
                if lock_node is None:
                    # Well, the node has disappeared while we iterated through this loop.
                    # That's okay, just continue.
                    continue
                contender = lock_node.data
                contenders.append(contender)

            rv[service_name] = contenders

        return rv

    @requires([objects.ComponentDescriptor.zk_prefix])
    def get_component(self, component_type, version):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :rtype: model_pb2.Component | None
        """
        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        path = objects.ComponentDescriptor.uid_to_zk_path(component_type_str, version)
        return self._components.get(path)

    @requires([objects.ComponentDescriptor.zk_prefix])
    def must_get_component(self, component_type, version):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :rtype: model_pb2.Component
        :raises: errors.NotFoundError
        """
        component_pb = self.get_component(component_type, version)
        if not component_pb:
            component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
            raise errors.NotFoundError(u'Component "{}" of version "{}" not found'.format(component_type_str, version))
        return component_pb

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def get_namespace(self, namespace_id):
        path = objects.NamespaceDescriptor.uid_to_zk_path(namespace_id)
        return self._namespaces.get(path)

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def must_get_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: model_pb2.Namespace
        :raises: errors.NotFoundError
        """
        namespace_pb = self.get_namespace(namespace_id)
        if not namespace_pb:
            raise errors.NotFoundError('Namespace "{}" not found'.format(namespace_id))
        return namespace_pb

    @requires([objects.NamespaceAspectSetDescriptor.zk_prefix])
    def get_namespace_aspects_set(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: model_pb2.NamespaceAspectsSet | None
        """
        path = objects.NamespaceAspectSetDescriptor.uid_to_zk_path(namespace_id)
        return self._namespace_aspects_sets.get(path)

    @requires([objects.NamespaceAspectSetDescriptor.zk_prefix])
    def must_get_namespace_aspects_set(self, namespace_id):
        """
        :param six.text_type namespace_id:
        :rtype: model_pb2.NamespaceAspectsSet
        :raises: errors.NotFoundError
        """
        namespace_aspects_set_pb = self.get_namespace_aspects_set(namespace_id)
        if not namespace_aspects_set_pb:
            raise errors.NotFoundError('Namespace aspects set "{}" not found'.format(namespace_id))
        return namespace_aspects_set_pb

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def count_namespaces(self):
        return len(self._namespaces)

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def list_all_namespaces(self, query=None):
        """
        :rtype: list[model_pb2.Namespace]
        """
        namespace_paths = set(self._namespaces.keys())

        query = query or {}

        id_in = set(query.get(self.NamespacesQueryTarget.ID_IN, []))
        if id_in:
            filtered_paths = set()
            for path in namespace_paths:
                namespace_id = self._namespaces[path].meta.id
                if namespace_id in id_in:
                    filtered_paths.add(path)
            namespace_paths = filtered_paths

        category_in = set(query.get(self.NamespacesQueryTarget.CATEGORY_IN, []))
        if category_in:
            filtered_paths = set()
            for path in namespace_paths:
                namespace_category = self._namespaces[path].meta.category
                if namespace_category in category_in:
                    filtered_paths.add(path)
            namespace_paths = filtered_paths

        abc_service_id_in = set(query.get(self.NamespacesQueryTarget.ABC_SERVICE_ID_IN, []))
        if abc_service_id_in:
            filtered_paths = set()
            for path in namespace_paths:
                namespace_abc_service_id = self._namespaces[path].meta.abc_service_id
                if namespace_abc_service_id in abc_service_id_in:
                    filtered_paths.add(path)
            namespace_paths = filtered_paths

        layout_type_in = query.get(self.NamespacesQueryTarget.LAYOUT_TYPE_IN)
        if layout_type_in:
            matched_paths = set()
            for layout_type in layout_type_in:
                matched_paths |= self._namespace_paths_by_layout_type[layout_type]
            namespace_paths &= matched_paths

        namespaces = list(map(self._namespaces.get, sorted(namespace_paths)))
        return namespaces

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def list_namespaces(self, skip=None, limit=None, query=None):
        namespace_pbs = self.list_all_namespaces(query=query)
        total = len(namespace_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_NAMESPACES_LIMIT,
                                        max_limit=self.MAX_NAMESPACES_LIMIT)
        return pagination.SliceResult(items=namespace_pbs[page], total=total)

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def list_all_balancer_states(self, namespace_id=None):
        """
        :type namespace_id: six.text_type | None
        :rtype: list[model_pb2.BalancerState]
        """
        if namespace_id:
            balancer_state_paths = self._balancer_state_paths_by_namespace_path[namespace_id]
        else:
            balancer_state_paths = list(self._balancer_states)
        return list(map(self._balancer_states.get, sorted(balancer_state_paths)))

    @requires([objects.L7BalancerDescriptor.zk_prefix])
    def count_balancers(self, namespace_id=None):
        if namespace_id:
            return len(self._balancer_paths_by_namespace_path.get(namespace_id) or frozenset())
        else:
            return len(self._balancers)

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def get_namespace_id_by_normalised_prj(self, prj):
        return self._namespace_id_by_normalised_prj.get(self.normalise_prj_tag(prj))

    @requires([objects.ComponentDescriptor.zk_prefix])
    def list_all_components(self, query=None, sort=(ComponentsSortTarget.TYPE, 1)):
        """
        :rtype: list[model_pb2.Component]
        """
        component_paths = list(self._components)

        query = query or {}

        type_in = set(query.get(self.ComponentQueryTarget.TYPE_IN, []))
        if type_in:
            filtered_paths = set()
            for path in component_paths:
                component_type = self._components[path].meta.type
                if component_type in type_in:
                    filtered_paths.add(path)
            component_paths = filtered_paths

        status_in = set(query.get(self.ComponentQueryTarget.STATUS_IN, []))
        if status_in:
            filtered_paths = set()
            for path in component_paths:
                component_status = self._components[path].status.status
                if component_status in status_in:
                    filtered_paths.add(path)
            component_paths = filtered_paths

        if sort[0] == self.ComponentsSortTarget.CTIME:
            key = self._component_ctimes.get
        elif sort[0] == self.ComponentsSortTarget.VERSION:
            def key(p):
                type_ = self._components[p].meta.type
                component_config = components.get_component_config(type_)
                version = component_config.parse_version(self._components[p].meta.version)
                return type_, version
        elif sort[0] == self.ComponentsSortTarget.STATUS:
            key = self._component_statuses.get
        else:
            key = identity
        sorted_paths = sorted(component_paths, key=key, reverse=(sort[-1] < 0))

        return list(map(self._components.get, sorted_paths))

    @requires([objects.ComponentDescriptor.zk_prefix])
    def list_components(self, skip=None, limit=None, query=None, sort=(ComponentsSortTarget.TYPE, 1)):
        component_pbs = self.list_all_components(query=query, sort=sort)

        total = len(component_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_COMPONENTS_LIMIT,
                                        max_limit=self.MAX_COMPONENTS_LIMIT)
        return pagination.SliceResult(items=component_pbs[page], total=total)

    @requires([objects.L7BalancerDescriptor.zk_prefix])
    def list_all_balancers(self, namespace_id=None, query=None):
        """
        :type namespace_id: six.text_type | None
        :type query: dict
        :rtype: list[model_pb2.Balancer]
        """
        balancer_paths = set(self._balancers)
        if namespace_id is not None:
            balancer_paths = self._balancer_paths_by_namespace_path.get(namespace_id) or set()

        query = query or {}
        nanny_service_id_in = set(query.get(self.BalancersQueryTarget.NANNY_SERVICE_ID_IN, []))
        if nanny_service_id_in:
            filtered_paths = set()
            for nanny_service_id in nanny_service_id_in:
                filtered_paths.update(self._balancer_paths_by_nanny_service_id[nanny_service_id])
            balancer_paths = balancer_paths & filtered_paths

        l7_macro_version_in = set(query.get(self.BalancersQueryTarget.L7_MACRO_VERSION_IN, []))
        if l7_macro_version_in:
            filtered_paths = set()
            for l7_macro_version in l7_macro_version_in:
                filtered_paths.update(self._balancer_paths_by_l7_macro_version[l7_macro_version])
            balancer_paths = balancer_paths & filtered_paths

        id_in = query.get(self.BalancersQueryTarget.ID_IN, [])
        if id_in:
            filtered_paths = set()
            for id_ in id_in:
                filtered_paths.update(self._balancer_paths_by_balancer_id[id_])
            balancer_paths = balancer_paths & filtered_paths

        canonical_id_in = query.get(self.BalancersQueryTarget.CANONICAL_ID_IN, [])
        if canonical_id_in:
            filtered_paths = set()
            for id_ in canonical_id_in:
                filtered_paths.update(self._balancer_paths_by_canonical_balancer_id[id_])
            balancer_paths = balancer_paths & filtered_paths

        component_in = query.get(self.BalancersQueryTarget.COMPONENT_IN, [])
        if component_in:
            component_balancer_paths = set()
            for component_ref_pb in component_in:
                if component_ref_pb.version:
                    path = self._balancer_paths_by_component_version[component_ref_pb.type][component_ref_pb.version]
                    component_balancer_paths |= path
                else:
                    paths_dict = self._balancer_paths_by_component_version[component_ref_pb.type]
                    for balancer_paths in six.itervalues(paths_dict):
                        component_balancer_paths |= balancer_paths
            balancer_paths &= component_balancer_paths

        balancers = list(map(self._balancers.get, sorted(balancer_paths)))
        return balancers

    @requires([objects.L7BalancerAspectSetDescriptor.zk_prefix])
    def list_all_balancer_aspects_sets(self, namespace_id=None):
        """
        :type namespace_id: six.text_type | None
        :rtype: list[model_pb2.BalancerAspectsSet]
        """
        balancer_aspects_set_paths = list(self._balancer_aspects_sets)
        aspects_set_pbs = list(map(self._balancer_aspects_sets.get, sorted(balancer_aspects_set_paths)))
        if namespace_id:
            aspects_set_pbs = [aspects_set_pb for aspects_set_pb in aspects_set_pbs if
                               aspects_set_pb.meta.namespace_id == namespace_id]
        return aspects_set_pbs

    @requires([objects.L7BalancerDescriptor.zk_prefix])
    def list_balancers(self, namespace_id=None, query=None, skip=None, limit=None):
        balancer_pbs = self.list_all_balancers(namespace_id=namespace_id, query=query)
        total = len(balancer_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_BALANCERS_LIMIT,
                                        max_limit=self.MAX_BALANCERS_LIMIT)
        return pagination.SliceResult(items=balancer_pbs[page], total=total)

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def list_full_balancer_ids_for_backends(self, full_backend_ids):
        """
        Returns a set of full ids of balancers that include backends with listed ids.
        :param iterable[(six.text_type, six.text_type)] full_backend_ids:
        :rtype: set[(six.text_type, six.text_type)]
        """
        full_balancer_ids_sets = list(map(self._full_backend_ids_to_full_balancer_id_sets.get, full_backend_ids))
        return set.union(*full_balancer_ids_sets)

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def list_full_balancer_ids_for_backend(self, namespace_id, backend_id):
        """
        Returns a set of full ids of balancers that include specified backend.
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        :rtype: set[(six.text_type, six.text_type)]
        """
        full_backend_id = (namespace_id, backend_id)
        return set(self._full_backend_ids_to_full_balancer_id_sets[full_backend_id])

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def list_full_balancer_ids_for_cert(self, namespace_id, cert_id):
        """
        Returns a set of full ids of balancers that include specified cert.
        :param six.text_type namespace_id:
        :param six.text_type cert_id:
        :rtype: set[(six.text_type, six.text_type)]
        """
        full_cert_id = (namespace_id, cert_id)
        return set(self._full_cert_ids_to_full_balancer_id_sets[full_cert_id])

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def list_full_balancer_ids_for_domain(self, namespace_id, domain_id):
        """
        Returns a set of full ids of balancers that include specified domain.
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :rtype: set[(six.text_type, six.text_type)]
        """
        full_domain_id = (namespace_id, domain_id)
        return set(self._full_domain_ids_to_full_balancer_id_sets[full_domain_id])

    @requires([objects.L3BalancerStateDescriptor.zk_prefix])
    def list_full_l3_balancer_ids_for_backend(self, namespace_id, backend_id):
        """
        Returns a set of full ids of l3 balancers that include specified backend.
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        :rtype: set[(six.text_type, six.text_type)]
        """
        full_backend_id = (namespace_id, backend_id)
        return set(self._full_backend_ids_to_full_l3_balancer_id_sets[full_backend_id])

    @requires([objects.DnsRecordStateDescriptor.zk_prefix])
    def list_full_dns_record_ids_for_backend(self, namespace_id, backend_id):
        """
        Returns a set of full ids of dns records that include specified backend.
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        :rtype: set[(six.text_type, six.text_type)]
        """
        full_backend_id = (namespace_id, backend_id)
        return set(self._full_backend_ids_to_full_dns_record_id_sets.get(full_backend_id, []))

    @requires([objects.L7BalancerDescriptor.zk_prefix])
    def get_balancer(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :rtype: model_pb2.Balancer | None
        """
        path = objects.L7BalancerDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        return self._balancers.get(path)

    @requires([objects.L7BalancerDescriptor.zk_prefix])
    def must_get_balancer(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :rtype: model_pb2.Balancer
        :raises: errors.NotFoundError
        """
        balancer_pb = self.get_balancer(namespace_id, balancer_id)
        if not balancer_pb:
            raise errors.NotFoundError(u'Balancer "{}:{}" not found'.format(namespace_id, balancer_id))
        return balancer_pb

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def get_balancer_state(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :rtype: model_pb2.BalancerState | None
        """
        path = objects.L7BalancerStateDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        return self._balancer_states.get(path)

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def must_get_balancer_state(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :rtype: model_pb2.BalancerState
        :raises: errors.NotFoundError
        """
        balancer_state_pb = self.get_balancer_state(namespace_id, balancer_id)
        if not balancer_state_pb:
            raise errors.NotFoundError('Balancer state "{}:{}" not found'.format(namespace_id, balancer_id))
        return balancer_state_pb

    @requires([objects.L7BalancerAspectSetDescriptor.zk_prefix])
    def get_balancer_aspects_set(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :rtype: model_pb2.BalancerAspectsSet | None
        """
        path = objects.L7BalancerAspectSetDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        return self._balancer_aspects_sets.get(path)

    @requires([objects.L7BalancerAspectSetDescriptor.zk_prefix])
    def must_get_balancer_aspects_set(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :rtype: model_pb2.BalancerAspectsSet
        :raises: errors.NotFoundError
        """
        balancer_aspects_set_pb = self.get_balancer_aspects_set(namespace_id, balancer_id)
        if not balancer_aspects_set_pb:
            raise errors.NotFoundError(u'Balancer aspects set "{}:{}" not found'.format(namespace_id, balancer_id))
        return balancer_aspects_set_pb

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def get_balancer_state_or_empty(self, namespace_id, balancer_id):
        rv = self.get_balancer_state(namespace_id, balancer_id)
        if not rv:
            rv = model_pb2.BalancerState(namespace_id=namespace_id, balancer_id=balancer_id)
        return rv

    @requires([objects.L3BalancerStateDescriptor.zk_prefix])
    def list_all_l3_balancer_states(self, namespace_id=None):
        l3_balancer_state_paths = list(self._l3_balancer_states)
        state_pbs = list(map(self._l3_balancer_states.get, sorted(l3_balancer_state_paths)))
        if namespace_id:
            state_pbs = [state_pb for state_pb in state_pbs if state_pb.namespace_id == namespace_id]
        return state_pbs

    @requires([objects.NameServerDescriptor.zk_prefix])
    def list_all_name_servers(self, namespace_id=None):
        name_server_paths = self._name_server_paths_by_namespace_path
        if namespace_id:
            paths = name_server_paths.get(namespace_id, set())
        else:
            paths = list(self._name_servers)
        return list(map(self._name_servers.get, paths))

    @requires([objects.NameServerDescriptor.zk_prefix])
    def get_name_server(self, namespace_id, name_server_id):
        """
        :rtype: model_pb2.NameServer | None
        """
        path = objects.NameServerDescriptor.uid_to_zk_path(namespace_id, name_server_id)
        return self._name_servers.get(path)

    @requires([objects.NameServerDescriptor.zk_prefix])
    def must_get_name_server(self, namespace_id, name_server_id):
        """
        :rtype: model_pb2.NameServer
        """
        name_server_pb = self.get_name_server(namespace_id, name_server_id)
        if not name_server_pb:
            raise errors.NotFoundError('Name server "{}:{}" not found'.format(namespace_id, name_server_id))
        return name_server_pb

    @requires([objects.DnsRecordDescriptor.zk_prefix, objects.NamespaceDescriptor.zk_prefix])
    def list_dns_record_fqdns(self):
        return self._dns_record_fqdns

    @requires([objects.DnsRecordDescriptor.zk_prefix])
    def count_dns_records(self, namespace_id=None, name_server_full_id=None):
        assert not (namespace_id and name_server_full_id)
        if namespace_id:
            return len(self._dns_record_paths_by_namespace_id.get(namespace_id, ()))
        elif name_server_full_id:
            return len(self._dns_record_paths_by_name_server_id.get(name_server_full_id, ()))
        else:
            return len(self._dns_records)

    @requires([objects.DnsRecordDescriptor.zk_prefix])
    def list_all_dns_records(self, namespace_id=None, name_server_full_id=None):
        """
        :rtype: list[model_pb2.DnsRecord]
        """
        assert not (namespace_id and name_server_full_id)
        if namespace_id:
            paths = self._dns_record_paths_by_namespace_id.get(namespace_id, ())
        elif name_server_full_id:
            paths = self._dns_record_paths_by_name_server_id.get(name_server_full_id, ())
        else:
            paths = list(self._dns_records)
        rv = []
        error_messages = []
        for path in sorted(paths):
            dns_record_pb = self._dns_records.get(path)
            if dns_record_pb is None:
                error_messages.append(
                    'dns record with path "{}" is not in self._dns_records. '
                    'namespace_id={}, name_server_full_id={}'.format(
                        path, namespace_id, name_server_full_id))
            else:
                rv.append(dns_record_pb)
        if error_messages:
            log = logging.getLogger('cache.list_all_dns_errors')
            for err in error_messages:
                log.error(err)
        return rv

    @requires([objects.DnsRecordDescriptor.zk_prefix])
    def get_dns_record(self, namespace_id, dns_record_id):
        """
        :rtype: model_pb2.DnsRecord | None
        """
        path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        return self._dns_records.get(path)

    @requires([objects.DnsRecordDescriptor.zk_prefix])
    def must_get_dns_record(self, namespace_id, dns_record_id):
        dns_record_pb = self.get_dns_record(namespace_id, dns_record_id)
        if not dns_record_pb:
            raise errors.NotFoundError('DNS record "{}:{}" not found'.format(namespace_id, dns_record_id))
        return dns_record_pb

    def _set_dns_record_pb(self, path, dns_record_pb):
        """
        :type dns_record_pb: model_pb2.DnsRecord
        """
        if self._must_be_ignored(path, dns_record_pb):
            return

        self._dns_records[path] = dns_record_pb
        namespace_id = dns_record_pb.meta.namespace_id
        dns_record_id = dns_record_pb.meta.id
        fqdn_info = dns_record_fqdn_info_tuple('dns_record', namespace_id, dns_record_id)

        if dns_record_pb.spec.incomplete or dns_record_pb.order.cancelled.value:
            name_server_pb = dns_record_pb.order.content.name_server
            zone = dns_record_pb.order.content.address.zone
            selected_l3_balancers_ids = [l3_bal_pb.id for l3_bal_pb in dns_record_pb.order.content.address.backends.l3_balancers]
        else:
            name_server_pb = dns_record_pb.spec.name_server
            zone = dns_record_pb.spec.address.zone
            selected_l3_balancers_ids = [l3_bal_pb.id for l3_bal_pb in dns_record_pb.spec.address.backends.l3_balancers]

        for l3_id in selected_l3_balancers_ids:
            self._dns_records_paths_by_l3_balancer_full_id[(namespace_id, l3_id)].add(path)
        self._dns_record_paths_by_namespace_id[namespace_id].add(path)
        self._dns_record_paths_by_name_server_id[(name_server_pb.namespace_id, name_server_pb.id)].add(path)
        # assume that name server ID matches its zone, since we can't get name_server_pb from cache on start
        self._dns_record_fqdns[name_server_pb.id][zone] = fqdn_info

        fqdn = zone + "." + name_server_pb.id
        self._fill_dns_records_wildcards_coverage_indexes(fqdn, namespace_id)

        event = events.DnsRecordUpdate(path=path, pb=dns_record_pb)
        self._run_callbacks(event)

    def _fill_dns_records_wildcards_coverage_indexes(self, fqdn, namespace_id):
        splitted_fqdn = fqdn.split(".")
        if splitted_fqdn[0] == "*":
            fqdn_part = ".".join(splitted_fqdn[1:])
            self._namespace_id_by_dns_record_wildcard[fqdn_part] = namespace_id

        for i in range(1, len(splitted_fqdn)):
            fqdn_part = ".".join(splitted_fqdn[i:])
            self.namespaces_covered_fqdns_count_by_dns_record_potential_wildcard[fqdn_part][namespace_id] += 1

    def _clean_dns_records_wildcards_coverage_indexes(self, fqdn, namespace_id):
        splitted_fqdn = fqdn.split(".")
        if splitted_fqdn[0] == "*":
            fqdn_part = ".".join(splitted_fqdn[1:])
            self._namespace_id_by_dns_record_wildcard.pop(fqdn_part, None)
        for i in range(1, len(splitted_fqdn)):
            fqdn_part = ".".join(splitted_fqdn[i:])
            self.namespaces_covered_fqdns_count_by_dns_record_potential_wildcard[fqdn_part][namespace_id] -= 1
            if self.namespaces_covered_fqdns_count_by_dns_record_potential_wildcard[fqdn_part][namespace_id] == 0:
                self.namespaces_covered_fqdns_count_by_dns_record_potential_wildcard[fqdn_part].pop(namespace_id)

    def _del_dns_record_pb(self, path):
        if path in self._dns_records:
            dns_record_pb = self._dns_records[path]
            namespace_id = dns_record_pb.meta.namespace_id
            if dns_record_pb.spec.incomplete or dns_record_pb.order.cancelled.value:
                name_server_pb = dns_record_pb.order.content.name_server
                zone = dns_record_pb.order.content.address.zone
                selected_l3_balancers_ids = [l3_bal_pb.id for l3_bal_pb in dns_record_pb.order.content.address.backends.l3_balancers]
            else:
                name_server_pb = dns_record_pb.spec.name_server
                zone = dns_record_pb.spec.address.zone
                selected_l3_balancers_ids = [l3_bal_pb.id for l3_bal_pb in dns_record_pb.spec.address.backends.l3_balancers]

            for l3_id in selected_l3_balancers_ids:
                self._dns_records_paths_by_l3_balancer_full_id[(namespace_id, l3_id)].discard(path)

            # assume that name server ID matches its zone, since we can't get name_server_pb from cache on start
            self._dns_record_fqdns[name_server_pb.id].pop(zone, None)
            fqdn = zone + "." + name_server_pb.id
            self._clean_dns_records_wildcards_coverage_indexes(fqdn, namespace_id)
            self._dns_record_paths_by_namespace_id[namespace_id].remove(path)
            self._dns_record_paths_by_name_server_id[(name_server_pb.namespace_id, name_server_pb.id)].remove(path)
            del self._dns_records[path]
        event = events.DnsRecordRemove(path=path)
        self._run_callbacks(event)

    def list_all_dns_record_operations(self, namespace_id, dns_record_id=None):
        """
        :rtype: list[model_pb2.DnsRecordOperation]
        """
        if dns_record_id:
            paths = self._dns_record_op_paths_by_dns_record_id.get((namespace_id, dns_record_id), set())
        else:
            paths = self._dns_record_op_paths_by_namespace_id.get(namespace_id, set())
        return list(map(self._dns_record_operations.get, paths))

    def get_dns_record_operation(self, namespace_id, dns_record_op_id):
        """
        :rtype: model_pb2.DnsRecordOperation | None
        """
        path = objects.DnsRecordOperationDescriptor.uid_to_zk_path(namespace_id, dns_record_op_id)
        return self._dns_record_operations.get(path)

    def must_get_dns_record_operation(self, namespace_id, dns_record_op_id):
        """
        :rtype: model_pb2.DnsRecordOperation
        """
        dns_record_op_pb = self.get_dns_record_operation(namespace_id, dns_record_op_id)
        if not dns_record_op_pb:
            raise errors.NotFoundError('DNS record operation "{}:{}" not found'.format(namespace_id, dns_record_op_id))
        return dns_record_op_pb

    def _set_dns_record_op_pb(self, path, dns_record_op_pb):
        """
        :type dns_record_op_pb: model_pb2.DnsRecordOperation
        """
        if (path in self._dns_record_operations and
                self._dns_record_operations[path].meta.generation >= dns_record_op_pb.meta.generation):
            return
        self._dns_record_operations[path] = dns_record_op_pb
        dns_record_id = dns_record_op_pb.meta.dns_record_id
        namespace_id = dns_record_op_pb.meta.namespace_id
        self._dns_record_op_paths_by_dns_record_id[(namespace_id, dns_record_id)].add(path)
        self._dns_record_op_paths_by_namespace_id[namespace_id].add(path)
        event = events.DnsRecordOperationUpdate(path=path, pb=dns_record_op_pb)
        self._run_callbacks(event)

    def _del_dns_record_op_pb(self, path):
        if path in self._dns_record_operations:
            dns_record_op_pb = self._dns_record_operations[path]
            dns_record_id = dns_record_op_pb.meta.dns_record_id
            namespace_id = dns_record_op_pb.meta.namespace_id
            self._dns_record_op_paths_by_dns_record_id[(namespace_id, dns_record_id)].remove(path)
            self._dns_record_op_paths_by_namespace_id[namespace_id].remove(path)
            del self._dns_record_operations[path]
        event = events.DnsRecordOperationRemove(path=path)
        self._run_callbacks(event)

    def _set_name_server_pb(self, path, name_server_pb):
        """
        :type name_server_pb: model_pb2.NameServer
        """
        if self._must_be_ignored(path, name_server_pb):
            return

        self._name_servers[path] = name_server_pb
        namespace_id = name_server_pb.meta.namespace_id
        self._name_server_paths_by_namespace_path[namespace_id].add(path)
        event = events.NameServerUpdate(path=path, pb=name_server_pb)
        self._run_callbacks(event)

    def _del_name_server_pb(self, path):
        if path in self._name_servers:
            name_server_pb = self._name_servers[path]
            namespace_id = name_server_pb.meta.namespace_id
            self._name_server_paths_by_namespace_path[namespace_id].remove(path)
            del self._name_servers[path]
        event = events.NameServerRemove(path=path)
        self._run_callbacks(event)

    @requires([objects.DnsRecordStateDescriptor.zk_prefix])
    def list_all_dns_record_states(self, namespace_id=None):
        """
        :type namespace_id: six.text_type | None
        :rtype: list[model_pb2.DnsRecordState]
        """
        dns_record_state_paths = list(self._dns_record_states)
        state_pbs = list(map(self._dns_record_states.get, sorted(dns_record_state_paths)))
        if namespace_id:
            state_pbs = [state_pb for state_pb in state_pbs if state_pb.namespace_id == namespace_id]
        return state_pbs

    @requires([objects.DnsRecordStateDescriptor.zk_prefix])
    def get_dns_record_state(self, namespace_id, dns_record_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :rtype: model_pb2.DnsRecordState | None
        """
        path = objects.DnsRecordStateDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        return self._dns_record_states.get(path)

    @requires([objects.DnsRecordStateDescriptor.zk_prefix])
    def must_get_dns_record_state(self, namespace_id, dns_record_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :rtype: model_pb2.DnsRecordState
        :raises: errors.NotFoundError
        """
        dns_record_state_pb = self.get_dns_record_state(namespace_id, dns_record_id)
        if not dns_record_state_pb:
            raise errors.NotFoundError('DNS record state "{}:{}" not found'.format(namespace_id, dns_record_id))
        return dns_record_state_pb

    @requires([objects.DnsRecordStateDescriptor.zk_prefix])
    def get_dns_record_state_or_empty(self, namespace_id, dns_record_id):
        rv = self.get_dns_record_state(namespace_id, dns_record_id)
        if not rv:
            rv = model_pb2.DnsRecordState(namespace_id=namespace_id, dns_record_id=dns_record_id)
        return rv

    def _set_dns_record_state_pb(self, path, dns_record_state_pb):
        """
        :type dns_record_state_pb: model_pb2.DnsRecordState
        """
        if self._must_be_ignored(path, dns_record_state_pb, meta_field=None):
            return

        self._dns_record_states[path] = dns_record_state_pb

        full_dns_record_id = (dns_record_state_pb.namespace_id, dns_record_state_pb.dns_record_id)
        self._update_backends_indexes_on_state_update(
            full_dns_record_id,
            dns_record_state_pb,
            self._full_backend_ids_to_full_dns_record_id_sets,
            self._full_dns_record_ids_to_full_backend_id_sets)

        event = events.DnsRecordStateUpdate(path=path, pb=dns_record_state_pb)
        self._run_callbacks(event)

    def _del_dns_record_state_pb(self, path):
        if path in self._dns_record_states:
            dns_record_state_pb = self._dns_record_states[path]
            full_dns_record_id = (dns_record_state_pb.namespace_id, dns_record_state_pb.dns_record_id)
            del self._dns_record_states[path]
            self._update_backends_indexes_on_state_removal(
                full_dns_record_id,
                self._full_backend_ids_to_full_dns_record_id_sets,
                self._full_dns_record_ids_to_full_backend_id_sets)

        event = events.DnsRecordStateRemove(path=path)
        self._run_callbacks(event)

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def count_l3_balancers(self, namespace_id=None):
        if namespace_id:
            return len(self._l3_balancer_paths_by_namespace_path.get(namespace_id) or frozenset())
        else:
            return len(self._l3_balancers)

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def list_all_l3_balancers(self, namespace_id=None, query=None):
        """
        :type namespace_id: six.text_type | None
        :type query: dict
        :rtype: list[model_pb2.L3Balancer]
        """
        l3_balancer_paths = list(self._l3_balancers)
        if namespace_id:
            l3_balancer_paths = self._l3_balancer_paths_by_namespace_path.get(namespace_id) or set()

        query = query or {}
        l3mgr_service_id_in = set(query.get(self.L3BalancersQueryTarget.L3MGR_SERVICE_ID_IN, []))
        if l3mgr_service_id_in:
            filtered_paths = set()
            for path in l3_balancer_paths:
                l3_balancer_spec_pb = self._l3_balancers[path].spec  # type: model_pb2.L3BalancerSpec
                if l3_balancer_spec_pb.l3mgr_service_id not in l3mgr_service_id_in:
                    continue
                filtered_paths.add(path)
            l3_balancer_paths = filtered_paths

        l3_balancers = list(map(self._l3_balancers.get, sorted(l3_balancer_paths)))
        return l3_balancers

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def list_l3_balancers(self, namespace_id=None, skip=None, limit=None):
        l3_balancer_pbs = self.list_all_l3_balancers(namespace_id=namespace_id)
        total = len(l3_balancer_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_BALANCERS_LIMIT,
                                        max_limit=self.MAX_BALANCERS_LIMIT)
        return pagination.SliceResult(items=l3_balancer_pbs[page], total=total)

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def get_l3_balancer(self, namespace_id, l3_balancer_id):
        """
        :rtype: model_pb2.L3Balancer
        """
        path = objects.L3BalancerDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        return self._l3_balancers.get(path)

    @requires([objects.L3BalancerDescriptor.zk_prefix])
    def must_get_l3_balancer(self, namespace_id, l3_balancer_id):
        """
        :rtype: model_pb2.L3Balancer
        """
        l3_balancer_pb = self.get_l3_balancer(namespace_id, l3_balancer_id)
        if not l3_balancer_pb:
            raise errors.NotFoundError('L3 balancer "{}:{}" not found'.format(namespace_id, l3_balancer_id))
        return l3_balancer_pb

    @requires([objects.L3BalancerDescriptor.zk_prefix, objects.DnsRecordDescriptor.zk_prefix])
    def list_dns_records_for_l3_balancer(self, namespace_id, l3_balancer_id):
        """
        :rtype: list[model_pb2.DnsRecord]
        """
        dns_record_pbs = []
        for path in self._dns_records_paths_by_l3_balancer_full_id[(namespace_id, l3_balancer_id)]:
            dns_record_pbs.append(self._dns_records[path])
        return dns_record_pbs

    @requires([objects.L3BalancerStateDescriptor.zk_prefix])
    def get_l3_balancer_state(self, namespace_id, l3_balancer_id):
        """
        :rtype: model_pb2.L3BalancerState
        """
        path = objects.L3BalancerStateDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        return self._l3_balancer_states.get(path)

    @requires([objects.L3BalancerStateDescriptor.zk_prefix])
    def must_get_l3_balancer_state(self, namespace_id, l3_balancer_id):
        l3_balancer_state_pb = self.get_l3_balancer_state(namespace_id, l3_balancer_id)
        if not l3_balancer_state_pb:
            raise errors.NotFoundError('L3 balancer state "{}:{}" not found'.format(namespace_id, l3_balancer_id))
        return l3_balancer_state_pb

    @requires([objects.L3BalancerStateDescriptor.zk_prefix])
    def get_l3_balancer_state_or_empty(self, namespace_id, l3_balancer_id):
        rv = self.get_l3_balancer_state(namespace_id, l3_balancer_id)
        if not rv:
            rv = model_pb2.L3BalancerState(namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
        return rv

    @requires([objects.UpstreamDescriptor.zk_prefix])
    def count_upstreams(self, namespace_id=None):
        if namespace_id:
            return len(self._upstream_paths_by_namespace_path.get(namespace_id) or frozenset())
        else:
            return len(self._upstreams)

    @requires([objects.UpstreamDescriptor.zk_prefix])
    def list_all_upstreams(self, namespace_id, query=None, sort=(UpstreamsSortTarget.ID, 1)):
        paths = self._upstream_paths_by_namespace_path.get(namespace_id) or set()
        query = query or {}

        id_in = query.get(self.UpstreamsQueryTarget.ID_IN, [])
        if id_in:
            paths = {p for p in paths if self._upstreams[p].meta.id in id_in}

        id_regexp = query.get(self.UpstreamsQueryTarget.ID_REGEXP)
        if id_regexp:
            paths = {p for p in paths if id_regexp.search(self._upstreams[p].meta.id)}

        type_in = query.get(self.UpstreamsQueryTarget.TYPE_IN)
        if type_in:
            filtered = set()
            for type_, paths_by_type in six.iteritems(self._upstream_paths_by_type):
                if type_ not in type_in:
                    continue
                filtered |= paths & paths_by_type
            paths = filtered

        validated_status_in = query.get(self.UpstreamsQueryTarget.VALIDATED_STATUS_IN, [])
        if validated_status_in:
            paths = {p for p in paths if self._upstreams[p].status.validated.status in validated_status_in}

        in_progress_status_in = query.get(self.UpstreamsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        if in_progress_status_in:
            paths = {p for p in paths if self._upstreams[p].status.in_progress.status in in_progress_status_in}

        active_status_in = query.get(self.UpstreamsQueryTarget.ACTIVE_STATUS_IN, [])
        if active_status_in:
            paths = {p for p in paths if self._upstreams[p].status.active.status in active_status_in}

        deleted = query.get(self.UpstreamsQueryTarget.DELETED, False)
        if deleted:
            paths = paths & self._deleted_upstream_paths

        if sort[0] == self.UpstreamsSortTarget.ORDER_LABEL:
            key = self._upstream_order_labels.get
        elif sort[0] == self.UpstreamsSortTarget.MTIME:
            key = self._upstream_mtimes.get
        else:
            key = identity
        sorted_paths = sorted(paths, key=key)
        if sort[-1] < 0:
            sorted_paths.reverse()

        return list(map(self._upstreams.get, sorted_paths))

    @requires([objects.UpstreamDescriptor.zk_prefix])
    def list_upstreams(self, namespace_id, query=None, sort=(UpstreamsSortTarget.ID, 1), skip=None, limit=None):
        upstream_pbs = self.list_all_upstreams(namespace_id, query=query, sort=sort)
        total = len(upstream_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_UPSTREAMS_LIMIT,
                                        max_limit=self.MAX_UPSTREAMS_LIMIT)
        return pagination.SliceResult(items=upstream_pbs[page], total=total)

    @requires([objects.UpstreamDescriptor.zk_prefix])
    def get_upstream(self, namespace_id, upstream_id):
        path = objects.UpstreamDescriptor.uid_to_zk_path(namespace_id, upstream_id)
        return self._upstreams.get(path)

    @requires([objects.UpstreamDescriptor.zk_prefix])
    def must_get_upstream(self, namespace_id, upstream_id):
        upstream_pb = self.get_upstream(namespace_id, upstream_id)
        if not upstream_pb:
            raise errors.NotFoundError('Upstream "{}:{}" not found'.format(namespace_id, upstream_id))
        return upstream_pb

    @requires([objects.DomainDescriptor.zk_prefix, objects.DomainOperationDescriptor.zk_prefix])
    def list_domain_fqdns(self):
        return self._domain_fqdns

    @requires([objects.DomainDescriptor.zk_prefix])
    def count_domains(self, namespace_id=None):
        if namespace_id:
            return len(self._domain_paths_by_namespace_path.get(namespace_id) or ())
        return len(self._domains)

    @requires([objects.DomainDescriptor.zk_prefix])
    def list_all_domains(self, namespace_id, query=None, sort=(DomainsSortTarget.ID, 1)):
        paths = self._domain_paths_by_namespace_path.get(namespace_id) or frozenset()

        query = query or {}
        id_in = query.get(self.DomainsQueryTarget.ID_IN, [])
        id_regexp = query.get(self.DomainsQueryTarget.ID_REGEXP)
        validated_status_in = query.get(self.DomainsQueryTarget.VALIDATED_STATUS_IN, [])
        in_progress_status_in = query.get(self.DomainsQueryTarget.IN_PROGRESS_STATUS_IN, [])
        active_status_in = query.get(self.DomainsQueryTarget.ACTIVE_STATUS_IN, [])

        if id_in:
            paths = {p for p in paths if self._domains[p].meta.id in id_in}
        if id_regexp:
            paths = {p for p in paths if id_regexp.search(self._domains[p].meta.id)}
        if validated_status_in:
            paths = {p for p in paths if self._domains[p].status.validated.status in validated_status_in}
        if in_progress_status_in:
            paths = {p for p in paths if self._domains[p].status.in_progress.status in in_progress_status_in}
        if active_status_in:
            paths = {p for p in paths if self._domains[p].status.active.status in active_status_in}

        if sort[0] == self.DomainsSortTarget.MTIME:
            key = self._domain_mtimes.get
        else:
            key = identity
        sorted_paths = sorted(paths, key=key)
        if sort[-1] < 0:
            sorted_paths.reverse()

        return list(map(self._domains.get, sorted_paths))

    @requires([objects.DomainOperationDescriptor.zk_prefix])
    def list_all_domain_operations(self, namespace_id, query=None):
        paths = self._domain_operation_paths_by_namespace_path.get(namespace_id) or frozenset()

        query = query or {}
        id_in = query.get(self.DomainsQueryTarget.ID_IN, [])
        id_regexp = query.get(self.DomainsQueryTarget.ID_REGEXP)

        if id_in:
            paths = {p for p in paths if self._domain_operations[p].meta.id in id_in}
        if id_regexp:
            paths = {p for p in paths if id_regexp.search(self._domain_operations[p].meta.id)}

        sorted_paths = sorted(paths, key=identity)

        return list(map(self._domain_operations.get, sorted_paths))

    @requires([objects.DomainDescriptor.zk_prefix])
    def list_domains(self, namespace_id, query=None, sort=(DomainsSortTarget.ID, 1), skip=None, limit=None):
        domain_pbs = self.list_all_domains(namespace_id, query=query, sort=sort)
        total = len(domain_pbs)
        page = pagination.compute_slice(skip, limit,
                                        default_limit=self.DEFAULT_DOMAINS_LIMIT,
                                        max_limit=self.MAX_DOMAINS_LIMIT)
        return pagination.SliceResult(items=domain_pbs[page], total=total)

    @requires([objects.DomainDescriptor.zk_prefix])
    def list_namespace_domain_config_pbs(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: dict[tuple[six.text_type, six.text_type], model_pb2.DomainSpec.Config]
        """
        domain_pbs = self.list_all_domains(namespace_id)
        return {(d_pb.meta.namespace_id, d_pb.meta.id): d_pb.spec.yandex_balancer.config for d_pb in domain_pbs}

    @requires([objects.DomainDescriptor.zk_prefix])
    def get_domain(self, namespace_id, domain_id):
        path = objects.DomainDescriptor.uid_to_zk_path(namespace_id, domain_id)
        return self._domains.get(path)

    @requires([objects.DomainDescriptor.zk_prefix])
    def does_domain_exist(self, namespace_id, domain_id):
        zk_path = objects.DomainDescriptor.uid_to_zk_path(namespace_id, domain_id)
        return zk_path in self._domains

    @requires([objects.NamespaceDescriptor.zk_prefix])
    def does_namespace_normalised_name_exist(self, namespace_id):
        norm_name = self.normalise_namespace_name(namespace_id)
        return norm_name in self._namespace_normalised_names

    @requires([objects.DomainDescriptor.zk_prefix])
    def must_get_domain(self, namespace_id, domain_id):
        domain_pb = self.get_domain(namespace_id, domain_id)
        if not domain_pb:
            raise errors.NotFoundError('Domain "{}:{}" not found'.format(namespace_id, domain_id))
        return domain_pb

    @requires([objects.DomainOperationDescriptor.zk_prefix])
    def get_domain_operation(self, namespace_id, domain_id):
        path = objects.DomainOperationDescriptor.uid_to_zk_path(namespace_id, domain_id)
        return self._domain_operations.get(path)

    @requires([objects.DomainOperationDescriptor.zk_prefix])
    def must_get_domain_operation(self, namespace_id, domain_id):
        domain_op_pb = self.get_domain_operation(namespace_id, domain_id)
        if not domain_op_pb:
            raise errors.NotFoundError('Domain operation "{}:{}" not found'.format(namespace_id, domain_id))
        return domain_op_pb

    @requires([objects.DomainOperationDescriptor.zk_prefix])
    def does_domain_operation_exist(self, namespace_id, domain_id):
        zk_path = objects.DomainOperationDescriptor.uid_to_zk_path(namespace_id, domain_id)
        return zk_path in self._domain_operations

    @requires([objects.L7BalancerOperationDescriptor.zk_prefix])
    def get_balancer_operation(self, namespace_id, balancer_id):
        path = objects.L7BalancerOperationDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        return self._balancer_operations.get(path)

    @requires([objects.L7BalancerOperationDescriptor.zk_prefix])
    def must_get_balancer_operation(self, namespace_id, balancer_id):
        op_pb = self.get_balancer_operation(namespace_id, balancer_id)
        if not op_pb:
            raise errors.NotFoundError('Balancer operation "{}:{}" not found'.format(namespace_id, balancer_id))
        return op_pb

    @requires([objects.L7BalancerOperationDescriptor.zk_prefix])
    def list_all_balancer_operations(self, namespace_id, query=None):
        paths = self._balancer_operation_paths_by_namespace_path.get(namespace_id) or frozenset()

        query = query or {}
        id_in = query.get(self.BalancerOpsQueryTarget.ID_IN, [])

        if id_in:
            paths = {p for p in paths if self._balancer_operations[p].meta.id in id_in}

        sorted_paths = sorted(paths, key=identity)

        return list(map(self._balancer_operations.get, sorted_paths))

    @requires([objects.L7BalancerOperationDescriptor.zk_prefix])
    def does_balancer_operation_exist(self, namespace_id, balancer_id):
        zk_path = objects.L7BalancerOperationDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        return zk_path in self._balancer_operations

    @requires([objects.BackendDescriptor.zk_prefix])
    def list_backend_ids(self):
        return set(self._backends)

    @requires([objects.L7BalancerStateDescriptor.zk_prefix])
    def list_included_backend_ids(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :rtype: set[(six.text_type, six.text_type)]
        """
        full_balancer_id = (namespace_id, balancer_id)
        return set(self._full_balancer_ids_to_full_backend_id_sets[full_balancer_id])

    @requires([objects.UpstreamDescriptor.zk_prefix])
    def list_easy_mode_upstreams_ids(self, namespace_id, backend_id):
        full_backend_id = (namespace_id, backend_id)
        return self._easy_mode_upstream_ids_by_backend_id[full_backend_id]

    @requires([objects.BackendDescriptor.zk_prefix])
    def count_backends(self, namespace_id=None):
        if namespace_id:
            return len(self._backend_paths_by_namespace_path.get(namespace_id) or frozenset())
        else:
            return len(self._backends)

    @requires([objects.BackendDescriptor.zk_prefix])
    def list_all_backends(self, namespace_id=None, sort=(BackendsSortTarget.ID, 1), query=None):
        """
        :type namespace_id: six.text_type
        :type sort: Iterable
        :type query: dict
        :rtype: list[model_pb2.Backend]
        """
        if namespace_id:
            backend_paths = self._backend_paths_by_namespace_path.get(namespace_id) or set()
        else:
            backend_paths = set(self._backends)

        query = query or {}
        id_in = query.get(self.BackendsQueryTarget.ID_IN, [])
        if id_in:
            filtered_paths = set()
            for id_ in id_in:
                filtered_paths.update(self._backend_paths_by_backend_id[id_])
            backend_paths = backend_paths & filtered_paths

        only_system = query.get(self.BackendsQueryTarget.ONLY_SYSTEM, False)
        if only_system:
            backend_paths = backend_paths & self._backend_paths_system

        exclude_system = query.get(self.BackendsQueryTarget.EXCLUDE_SYSTEM, False)
        if exclude_system:
            backend_paths = backend_paths & self._backend_paths_not_system

        nanny_service_id_in = query.get(self.BackendsQueryTarget.NANNY_SERVICE_ID_IN, [])
        if nanny_service_id_in:
            filtered_paths = set()
            for nanny_service_id in nanny_service_id_in:
                filtered_paths.update(self._backend_paths_by_nanny_service_id[nanny_service_id])
            backend_paths = backend_paths & filtered_paths

        gencfg_group_name_in = query.get(self.BackendsQueryTarget.GENCFG_GROUP_NAME_IN, [])
        if gencfg_group_name_in:
            filtered_paths = set()
            for gencfg_group_name in gencfg_group_name_in:
                filtered_paths.update(self._backend_paths_by_gencfg_group_name[gencfg_group_name])
            backend_paths = backend_paths & filtered_paths

        yp_endpoint_set_full_id_in = query.get(self.BackendsQueryTarget.YP_ENDPOINT_SET_FULL_ID_IN, [])
        if yp_endpoint_set_full_id_in:
            filtered_paths = set()
            for yp_endpoint_set_full_id in yp_endpoint_set_full_id_in:
                filtered_paths.update(self._backend_paths_by_yp_endpoint_set_full_id[
                    yp_endpoint_set_full_id.cluster, yp_endpoint_set_full_id.id])

                full_id = (yp_endpoint_set_full_id.cluster, yp_endpoint_set_full_id.id)
                full_balancer_id = self._full_balancer_id_by_yp_endpoint_set_full_id.get(full_id)
                if full_balancer_id:
                    filtered_paths.update(self._backend_paths_by_full_balancer_id[full_balancer_id])
            backend_paths = backend_paths & filtered_paths

        id_regexp = query.get(self.BackendsQueryTarget.ID_REGEXP)
        if id_regexp:
            backend_paths = {p for p in backend_paths if id_regexp.search(self._backends[p].meta.id)}

        if sort[0] == self.BackendsSortTarget.MTIME:
            key = self._backend_mtimes.get
        else:
            key = identity
        sorted_paths = sorted(backend_paths, key=key)
        if sort[-1] < 0:
            sorted_paths.reverse()
        return list(map(self._backends.get, sorted_paths))

    @requires([objects.BackendDescriptor.zk_prefix])
    def get_system_backend_for_balancer(self, namespace_id, balancer_id):
        """
        :rtype: model_pb2.Backend | None
        """
        backend_pb = self.get_backend(namespace_id, balancer_id)
        if backend_pb and backend_pb.meta.is_system.value:
            return backend_pb
        return None

    @requires([objects.BackendDescriptor.zk_prefix])
    def get_backend(self, namespace_id, backend_id):
        """
        :rtype: model_pb2.Backend | None
        """
        path = objects.BackendDescriptor.uid_to_zk_path(namespace_id, backend_id)
        return self._backends.get(path)

    @requires([objects.BackendDescriptor.zk_prefix])
    def must_get_backend(self, namespace_id, backend_id):
        backend_pb = self.get_backend(namespace_id, backend_id)
        if not backend_pb:
            raise errors.NotFoundError('Backend "{}:{}" not found'.format(namespace_id, backend_id))
        return backend_pb

    @requires([objects.EndpointSetDescriptor.zk_prefix])
    def count_endpoint_sets(self, namespace_id=None):
        if namespace_id:
            return len(self._endpoint_set_paths_by_namespace_path.get(namespace_id) or frozenset())
        else:
            return len(self._endpoint_sets)

    @requires([objects.EndpointSetDescriptor.zk_prefix])
    def list_all_endpoint_sets(self, namespace_id):
        paths = self._endpoint_set_paths_by_namespace_path.get(namespace_id) or set()
        return list(map(self._endpoint_sets.get, paths))

    @requires([objects.EndpointSetDescriptor.zk_prefix])
    def get_endpoint_set(self, namespace_id, endpoint_set_id):
        path = objects.EndpointSetDescriptor.uid_to_zk_path(namespace_id, endpoint_set_id)
        return self._endpoint_sets.get(path)

    @requires([objects.EndpointSetDescriptor.zk_prefix])
    def must_get_endpoint_set(self, namespace_id, endpoint_set_id):
        endpoint_set_pb = self.get_endpoint_set(namespace_id, endpoint_set_id)
        if not endpoint_set_pb:
            raise errors.NotFoundError('Endpoint set "{}:{}" not found'.format(namespace_id, endpoint_set_id))
        return endpoint_set_pb

    @requires([objects.KnobDescriptor.zk_prefix])
    def count_knobs(self, namespace_id=None):
        if namespace_id:
            return len(self._knob_paths_by_namespace_path.get(namespace_id) or frozenset())
        else:
            return len(self._knobs)

    @requires([objects.KnobDescriptor.zk_prefix])
    def list_all_knobs(self, namespace_id=None, sort=(KnobsSortTarget.ID, 1)):
        """
        :type namespace_id: six.text_type
        :type sort: Iterable
        :rtype: list[model_pb2.Knob]
        """
        knob_paths = self._knob_paths_by_namespace_path
        if namespace_id:
            paths = knob_paths.get(namespace_id, set())
        elif knob_paths:
            paths = set.union(*six.itervalues(knob_paths))
        else:
            paths = set()

        if sort[0] == self.KnobsSortTarget.MTIME:
            key = self._knob_mtimes.get
        else:
            key = identity
        sorted_paths = sorted(paths, key=key)
        if sort[-1] < 0:
            sorted_paths.reverse()
        return list(map(self._knobs.get, sorted_paths))

    @requires([objects.KnobDescriptor.zk_prefix])
    def get_knob(self, namespace_id, knob_id):
        """
        :rtype: model_pb2.Knob | None
        """
        path = objects.KnobDescriptor.uid_to_zk_path(namespace_id, knob_id)
        return self._knobs.get(path)

    @requires([objects.KnobDescriptor.zk_prefix])
    def must_get_knob(self, namespace_id, knob_id):
        knob_pb = self.get_knob(namespace_id, knob_id)
        if not knob_pb:
            raise errors.NotFoundError('Knob "{}:{}" not found'.format(namespace_id, knob_id))
        return knob_pb

    @requires([objects.CertDescriptor.zk_prefix])
    def count_certs(self, namespace_id=None):
        if namespace_id:
            return len(self._cert_paths_by_namespace_path.get(namespace_id) or frozenset())
        else:
            return len(self._certs)

    @requires([objects.CertDescriptor.zk_prefix])
    def list_all_cert_ids(self, namespace_id=None):
        for path in self._list_all_cert_paths(namespace_id):
            yield path.split('/')[-1]

    @requires([objects.CertDescriptor.zk_prefix])
    def _list_all_cert_paths(self, namespace_id=None):
        cert_paths = self._cert_paths_by_namespace_path
        if namespace_id:
            return cert_paths.get(namespace_id, set())
        elif cert_paths:
            return set.union(*six.itervalues(cert_paths))
        else:
            return set()

    @requires([objects.CertDescriptor.zk_prefix])
    def list_all_certs(self, namespace_id=None, sort=(CertsSortTarget.ID, 1)):
        """
        :type namespace_id: six.text_type
        :type sort: Iterable
        :rtype: list[model_pb2.Certificate]
        """
        paths = self._list_all_cert_paths(namespace_id)

        if sort[0] == self.CertsSortTarget.MTIME:
            key = self._cert_mtimes.get
        else:
            key = identity
        sorted_paths = sorted(paths, key=key)
        if sort[-1] < 0:
            sorted_paths.reverse()
        return list(map(self._certs.get, sorted_paths))

    @requires([objects.CertDescriptor.zk_prefix])
    def get_cert(self, namespace_id, cert_id):
        """
        :rtype: model_pb2.Certificate | None
        """
        path = objects.CertDescriptor.uid_to_zk_path(namespace_id, cert_id)
        return self._certs.get(path)

    @requires([objects.CertDescriptor.zk_prefix])
    def must_get_cert(self, namespace_id, cert_id):
        """
        :rtype: model_pb2.Certificate
        """
        cert_pb = self.get_cert(namespace_id, cert_id)
        if not cert_pb:
            raise errors.NotFoundError('Certificate "{}:{}" not found'.format(namespace_id, cert_id))
        return cert_pb

    @requires([objects.CertRenewalDescriptor.zk_prefix])
    def does_cert_renewal_exist(self, namespace_id, cert_renewal_id):
        """
        :type namespace_id: six.text_type
        :type cert_renewal_id: six.text_type
        :rtype: bool
        """
        zk_path = objects.CertRenewalDescriptor.uid_to_zk_path(namespace_id, cert_renewal_id)
        return zk_path in self._cert_renewals

    @requires([objects.CertRenewalDescriptor.zk_prefix])
    def get_cert_renewal(self, namespace_id, cert_renewal_id):
        """
        :rtype: model_pb2.CertificateRenewal | None
        """
        path = objects.CertRenewalDescriptor.uid_to_zk_path(namespace_id, cert_renewal_id)
        return self._cert_renewals.get(path)

    @requires([objects.CertRenewalDescriptor.zk_prefix])
    def must_get_cert_renewal(self, namespace_id, cert_renewal_id):
        """
        :rtype: model_pb2.CertificateRenewal
        """
        cert_renewal_pb = self.get_cert_renewal(namespace_id, cert_renewal_id)
        if not cert_renewal_pb:
            raise errors.NotFoundError('Certificate renewal "{}:{}" not found'.format(namespace_id, cert_renewal_id))
        return cert_renewal_pb

    @requires([objects.CertRenewalDescriptor.zk_prefix])
    def list_all_cert_renewals(self, namespace_id=None, sort=(CertRenewalsSortTarget.ID, 1)):
        """
        :type namespace_id: six.text_type
        :type sort: Iterable
        :rtype: list[model_pb2.CertificateRenewal]
        """
        cert_renewal_paths = self._cert_renewal_paths_by_namespace_path
        if namespace_id:
            paths = cert_renewal_paths.get(namespace_id, set())
        elif cert_renewal_paths:
            paths = set.union(*six.itervalues(cert_renewal_paths))
        else:
            paths = set()

        if sort[0] == self.CertRenewalsSortTarget.TARGET_CERT_VALIDITY_NOT_AFTER:
            key = self._cert_not_afters.get
        else:
            key = identity
        sorted_paths = sorted(paths, key=key)
        if sort[-1] < 0:
            sorted_paths.reverse()

        return list(map(self._cert_renewals.get, sorted_paths))

    def _set_namespace_pb(self, path, namespace_pb):
        """
        :type namespace_pb: model_pb2.Namespace
        """
        if self._must_be_ignored(path, namespace_pb):
            return

        self._namespaces[path] = namespace_pb

        for paths in six.itervalues(self._namespace_paths_by_layout_type):
            paths.discard(path)
        self._namespace_paths_by_layout_type[namespace_pb.spec.layout_type].add(path)

        if namespace_pb.spec.balancer_constraints.instance_tags.prj:
            normalised_prj = self.normalise_prj_tag(namespace_pb.spec.balancer_constraints.instance_tags.prj)
            self._namespace_id_by_normalised_prj[normalised_prj] = namespace_pb.meta.id
        if namespace_pb.spec.HasField('its'):
            self._namespaces_with_enabled_its.add(namespace_pb.meta.id)
        else:
            self._namespaces_with_enabled_its.discard(namespace_pb.meta.id)
        dns_record_req = namespace_pb.order.content.dns_record_request.default
        if is_order_in_progress(namespace_pb):
            if namespace_pb.meta.id in self._namespace_being_created_states:
                known_since = self._namespace_being_created_states[namespace_pb.meta.id].known_since
            else:
                known_since = datetime.utcnow()
            state = NamespaceState(known_since=known_since,
                                   ctime=namespace_pb.meta.ctime.ToDatetime(),
                                   state=namespace_pb.order.progress.state.id)
            self._namespace_being_created_states[namespace_pb.meta.id] = state
            if dns_record_req.zone:
                # assume that name server ID matches its zone, since we can't get name_server_pb from cache on start
                fqdn = dns_record_req.zone + "." + dns_record_req.name_server.id
                dns_record_id = fqdn
                fqdn_info = dns_record_fqdn_info_tuple('namespace', namespace_pb.meta.id, dns_record_id)
                self._dns_record_fqdns[dns_record_req.name_server.id][dns_record_req.zone] = fqdn_info
                self._fill_dns_records_wildcards_coverage_indexes(fqdn, namespace_pb.meta.id)
        elif namespace_pb.meta.id in self._namespace_being_created_states:
            del self._namespace_being_created_states[namespace_pb.meta.id]
            fqdn = dns_record_req.zone + "." + dns_record_req.name_server.id
            self._clean_dns_records_wildcards_coverage_indexes(fqdn, namespace_pb.meta.id)
        if namespace_pb.alerting_sync_status.last_attempt.HasField('finished_at'):
            last_attempt = namespace_pb.alerting_sync_status.last_attempt.finished_at.ToDatetime()
            if not namespace_pb.alerting_sync_status.last_successful_attempt.HasField('finished_at'):
                self._namespaces_with_stuck_alerting_states[namespace_pb.meta.id] = AlertingState(
                    last_sync_attempt_time=last_attempt,
                    last_successful_sync_attempt_time=None,
                )
            elif namespace_pb.alerting_sync_status.last_attempt.succeeded.status == 'True':
                self._namespaces_with_stuck_alerting_states.pop(namespace_pb.meta.id, None)
            else:
                finished_at = namespace_pb.alerting_sync_status.last_successful_attempt.finished_at
                last_successful_attempt = finished_at.ToDatetime()
                if last_attempt - last_successful_attempt > self.STUCK_ALERTING_THRESHOLD:
                    self._namespaces_with_stuck_alerting_states[namespace_pb.meta.id] = AlertingState(
                        last_sync_attempt_time=last_attempt,
                        last_successful_sync_attempt_time=last_successful_attempt,
                    )
                else:
                    self._namespaces_with_stuck_alerting_states.pop(namespace_pb.meta.id, None)

        else:
            self._namespaces_with_stuck_alerting_states.pop(namespace_pb.meta.id, None)

        event = events.NamespaceUpdate(path=path, pb=namespace_pb)
        self._run_callbacks(event)

    def _del_namespace_pb(self, path):
        namespace_id = None
        if path in self._namespaces:
            namespace_id = self._namespaces[path].meta.id
            namespace_pb = self._namespaces[path]

            for paths in six.itervalues(self._namespace_paths_by_layout_type):
                paths.discard(path)

            normalised_prj = self.normalise_prj_tag(namespace_pb.spec.balancer_constraints.instance_tags.prj)
            if normalised_prj in self._namespace_id_by_normalised_prj:
                self._namespace_id_by_normalised_prj.pop(normalised_prj)
            self._namespaces_with_enabled_its.discard(namespace_pb.meta.id)

            dns_record_req = namespace_pb.order.content.dns_record_request.default
            if is_order_cancelled(namespace_pb):
                if dns_record_req.zone:
                    # assume that name server ID matches its zone, since we can't get name_server_pb from cache on start
                    fqdn_info = self._dns_record_fqdns[dns_record_req.name_server.id].get(dns_record_req.zone)
                    if fqdn_info and fqdn_info.entity_type == 'namespace':
                        self._dns_record_fqdns[dns_record_req.name_server.id].pop(dns_record_req.zone, None)
            fqdn = dns_record_req.zone + "." + dns_record_req.name_server.id
            self._clean_dns_records_wildcards_coverage_indexes(fqdn, namespace_id)
            if namespace_id in self._namespace_being_created_states:
                del self._namespace_being_created_states[namespace_id]
            del self._namespaces[path]
        event = events.NamespaceRemove(path=path)
        self._run_callbacks(event)

    def _set_namespace_normalised_name(self, path):
        norm_namespace_name = self.normalise_namespace_name(path)
        self._namespace_normalised_names.add(norm_namespace_name)

    def _del_namespace_normalised_name(self, path):
        norm_namespace_name = self.normalise_namespace_name(path)
        if norm_namespace_name in self._namespace_normalised_names:
            self._namespace_normalised_names.remove(norm_namespace_name)

    def _set_namespace_aspects_set_pb(self, path, namespace_aspects_set_pb):
        """
        :type namespace_aspects_set_pb: model_pb2.NamespaceAspectsSet
        """
        if self._must_be_ignored(path, namespace_aspects_set_pb):
            return

        self._namespace_aspects_sets[path] = namespace_aspects_set_pb
        event = events.NamespaceAspectsSetUpdate(path=path, pb=namespace_aspects_set_pb)
        self._run_callbacks(event)

    def _del_namespace_aspects_set_pb(self, path):
        if path in self._namespace_aspects_sets:
            del self._namespace_aspects_sets[path]
        event = events.NamespaceAspectsSetRemove(path=path)
        self._run_callbacks(event)

    def _set_component_pb(self, path, component_pb):
        """
        :type component_pb: model_pb2.Component
        """
        if path in self._components and self._components[path].meta.generation >= component_pb.meta.generation:
            return
        self._components[path] = component_pb
        self._component_ctimes[path] = component_pb.status.drafted.at.ToDatetime()
        self._component_statuses[path] = component_pb.status.status
        for cluster in self._component_default_versions[component_pb.meta.type]:
            if cluster not in component_pb.status.published.marked_as_default:
                self._component_default_versions[component_pb.meta.type][cluster].discard(component_pb.meta.version)
        for cluster in component_pb.status.published.marked_as_default:
            self._component_default_versions[component_pb.meta.type][cluster].add(component_pb.meta.version)
        event = events.ComponentUpdate(path=path, pb=component_pb)
        self._run_callbacks(event)

    def _del_component_pb(self, path):
        if path in self._components:
            del self._component_ctimes[path]
            del self._component_statuses[path]
            del self._components[path]
        event = events.ComponentRemove(path=path)
        self._run_callbacks(event)

    def _set_knob_pb(self, path, knob_pb):
        """
        :type knob_pb: model_pb2.Knob
        """
        if self._must_be_ignored(path, knob_pb):
            return

        self._knobs[path] = knob_pb
        namespace_id = knob_pb.meta.namespace_id
        self._knob_paths_by_namespace_path[namespace_id].add(path)
        self._knob_mtimes[path] = knob_pb.meta.mtime.ToDatetime()
        event = events.KnobUpdate(path=path, pb=knob_pb)
        self._run_callbacks(event)

    def _del_knob_pb(self, path):
        if path in self._knobs:
            knob_pb = self._knobs[path]
            namespace_id = knob_pb.meta.namespace_id
            self._knob_paths_by_namespace_path[namespace_id].remove(path)
            del self._knobs[path]
        event = events.KnobRemove(path=path)
        self._run_callbacks(event)

    def _set_cert_pb(self, path, cert_pb):
        """
        :type cert_pb: model_pb2.Certificate
        """
        if self._must_be_ignored(path, cert_pb):
            return

        self._certs[path] = cert_pb
        namespace_id = cert_pb.meta.namespace_id
        self._cert_paths_by_namespace_path[namespace_id].add(path)
        self._cert_mtimes[path] = cert_pb.meta.mtime.ToDatetime()
        if not cert_pb.spec.incomplete:
            self._cert_not_afters[path] = cert_pb.spec.fields.validity.not_after.ToDatetime()
        event = events.CertUpdate(path=path, pb=cert_pb)
        self._run_callbacks(event)

    def _del_cert_pb(self, path):
        if path in self._certs:
            cert_pb = self._certs[path]
            namespace_id = cert_pb.meta.namespace_id
            self._cert_paths_by_namespace_path[namespace_id].remove(path)
            del self._certs[path]
            del self._cert_mtimes[path]
            if path in self._cert_not_afters:
                del self._cert_not_afters[path]
        event = events.CertRemove(path=path)
        self._run_callbacks(event)

    def _set_cert_renewal_pb(self, path, cert_renewal_pb):
        """
        :type cert_renewal_pb: model_pb2.CertificateRenewal
        """
        if self._must_be_ignored(path, cert_renewal_pb):
            return

        self._cert_renewals[path] = cert_renewal_pb
        namespace_id = cert_renewal_pb.meta.namespace_id
        self._cert_renewal_paths_by_namespace_path[namespace_id].add(path)
        event = events.CertRenewalUpdate(path=path, pb=cert_renewal_pb)
        self._run_callbacks(event)

    def _del_cert_renewal_pb(self, path):
        if path in self._cert_renewals:
            cert_renewal_pb = self._cert_renewals[path]
            namespace_id = cert_renewal_pb.meta.namespace_id
            self._cert_renewal_paths_by_namespace_path[namespace_id].remove(path)
            del self._cert_renewals[path]
        event = events.CertRenewalRemove(path=path)
        self._run_callbacks(event)

    def _set_balancer_pb(self, path, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        """
        if self._must_be_ignored(path, balancer_pb):
            return

        self._balancers[path] = balancer_pb
        namespace_id = balancer_pb.meta.namespace_id
        self._balancer_paths_by_namespace_path[namespace_id].add(path)
        self._balancer_paths_by_balancer_id[balancer_pb.meta.id].add(path)
        self._balancer_paths_by_canonical_balancer_id[balancer_pb.meta.id.replace('_', '-')].add(path)
        if balancer_pb.spec.config_transport.type == model_pb2.NANNY_STATIC_FILE:
            service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
            self._balancer_paths_by_nanny_service_id[service_id].add(path)

            endpoint_set_id = util.make_system_endpoint_set_id(service_id)
            cluster = balancer_pb.meta.location.yp_cluster.lower()
            key = cluster, endpoint_set_id
            self._full_balancer_id_by_yp_endpoint_set_full_id[key] = (namespace_id, balancer_pb.meta.id)

        components_pb = balancer_pb.spec.components
        for component_config, component_pb in components.iter_balancer_components(components_pb):
            component_version = component_pb.version or 'none'
            component_type = component_config.type
            if not balancer_pb.spec.incomplete:
                self._balancer_paths_by_component_version[component_type][component_version].add(path)
            for version in self._balancer_paths_by_component_version[component_type]:
                if version != component_version:
                    self._balancer_paths_by_component_version[component_type][version].discard(path)

        config_mode = balancer_pb.spec.yandex_balancer.mode
        if config_mode != model_pb2.YandexBalancerSpec.EASY_MODE:
            l7_macro_version = 'none'
        else:
            l7_macro_version = balancer_pb.spec.yandex_balancer.config.l7_macro.version
        self._balancer_paths_by_l7_macro_version[l7_macro_version].add(path)
        for version in self._balancer_paths_by_l7_macro_version:
            if version != l7_macro_version:
                self._balancer_paths_by_l7_macro_version[version].discard(path)

        event = events.BalancerUpdate(path=path, pb=balancer_pb)
        self._run_callbacks(event)

    def _del_balancer_pb(self, path):
        namespace_id = None
        if path in self._balancers:
            balancer_pb = self._balancers[path]
            namespace_id = balancer_pb.meta.namespace_id
            self._balancer_paths_by_namespace_path[namespace_id].remove(path)
            self._balancer_paths_by_balancer_id[balancer_pb.meta.id].remove(path)
            self._balancer_paths_by_canonical_balancer_id[balancer_pb.meta.id.replace('_', '-')].remove(path)
            if balancer_pb.spec.config_transport.type == model_pb2.NANNY_STATIC_FILE:
                service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
                self._balancer_paths_by_nanny_service_id[service_id].remove(path)

                endpoint_set_id = util.make_system_endpoint_set_id(service_id)
                cluster = balancer_pb.meta.location.yp_cluster.lower()
                self._full_balancer_id_by_yp_endpoint_set_full_id.pop((cluster, endpoint_set_id), None)
            del self._balancers[path]
            for component_versions in six.itervalues(self._balancer_paths_by_component_version):
                for version in component_versions:
                    component_versions[version].discard(path)
            for version in self._balancer_paths_by_l7_macro_version:
                self._balancer_paths_by_l7_macro_version[version].discard(path)

        event = events.BalancerRemove(path=path, namespace_id=namespace_id)
        self._run_callbacks(event)

    def _set_l3_balancer_pb(self, path, l3_balancer_pb):
        """
        :type l3_balancer_pb: model_pb2.L3Balancer
        """
        if self._must_be_ignored(path, l3_balancer_pb):
            return

        self._l3_balancers[path] = l3_balancer_pb
        full_l3_balancer_id = (l3_balancer_pb.meta.namespace_id, l3_balancer_pb.meta.id)
        if is_order_in_progress(l3_balancer_pb):
            if full_l3_balancer_id in self._l3_balancer_being_created_states:
                known_since = self._l3_balancer_being_created_states[full_l3_balancer_id].known_since
            else:
                known_since = datetime.utcnow()
            state = L3State(known_since=known_since,
                            ctime=l3_balancer_pb.meta.ctime.ToDatetime(),
                            state=l3_balancer_pb.order.progress.state.id)
            self._l3_balancer_being_created_states[full_l3_balancer_id] = state
        elif full_l3_balancer_id in self._l3_balancer_being_created_states:
            del self._l3_balancer_being_created_states[full_l3_balancer_id]
        namespace_id = l3_balancer_pb.meta.namespace_id
        self._l3_balancer_paths_by_namespace_path[namespace_id].add(path)
        event = events.L3BalancerUpdate(path=path, pb=l3_balancer_pb)
        self._run_callbacks(event)

    def _del_l3_balancer_pb(self, path):
        namespace_id = None
        if path in self._l3_balancers:
            l3_balancer_pb = self._l3_balancers[path]
            namespace_id = l3_balancer_pb.meta.namespace_id
            self._l3_balancer_paths_by_namespace_path[namespace_id].remove(path)
            del self._l3_balancers[path]
            self._l3_balancer_being_created_states.pop((namespace_id, l3_balancer_pb.meta.id), None)
        event = events.L3BalancerRemove(path=path)
        self._run_callbacks(event)

    @classmethod
    def _list_full_cert_ids(cls, entity_state_pb):
        """
        :type entity_state_pb: Union[model_pb2.BalancerState]
        """
        rv = set()
        for cert_id, statuses in six.iteritems(entity_state_pb.certificates):
            if isinstance(entity_state_pb, model_pb2.BalancerState):
                status_pbs = statuses.statuses
            else:
                raise RuntimeError('Unexpected "entity_state_pb" type: {!r}'.format(type(entity_state_pb)))
            if status_pbs:
                full_cert_id = to_full_id(entity_state_pb.namespace_id, cert_id)
                rv.add(full_cert_id)
        return rv

    @classmethod
    def _list_full_domain_ids(cls, entity_state_pb):
        """
        :type entity_state_pb: Union[model_pb2.BalancerState]
        """
        rv = set()
        for domain_id, statuses in six.iteritems(entity_state_pb.domains):
            if isinstance(entity_state_pb, model_pb2.BalancerState):
                status_pbs = statuses.statuses
            else:
                raise RuntimeError('Unexpected "entity_state_pb" type: {!r}'.format(type(entity_state_pb)))
            if status_pbs:
                full_domain_id = to_full_id(entity_state_pb.namespace_id, domain_id)
                rv.add(full_domain_id)
        return rv

    @classmethod
    def _list_full_backend_ids(cls, entity_state_pb):
        """
        :type entity_state_pb: Union[model_pb2.BalancerState, model_pb2.L3BalancerState | model_pb2.DnsRecordState]
        """
        rv = set()
        for backend_id, statuses in six.iteritems(entity_state_pb.backends):
            if isinstance(entity_state_pb, model_pb2.BalancerState):
                status_pbs = statuses.statuses
            elif isinstance(entity_state_pb, model_pb2.L3BalancerState):
                status_pbs = statuses.l3_statuses
            elif isinstance(entity_state_pb, model_pb2.DnsRecordState):
                status_pbs = statuses.statuses
            else:
                raise RuntimeError('Unexpected "entity_state_pb" type: {!r}'.format(type(entity_state_pb)))
            if status_pbs:
                full_backend_id = to_full_id(entity_state_pb.namespace_id, backend_id)
                rv.add(full_backend_id)
        return rv

    def _update_backends_indexes_on_state_update(self, full_entity_id, entity_state_pb,
                                                 backends_to_entities_idx, entities_to_backends_idx):
        """
        :type full_entity_id: (six.text_type, six.text_type)
        :type (entity_state_pb: model_pb2.BalancerState | model_pb2.L3BalancerState | model_pb2.DnsRecordState)
        :type backends_to_entities_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :type entities_to_backends_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :return:
        """
        included_full_backend_ids = self._list_full_backend_ids(entity_state_pb)
        for included_full_backend_id in included_full_backend_ids:
            backends_to_entities_idx[included_full_backend_id].add(full_entity_id)
        current_included_full_backend_ids = entities_to_backends_idx[full_entity_id]
        removed_full_backend_ids = current_included_full_backend_ids - included_full_backend_ids
        for removed_full_backend_id in removed_full_backend_ids:
            backends_to_entities_idx[removed_full_backend_id].discard(full_entity_id)
        entities_to_backends_idx[full_entity_id] = included_full_backend_ids

    @staticmethod
    def _update_backends_indexes_on_state_removal(full_entity_id, backends_to_entities_idx, entities_to_backends_idx):
        """
        :type full_entity_id: (six.text_type, six.text_type)
        :type backends_to_entities_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :type entities_to_backends_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :return:
        """
        # update indexes:
        included_full_backend_ids = entities_to_backends_idx[full_entity_id]
        del entities_to_backends_idx[full_entity_id]
        for full_backend_id in included_full_backend_ids:
            backends_to_entities_idx[full_backend_id].discard(full_entity_id)

    def _update_certs_indexes_on_state_update(self, full_entity_id, entity_state_pb,
                                              certs_to_entities_idx, entities_to_certs_idx):
        """
        :type full_entity_id: (six.text_type, six.text_type)
        :type entity_state_pb: model_pb2.BalancerState
        :type certs_to_entities_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :type entities_to_certs_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :return:
        """
        included_full_cert_ids = self._list_full_cert_ids(entity_state_pb)
        for included_full_cert_id in included_full_cert_ids:
            certs_to_entities_idx[included_full_cert_id].add(full_entity_id)
        current_included_full_cert_ids = entities_to_certs_idx[full_entity_id]
        removed_full_cert_ids = current_included_full_cert_ids - included_full_cert_ids
        for removed_full_cert_id in removed_full_cert_ids:
            certs_to_entities_idx[removed_full_cert_id].discard(full_entity_id)
        entities_to_certs_idx[full_entity_id] = included_full_cert_ids

    @staticmethod
    def _update_certs_indexes_on_state_removal(full_entity_id, certs_to_entities_idx, entities_to_certs_idx):
        """
        :type full_entity_id: (six.text_type, six.text_type)
        :type certs_to_entities_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :type entities_to_certs_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :return:
        """
        # update indexes:
        included_full_cert_ids = entities_to_certs_idx[full_entity_id]
        del entities_to_certs_idx[full_entity_id]
        for full_cert_id in included_full_cert_ids:
            certs_to_entities_idx[full_cert_id].discard(full_entity_id)

    def _update_domains_indexes_on_state_update(self, full_entity_id, entity_state_pb,
                                                domains_to_entities_idx, entities_to_domains_idx):
        """
        :type full_entity_id: (six.text_type, six.text_type)
        :type entity_state_pb: model_pb2.BalancerState
        :type domains_to_entities_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :type entities_to_domains_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :return:
        """
        included_full_domain_ids = self._list_full_domain_ids(entity_state_pb)
        for included_full_domain_id in included_full_domain_ids:
            domains_to_entities_idx[included_full_domain_id].add(full_entity_id)
        current_included_full_domain_ids = entities_to_domains_idx[full_entity_id]
        removed_full_domain_ids = current_included_full_domain_ids - included_full_domain_ids
        for removed_full_domain_id in removed_full_domain_ids:
            domains_to_entities_idx[removed_full_domain_id].discard(full_entity_id)
        entities_to_domains_idx[full_entity_id] = included_full_domain_ids

    @staticmethod
    def _update_domains_indexes_on_state_removal(full_entity_id, domains_to_entities_idx, entities_to_domains_idx):
        """
        :type full_entity_id: (six.text_type, six.text_type)
        :type domains_to_entities_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :type entities_to_domains_idx: defaultdict[(six.text_type, six.text_type), (six.text_type, six.text_type)]
        :return:
        """
        included_full_domain_ids = entities_to_domains_idx[full_entity_id]
        del entities_to_domains_idx[full_entity_id]
        for full_domain_id in included_full_domain_ids:
            domains_to_entities_idx[full_domain_id].discard(full_entity_id)

    def _set_balancer_state_pb(self, path, balancer_state_pb):
        """
        :type balancer_state_pb: model_pb2.BalancerState
        """
        if self._must_be_ignored(path, balancer_state_pb, meta_field=None):
            return

        self._balancer_states[path] = balancer_state_pb
        namespace_id = balancer_state_pb.namespace_id
        self._balancer_state_paths_by_namespace_path[namespace_id].add(path)

        full_balancer_id = (balancer_state_pb.namespace_id, balancer_state_pb.balancer_id)
        self._update_backends_indexes_on_state_update(
            full_balancer_id,
            balancer_state_pb,
            self._full_backend_ids_to_full_balancer_id_sets,
            self._full_balancer_ids_to_full_backend_id_sets)
        self._update_certs_indexes_on_state_update(
            full_balancer_id,
            balancer_state_pb,
            self._full_cert_ids_to_full_balancer_id_sets,
            self._full_balancer_ids_to_full_cert_id_sets)
        self._update_domains_indexes_on_state_update(
            full_balancer_id,
            balancer_state_pb,
            self._full_domain_ids_to_full_balancer_id_sets,
            self._full_balancer_ids_to_full_domain_id_sets)

        in_progress_configs = set()
        now = None
        for rev_status_pb in balancer_state_pb.balancer.statuses:
            if rev_status_pb.in_progress.status != 'False':
                if now is None:
                    # the function called often, let's call .utcnow() only when needed
                    now = datetime.utcnow()
                snapshot_pbs = rev_status_pb.in_progress.meta.nanny_static_file.snapshots
                for snapshot_pb in snapshot_pbs:
                    in_progress_configs.add(L7Config(known_since=now,
                                                     ctime=snapshot_pb.ctime.ToDatetime(),
                                                     service_id=snapshot_pb.service_id,
                                                     snapshot_id=snapshot_pb.snapshot_id))
        if in_progress_configs:
            self._balancer_to_in_progress_configs[full_balancer_id] = in_progress_configs
        else:
            if full_balancer_id in self._balancer_to_in_progress_configs:
                del self._balancer_to_in_progress_configs[full_balancer_id]

        event = events.BalancerStateUpdate(path=path, pb=balancer_state_pb)
        self._run_callbacks(event)

    def _del_balancer_state_pb(self, path):
        namespace_id = None
        if path in self._balancer_states:
            balancer_state_pb = self._balancer_states[path]
            namespace_id = balancer_state_pb.namespace_id
            balancer_id = balancer_state_pb.balancer_id
            self._balancer_state_paths_by_namespace_path[namespace_id].remove(path)
            del self._balancer_states[path]
            full_balancer_id = (namespace_id, balancer_id)
            self._update_backends_indexes_on_state_removal(
                full_balancer_id,
                self._full_backend_ids_to_full_balancer_id_sets,
                self._full_balancer_ids_to_full_backend_id_sets)
            self._update_certs_indexes_on_state_removal(
                full_balancer_id,
                self._full_cert_ids_to_full_balancer_id_sets,
                self._full_balancer_ids_to_full_cert_id_sets)
            self._update_domains_indexes_on_state_removal(
                full_balancer_id,
                self._full_domain_ids_to_full_balancer_id_sets,
                self._full_balancer_ids_to_full_domain_id_sets)
            if full_balancer_id in self._balancer_to_in_progress_configs:
                del self._balancer_to_in_progress_configs[full_balancer_id]
        event = events.BalancerStateRemove(path=path, namespace_id=namespace_id)
        self._run_callbacks(event)

    def _set_balancer_aspects_set_pb(self, path, balancer_aspects_set_pb):
        """
        :type balancer_aspects_set_pb: model_pb2.BalancerAspectsSet
        """
        if self._must_be_ignored(path, balancer_aspects_set_pb):
            return

        self._balancer_aspects_sets[path] = balancer_aspects_set_pb
        event = events.BalancerAspectsSetUpdate(path=path, pb=balancer_aspects_set_pb)
        self._run_callbacks(event)

    def _del_balancer_aspects_set_pb(self, path):
        if path in self._balancer_aspects_sets:
            del self._balancer_aspects_sets[path]
        event = events.BalancerAspectsSetRemove(path=path)
        self._run_callbacks(event)

    def _set_l3_balancer_state_pb(self, path, l3_balancer_state_pb):
        """
        :type l3_balancer_state_pb: model_pb2.L3BalancerState
        """
        if self._must_be_ignored(path, l3_balancer_state_pb, meta_field=None):
            return

        self._l3_balancer_states[path] = l3_balancer_state_pb
        namespace_id = l3_balancer_state_pb.namespace_id
        self._l3_balancer_state_paths_by_namespace_path[namespace_id].add(path)

        full_l3_balancer_id = (l3_balancer_state_pb.namespace_id, l3_balancer_state_pb.l3_balancer_id)
        self._update_backends_indexes_on_state_update(
            full_l3_balancer_id,
            l3_balancer_state_pb,
            self._full_backend_ids_to_full_l3_balancer_id_sets,
            self._full_l3_balancer_ids_to_full_backend_id_sets)

        in_progress_configs = set()
        now = None
        for rev_l3_status_pb in l3_balancer_state_pb.l3_balancer.l3_statuses:
            if rev_l3_status_pb.in_progress.status != 'False':
                if now is None:
                    # the function called often, let's call .utcnow() only when needed
                    now = datetime.utcnow()
                config_pbs = rev_l3_status_pb.in_progress.meta.l3mgr.configs
                for config_pb in config_pbs:
                    in_progress_configs.add(L3Config(known_since=now,
                                                     ctime=config_pb.ctime.ToDatetime(),
                                                     service_id=config_pb.service_id,
                                                     config_id=config_pb.config_id))
        if in_progress_configs:
            self._l3_balancer_to_in_progress_configs[full_l3_balancer_id] = in_progress_configs
        else:
            if full_l3_balancer_id in self._l3_balancer_to_in_progress_configs:
                del self._l3_balancer_to_in_progress_configs[full_l3_balancer_id]

        event = events.L3BalancerStateUpdate(path=path, pb=l3_balancer_state_pb)
        self._run_callbacks(event)

    def _del_l3_balancer_state_pb(self, path):
        namespace_id = None
        if path in self._l3_balancer_states:
            l3_balancer_state_pb = self._l3_balancer_states[path]
            namespace_id = l3_balancer_state_pb.namespace_id
            l3_balancer_id = l3_balancer_state_pb.l3_balancer_id
            self._l3_balancer_state_paths_by_namespace_path[namespace_id].remove(path)
            del self._l3_balancer_states[path]

            # update indexes:
            full_l3_balancer_id = (namespace_id, l3_balancer_id)
            self._update_backends_indexes_on_state_removal(
                full_l3_balancer_id,
                self._full_backend_ids_to_full_l3_balancer_id_sets,
                self._full_l3_balancer_ids_to_full_backend_id_sets)

            if full_l3_balancer_id in self._l3_balancer_to_in_progress_configs:
                del self._l3_balancer_to_in_progress_configs[full_l3_balancer_id]

        event = events.L3BalancerStateRemove(path=path)
        self._run_callbacks(event)

    def _set_upstream_pb(self, path, upstream_pb):
        """
        :type upstream_pb: model_pb2.Upstream
        """
        if self._must_be_ignored(path, upstream_pb):
            return

        self._upstreams[path] = upstream_pb
        self._upstream_order_labels[path] = upstream_pb.spec.labels.get('order', '')
        self._upstream_mtimes[path] = upstream_pb.meta.mtime.ToDatetime()
        if upstream_pb.spec.deleted:
            self._deleted_upstream_paths.add(path)
        else:
            self._deleted_upstream_paths.discard(path)

        namespace_id = upstream_pb.meta.namespace_id
        existing_upstreams = self._upstream_paths_by_namespace_path[namespace_id]
        if path not in existing_upstreams:
            self._upstream_paths_by_namespace_path[namespace_id].add(path)
            self._namespace_objects_count[namespace_id]['upstreams'] += 1
        for v in six.itervalues(self._upstream_paths_by_type):
            v.discard(path)
        self._upstream_paths_by_type[upstream_pb.spec.yandex_balancer.type].add(path)

        if upstream_pb.spec.yandex_balancer.mode == upstream_pb.spec.yandex_balancer.EASY_MODE2:
            full_upstream_id = (namespace_id, upstream_pb.meta.id)
            backend_ids = set()
            if upstream_pb.spec.yandex_balancer.config.l7_upstream_macro.HasField('flat_scheme'):
                backend_ids.update(upstream_pb.spec.yandex_balancer.config.l7_upstream_macro.flat_scheme.backend_ids)
            elif upstream_pb.spec.yandex_balancer.config.l7_upstream_macro.HasField('by_dc_scheme'):
                for dc_pb in upstream_pb.spec.yandex_balancer.config.l7_upstream_macro.by_dc_scheme.dcs:
                    backend_ids.update(dc_pb.backend_ids)

            for full_backend_id in self._backend_ids_by_easy_mode_upstream_id[full_upstream_id]:
                self._easy_mode_upstream_ids_by_backend_id[full_backend_id].discard(full_upstream_id)
            self._backend_ids_by_easy_mode_upstream_id[full_upstream_id].clear()

            for backend_id in backend_ids:
                full_backend_id = (namespace_id, backend_id)
                self._backend_ids_by_easy_mode_upstream_id[full_upstream_id].add(full_backend_id)
                self._easy_mode_upstream_ids_by_backend_id[full_backend_id].add(full_upstream_id)

        event = events.UpstreamUpdate(path=path, pb=upstream_pb)
        self._run_callbacks(event)

    def _del_upstream_pb(self, path):
        if path in self._upstreams:
            upstream_pb = self._upstreams[path]
            del self._upstream_order_labels[path]
            del self._upstream_mtimes[path]
            del self._upstreams[path]
            self._deleted_upstream_paths.discard(path)
            for v in six.itervalues(self._upstream_paths_by_type):
                v.discard(path)

            namespace_id = upstream_pb.meta.namespace_id
            existing_upstreams = self._upstream_paths_by_namespace_path[namespace_id]
            if path in existing_upstreams:
                self._upstream_paths_by_namespace_path[namespace_id].remove(path)
                self._namespace_objects_count[namespace_id]['upstreams'] -= 1

            if upstream_pb.spec.yandex_balancer.mode == upstream_pb.spec.yandex_balancer.EASY_MODE2:
                full_upstream_id = (namespace_id, upstream_pb.meta.id)
                for full_backend_id in self._backend_ids_by_easy_mode_upstream_id[full_upstream_id]:
                    self._easy_mode_upstream_ids_by_backend_id[full_backend_id].discard(full_upstream_id)
                self._backend_ids_by_easy_mode_upstream_id.pop(full_upstream_id)

        event = events.UpstreamRemove(path=path)
        self._run_callbacks(event)

    def _add_fqdn_to_cache(self, fqdn, fqdn_info):
        last_level_part, higher_level_part = fqdn.split('.', 1)
        if last_level_part not in self._domain_fqdns[higher_level_part]:
            self._domain_fqdns[higher_level_part][last_level_part] = set()
        self._domain_fqdns[higher_level_part][last_level_part].add(fqdn_info)

    def _remove_fqdn_from_cache(self, fqdn, fqdn_info):
        """
        :type fqdn: six.text_type
        :type fqdn_info: fqdn_info_tuple
        :rtype: FqdnRemovalResult
        """
        last_level_part, higher_level_part = fqdn.split('.', 1)
        result = FqdnRemovalResult(last_level_part=last_level_part,
                                   higher_level_part=higher_level_part,
                                   removed=False,
                                   message=u'')
        last_level_parts = self._domain_fqdns[higher_level_part]
        if last_level_part not in last_level_parts:
            result.message = u'last_level_part not in last_level_parts: {}'.format(last_level_parts)
            return result
        fqdn_infos = last_level_parts[last_level_part]
        if fqdn_info not in fqdn_infos:
            result.message = u'fqdn_info "{}" not in fqdn_infos: {}'.format(fqdn_info, fqdn_infos)
            return result
        fqdn_infos.remove(fqdn_info)
        if len(fqdn_infos) == 0:
            last_level_parts.pop(last_level_part)
        if len(last_level_parts) == 0:
            self._domain_fqdns.pop(higher_level_part)
        result.removed = True
        result.message = fqdn_info
        return result

    def _set_domain_pb(self, path, domain_pb):
        """
        :type domain_pb: model_pb2.Domain
        """
        if self._must_be_ignored(path, domain_pb):
            return

        namespace_id = domain_pb.meta.namespace_id
        domain_id = domain_pb.meta.id
        self._domains[path] = domain_pb
        self._domain_mtimes[path] = domain_pb.meta.mtime.ToDatetime()
        existing_domains = self._domain_paths_by_namespace_path[namespace_id]
        if path not in existing_domains:
            self._domain_paths_by_namespace_path[namespace_id].add(path)
            self._namespace_objects_count[namespace_id]['domains'] += 1
        if domain_pb.spec.incomplete:
            fqdns = domain_pb.order.content.fqdns
        else:
            fqdns = domain_pb.spec.yandex_balancer.config.fqdns
        fqdn_info = fqdn_info_tuple('domain', namespace_id, domain_id)
        for fqdn in fqdns:
            self._add_fqdn_to_cache(fqdn, fqdn_info)
        event = events.DomainUpdate(path=path, pb=domain_pb)
        self._run_callbacks(event)

    def _del_domain_pb(self, path):
        domain_pb = self._domains.get(path)
        was_present = spec_is_complete = order_is_cancelled = False
        removal_results = []
        fqdns = []
        if domain_pb:
            was_present = True
            del self._domain_mtimes[path]
            del self._domains[path]
            namespace_id = domain_pb.meta.namespace_id
            existing_domains = self._domain_paths_by_namespace_path[namespace_id]
            if path in existing_domains:
                self._domain_paths_by_namespace_path[namespace_id].remove(path)
                self._namespace_objects_count[namespace_id]['domains'] -= 1

            if domain_pb.spec.incomplete:
                fqdns = domain_pb.order.content.fqdns
            elif domain_pb.order.cancelled.value:
                order_is_cancelled = True
                fqdns = domain_pb.order.content.fqdns
            else:
                spec_is_complete = True
                fqdns = domain_pb.spec.yandex_balancer.config.fqdns
            fqdn_info = fqdn_info_tuple('domain', namespace_id, domain_pb.meta.id)
            for fqdn in fqdns:
                removal_results.append(self._remove_fqdn_from_cache(fqdn, fqdn_info))

        event = events.DomainRemove(path=path)
        self._run_callbacks(event)

        log = get_op_log(self._log, op_id=None)
        if was_present:
            log.debug(u'domain pb removed from cache: %s', path)
            if spec_is_complete:
                log.debug(u'domain spec is complete, getting fqdns from spec')
            elif order_is_cancelled:
                log.debug(u'domain order was cancelled, getting fqdns from order')
            else:
                log.debug(u'domain spec is incomplete, getting fqdns from order')
        else:
            log.debug(u'domain pb was already removed from cache: %s', path)
        log.debug(u'fqdns to process: %s', u', '.join(fqdns))
        for result in removal_results:
            msg = u'result for fqdn "{} . {}": '.format(result.last_level_part, result.higher_level_part)
            if result.removed:
                msg += u'removed record {}'.format(result.message)
            else:
                msg += u'not removed, reason: {}'.format(result.message)
            log.debug(msg)

    def _set_domain_operation_pb(self, path, domain_operation_pb):
        """
        :type domain_operation_pb: model_pb2.DomainOperation
        """
        if self._must_be_ignored(path, domain_operation_pb):
            return

        namespace_id = domain_operation_pb.meta.namespace_id
        self._domain_operations[path] = domain_operation_pb
        self._domain_operation_paths_by_namespace_path[namespace_id].add(path)
        if domain_operation_pb.order.content.HasField('set_fqdns'):
            fqdn_info = fqdn_info_tuple('domain op', namespace_id, domain_operation_pb.meta.id)
            for fqdn in domain_operation_pb.order.content.set_fqdns.fqdns:
                self._add_fqdn_to_cache(fqdn, fqdn_info)
        event = events.DomainOperationUpdate(path=path, pb=domain_operation_pb)
        self._run_callbacks(event)

    def _del_domain_operation_pb(self, path):
        domain_op_pb = self._domain_operations.get(path)
        was_present = False
        removal_results = []
        clean_up_fqdns = domain_op_pb.order.content.HasField('set_fqdns')
        old_fqdns = []
        new_fqdns = []
        if domain_op_pb:
            was_present = True
            ns_id = domain_op_pb.meta.namespace_id
            domain_id = domain_op_pb.meta.id
            self._domain_operation_paths_by_namespace_path[ns_id].remove(path)
            del self._domain_operations[path]

            if clean_up_fqdns:
                old_fqdns = domain_op_pb.spec.old_fqdns
                new_fqdns = domain_op_pb.order.content.set_fqdns.fqdns
                # "set_fqdns" operation changes domain fqdns, so we need to perform 3 actions:
                # 1. Remove "domain" FQDNs from cache that are no longer present in this domain
                fqdn_info = fqdn_info_tuple('domain', ns_id, domain_id)
                for fqdn in old_fqdns:
                    removal_results.append(self._remove_fqdn_from_cache(fqdn, fqdn_info))
                op_fqdn_info = fqdn_info_tuple('domain op', ns_id, domain_id)
                for fqdn in new_fqdns:
                    # 2. Remove "domain op" records for FQDNs that didn't change during the operation
                    removal_results.append(self._remove_fqdn_from_cache(fqdn, op_fqdn_info))
                    # 3. Replace "domain op" records with "domain" records for new FQDNs
                    if fqdn not in old_fqdns:
                        self._add_fqdn_to_cache(fqdn, fqdn_info)

        event = events.DomainOperationRemove(path=path)
        self._run_callbacks(event)

        if not clean_up_fqdns:
            return
        log = get_op_log(self._log, op_id=None)
        if was_present:
            log.debug(u'domain op pb removed from cache: %s', path)
        else:
            log.debug(u'domain op pb was already removed from cache: %s', path)
        log.debug(u'old fqdns to process: %s', u', '.join(old_fqdns))
        log.debug(u'new fqdns to process: %s', u', '.join(new_fqdns))
        for result in removal_results:
            msg = u'result for fqdn "{} . {}": '.format(result.last_level_part, result.higher_level_part)
            if result.removed:
                msg += u'removed record {}'.format(result.message)
            else:
                msg += u'not removed, reason: {}'.format(result.message)
            log.debug(msg)

    def _set_balancer_operation_pb(self, path, balancer_operation_pb):
        """
        :type balancer_operation_pb: model_pb2.BalancerOperation
        """
        existing_pb = self._balancer_operations.get(path)
        if existing_pb and existing_pb.meta.generation >= balancer_operation_pb.meta.generation:
            return
        namespace_id = balancer_operation_pb.meta.namespace_id
        self._balancer_operations[path] = balancer_operation_pb
        self._balancer_operation_paths_by_namespace_path[namespace_id].add(path)
        event = events.BalancerOperationUpdate(path=path, pb=balancer_operation_pb)
        self._run_callbacks(event)

    def _del_balancer_operation_pb(self, path):
        balancer_operation_pb = self._balancer_operations.get(path)
        if balancer_operation_pb:
            self._balancer_operation_paths_by_namespace_path[balancer_operation_pb.meta.namespace_id].remove(path)
            del self._balancer_operations[path]
        event = events.BalancerOperationRemove(path=path)
        self._run_callbacks(event)

    def _clean_backend_selector_indexes(self, namespace_id, path, selector_pb):
        if selector_pb.type == selector_pb.NANNY_SNAPSHOTS:
            for item_pb in selector_pb.nanny_snapshots:
                self._backend_paths_by_nanny_service_id[item_pb.service_id].discard(path)
        elif selector_pb.type == selector_pb.GENCFG_GROUPS:
            for item_pb in selector_pb.gencfg_groups:
                self._backend_paths_by_gencfg_group_name[item_pb.name].discard(path)
        elif selector_pb.type in (selector_pb.YP_ENDPOINT_SETS, selector_pb.YP_ENDPOINT_SETS_SD):
            for item_pb in selector_pb.yp_endpoint_sets:
                self._backend_paths_by_yp_endpoint_set_full_id[item_pb.cluster, item_pb.endpoint_set_id].discard(path)
        elif selector_pb.type == selector_pb.BALANCERS:
            for item_pb in selector_pb.balancers:
                self._backend_paths_by_full_balancer_id[namespace_id, item_pb.id].discard(path)

    def _set_backend_pb(self, path, backend_pb):
        """
        :type backend_pb: model_pb2.Backend
        """
        if self._must_be_ignored(path, backend_pb):
            return

        if path in self._backends:
            self._clean_backend_selector_indexes(backend_pb.meta.namespace_id, path, self._backends[path].spec.selector)
        self._backends[path] = backend_pb
        self._backend_mtimes[path] = backend_pb.meta.mtime.ToDatetime()

        namespace_id = backend_pb.meta.namespace_id
        existing_backends = self._backend_paths_by_namespace_path[namespace_id]
        if path not in existing_backends:
            self._backend_paths_by_namespace_path[namespace_id].add(path)
            self._namespace_objects_count[namespace_id]['backends'] += 1

        self._backend_paths_by_backend_id[backend_pb.meta.id].add(path)
        if backend_pb.meta.is_system.value:
            self._backend_paths_system.add(path)
        else:
            self._backend_paths_not_system.add(path)

        selector_pb = backend_pb.spec.selector
        if selector_pb.type == selector_pb.NANNY_SNAPSHOTS:
            for item_pb in selector_pb.nanny_snapshots:
                self._backend_paths_by_nanny_service_id[item_pb.service_id].add(path)
        elif selector_pb.type == selector_pb.GENCFG_GROUPS:
            for item_pb in selector_pb.gencfg_groups:
                self._backend_paths_by_gencfg_group_name[item_pb.name].add(path)
        elif selector_pb.type in (selector_pb.YP_ENDPOINT_SETS, selector_pb.YP_ENDPOINT_SETS_SD):
            for item_pb in selector_pb.yp_endpoint_sets:
                self._backend_paths_by_yp_endpoint_set_full_id[item_pb.cluster, item_pb.endpoint_set_id].add(path)
        elif selector_pb.type == selector_pb.BALANCERS:
            for item_pb in selector_pb.balancers:
                self._backend_paths_by_full_balancer_id[namespace_id, item_pb.id].add(path)

        event = events.BackendUpdate(path=path, pb=backend_pb)
        self._run_callbacks(event)

    def _del_backend_pb(self, path):
        if path in self._backends:
            backend_pb = self._backends[path]
            del self._backends[path]
            del self._backend_mtimes[path]

            namespace_id = backend_pb.meta.namespace_id
            self._clean_backend_selector_indexes(namespace_id, path, backend_pb.spec.selector)

            existing_backends = self._backend_paths_by_namespace_path[namespace_id]
            if path in existing_backends:
                self._backend_paths_by_namespace_path[namespace_id].remove(path)
                self._namespace_objects_count[namespace_id]['backends'] -= 1

            self._backend_paths_by_backend_id[backend_pb.meta.id].discard(path)
            if backend_pb.meta.is_system.value:
                self._backend_paths_system.discard(path)
            else:
                self._backend_paths_not_system.discard(path)

        event = events.BackendRemove(path=path)
        self._run_callbacks(event)

    def _set_endpoint_set_pb(self, path, endpoint_set_pb):
        """
        :type endpoint_set_pb: model_pb2.EndpointSet
        """
        if self._must_be_ignored(path, endpoint_set_pb):
            return

        self._endpoint_sets[path] = endpoint_set_pb
        self._endpoint_set_paths_by_namespace_path[endpoint_set_pb.meta.namespace_id].add(path)

        event = events.EndpointSetUpdate(path=path, pb=endpoint_set_pb)
        self._run_callbacks(event)

    def _del_endpoint_set_pb(self, path):
        if path in self._endpoint_sets:
            endpoint_set_pb = self._endpoint_sets[path]
            self._endpoint_set_paths_by_namespace_path[endpoint_set_pb.meta.namespace_id].remove(path)
            del self._endpoint_sets[path]

        event = events.EndpointSetRemove(path=path)
        self._run_callbacks(event)

    def _handle_namespace_update(self, path, namespace_pb):
        if namespace_pb is None:
            self._del_namespace_pb(path)
            self._del_namespace_normalised_name(path)
        else:
            self._set_namespace_pb(path, namespace_pb)
            self._set_namespace_normalised_name(path)

    def _handle_namespace_aspects_set_update(self, path, namespace_aspects_set_pb):
        if namespace_aspects_set_pb is None:
            self._del_namespace_aspects_set_pb(path)
        else:
            self._set_namespace_aspects_set_pb(path, namespace_aspects_set_pb)

    def _handle_component_update(self, path, component_pb):
        if component_pb is None:
            self._del_component_pb(path)
        else:
            self._set_component_pb(path, component_pb)

    def _handle_knob_update(self, path, knob_pb):
        if knob_pb is None:
            self._del_knob_pb(path)
        else:
            self._set_knob_pb(path, knob_pb)

    def _handle_cert_update(self, path, cert_pb):
        if cert_pb is None:
            self._del_cert_pb(path)
        else:
            self._set_cert_pb(path, cert_pb)

    def _handle_cert_renewal_update(self, path, cert_renewal_pb):
        if cert_renewal_pb is None:
            self._del_cert_renewal_pb(path)
        else:
            self._set_cert_renewal_pb(path, cert_renewal_pb)

    def _handle_name_server_update(self, path, name_server_pb):
        if name_server_pb is None:
            self._del_name_server_pb(path)
        else:
            self._set_name_server_pb(path, name_server_pb)

    def _handle_dns_record_update(self, path, dns_record_pb):
        if dns_record_pb is None:
            self._del_dns_record_pb(path)
        else:
            self._set_dns_record_pb(path, dns_record_pb)

    def _handle_dns_record_op_update(self, path, dns_record_op_pb):
        if dns_record_op_pb is None:
            self._del_dns_record_op_pb(path)
        else:
            self._set_dns_record_op_pb(path, dns_record_op_pb)

    def _handle_dns_record_state_update(self, path, dns_record_state_pb):
        if dns_record_state_pb is None:
            self._del_dns_record_state_pb(path)
        else:
            self._set_dns_record_state_pb(path, dns_record_state_pb)

    def _handle_balancer_update(self, path, balancer_pb):
        if balancer_pb is None:
            self._del_balancer_pb(path)
        else:
            self._set_balancer_pb(path, balancer_pb)

    def _handle_balancer_state_update(self, path, balancer_state_pb):
        if balancer_state_pb is None:
            self._del_balancer_state_pb(path)
        else:
            self._set_balancer_state_pb(path, balancer_state_pb)

    def _handle_balancer_aspects_set_update(self, path, balancer_aspects_set_pb):
        if balancer_aspects_set_pb is None:
            self._del_balancer_aspects_set_pb(path)
        else:
            self._set_balancer_aspects_set_pb(path, balancer_aspects_set_pb)

    def _handle_l3_balancer_update(self, path, l3_balancer_pb):
        if l3_balancer_pb is None:
            self._del_l3_balancer_pb(path)
        else:
            self._set_l3_balancer_pb(path, l3_balancer_pb)

    def _handle_l3_balancer_state_update(self, path, l3_balancer_state_pb):
        if l3_balancer_state_pb is None:
            self._del_l3_balancer_state_pb(path)
        else:
            self._set_l3_balancer_state_pb(path, l3_balancer_state_pb)

    def _handle_upstream_update(self, path, upstream_pb):
        if upstream_pb is None:
            self._del_upstream_pb(path)
        else:
            self._set_upstream_pb(path, upstream_pb)

    def _handle_domain_update(self, path, domain_pb):
        if domain_pb is None:
            self._del_domain_pb(path)
        else:
            self._set_domain_pb(path, domain_pb)

    def _handle_domain_operation_update(self, path, domain_operation_pb):
        if domain_operation_pb is None:
            self._del_domain_operation_pb(path)
        else:
            self._set_domain_operation_pb(path, domain_operation_pb)

    def _handle_balancer_operation_update(self, path, balancer_operation_pb):
        if balancer_operation_pb is None:
            self._del_balancer_operation_pb(path)
        else:
            self._set_balancer_operation_pb(path, balancer_operation_pb)

    def _handle_backend_update(self, path, backend_pb):
        if backend_pb is None:
            self._del_backend_pb(path)
        else:
            self._set_backend_pb(path, backend_pb)

    def _handle_endpoint_set_update(self, path, endpoint_set_pb):
        if endpoint_set_pb is None:
            self._del_endpoint_set_pb(path)
        else:
            self._set_endpoint_set_pb(path, endpoint_set_pb)

    def _handle_party_member_update(self, party_id, member_id):
        event = events.PartyMemberUpdate(party_id, member_id)
        self._run_callbacks(event)

    def _handle_party_member_remove(self, party_id, member_id):
        event = events.PartyMemberRemove(party_id, member_id)
        self._run_callbacks(event)

    def _handle_tree_event(self, event):
        event = events.TreeEvent(event_type=event.event_type)
        self._run_callbacks(event)

    def _process_event(self, event):
        if isinstance(event, _NamespaceUpdate):
            self._handle_namespace_update(event.path, event.namespace_pb)
        elif isinstance(event, _ComponentUpdate):
            self._handle_component_update(event.path, event.component_pb)
        elif isinstance(event, _KnobUpdate):
            self._handle_knob_update(event.path, event.knob_pb)
        elif isinstance(event, _CertUpdate):
            self._handle_cert_update(event.path, event.cert_pb)
        elif isinstance(event, _CertRenewalUpdate):
            self._handle_cert_renewal_update(event.path, event.cert_renewal_pb)
        elif isinstance(event, _DnsRecordUpdate):
            self._handle_dns_record_update(event.path, event.dns_record_pb)
        elif isinstance(event, _DnsRecordOperationUpdate):
            self._handle_dns_record_op_update(event.path, event.dns_record_operation_pb)
        elif isinstance(event, _DnsRecordStateUpdate):
            self._handle_dns_record_state_update(event.path, event.dns_record_state_pb)
        elif isinstance(event, _NameServerUpdate):
            self._handle_name_server_update(event.path, event.name_server_pb)
        elif isinstance(event, _BalancerUpdate):
            self._handle_balancer_update(event.path, event.balancer_pb)
        elif isinstance(event, _BalancerOperationUpdate):
            self._handle_balancer_operation_update(event.path, event.balancer_pb)
        elif isinstance(event, _BalancerStateUpdate):
            self._handle_balancer_state_update(event.path, event.balancer_state_pb)
        elif isinstance(event, _BalancerAspectsSetUpdate):
            self._handle_balancer_aspects_set_update(event.path, event.balancer_aspects_set_pb)
        elif isinstance(event, _NamespaceAspectsSetUpdate):
            self._handle_namespace_aspects_set_update(event.path, event.namespace_aspects_set_pb)
        elif isinstance(event, _L3BalancerUpdate):
            self._handle_l3_balancer_update(event.path, event.l3_balancer_pb)
        elif isinstance(event, _L3BalancerStateUpdate):
            self._handle_l3_balancer_state_update(event.path, event.l3_balancer_state_pb)
        elif isinstance(event, _UpstreamUpdate):
            self._handle_upstream_update(event.path, event.upstream_pb)
        elif isinstance(event, _DomainUpdate):
            self._handle_domain_update(event.path, event.domain_pb)
        elif isinstance(event, _DomainOperationUpdate):
            self._handle_domain_operation_update(event.path, event.domain_operation_pb)
        elif isinstance(event, _BackendUpdate):
            self._handle_backend_update(event.path, event.backend_pb)
        elif isinstance(event, _EndpointSetUpdate):
            self._handle_endpoint_set_update(event.path, event.endpoint_set_pb)
        elif isinstance(event, _PartyMemberUpdate):
            party_id, node_name = event.path.split('/')
            member_id = node_name_to_member_id(node_name)
            self._handle_party_member_update(party_id, member_id)
        elif isinstance(event, _PartyMemberRemove):
            party_id, node_name = event.path.split('/')
            member_id = node_name_to_member_id(node_name)
            self._handle_party_member_remove(party_id, member_id)
        elif isinstance(event, treecache.TreeEvent):
            self._handle_tree_event(event)
        else:
            raise RuntimeError('Unknown event type {}'.format(type(event)))

    def run(self):
        self._stopped.wait()

    def start(self, stacksample=False):
        self._stopped.clear()

        is_initialized = gevent.event.Event()

        def listener(event):
            node_removed = False
            if event.event_type == event.NODE_UPDATED or event.event_type == event.NODE_ADDED:
                data = event.event_data.data
            elif event.event_type == event.NODE_REMOVED:
                data = None
                node_removed = True
            elif event.event_type == event.INITIALIZED:
                self._log.info('listener: connection event {}'.format(event.event_type))
                is_initialized.set()
                self._process_event(event)
                return
            elif event.event_type in (event.CONNECTION_SUSPENDED, event.CONNECTION_RECONNECTED, event.CONNECTION_LOST):
                self._log.info('listener: connection event {}'.format(event.event_type))
                self._process_event(event)
                return
            else:
                self._log.warn('listener: unknown event type %s', event.event_type)
                return

            event_path = event.event_data.path
            if event_path.startswith(self._path_w_trailing_slash):
                event_path = event_path[len(self._path_w_trailing_slash):]

            stripped_path = event_path.strip('/')
            parts = stripped_path.split('/') if '/' in stripped_path else []

            rv = None
            if not parts:
                pass
            elif parts[0] == objects.NamespaceDescriptor.zk_prefix:
                if len(parts) == 2:
                    namespace_id = parts[1]
                    rv = _NamespaceUpdate(namespace_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.ComponentDescriptor.zk_prefix:
                if len(parts) == 3:
                    component_type = parts[1]
                    version = parts[2]
                    rv = _ComponentUpdate(component_type + '/' + version, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.NamespaceAspectSetDescriptor.zk_prefix:
                if len(parts) == 2:
                    namespace_id = parts[1]
                    rv = _NamespaceAspectsSetUpdate(namespace_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.KnobDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    knob_id = parts[2]
                    rv = _KnobUpdate(namespace_id + '/' + knob_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.CertDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    cert_id = parts[2]
                    rv = _CertUpdate(namespace_id + '/' + cert_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.CertRenewalDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    cert_renewal_id = parts[2]
                    rv = _CertRenewalUpdate(namespace_id + '/' + cert_renewal_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.NameServerDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    name_server_id = parts[2]
                    rv = _NameServerUpdate(namespace_id + '/' + name_server_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.DnsRecordDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    dns_record_id = parts[2]
                    rv = _DnsRecordUpdate(namespace_id + '/' + dns_record_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.DnsRecordOperationDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    dns_record_op_id = parts[2]
                    rv = _DnsRecordOperationUpdate(namespace_id + '/' + dns_record_op_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.DnsRecordStateDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    dns_record_id = parts[2]
                    rv = _DnsRecordStateUpdate(namespace_id + '/' + dns_record_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.L7BalancerDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    balancer_id = parts[2]
                    rv = _BalancerUpdate(namespace_id + '/' + balancer_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.L7BalancerOperationDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    balancer_op_id = parts[2]
                    rv = _BalancerOperationUpdate(namespace_id + '/' + balancer_op_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.L7BalancerStateDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    balancer_id = parts[2]
                    rv = _BalancerStateUpdate(namespace_id + '/' + balancer_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.L7BalancerAspectSetDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    balancer_id = parts[2]
                    rv = _BalancerAspectsSetUpdate(namespace_id + '/' + balancer_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.L3BalancerDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    l3_balancer_id = parts[2]
                    rv = _L3BalancerUpdate(namespace_id + '/' + l3_balancer_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.L3BalancerStateDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    l3_balancer_id = parts[2]
                    rv = _L3BalancerStateUpdate(namespace_id + '/' + l3_balancer_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.UpstreamDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    upstream_id = parts[2]
                    rv = _UpstreamUpdate(namespace_id + '/' + upstream_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.DomainDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    domain_id = parts[2]
                    rv = _DomainUpdate(namespace_id + '/' + domain_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.DomainOperationDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    domain_operation_id = parts[2]
                    rv = _DomainOperationUpdate(namespace_id + '/' + domain_operation_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.BackendDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    backend_id = parts[2]
                    rv = _BackendUpdate(namespace_id + '/' + backend_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif parts[0] == objects.EndpointSetDescriptor.zk_prefix:
                if len(parts) == 3:
                    namespace_id = parts[1]
                    endpoint_set_id = parts[2]
                    rv = _EndpointSetUpdate(namespace_id + '/' + endpoint_set_id, data)
                    assert data or event.event_type == event.NODE_REMOVED
                else:
                    assert not data
            elif get_model_by_zk_prefix(parts[0]) is not None:
                model = get_model_by_zk_prefix(parts[0])
                full_uid = tuple(parts[1:])
                if data:
                    zk_path = model.desc.uid_to_zk_path(*full_uid)
                    model.cache.add(zk_path, data)

                    if model.cache.legacy_update_event is not None:
                        event = model.cache.legacy_update_event(path=zk_path, pb=data)
                        self._run_callbacks(event)
                else:  # either a parent node (skip it), or a valid object path (remove it from cache)
                    try:
                        zk_path = model.desc.uid_to_zk_path(*full_uid)
                    except AssertionError:  # parent node, e.g. /namespaces
                        return
                    else:  # actual object node, e.g. /namespace/my_ns_1
                        model.cache.discard(zk_path)

                        if model.cache.legacy_remove_event is not None:
                            event = model.cache.legacy_remove_event(path=zk_path)
                            self._run_callbacks(event)
                return self._run_subscribers(model=model, full_uid=full_uid, pb=data)
            elif parts[0] == storage_modern.EXCLUSIVE_SERVICES_NODE_ZK_PREFIX:
                pass
            elif parts[0] == storage_modern.PARTIES_NODE_ZK_PREFIX:
                if len(parts) == 3:
                    path = parts[1] + '/' + parts[2]
                    if node_removed:
                        rv = _PartyMemberRemove(path)
                    else:
                        rv = _PartyMemberUpdate(path)
            else:
                raise RuntimeError('Unknown path: {}'.format(parts))

            if rv:
                self._process_event(rv)

        b = time.time()
        if stacksample:
            s = Sampler()
            s.start()
        self._tree_cache.listen(listener)
        self._tree_cache.start()
        self._log.debug('Waiting for cache initialization...')
        is_initialized.wait()
        a = time.time()
        self._log.debug('Cache initialized in %s seconds', a - b)
        if stacksample:
            with open('./cache-init.stats', 'w') as f:
                f.write(s.output_stats())
            s.stop()
        super(AwacsCache, self).start()

    def stop(self, **kwargs):
        if self._stopped.is_set():
            return
        self._stopped.set()
        self._tree_cache.close()

    def subscribe_to_updates(self, cb, zk_path_chunks, model):
        self._update_subscribers[zk_path_chunks].add((cb, model))

    def unsubscribe_from_updates(self, cb, zk_path_chunks, model):
        self._update_subscribers[zk_path_chunks].discard((cb, model))

    def subscribe_to_removals(self, cb, zk_path_chunks, model):
        self._removal_subscribers[zk_path_chunks].add((cb, model))

    def unsubscribe_from_removals(self, cb, zk_path_chunks, model):
        self._removal_subscribers[zk_path_chunks].discard((cb, model))

    def _run_subscribers(self, model, full_uid, pb=None):
        self._run_subscribers_counter.inc(1)
        zk_path_chunks = [model.desc.zk_prefix]
        zk_path_chunks.extend(full_uid)
        with self._run_subscribers_timer.timer():
            while zk_path_chunks:
                if pb is not None:
                    handlers = self._update_subscribers.get(tuple(zk_path_chunks), ())
                else:
                    handlers = self._removal_subscribers.get(tuple(zk_path_chunks), ())
                handlers = copy.copy(handlers)
                self._run_subscribers_callback_counter.inc(len(handlers))
                for cb, model in handlers:
                    try:
                        cb(full_uid, model, pb)
                    except Exception as e:
                        self._run_subscribers_failure_counter.inc(1)
                        self._log.exception(u'subscriber callback failed: %s', e)
                        return
                zk_path_chunks.pop()

    def bind_on_specific_events(self, cb, event_types):
        # lock to avoid "Set changed size during iteration" errors
        with self._filtered_callbacks_lock:
            for event in event_types:
                self._filtered_callbacks[event.__name__].add(cb)

    def unbind_from_specific_events(self, cb, event_types):
        # lock to avoid "Set changed size during iteration" errors
        with self._filtered_callbacks_lock:
            for event in event_types:
                self._filtered_callbacks[event.__name__].discard(cb)

    def bind(self, cb):
        # lock to avoid "Set changed size during iteration" errors
        with self._callbacks_lock:
            self._callbacks.add(cb)

    def unbind(self, cb):
        # lock to avoid "Set changed size during iteration" errors
        with self._callbacks_lock:
            self._callbacks.discard(cb)

    def sync(self):
        return self._zk_client.client.sync(self._path)

    def _run_callbacks(self, event):
        self._run_callbacks_calls_counter.inc(1)
        with self._callbacks_lock:
            with self._run_callbacks_calls_timer.timer():
                self._callback_calls_counter.inc(len(self._callbacks))
                for cb in self._callbacks:
                    try:
                        cb(event)
                    except Exception as e:
                        self._run_callbacks_failures_counter.inc(1)
                        self._log.exception('cache callback failed: {}'.format(e))
                        return
        self._run_filtered_callbacks(event)

    def _run_filtered_callbacks(self, event):
        self._run_filtered_callbacks_calls_counter.inc(1)
        with self._filtered_callbacks_lock:
            with self._run_filtered_callbacks_calls_timer.timer():
                callbacks = self._filtered_callbacks[event.__class__.__name__]
                self._filtered_callback_calls_counter.inc(len(callbacks))
                for cb in callbacks:
                    try:
                        cb(event)
                    except Exception as e:
                        self._run_filtered_callbacks_failures_counter.inc(1)
                        self._log.exception('cache filtered callback failed: {}'.format(e))
                        return


_NamespaceUpdate = collections.namedtuple('_NamespaceUpdate', ('path', 'namespace_pb'))
_ComponentUpdate = collections.namedtuple('_ComponentUpdate', ('path', 'component_pb'))
_BalancerStateUpdate = collections.namedtuple('_BalancerStateUpdate', ('path', 'balancer_state_pb'))
_BalancerAspectsSetUpdate = collections.namedtuple('_BalancerAspectsSetUpdate', ('path', 'balancer_aspects_set_pb'))
_NamespaceAspectsSetUpdate = collections.namedtuple('_NamespaceAspectsSetUpdate', ('path', 'namespace_aspects_set_pb'))
_KnobUpdate = collections.namedtuple('_KnobUpdate', ('path', 'knob_pb'))
_CertUpdate = collections.namedtuple('_CertUpdate', ('path', 'cert_pb'))
_CertRenewalUpdate = collections.namedtuple('_CertRenewalUpdate', ('path', 'cert_renewal_pb'))
_NameServerUpdate = collections.namedtuple('_NameServerUpdate', ('path', 'name_server_pb'))
_DnsRecordUpdate = collections.namedtuple('_DnsRecordUpdate', ('path', 'dns_record_pb'))
_DnsRecordOperationUpdate = collections.namedtuple('_DnsRecordOperationUpdate', ('path', 'dns_record_operation_pb'))
_DnsRecordStateUpdate = collections.namedtuple('_DnsRecordStateUpdate', ('path', 'dns_record_state_pb'))
_BalancerUpdate = collections.namedtuple('_BalancerUpdate', ('path', 'balancer_pb'))
_BalancerOperationUpdate = collections.namedtuple('_BalancerOperationUpdate', ('path', 'balancer_pb'))
_L3BalancerStateUpdate = collections.namedtuple('_L3BalancerStateUpdate', ('path', 'l3_balancer_state_pb'))
_L3BalancerUpdate = collections.namedtuple('_L3BalancerUpdate', ('path', 'l3_balancer_pb'))
_UpstreamUpdate = collections.namedtuple('_UpstreamUpdate', ('path', 'upstream_pb'))
_DomainUpdate = collections.namedtuple('_DomainUpdate', ('path', 'domain_pb'))
_DomainOperationUpdate = collections.namedtuple('_DomainOperationUpdate', ('path', 'domain_operation_pb'))
_BackendUpdate = collections.namedtuple('_BackendUpdate', ('path', 'backend_pb'))
_EndpointSetUpdate = collections.namedtuple('_EndpointSetUpdate', ('path', 'endpoint_set_pb'))
_PartyMemberUpdate = collections.namedtuple('_PartyMemberUpdate', ('path',))
_PartyMemberRemove = collections.namedtuple('_PartyMemberRemove', ('path',))
