from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import is_order_in_progress, has_actionable_spec
from awacs.model import events
from awacs.model.certs.ctl import CertCtl
from awacs.model.certs.order_ctl import CertOrderCtl


class CertCtlManager(ctlmanager.PartyingCtlManagerV2):
    CONTENDING_MEMBERS = 2

    def __init__(self, coord, member_id, party_suffix, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(CertCtlManager, self).__init__(
            coord=coord,
            member_id=member_id,
            party_suffix=party_suffix,
            cache=cache,
            entity_name='cert',
            subscribed_events=(events.CertUpdate, events.CertRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.CertRemove):
            return False
        return has_actionable_spec(event.pb) and not is_order_in_progress(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for cert_pb in self._cache.list_all_certs(namespace_id):
                yield events.CertUpdate(path=self._make_event_path_from_ctl_id(namespace_id, cert_pb.meta.id),
                                        pb=cert_pb)

    def _create_ctl(self, ctl_id):
        return CertCtl(*ctl_id)


class CertOrderCtlManager(ctlmanager.PartyingCtlManagerV2):
    CONTENDING_MEMBERS = 2

    def __init__(self, coord, member_id, party_suffix, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(CertOrderCtlManager, self).__init__(
            coord=coord,
            member_id=member_id,
            party_suffix=party_suffix,
            cache=cache,
            entity_name='cert-order',
            subscribed_events=(events.CertUpdate, events.CertRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.CertRemove):
            return False
        return is_order_in_progress(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for cert_pb in self._cache.list_all_certs(namespace_id):
                yield events.CertUpdate(path=self._make_event_path_from_ctl_id(namespace_id, cert_pb.meta.id),
                                        pb=cert_pb)

    def _create_ctl(self, ctl_id):
        return CertOrderCtl(*ctl_id)
