# coding: utf-8

import inject

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import is_spec_complete
from awacs.lib.order_processor.runner import StateRunner
from awacs.model import events, cache
from awacs.model.certs.processors.processors import get_cert_state_processors, CertOrder
from awacs.model.zk import IZkStorage, ZkStorage
from infra.awacs.proto import model_pb2


class CertOrderCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(IZkStorage)  # type: ZkStorage

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    # sleep more after unexpected exceptions such as 429 TOO MANY REQUESTS (see AWACS-652 for details)
    SLEEP_AFTER_EXCEPTION_TIMEOUT = 3 * 60  # 3 min
    SLEEP_AFTER_EXCEPTION_TIMEOUT_JITTER = 2 * 60  # 2 min

    _state_runner = StateRunner(entity_class=CertOrder,
                                initial_state=CertOrder.states.SENDING_CREATE_REQUEST_TO_CERTIFICATOR,
                                final_state=CertOrder.states.FINISH,
                                final_cancelled_state=CertOrder.states.CANCELLED,
                                processors=get_cert_state_processors(),
                                processing_interval=PROCESSING_INTERVAL)

    def __init__(self, namespace_id, cert_id):
        name = 'cert-order-ctl("{}:{}")'.format(namespace_id, cert_id)
        super(CertOrderCtl, self).__init__(name)
        self._cert_id = cert_id
        self._namespace_id = namespace_id
        self._full_cert_id = (self._namespace_id, self._cert_id)
        self._pb = None  # type: model_pb2.Certificate or None

    def _accept_event(self, event):
        return (isinstance(event, events.CertUpdate) and
                event.pb.meta.id == self._cert_id and
                not is_spec_complete(event.pb))

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process cert order on start: %s', e)
        self._cache.bind(self._callback)

    def _stop(self):
        self._cache.unbind(self._callback)

    def _process(self, ctx, event=None):
        """
        :type event: events.CertUpdate
        :type ctx: context.OpCtx
        """
        cached_pb = self._cache.must_get_cert(self._namespace_id, self._cert_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if not self._pb.HasField('order'):
            raise RuntimeError('Certificate order has no "order" field')

        return self._state_runner.process(ctx, self._pb)

    def _process_event(self, ctx, event):
        assert isinstance(event, events.CertUpdate)
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
