# coding: utf-8
import textwrap

import abc
import enum
import inject
import itertools

import six
from sepelib.core import config


try:
    from vault_client.errors import ClientError
except ImportError:
    # for Arcadia:
    from library.python.vault_client.errors import ClientError

from awacs.lib import startrekclient, certificator, certs, ya_vault
from awacs.lib.order_processor.model import BaseProcessor, WithOrder
from awacs.model import dao, zk, util
from infra.awacs.proto import model_pb2


CERT_ORDER_SOURCES = {
    'production': model_pb2.CertificateSpec.CERTIFICATOR,
    'testing': model_pb2.CertificateSpec.CERTIFICATOR_TESTING,
}
PRODUCTION_CA = 'CertumProductionCA'

SIGNATURE_ALGO_EC = 'ec'
SIGNATURE_ALGO_RSA = 'rsa'

SIX_MONTHS_IN_DAYS = 182


class State(enum.Enum):
    START = 1
    SENDING_CREATE_REQUEST_TO_CERTIFICATOR = 2
    SENT_REQUEST_TO_CERTIFICATOR = 3
    POLLING_CERTIFICATOR_FOR_STORAGE_INFO = 4
    GOT_STORAGE_INFO = 5
    FETCHING_CERT_INFO_FROM_STORAGE = 6
    GOT_CERT_INFO = 7
    FINISH = 8
    CANCELLING = 9
    CANCELLED = 10
    ADDING_COMMENT_TO_SECTASK = 11
    ADDING_CERT_OWNERS_TO_SECTASK = 12
    MODIFYING_YAV_SECRET_ACCESS = 13
    ADDING_AWACS_ON_DUTY_TO_SECTASK = 14


def get_cert_state_processors():
    return CertOrderProcessor.__subclasses__()


class CertOrder(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = 'Certificate'
    states = State

    def zk_update(self):
        return self.zk.update_cert(namespace_id=self.namespace_id,
                                   cert_id=self.id)

    def dao_update(self, comment):
        return self.dao.update_cert(namespace_id=self.namespace_id,
                                    cert_id=self.id,
                                    version=self.pb.meta.version,
                                    comment=comment,
                                    login=util.NANNY_ROBOT_LOGIN,
                                    updated_spec_pb=self.pb.spec)


class CertOrderProcessor(six.with_metaclass(abc.ABCMeta, BaseProcessor)):
    __slots__ = ('cert',)

    def __init__(self, entity):
        super(CertOrderProcessor, self).__init__(entity)
        self.cert = self.entity  # type: CertOrder


def get_certificator_order_id_from_url(url):
    """
    :type url: six.text_type
    :return:
    """
    _, order_id = url.rstrip('/').rsplit('/', 1)  # Temporary hack. See CERTOR-1410
    if not order_id:
        raise ValueError('Cannot parse order_id from url: {}'.format(url))
    return order_id


class Start(CertOrderProcessor):
    __slots__ = ()
    state = State.START
    next_state = State.SENDING_CREATE_REQUEST_TO_CERTIFICATOR
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class SendingCreateRequestToCertificator(CertOrderProcessor):
    __slots__ = ()
    state = State.SENDING_CREATE_REQUEST_TO_CERTIFICATOR
    next_state = State.ADDING_COMMENT_TO_SECTASK
    cancelled_state = State.CANCELLING

    _certificator_client = inject.attr(certificator.ICertificatorClient)  # type: certificator.CertificatorClient

    def _check_owners(self, ctx):
        if not self.cert.pb.meta.auth.staff.owners.logins:
            raise RuntimeError('{}: cert has no owners'.format(ctx.id()))

    def _get_cert_ttl(self):
        if self.cert.pb.order.content.ttl:
            return min(self.cert.pb.order.content.ttl, SIX_MONTHS_IN_DAYS)
        elif self.cert.pb.order.content.ca_name == PRODUCTION_CA:
            return SIX_MONTHS_IN_DAYS
        else:  # internal certs
            return None  # Certificator default

    def process(self, ctx):
        self._check_owners(ctx)
        if self.cert.pb.spec.certificator.order_id and 'status' in self.cert.context:
            return self.next_state

        resp = self._certificator_client.send_create_request(
            ca_name=self.cert.pb.order.content.ca_name,
            common_name=self.cert.pb.order.content.common_name,
            subject_alternative_names=self.cert.pb.order.content.subject_alternative_names,
            abc_service_id=self.cert.pb.order.content.abc_service_id,
            is_ecc=self.cert.pb.order.content.public_key_algorithm_id == SIGNATURE_ALGO_EC,
            desired_ttl_days=self._get_cert_ttl(),
        )
        self.cert.context['status'] = resp['status']

        self.cert.pb.spec.source = CERT_ORDER_SOURCES[self._certificator_client.environment]
        self.cert.pb.spec.certificator.order_id = get_certificator_order_id_from_url(resp['url'])
        self.cert.pb.spec.certificator.abc_service_id = self.cert.pb.order.content.abc_service_id
        self.cert.pb.spec.certificator.ca_name = self.cert.pb.order.content.ca_name
        if resp['approve_request'] and resp['st_issue_key']:
            self.cert.pb.spec.certificator.approval.startrek.issue_id = resp['st_issue_key']
        self.cert.pb = self.cert.dao_update(comment='Saved certificate order info')
        return self.next_state


class SentRequestToCertificator(CertOrderProcessor):
    __slots__ = ()
    state = State.SENT_REQUEST_TO_CERTIFICATOR
    next_state = State.ADDING_COMMENT_TO_SECTASK
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class AddingCommentToSectask(CertOrderProcessor):
    __slots__ = ()
    state = State.ADDING_COMMENT_TO_SECTASK
    next_state = State.ADDING_CERT_OWNERS_TO_SECTASK
    cancelled_state = State.CANCELLING

    startrek_client = inject.attr(startrekclient.IStartrekClient)  # type: startrekclient.StartrekClient

    sectask_comment_template = textwrap.dedent(
        """
        Сертификат заказан через awacs - новый заказ.
        Заказал(a): **кто:{author} ({author})**
        awacs namespace: https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{namespace}/show/
        """
    )

    def _get_sectask_comment_text(self):
        return self.sectask_comment_template.format(
            author=self.cert.pb.meta.auth.staff.owners.logins[0],
            namespace=self.cert.namespace_id,
        )

    def _add_comment_to_sectask(self, ctx, sectask_id, ticket):
        if len(list((ticket.comments.get_all()))) > 30:
            ctx.log.debug('Not leaving comment in %s - is already has more than 30 comments', sectask_id)
            return
        ctx.log.debug('Adding comment to %s', sectask_id)
        try:
            resp = ticket.comments.create(text=self._get_sectask_comment_text())
            self.cert.context['sectask_comment_id'] = resp.id
        except Exception as e:
            ctx.log.exception('Failed to post comment in ticket %s: %s', sectask_id, e)

    def process(self, ctx):
        if 'sectask_comment_id' in self.cert.context:
            return self.next_state
        sectask_id = self.cert.pb.spec.certificator.approval.startrek.issue_id
        if not sectask_id:
            return self.next_state
        ticket = self.startrek_client.issues[sectask_id]
        self._add_comment_to_sectask(ctx, sectask_id, ticket)
        return self.next_state


def add_users_to_ticket_followers(ctx, startrek_client, users, ticket_id):
    attempt = 0
    while attempt < 3:
        ticket = startrek_client.issues[ticket_id]
        followers = set(f.login for f in ticket.followers)
        users_without_access = users - followers
        if not users_without_access:
            return
        if attempt > 0:
            ctx.log.warning('Attempt #%s: %s was successfully updated, but some followers are still missing: %s',
                            attempt, ticket_id, ', '.join(sorted(users_without_access)))
        for login in users_without_access:
            user = startrek_client.users[login]
            ticket.followers.append(user)
        ticket.update(followers=ticket.followers)
        ctx.log.debug('Added followers to %s: %s', ticket_id, ', '.join(users_without_access))
        attempt += 1


class AddingCertOwnersToSectask(CertOrderProcessor):
    __slots__ = ()
    state = State.ADDING_CERT_OWNERS_TO_SECTASK
    next_state = State.ADDING_AWACS_ON_DUTY_TO_SECTASK
    cancelled_state = State.CANCELLING

    startrek_client = inject.attr(startrekclient.IStartrekClient)  # type: startrekclient.StartrekClient

    def _get_cert_owners(self):
        return set(self.cert.pb.meta.auth.staff.owners.logins)

    def process(self, ctx):
        sectask_id = self.cert.pb.spec.certificator.approval.startrek.issue_id
        if not sectask_id:
            return self.next_state
        add_users_to_ticket_followers(ctx, self.startrek_client, users=self._get_cert_owners(), ticket_id=sectask_id)
        return self.next_state


class AddingAwacsOnDutyToSectask(CertOrderProcessor):
    __slots__ = ()
    state = State.ADDING_AWACS_ON_DUTY_TO_SECTASK
    next_state = State.POLLING_CERTIFICATOR_FOR_STORAGE_INFO
    cancelled_state = State.CANCELLING

    startrek_client = inject.attr(startrekclient.IStartrekClient)  # type: startrekclient.StartrekClient

    @staticmethod
    def _get_awacs_on_duty_users():
        return set(config.get_value('run.on_duty_users', ()))

    def process(self, ctx):
        sectask_id = self.cert.pb.spec.certificator.approval.startrek.issue_id
        if not sectask_id:
            return self.next_state
        add_users_to_ticket_followers(ctx, self.startrek_client,
                                      users=self._get_awacs_on_duty_users(),
                                      ticket_id=sectask_id)
        return self.next_state


class PollingCertificatorForStorageInfo(CertOrderProcessor):
    __slots__ = ()
    state = State.POLLING_CERTIFICATOR_FOR_STORAGE_INFO
    next_state = State.FETCHING_CERT_INFO_FROM_STORAGE
    cancelled_state = State.CANCELLING

    _certificator_client = inject.attr(certificator.ICertificatorClient)  # type: certificator.CertificatorClient

    def process(self, ctx):
        if 'serial_number' in self.cert.context:
            return self.next_state
        cert = self._certificator_client.get_cert(self.cert.pb.spec.certificator.order_id)
        if not cert['uploaded_to_yav']:
            if cert['status'] == 'need_approve':
                self.cert.context['status'] = cert['status']
            if cert['status'] == 'error':
                raise RuntimeError('Something went wrong during the cert order, got "status=error" from certificator')
            return self.state

        self.cert.pb.spec.storage.ya_vault_secret.secret_id = cert['yav_secret_id']
        self.cert.pb.spec.storage.ya_vault_secret.secret_ver = cert['yav_secret_version']
        self.cert.pb = self.cert.dao_update(comment='Saved certificate storage info')
        self.cert.context['serial_number'] = cert['serial_number']
        return self.next_state


class GotStorageInfo(CertOrderProcessor):
    __slots__ = ()
    state = State.GOT_STORAGE_INFO
    next_state = State.FETCHING_CERT_INFO_FROM_STORAGE
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class FetchingCertInfoFromStorage(CertOrderProcessor):
    __slots__ = ()
    state = State.FETCHING_CERT_INFO_FROM_STORAGE
    next_state = State.MODIFYING_YAV_SECRET_ACCESS
    cancelled_state = State.CANCELLING

    _yav_client = inject.attr(ya_vault.IYaVaultClient)  # type: ya_vault.YaVaultClient

    def process(self, ctx):
        if 'got_cert_secret' in self.cert.context:
            return self.next_state

        cert_secret = self._yav_client.get_version(version=self.cert.pb.spec.storage.ya_vault_secret.secret_ver)
        flat_cert_id = '{}/{}'.format(self.cert.pb.meta.namespace_id, self.cert.pb.meta.id)
        public_key, _, _ = certs.extract_certs_from_yav_secret(log=ctx.log,
                                                               flat_cert_id=flat_cert_id,
                                                               serial_number=self.cert.context['serial_number'],
                                                               cert_secret=cert_secret['value'])
        certs.fill_cert_fields(self.cert.pb.spec.fields, certs.get_end_entity_cert(public_key))
        self.cert.pb = self.cert.dao_update(comment='Saved certificate info')
        self.cert.context['got_cert_secret'] = True
        return self.next_state


class ModifyingYavSecretAccess(CertOrderProcessor):
    __slots__ = ()
    state = State.MODIFYING_YAV_SECRET_ACCESS
    next_state = State.FINISH
    cancelled_state = State.CANCELLING

    _yav_client = inject.attr(ya_vault.IYaVaultClient)  # type: ya_vault.YaVaultClient

    ABC_SCOPE_IDS = {
        8: 'administration',
        85: 'cert'
    }

    def process(self, ctx):
        owners = self._yav_client.get_owners(self.cert.pb.spec.storage.ya_vault_secret.secret_id)
        readers = self._yav_client.get_readers(self.cert.pb.spec.storage.ya_vault_secret.secret_id)
        users_to_delete = set()
        for user in itertools.chain(owners, readers):
            if user.get(u'abc_id') == self.cert.pb.spec.certificator.abc_service_id:
                users_to_delete.add((user[u'abc_id'],
                                     self.ABC_SCOPE_IDS.get(user[u'abc_scope_id']),
                                     user[u'role_slug']))
            elif user.get(u'login') != u'robot-awacs-certs':
                ctx.log.error('Unexpected user with access to certificate secret: %s', user)
        for abc_id, abc_scope, role in users_to_delete:
            self._yav_client.delete_user_role_from_secret(
                secret_id=self.cert.pb.spec.storage.ya_vault_secret.secret_id,
                abc_service_id=abc_id,
                abc_scope=abc_scope,
                role=role)
            ctx.log.info('Removed access to cert secret: "abc_id=%s, scope=%s, role=%s', abc_id, abc_scope, role)
        self.cert.pb.spec.incomplete = False
        self.cert.pb = self.cert.dao_update(comment='Marked cert as complete')
        return self.next_state


class GotCertInfo(CertOrderProcessor):
    __slots__ = ()
    state = State.GOT_CERT_INFO
    next_state = State.FINISH
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class Cancelling(CertOrderProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    _startrek_client = inject.attr(startrekclient.IStartrekClient)  # type: startrekclient.StartrekClient

    sectask_cancel_comment_template = textwrap.dedent(
        """
        Заказ сертификата был отменён в awacs.
        Отменил(a): **кто:{author} ({author})**
        awacs namespace: https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{namespace}/show/
        """
    )

    def _add_cancel_comment_to_sectask(self, ctx):
        sectask_id = self.cert.pb.spec.certificator.approval.startrek.issue_id
        if not sectask_id:
            return
        existing_comment = self.cert.context.get('sectask_cancel_comment_id')
        if existing_comment:
            return
        ticket = self._startrek_client.issues[sectask_id]
        if len(list((ticket.comments.get_all()))) > 30:
            ctx.log.debug('Not leaving cancel comment in %s - is already has more than 30 comments', sectask_id)
            return
        ctx.log.debug('Adding cancel comment to %s', sectask_id)
        try:
            resp = ticket.comments.create(text=self.sectask_cancel_comment_template.format(
                author=self.cert.pb.order.cancelled.author,
                namespace=self.cert.namespace_id,
            ))
            self.cert.context['sectask_cancel_comment_id'] = resp.id
        except Exception as e:
            ctx.log.exception('Failed to post cancel comment in ticket %s: %s', sectask_id, e)

    def process(self, ctx):
        self._add_cancel_comment_to_sectask(ctx)
        return self.next_state
