# coding=utf-8
import textwrap

import enum
import inject

from awacs.lib.order_processor.model import WithOrder
from awacs.model import dao, zk, util
from awacs.model.certs.processors import processors


class State(enum.Enum):
    SENDING_CREATE_REQUEST_TO_CERTIFICATOR = 1
    POLLING_CERTIFICATOR_FOR_STORAGE_INFO = 2
    FETCHING_CERT_INFO_FROM_STORAGE = 3
    FINISH = 4
    CANCELLING = 5
    CANCELLED = 6
    ADDING_COMMENT_TO_SECTASK = 7
    ADDING_CERT_OWNERS_TO_SECTASK = 8
    MODIFYING_YAV_SECRET_ACCESS = 9
    ADDING_AWACS_ON_DUTY_TO_SECTASK = 10


def get_cert_renewal_state_processors():
    return (
        SendingCreateRequestToCertificator,
        PollingCertificatorForStorageInfo,
        FetchingCertInfoFromStorage,
        Cancelling,
        AddingCommentToSectask,
        AddingCertOwnersToSectask,
        AddingAwacsOnDutyToSectask,
        ModifyingYavSecretAccess,
    )


class CertRenewalOrder(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = 'Certificate renewal'
    states = State

    def zk_update(self):
        return self.zk.update_cert_renewal(namespace_id=self.pb.meta.namespace_id,
                                           cert_renewal_id=self.pb.meta.id)

    def dao_update(self, comment):
        return self.dao.update_cert_renewal(namespace_id=self.pb.meta.namespace_id,
                                            cert_renewal_id=self.pb.meta.id,
                                            version=self.pb.meta.version,
                                            comment=comment,
                                            login=util.NANNY_ROBOT_LOGIN,
                                            updated_spec_pb=self.pb.spec)


class SendingCreateRequestToCertificator(processors.SendingCreateRequestToCertificator):
    __slots__ = ()
    state = State.SENDING_CREATE_REQUEST_TO_CERTIFICATOR
    next_state = State.ADDING_COMMENT_TO_SECTASK
    cancelled_state = State.CANCELLING

    def _check_owners(self, ctx):
        cert_pb = self.cert.zk.must_get_cert(self.cert.namespace_id, self.cert.id)
        if not cert_pb.meta.auth.staff.owners.logins:
            raise RuntimeError('{}: original cert has no owners'.format(ctx.id()))


class AddingCommentToSectask(processors.AddingCommentToSectask):
    __slots__ = ()
    state = State.ADDING_COMMENT_TO_SECTASK
    next_state = State.ADDING_CERT_OWNERS_TO_SECTASK
    cancelled_state = State.CANCELLING

    sectask_comment_template = textwrap.dedent(
        """
        Сертификат заказан через awacs - перезаказ в связи с истечением срока годности.
        Предыдущий тикет на заказ сертификата: {previous_sectask_id}
        Заказал(a): **кто:{author} ({author})**
        awacs namespace: https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{namespace}/show/
        """
    )

    def _get_sectask_comment_text(self):
        cert_pb = self.cert.zk.must_get_cert(self.cert.namespace_id, self.cert.id)
        if cert_pb.spec.certificator.approval.startrek.issue_id:
            previous_sectask_id = cert_pb.spec.certificator.approval.startrek.issue_id
        else:
            previous_sectask_id = '-'
        return self.sectask_comment_template.format(
            author=cert_pb.meta.auth.staff.owners.logins[0],
            namespace=self.cert.namespace_id,
            previous_sectask_id=previous_sectask_id
        )


class AddingCertOwnersToSectask(processors.AddingCertOwnersToSectask):
    __slots__ = ()
    state = State.ADDING_CERT_OWNERS_TO_SECTASK
    next_state = State.ADDING_AWACS_ON_DUTY_TO_SECTASK
    cancelled_state = State.CANCELLING

    def _get_cert_owners(self):
        cert_pb = self.cert.zk.must_get_cert(self.cert.namespace_id, self.cert.id)
        return set(cert_pb.meta.auth.staff.owners.logins)


class AddingAwacsOnDutyToSectask(processors.AddingAwacsOnDutyToSectask):
    __slots__ = ()
    state = State.ADDING_AWACS_ON_DUTY_TO_SECTASK
    next_state = State.POLLING_CERTIFICATOR_FOR_STORAGE_INFO
    cancelled_state = State.CANCELLING


class PollingCertificatorForStorageInfo(processors.PollingCertificatorForStorageInfo):
    __slots__ = ()
    state = State.POLLING_CERTIFICATOR_FOR_STORAGE_INFO
    next_state = State.FETCHING_CERT_INFO_FROM_STORAGE
    cancelled_state = State.CANCELLING


class FetchingCertInfoFromStorage(processors.FetchingCertInfoFromStorage):
    __slots__ = ()
    state = State.FETCHING_CERT_INFO_FROM_STORAGE
    next_state = State.MODIFYING_YAV_SECRET_ACCESS
    cancelled_state = State.CANCELLING


class ModifyingYavSecretAccess(processors.ModifyingYavSecretAccess):
    __slots__ = ()
    state = State.MODIFYING_YAV_SECRET_ACCESS
    next_state = State.FINISH
    cancelled_state = State.CANCELLING


class Cancelling(processors.Cancelling):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None
