from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import has_actionable_spec, is_order_in_progress
from awacs.model import events
from awacs.model.certs.renewal_ctl import CertRenewalCtl
from awacs.model.certs.renewal_order_ctl import CertRenewalOrderCtl


class CertRenewalCtlManager(ctlmanager.PartyingCtlManagerV2):
    CONTENDING_MEMBERS = 2

    def __init__(self, coord, member_id, party_suffix, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type party_suffix: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(CertRenewalCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            member_id=member_id,
            party_suffix=party_suffix,
            entity_name='cert-renewal',
            subscribed_events=(events.CertRenewalUpdate, events.CertRenewalRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.CertRenewalRemove):
            return False
        return has_actionable_spec(event.pb) and not is_order_in_progress(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for cert_renewal_pb in self._cache.list_all_cert_renewals(namespace_id):
                yield events.CertRenewalUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, cert_renewal_pb.meta.id),
                    pb=cert_renewal_pb)

    def _create_ctl(self, ctl_id):
        namespace_id, cert_renewal_id = ctl_id
        try:
            return CertRenewalCtl(namespace_id, cert_renewal_id)
        except Exception as e:
            self._log.exception('%s:%s failed to start controller: %s', namespace_id, cert_renewal_id, e)


class CertRenewalOrderCtlManager(ctlmanager.PartyingCtlManagerV2):
    CONTENDING_MEMBERS = 2

    def __init__(self, coord, member_id, party_suffix, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type party_suffix: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(CertRenewalOrderCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            member_id=member_id,
            party_suffix=party_suffix,
            entity_name='cert-renewal',
            subscribed_events=(events.CertRenewalUpdate, events.CertRenewalRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.CertRenewalRemove):
            return False
        return is_order_in_progress(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for cert_renewal_pb in self._cache.list_all_cert_renewals(namespace_id):
                yield events.CertRenewalUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, cert_renewal_pb.meta.id),
                    pb=cert_renewal_pb)

    def _create_ctl(self, ctl_id):
        return CertRenewalOrderCtl(*ctl_id)
