import functools

import abc
import enum
import re
import semantic_version
import six

from awacs.lib import OrderedDict
from awacs.lib.rpc import exceptions
from infra.awacs.proto import model_pb2


LAYERS_SEQUENCE = [
    model_pb2.ComponentMeta.BASE_LAYER,
    model_pb2.ComponentMeta.UTILITY_LAYER,
    model_pb2.ComponentMeta.SHAWSHANK_LAYER,
    model_pb2.ComponentMeta.TANK_LAYER
]


@functools.total_ordering
class Version(six.with_metaclass(abc.ABCMeta, object)):
    __slots__ = ()

    def __eq__(self, other):
        return type(self) is type(other) and self.to_key() == other.to_key()

    def __ne__(self, other):
        # https://bugs.python.org/issue25732
        return not self.__eq__(other)

    def __lt__(self, other):
        if type(self) is type(other):
            return self.to_key() < other.to_key()
        else:
            raise ValueError(u'{!r} is not comparable to {!r}'.format(self, other))

    def __hash__(self):
        return hash(self.to_key())

    @abc.abstractmethod
    def to_key(self):
        raise NotImplementedError

    @classmethod
    @abc.abstractmethod
    def parse(cls, s):
        raise NotImplementedError

    @classmethod
    @abc.abstractmethod
    def validate(cls, s):
        raise NotImplementedError


class MajorMinorVersion(six.with_metaclass(abc.ABCMeta, Version)):
    __slots__ = (u'major', u'minor')

    def __init__(self, major, minor):
        """
        :type major: int
        :type minor: int
        """
        self.major = major
        self.minor = minor

    @classmethod
    @abc.abstractmethod
    def get_pattern(cls):
        raise NotImplementedError

    @classmethod
    @abc.abstractmethod
    def get_separator(cls):
        raise NotImplementedError

    def to_key(self):
        return self.major, self.minor

    @classmethod
    def parse(cls, s):
        assert re.match(cls.get_pattern(), s)
        major_s, minor_s = s.split(cls.get_separator())
        return cls(int(major_s), int(minor_s))

    @classmethod
    def validate(cls, s):
        if not re.match(cls.get_pattern(), s):
            raise ValueError(u'is not {}'.format(cls.get_pattern()))

    def __str__(self):
        return '{}{}{}'.format(self.major, self.get_separator(), self.minor)


class DotSeparatedMajorMinorVersion(MajorMinorVersion):
    __slots__ = ()

    @classmethod
    def get_separator(cls):
        return u'.'

    @classmethod
    def get_pattern(cls):
        return r'^[0-9]+\.[0-9]+$'


class DashSeparatedMajorMinorVersion(MajorMinorVersion):
    @classmethod
    def get_separator(cls):
        return u'-'

    @classmethod
    def get_pattern(cls):
        return u'^[0-9]+-[0-9]+$'


class SemanticVersion(Version):
    __slots__ = (u'semver',)

    def __init__(self, semver):
        """
        :type semver: semantic_version.Version
        """
        self.semver = semver

    def to_key(self):
        # We often use prerelease part to distinguish between two different branches of component
        # (e.g. 0.0.1 and 0.0.1-pushclient). Let's consider them equal
        truncated_v = '.'.join(six.text_type(x) for x in [self.semver.major, self.semver.minor, self.semver.patch])
        return semantic_version.Version(truncated_v)

    @classmethod
    def parse(cls, s):
        return cls(semantic_version.Version(s))

    @classmethod
    def validate(cls, s):
        if not semantic_version.validate(s):
            raise ValueError(u'is not valid semantic version')


class BaseLayerVersion(Version):
    __slots__ = (u'name', u'version')

    pattern = r'[a-z]+-[0-9]+'
    sequence = [u'precise', u'trusty', u'xenial', u'bionic', u'focal']

    def __init__(self, name, version):
        """
        :type name: six.text_type
        :type version: int
        """
        self.name = name
        self.version = version

    def to_key(self):
        return self.sequence.index(self.name), self.version

    @classmethod
    def parse(cls, s):
        name, version = s.split('-')
        return cls(name, int(version))

    @classmethod
    def validate(cls, s):
        if not re.match(cls.pattern, s):
            raise ValueError(u'is not {}'.format(cls.pattern))
        name = s.split('-')[0]
        if name not in cls.sequence:
            raise ValueError(u'name must be one of "{}"'.format('", "'.join(cls.sequence)))


class ComponentSandboxResource(object):
    __slots__ = (u'task_types', u'resource_types')

    def __init__(self, task_types, resource_types):
        """
        :type task_types: set[six.text_type]
        :type resource_types: set[six.text_type]
        """
        self.task_types = task_types
        self.resource_types = resource_types


class SourceType(enum.Enum):
    SANDBOX_RESOURCE = 1
    URL_FILE = 2


class ComponentConfig(object):
    __slots__ = ('type', 'pb_field_name', 'local_file_path', 'version_class', 'sandbox_resource',
                 'can_be_removed_from_spec', 'str_type', 'is_layer', 'make_nanny_resource_readonly',
                 'possible_source_types')

    SOURCE_TYPE_ENUM_TO_VALUE = {
        SourceType.SANDBOX_RESOURCE: 'sandbox_resource',
        SourceType.URL_FILE: 'url_file'
    }

    def __init__(self, component_type, pb_field_name, local_file_path, version_class, sandbox_resource,
                 can_be_removed_from_spec, is_layer, make_nanny_resource_readonly, possible_source_types):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type pb_field_name: six.text_type
        :type local_file_path: Optional[six.text_type]
        :type version_class: Type[Version]
        :type sandbox_resource: ComponentSandboxResource
        :type can_be_removed_from_spec: bool
        :type is_layer: bool
        :type make_nanny_resource_readonly: bool
        :type possible_source_types: set[SourceType]
        """
        self.type = component_type
        self.pb_field_name = pb_field_name
        self.local_file_path = local_file_path
        self.version_class = version_class
        self.sandbox_resource = sandbox_resource
        self.can_be_removed_from_spec = can_be_removed_from_spec
        self.str_type = model_pb2.ComponentMeta.Type.Name(component_type)
        self.is_layer = is_layer
        if is_layer:
            assert self.type in LAYERS_SEQUENCE
        else:
            assert self.type not in LAYERS_SEQUENCE
        self.make_nanny_resource_readonly = make_nanny_resource_readonly
        self.possible_source_types = possible_source_types

    def parse_version(self, version):
        return self.version_class.parse(version)

    def validate_version(self, version):
        return self.version_class.validate(version)

    def validate_source_type(self, source_type, field_name='spec.source'):
        """
        :type source_type: six.text_type
        :raises: exceptions.BadRequestError
        """
        possible_source_type_strs = []
        for possible_source_type in self.possible_source_types:
            possible_source_type_strs.append(self.SOURCE_TYPE_ENUM_TO_VALUE[possible_source_type])
        if source_type not in possible_source_type_strs:
            raise exceptions.BadRequestError('one of ("{}") must be set for selected component type'.format(
                '", "'.join("{}.{}".format(field_name, source_type_str)
                            for source_type_str in sorted(possible_source_type_strs))
            ))

    def get_component_pb(self, cache, version):
        """
        :type cache: awacs.model.cache.AwacsCache
        :type version: six.text_type
        :rtype: model_pb2.Component | None
        """
        return cache.get_component(self.type, version)

    def get_default_version(self, cache, cluster):
        """
        :type cache: awacs.model.cache.AwacsCache
        :type cluster: six.text_type
        :rtype: six.text_type | None
        """
        return cache.get_component_default_version(self.type, cluster)

    def must_get_default_version(self, cache, cluster):
        """
        :type cache: awacs.model.cache.AwacsCache
        :type cluster: six.text_type
        :rtype: six.text_type
        """
        return cache.must_get_component_default_version(self.type, cluster)

    def get_latest_published_version(self, cache):
        """
        :type cache: awacs.model.cache.AwacsCache
        :rtype: six.text_type | None
        """
        all_pbs = cache.list_all_components(
            query={cache.ComponentQueryTarget.TYPE_IN: [self.type],
                   cache.ComponentQueryTarget.STATUS_IN: [model_pb2.ComponentStatus.PUBLISHED]},
            sort=(cache.ComponentsSortTarget.VERSION, 1))
        if not all_pbs:
            return None
        return all_pbs[-1].meta.version


COMPONENT_CONFIGS = [
    ComponentConfig(
        component_type=model_pb2.ComponentMeta.INSTANCECTL,
        pb_field_name='instancectl',
        local_file_path=None,
        version_class=DotSeparatedMajorMinorVersion,
        sandbox_resource=ComponentSandboxResource({'BUILD_INSTANCE_CTL'}, {'INSTANCECTL'}),
        can_be_removed_from_spec=False,
        is_layer=False,
        make_nanny_resource_readonly=False,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.INSTANCECTL_CONF,
        pb_field_name='instancectl_conf',
        local_file_path='instancectl.conf',
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'NANNY_REMOTE_COPY_RESOURCE'}, {
            'RTC_MTN_BALANCER_INSTANCECTL_CONF', 'RTC_MTN_BALANCER_PLUS_PUSHCLIENT_INSTANCECTL_CONF'}),
        can_be_removed_from_spec=True,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.PGINX_BINARY,
        pb_field_name='pginx_binary',
        local_file_path='balancer',
        version_class=DashSeparatedMajorMinorVersion,
        sandbox_resource=ComponentSandboxResource({'BUILD_BALANCER_BUNDLE'}, {'BALANCER_EXECUTABLE'}),
        can_be_removed_from_spec=False,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.JUGGLER_CHECKS_BUNDLE,
        pb_field_name='juggler_checks_bundle',
        local_file_path='juggler-check-bundle-rtc-balancers.tar.gz',
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'BUILD_JUGGLER_CHECKS_BUNDLE'}, {'JUGGLER_CHECKS_BUNDLE'}),
        can_be_removed_from_spec=False,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.GET_WORKERS_PROVIDER,
        pb_field_name='get_workers_provider',
        local_file_path='dump_json_get_workers_provider.lua',
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'NANNY_REMOTE_COPY_RESOURCE'}, {'GET_WORKERS_PROVIDER'}),
        can_be_removed_from_spec=True,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.AWACSLET,
        pb_field_name='awacslet',
        local_file_path='awacslet',
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'YA_MAKE', 'YA_MAKE_2'}, {'AWACSLET_BINARY'}),
        can_be_removed_from_spec=True,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.AWACSLET_GET_WORKERS_PROVIDER,
        pb_field_name='awacslet_get_workers_provider',
        local_file_path='awacslet_get_workers_provider.lua',
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'NANNY_REMOTE_COPY_RESOURCE'}, {'AWACSLET_GET_WORKERS_PROVIDER'}),
        can_be_removed_from_spec=True,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.SHAWSHANK_LAYER,
        pb_field_name='shawshank_layer',
        local_file_path=None,
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'YA_PACKAGE'}, {'SHAWSHANK_LAYER', 'YA_PACKAGE'}),
        can_be_removed_from_spec=True,
        is_layer=True,
        make_nanny_resource_readonly=False,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.BASE_LAYER,
        pb_field_name='base_layer',
        local_file_path=None,
        version_class=BaseLayerVersion,
        sandbox_resource=ComponentSandboxResource({'BUILD_PORTO_LAYER', 'YA_MAKE_TGZ'},
                                                  {'PORTO_LAYER_SEARCH_UBUNTU_PRECISE_APP',
                                                   'PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP',
                                                   'PORTO_LAYER_SEARCH_UBUNTU_BIONIC_APP'}),
        can_be_removed_from_spec=False,
        is_layer=True,
        make_nanny_resource_readonly=False,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.UTILITY_LAYER,
        pb_field_name='utility_layer',
        local_file_path=None,
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'BUILD_PORTO_LAYER'}, {'PORTO_LAYER_BALANCER_MTN'}),
        can_be_removed_from_spec=False,
        is_layer=True,
        make_nanny_resource_readonly=False,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.TANK_LAYER,
        pb_field_name='tank_layer',
        local_file_path=None,
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'BUILD_PORTO_LAYER'}, {'PORTO_LAYER_YANDEX_TANK'}),
        can_be_removed_from_spec=True,
        is_layer=True,
        make_nanny_resource_readonly=False,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.ENDPOINT_ROOT_CERTS,
        pb_field_name='endpoint_root_certs',
        local_file_path='allCAs.pem',
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'MDS_UPLOAD'}, {'AWACS_ENDPOINT_ROOT_CERTS'}),
        can_be_removed_from_spec=True,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE, SourceType.URL_FILE},
    ),

    ComponentConfig(
        component_type=model_pb2.ComponentMeta.PUSHCLIENT,
        pb_field_name='pushclient',
        local_file_path='push-client',
        version_class=SemanticVersion,
        sandbox_resource=ComponentSandboxResource({'BUILD_STATBOX_PUSHCLIENT'}, {'STATBOX_PUSHCLIENT'}),
        can_be_removed_from_spec=True,
        is_layer=False,
        make_nanny_resource_readonly=True,
        possible_source_types={SourceType.SANDBOX_RESOURCE},
    ),
]

LAYER_COMPONENT_TYPES_BY_RESOURCE_TYPE = {}
for component in COMPONENT_CONFIGS:
    if not component.is_layer:
        continue
    for resource_type in component.sandbox_resource.resource_types:
        assert resource_type not in LAYER_COMPONENT_TYPES_BY_RESOURCE_TYPE
        LAYER_COMPONENT_TYPES_BY_RESOURCE_TYPE[resource_type] = component.type

COMPONENT_CONFIGS_BY_TYPE = OrderedDict(
    (component_config.type, component_config)
    for component_config in COMPONENT_CONFIGS
)


def get_component_config(component_type):
    """
    :type component_type: model_pb2.ComponentMeta.Type
    :rtype: ComponentConfig
    :raises KeyError
    """
    return COMPONENT_CONFIGS_BY_TYPE[component_type]


def iter_balancer_components(balancer_components_spec_pb):
    """
    :type balancer_components_spec_pb: model_pb2.BalancerSpec.ComponentsSpec
    :rtype: generator[(model_pb2.ComponentMeta.Type, model_pb2.BalancerSpec.ComponentsSpec.Component)]
    """
    for component_config in COMPONENT_CONFIGS:
        yield component_config, getattr(balancer_components_spec_pb, component_config.pb_field_name)


def iter_changed_balancer_components(prev_components_spec_pb, updated_balancer_components_spec_pb):
    """
    :type prev_components_spec_pb: model_pb2.BalancerSpec.ComponentsSpec
    :type updated_balancer_components_spec_pb: model_pb2.BalancerSpec.ComponentsSpec
    :rtype: generator[(ComponentConfig,
                       model_pb2.BalancerSpec.ComponentsSpec.Component,
                       model_pb2.BalancerSpec.ComponentsSpec.Component)]
    """
    it = zip(iter_balancer_components(prev_components_spec_pb),
             iter_balancer_components(updated_balancer_components_spec_pb))
    for (component_cfg_1, prev_component_pb), (component_cfg_2, updated_component_pb) in it:
        assert component_cfg_1.type == component_cfg_2.type
        if prev_component_pb != updated_component_pb:
            yield component_cfg_1, prev_component_pb, updated_component_pb


def is_set(component_pb):
    return component_pb.state == model_pb2.BalancerSpec.ComponentsSpec.Component.SET


def is_removed(component_pb):
    return component_pb.state == model_pb2.BalancerSpec.ComponentsSpec.Component.REMOVED


def is_pushclient_enabled(component_type, component_version):
    if component_type not in (model_pb2.ComponentMeta.INSTANCECTL_CONF, model_pb2.ComponentMeta.AWACSLET):
        return False
    return component_version.endswith('-pushclient')


def get_pushclient_supervisor_component(components_pb):
    """
    :type components_pb: model_pb2.BalancerSpec.ComponentsSpec
    :rtype: model_pb2.BalancerSpec.ComponentsSpec.Component
    """
    supervisor_component_type = None
    supervisor_component_pb = None
    if is_set(components_pb.instancectl_conf):
        supervisor_component_type = model_pb2.ComponentMeta.INSTANCECTL_CONF
        supervisor_component_pb = components_pb.instancectl_conf
    elif is_set(components_pb.awacslet):
        supervisor_component_type = model_pb2.ComponentMeta.AWACSLET
        supervisor_component_pb = components_pb.awacslet
    return supervisor_component_type, supervisor_component_pb


def find_pushclient_supervisor_component_version(cache, supervisor_component_type, old_version, with_pushclient):
    old_version = semantic_version.Version(old_version)
    component_pbs = cache.list_all_components(
        query={
            cache.ComponentQueryTarget.TYPE_IN: [supervisor_component_type],
            cache.ComponentQueryTarget.STATUS_IN: [model_pb2.ComponentStatus.PUBLISHED]
        },
        sort=[cache.ComponentsSortTarget.VERSION, -1]
    )
    for component_pb in component_pbs:
        enabled = is_pushclient_enabled(supervisor_component_type, component_pb.meta.version)
        if (with_pushclient and enabled) or (not with_pushclient and not enabled):
            version = semantic_version.Version(component_pb.meta.version)
            if old_version.minor == version.minor:
                return component_pb.meta.version


def find_last_component_version(cache, component_type):
    component_pbs = cache.list_all_components(
        query={
            cache.ComponentQueryTarget.TYPE_IN: [component_type],
            cache.ComponentQueryTarget.STATUS_IN: [model_pb2.ComponentStatus.PUBLISHED]
        },
        sort=[cache.ComponentsSortTarget.VERSION, -1]
    )
    return component_pbs[0].meta.version if component_pbs else None
