import logging
import os.path

import inject
import six
import time
import yaml
from sepelib.core import config as appconfig

from awacs.model import cache
from infra.swatlib import metrics
from infra.swatlib.gevent import greenthread


class IControlsUpdater(object):
    @classmethod
    def instance(cls):
        """
        :rtype: ControlsUpdater
        """
        return inject.instance(cls)


class ControlsUpdater(greenthread.GreenThread):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    NOT_SET = object()
    PROCESS_INTERVAL = 15
    RUN_SECTION_CONFIG_FILENAME = u'run_section_config.yml'

    def __init__(self, controls_dir):
        """
        :type controls_dir: six.text_type
        """
        super(ControlsUpdater, self).__init__()

        self._controls_dir = controls_dir
        self._overridden_run_section_values = {}
        self._prev_run_section_config = {}

        registry = metrics.ROOT_REGISTRY.path(u'controls-updater')
        self._success_counter = registry.get_counter(u'success')
        self._failure_counter = registry.get_counter(u'failure')

        self._log = logging.getLogger(u'controls-updater')

    @classmethod
    def _shortened_repr(cls, v):
        if v is cls.NOT_SET:
            return u'NOT SET'
        rv = repr(v)
        if len(rv) > 100:
            rv = rv[:100] + u'...'
        return rv

    @classmethod
    def _get_run_value(cls, key):
        return appconfig.get_value(u'run.' + key, default=cls.NOT_SET)

    @classmethod
    def _set_run_value(cls, key, value):
        appconfig.set_value(u'run.' + key, value)

    @classmethod
    def _remove_run_value(cls, key):
        # we don't have `appconfig.delete_value` method at the moment, let's work around
        c = appconfig.get()  # type: dict
        del c[u'run'][key]

    def _override_run_section_values(self, run_section_config):
        n = 0

        for key, value in six.iteritems(run_section_config):
            curr_value = self._get_run_value(key)
            if curr_value == value:
                continue
            if key not in self._overridden_run_section_values:
                self._overridden_run_section_values[key] = curr_value
            self._set_run_value(key, value)
            self._log.debug(u'started overriding "run.%s": changed its value from %s to %s',
                            key, self._shortened_repr(curr_value), self._shortened_repr(value))
            n += 1

        for key, value in list(six.iteritems(self._overridden_run_section_values)):
            if key in run_section_config:
                continue
            original_value = self._overridden_run_section_values.pop(key)
            curr_value = self._get_run_value(key)
            if curr_value == original_value:
                continue
            if original_value is self.NOT_SET:
                self._remove_run_value(key)
            else:
                self._set_run_value(key, original_value)
            self._log.debug(u'stopped overriding "run.%s": changed its value from %s to %s',
                            key, self._shortened_repr(curr_value), self._shortened_repr(original_value))
            n += 1

        return n

    @staticmethod
    def _find_and_read_run_section_config(path):
        if not os.path.exists(path):
            return False, {}
        with open(path, u'r') as f:
            data = f.read()
        run_section_config = yaml.load(data, Loader=yaml.Loader)
        if not isinstance(run_section_config, dict):
            raise ValueError(u'{}\'s content is not a dict'.format(path))
        return True, run_section_config

    def process(self):
        try:
            run_section_config_path = os.path.join(self._controls_dir, self.RUN_SECTION_CONFIG_FILENAME)
            found, run_section_config = self._find_and_read_run_section_config(run_section_config_path)

            if self._prev_run_section_config == run_section_config:
                pass
            else:
                if not found:
                    self._log.debug(u'%s was not found', run_section_config_path)
                self._log.debug(u'%s has been changed or removed, proceeding', run_section_config_path)

                n = self._override_run_section_values(run_section_config)
                self._log.debug(u'changed %d run section values', n)
                self._prev_run_section_config = run_section_config
        except Exception:
            self._failure_counter.inc()
            self._log.exception(u'unexpected exception')
        else:
            self._success_counter.inc()

    def run(self):
        while 1:
            self.process()
            time.sleep(self.PROCESS_INTERVAL)
