# coding: utf-8
import logging
from datetime import datetime, timedelta

import six
from infra.swatlib.logutil import rndstr
from awacs.lib.context import OpLoggerAdapter
from infra.awacs.proto import model_pb2


class AspectsUpdaterError(Exception):
    pass


class BalancerIsTooLargeError(AspectsUpdaterError):
    pass


class BalancerAspectsUpdater(object):
    _log = logging.getLogger('balancer-aspects-updater')

    def get_aspects_name(self):
        """
        :rtype: str
        """
        raise NotImplementedError

    def prepare(self):
        pass

    def update(self, balancer_pb, aspects_set_content_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
        """
        raise NotImplementedError

    def get_update_interval(self):
        return timedelta(seconds=60 * 30)

    def process(self, balancer_pb, aspects_set_content_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
        """
        if balancer_pb.spec.incomplete:
            return
        namespace_id = balancer_pb.meta.namespace_id
        balancer_id = balancer_pb.meta.id
        flat_full_balancer_id = '{}/{}'.format(namespace_id, balancer_id)

        op_id = rndstr()
        op_log = OpLoggerAdapter(log=self._log, op_id=op_id)
        aspects_name = self.get_aspects_name()
        aspects_pb = getattr(aspects_set_content_pb, aspects_name)
        status_pb = aspects_pb.status  # type: model_pb2.AspectsResolverStatus
        if (
                not status_pb.HasField('last_attempt') or
                status_pb.invalidated or
                status_pb.last_attempt.succeeded.status != 'True' or
                status_pb.last_attempt.finished_at.ToDatetime() + self.get_update_interval() < datetime.utcnow()
        ):
            current_attempt_pb = status_pb.last_attempt
            current_attempt_pb.started_at.GetCurrentTime()
            current_attempt_pb.succeeded.Clear()
            op_log.info('Updating "{}" aspects of balancer "{}"'.format(aspects_name, flat_full_balancer_id))
            try:
                updated = self.update(balancer_pb, aspects_set_content_pb)
            except BalancerIsTooLargeError as e:
                current_attempt_pb.succeeded.status = 'False'
                current_attempt_pb.succeeded.message = six.text_type(e)
            except AspectsUpdaterError as e:
                current_attempt_pb.succeeded.status = 'False'
                current_attempt_pb.succeeded.message = six.text_type(e)
                op_log.warn('Failed to update "%s" aspects of balancer "%s"',
                            aspects_name, flat_full_balancer_id, exc_info=True)
            else:
                current_attempt_pb.succeeded.status = 'True'
                if updated:
                    status_pb.invalidated = False
                op_log.info('Successfully updated "{}" aspects of balancer "{}"'.format(
                    aspects_name, flat_full_balancer_id))
            finally:
                current_attempt_pb.finished_at.GetCurrentTime()

            if current_attempt_pb.succeeded.status == 'True':
                status_pb.last_successful_attempt.CopyFrom(current_attempt_pb)


class NamespaceAspectsUpdater(object):
    _log = logging.getLogger('namespace-aspects-updater')

    def get_aspects_name(self):
        """
        :rtype: str
        """
        raise NotImplementedError

    def prepare(self):
        pass

    def update(self, namespace_pb, aspects_set_content_pb):
        """
        :type namespace_pb: model_pb2.Namespace
        :type aspects_set_content_pb: model_pb2.NamespaceAspectsSetContent
        """
        raise NotImplementedError

    def get_update_interval(self):
        return timedelta(seconds=60 * 30)

    def process(self, namespace_pb, aspects_set_content_pb):
        """
        :type namespace_pb: model_pb2.Namespace
        :type aspects_set_content_pb: model_pb2.NamespaceAspectsSetContent
        """
        namespace_id = namespace_pb.meta.id

        op_id = rndstr()
        op_log = OpLoggerAdapter(log=self._log, op_id=op_id)
        aspects_name = self.get_aspects_name()
        aspects_pb = getattr(aspects_set_content_pb, aspects_name)
        status_pb = aspects_pb.status  # type: model_pb2.AspectsResolverStatus
        if (
                not status_pb.HasField('last_attempt') or
                status_pb.invalidated or
                status_pb.last_attempt.succeeded.status != 'True' or
                status_pb.last_attempt.finished_at.ToDatetime() + self.get_update_interval() < datetime.utcnow()
        ):
            current_attempt_pb = status_pb.last_attempt
            current_attempt_pb.started_at.GetCurrentTime()
            current_attempt_pb.succeeded.Clear()
            op_log.info('Updating "{}" aspects of namespace "{}"'.format(aspects_name, namespace_id))
            try:
                updated = self.update(namespace_pb, aspects_set_content_pb)
            except BalancerIsTooLargeError as e:
                current_attempt_pb.succeeded.status = 'False'
                current_attempt_pb.succeeded.message = six.text_type(e)
            except AspectsUpdaterError as e:
                current_attempt_pb.succeeded.status = 'False'
                current_attempt_pb.succeeded.message = six.text_type(e)
                op_log.warn('Failed to update "%s" aspects of namespace "%s"',
                            aspects_name, namespace_id, exc_info=True)
            else:
                current_attempt_pb.succeeded.status = 'True'
                if updated:
                    status_pb.invalidated = False
                op_log.info('Successfully updated "{}" aspects of namespace "{}"'.format(
                    aspects_name, namespace_id))
            finally:
                current_attempt_pb.finished_at.GetCurrentTime()

            if current_attempt_pb.succeeded.status == 'True':
                status_pb.last_successful_attempt.CopyFrom(current_attempt_pb)
