# coding: utf-8
import inject

from awacs.lib.nannyclient import INannyClient, NannyClient
from awacs.model import cache
from awacs.model.util import clone_pb
from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.model.db import find_balancer_revision_spec
from awacs.model.balancer.vector import BalancerVersion
from infra.awacs.proto import model_pb2
from .base import BalancerAspectsUpdater, AspectsUpdaterError


class ClusterAspectsUpdater(BalancerAspectsUpdater):
    _nanny_client = inject.attr(INannyClient)  # type: NannyClient
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def get_aspects_name(self):
        return 'cluster'

    @staticmethod
    def is_valid_tag_value(value):
        return value not in (None, 'none', 'unknown', '')

    def update(self, balancer_pb, aspects_set_content_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
        """
        namespace_id = balancer_pb.meta.namespace_id
        balancer_id = balancer_pb.meta.id
        full_balancer_id = (namespace_id, balancer_id)

        balancer_state_pb = self._cache.must_get_balancer_state(namespace_id, balancer_id)
        h = L7BalancerStateHandler(balancer_state_pb)
        b = h.select_balancer()
        if not b:
            raise AspectsUpdaterError('Balancer does not have an active revision')
        active_balancer_rev = b.select_active_rev()
        if not active_balancer_rev:
            raise AspectsUpdaterError('Balancer does not have an active revision')
        active_balancer_version = BalancerVersion.from_rev_status_pb(full_balancer_id, active_balancer_rev.pb)

        active_balancer_spec = find_balancer_revision_spec(active_balancer_version)
        assert active_balancer_spec.config_transport.type == model_pb2.NANNY_STATIC_FILE
        active_nanny_service_id = active_balancer_spec.config_transport.nanny_static_file.service_id

        for in_progress_balancer_rev in h.select_balancer().list_in_progress_revs():
            in_progress_balancer_version = BalancerVersion.from_rev_status_pb(
                full_balancer_id, in_progress_balancer_rev.pb)
            in_progress_balancer_spec = find_balancer_revision_spec(in_progress_balancer_version)
            in_progress_nanny_service_id = in_progress_balancer_spec.config_transport.nanny_static_file.service_id
            if in_progress_nanny_service_id != active_nanny_service_id:
                raise AspectsUpdaterError(
                    'Balancer revisions with different Nanny service '
                    'ids ("{}" and "{}") are being activated and active '
                    'at the same time'.format(in_progress_nanny_service_id, active_nanny_service_id))

        try:
            resp_data = self._nanny_client.get_active_revision_id(active_nanny_service_id)
        except Exception as e:
            raise AspectsUpdaterError(
                'Failed to call get_active_revision_id for {}: {}'.format(active_nanny_service_id, e))

        content_pb = aspects_set_content_pb.cluster.content
        prev_content_pb = clone_pb(content_pb)
        content_pb.Clear()

        if balancer_pb.meta.location.type in (model_pb2.BalancerMeta.Location.YP_CLUSTER,
                                              model_pb2.BalancerMeta.Location.AZURE_CLUSTER):
            content_pb.platform = model_pb2.ClusterAspectsContent.YP_LITE
        else:
            content_pb.platform = model_pb2.ClusterAspectsContent.GENCFG

        active_conf_id = resp_data.get('active_revision_id', '')
        if not active_conf_id:
            raise AspectsUpdaterError(
                'Nanny service "{}" does not have an active configuration'.format(active_nanny_service_id))
        content_pb.active_conf_id = active_conf_id

        locations_data = resp_data.get('locations', [])
        content_pb.locations.extend(sorted(locations_data))

        tags_data = resp_data.get('full_orthogonal_tags', {})

        itype_tags_combinations = tags_data.get('itype', [])
        itype_tags = set()
        for itype_tags_combination in itype_tags_combinations:
            for itype_tag in itype_tags_combination:
                if self.is_valid_tag_value(itype_tag):
                    itype_tags.add(itype_tag)
        if itype_tags:
            content_pb.tags.itype.extend(sorted(itype_tags))

        ctype_tags_combinations = tags_data.get('ctype', [])
        ctype_tags = set()
        for ctype_tags_combination in ctype_tags_combinations:
            for ctype_tag in ctype_tags_combination:
                if self.is_valid_tag_value(ctype_tag):
                    ctype_tags.add(ctype_tag)
        if ctype_tags:
            content_pb.tags.ctype.extend(sorted(ctype_tags))

        prj_tags_combinations = tags_data.get('prj', [])
        prj_tags = set()
        for prj_tags_combination in prj_tags_combinations:
            for prj_tag in prj_tags_combination:
                if self.is_valid_tag_value(prj_tag):
                    prj_tags.add(prj_tag)
        if prj_tags:
            content_pb.tags.prj.extend(sorted(prj_tags))

        metaprj_tags_combinations = tags_data.get('metaprj', [])
        metaprj_tags = set()
        for metaprj_tags_combination in metaprj_tags_combinations:
            for metaprj_tag in metaprj_tags_combination:
                if self.is_valid_tag_value(metaprj_tag):
                    metaprj_tags.add(metaprj_tag)
        if metaprj_tags:
            content_pb.tags.metaprj.extend(sorted(metaprj_tags))

        return prev_content_pb != content_pb
