# coding: utf-8
import inject

from awacs.lib.gutils import gevent_idle_iter
from awacs.model import cache
from awacs.model.util import clone_pb
from infra.awacs.proto import model_pb2
from awacs.wrappers.base import Holder, ValidationCtx
from awacs.wrappers.main import Report
from .base import BalancerAspectsUpdater, BalancerIsTooLargeError


class ConfigAspectsUpdater(BalancerAspectsUpdater):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    MAX_UPSTREAMS_NUMBER = 1000

    def get_aspects_name(self):
        return 'config'

    def update(self, balancer_pb, aspects_set_content_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
        """
        namespace_id = balancer_pb.meta.namespace_id
        balancer_id = balancer_pb.meta.id

        namespace_pb = self._cache.must_get_namespace(namespace_id)
        balancer_state_pb = self._cache.must_get_balancer_state(namespace_id, balancer_id)

        content_pb = aspects_set_content_pb.config.content
        prev_content_pb = clone_pb(content_pb)
        content_pb.Clear()

        balancer_pb = clone_pb(balancer_pb)
        holder = Holder(balancer_pb.spec.yandex_balancer.config)

        # hotfix by romanovich@:
        # let's create a fake domain for now
        domain_spec_pb = model_pb2.DomainSpec()
        domain_config_pb = domain_spec_pb.yandex_balancer.config
        domain_config_pb.protocol = domain_config_pb.HTTP_AND_HTTPS
        domain_config_pb.fqdns.append('cron')
        domain_config_pb.cert.id = 'cron'
        domain_config_pb.include_upstreams.ids.append('cron')
        domain_spec_pbs = {('cron', 'fake-domain'): domain_spec_pb}

        spec_pb = model_pb2.CertificateSpec()
        spec_pb.storage.type = model_pb2.CertificateSpec.Storage.YA_VAULT
        spec_pb.storage.ya_vault_secret.secret_id = 'some_secret_yav'
        spec_pb.storage.ya_vault_secret.secret_ver = 'some_ver'
        spec_pb.fields.subject_alternative_names.extend(["yandex.ru"])

        ctx = ValidationCtx.create_ctx_with_config_type_full(
            namespace_pb=namespace_pb,
            full_balancer_id=(namespace_id, balancer_id),
            balancer_spec_pb=balancer_pb.spec,
            knob_spec_pbs={},
            knob_version_hints={},
            cert_spec_pbs={
                (namespace_id, 'cron'): spec_pb,
            },
            weight_section_spec_pbs={},
            upstream_spec_pbs={},
            domain_spec_pbs=domain_spec_pbs,
            disable_gevent_idle=False
        )
        if holder.is_l7_macro():
            holder.expand_immediate_contained_macro(ctx=ctx)

        holder.expand_macroses(ctx=ctx)

        balancer_entry_pb = content_pb.balancer
        for module in holder.walk_chain(visit_branches=True):
            if not isinstance(module, Report) or not module.pb.uuid:
                continue
            balancer_entry_pb.report_uuids.append(module.pb.uuid)

        if len(balancer_state_pb.upstreams) > self.MAX_UPSTREAMS_NUMBER:
            raise BalancerIsTooLargeError(
                'Balancer is too big (number of upstreams is larger than {})'.format(self.MAX_UPSTREAMS_NUMBER))

        for upstream_id in gevent_idle_iter(balancer_state_pb.upstreams):
            upstream_pb = self._cache.must_get_upstream(namespace_id, upstream_id)
            upstream_pb = clone_pb(upstream_pb)
            holder = Holder(upstream_pb.spec.yandex_balancer.config)
            holder.expand_macroses(ctx=ctx)

            upstream_entry_pb = content_pb.upstreams.add(id=upstream_id)
            for module in holder.walk_chain(visit_branches=True):
                if not isinstance(module, Report) or not module.pb.uuid:
                    continue
                upstream_entry_pb.report_uuids.append(module.pb.uuid)

        return prev_content_pb != content_pb
