# coding: utf-8
import logging
import time
from datetime import datetime, timedelta

import gevent.threadpool
import requests
import inject
import google.protobuf.json_format

from infra.swatlib.gevent import exclusiveservice2
from infra.swatlib.logutil import rndstr
from awacs.lib.context import OpLoggerAdapter
from awacs.lib.ctlmanager import UNEXPECTED_EXCEPTIONS
from awacs.model.cron.configupdater import ConfigAspectsUpdater
from infra.awacs.proto import model_pb2
from awacs.model.util import clone_pb
from awacs.model import cache, zk, db
from sepelib.core import config
from .clusterupdater import ClusterAspectsUpdater
from .itsupdater import ItsAspectsUpdater
from .graphupdater import GraphAspectsUpdater
from .namespacestatisticsupdater import NamespaceStatisticsAspectsUpdater
from .notifications_statistics_updater import NotificationsStatisticsUpdater
from .statisticsupdater import UsageStatisticsUpdater
from .rpsstatisticsupdater import RpsStatisticsUpdater
from .webauthrolesidmsyncer import WebauthRolesIdmSyncer
from .base import BalancerAspectsUpdater, NamespaceAspectsUpdater
from .dismissedcleaner import DismissedCleaner


class Cron(object):
    _log = logging.getLogger(__name__)
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _db = inject.attr(db.IMongoStorage)  # type: db.MongoStorage

    PROPAGATION_DELAY = 10
    INTERVAL = 60

    def __init__(self, metrics_registry):
        self.metrics_registry = metrics_registry
        self.iteration_timer = self.metrics_registry.get_gauge('iteration-timer').timer()

    def _run(self):
        op_id = rndstr()
        op_log = OpLoggerAdapter(log=self._log, op_id=op_id)

        op_log.info('Preparing cluster aspects updater...')
        cluster_aspects_updater = ClusterAspectsUpdater()
        cluster_aspects_updater.prepare()

        op_log.info('Preparing ITS aspects updater...')
        its_aspects_updater = ItsAspectsUpdater()
        its_aspects_updater.prepare()

        op_log.info('Preparing config aspects updater...')
        config_aspects_updater = ConfigAspectsUpdater()
        config_aspects_updater.prepare()

        balancer_aspects_updaters = [
            cluster_aspects_updater,
            its_aspects_updater,
            config_aspects_updater,
        ]  # type: list[BalancerAspectsUpdater]

        op_log.info('Processing all balancers...')
        for namespace_pb in self._cache.list_all_namespaces():
            for balancer_pb in self._cache.list_all_balancers(namespace_id=namespace_pb.meta.id):
                if balancer_pb.spec.incomplete:
                    continue
                namespace_id = balancer_pb.meta.namespace_id
                balancer_id = balancer_pb.meta.id
                aspects_set_pb = self._cache.must_get_balancer_aspects_set(namespace_id, balancer_id)
                op_log.info('Processing balancer "{}/{}"'.format(namespace_id, balancer_id))

                updated_content_pb = clone_pb(aspects_set_pb.content)
                try:
                    for balancer_aspects_updater in balancer_aspects_updaters:
                        balancer_aspects_updater.process(balancer_pb, updated_content_pb)
                except UNEXPECTED_EXCEPTIONS:
                    op_log.exception('Failed to process balancer "%s/%s"', namespace_id, balancer_id)
                else:
                    if aspects_set_pb.content != updated_content_pb:
                        for pb in self._zk.update_balancer_aspects_set(namespace_id, balancer_id):
                            pb.content.CopyFrom(updated_content_pb)
                    op_log.info('Processed balancer "{}/{}"'.format(namespace_id, balancer_id))

        op_log.info('Processed all balancers.')

        op_log.info('Processing all namespaces...')

        namespace_aspects_updaters = [
            GraphAspectsUpdater(),
            NamespaceStatisticsAspectsUpdater()
        ]  # type: list[NamespaceAspectsUpdater]

        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            aspects_set_pb = self._cache.get_namespace_aspects_set(namespace_id)
            if aspects_set_pb is None:
                aspects_set_pb = model_pb2.NamespaceAspectsSet()
                aspects_set_pb.meta.namespace_id = namespace_id
                aspects_set_pb.meta.ctime.GetCurrentTime()
                self._zk.create_namespace_aspects_set(namespace_id, aspects_set_pb)
            op_log.info('Processing namespace "{}"'.format(namespace_id))

            updated_content_pb = clone_pb(aspects_set_pb.content)
            try:
                for namespace_aspects_updater in namespace_aspects_updaters:
                    namespace_aspects_updater.process(namespace_pb, updated_content_pb)
            except UNEXPECTED_EXCEPTIONS:
                op_log.exception('Failed to process namespace "%s"', namespace_id)
            else:
                if aspects_set_pb.content != updated_content_pb:
                    for pb in self._zk.update_namespace_aspects_set(namespace_id):
                        pb.content.CopyFrom(updated_content_pb)
                op_log.info('Processed namespace "{}"'.format(namespace_id))
        op_log.info('Processed all namespaces.')

        time.sleep(self.PROPAGATION_DELAY)  # let everything be propagated to cache
        utc_today_midnight = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)

        op_log.info('Processing statistics')
        entry_pb = self._db.get_usage_statistics_entry(span='DATE', start=utc_today_midnight)
        if entry_pb is None:
            entry_pb = model_pb2.UsageStatisticsEntry(span=model_pb2.UsageStatisticsEntry.DATE)
            entry_pb.start.FromDatetime(utc_today_midnight)
            try:
                UsageStatisticsUpdater().update_date_statistics(entry_pb.date_statistics)
                self._db.save_usage_statistics_entry(entry_pb)
            except UNEXPECTED_EXCEPTIONS:
                op_log.exception('Failed to process statistics')
            else:
                op_log.info('Successfully processed statistics')
        else:
            op_log.info("Today's statistics has already been processed")

        op_log.info('Sync webauth roles to IDM...')
        try:
            WebauthRolesIdmSyncer(self.metrics_registry, op_log).process()
        except UNEXPECTED_EXCEPTIONS:
            op_log.exception('Failed to sync webauth roles to IDM')
        else:
            op_log.info("Successfully synced webauth roles to IDM")

        op_log.info('Cleaning dismissed workers from alerting notification rules in namespaces')
        try:
            DismissedCleaner(self.metrics_registry, op_log).process()
        except UNEXPECTED_EXCEPTIONS:
            op_log.exception('Failed to clean dismissed workers')
        else:
            op_log.info("Dismissed workers were successfully cleaned")

        utc_yesterday_midnight = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0) - timedelta(days=1)
        self._collect_notifications_statistics(op_log, utc_yesterday_midnight)

        op_log.info('Processing load statistics')
        entry_pb = self._db.get_load_statistics_entry(span='DATE', start=utc_today_midnight)
        if entry_pb is None or config.get_value('run.force_cron_load_statistics_resync', False):
            if entry_pb is not None:
                self._db.remove_load_statistics_entry(span='DATE', start=utc_today_midnight)
            entry_pb = model_pb2.LoadStatisticsEntry(span=model_pb2.LoadStatisticsEntry.DATE)
            entry_pb.start.FromDatetime(utc_today_midnight)
            try:
                RpsStatisticsUpdater().update_statistics(entry_pb.date_statistics.by_balancer,
                                                         entry_pb.date_statistics.by_namespace,
                                                         entry_pb.date_statistics.by_upstream)
                self._db.save_load_statistics_entry(entry_pb)
            except UNEXPECTED_EXCEPTIONS:
                op_log.exception('Failed to process load statistics')
            else:
                op_log.info('Successfully processed load statistics')
        else:
            op_log.info("Today's load statistics has already been processed")

    def _collect_notifications_statistics(self, op_log, utc_yesterday_midnight):
        pool = gevent.threadpool.ThreadPool(1)
        yesterday_str = utc_yesterday_midnight.strftime('%Y-%m-%dT%H:%M:%SZ')
        op_log.info("Processing notifications statistics for %s", yesterday_str)
        entry_pb = self._db.get_notifications_statistics_entry(span='DATE', start=utc_yesterday_midnight)
        if entry_pb is None:
            entry_pb = model_pb2.NotificationsStatisticsEntry(span=model_pb2.NotificationsStatisticsEntry.DATE)
            entry_pb.start.FromDatetime(utc_yesterday_midnight)
            try:
                job = pool.spawn(lambda: NotificationsStatisticsUpdater().update_date_notifications_statistics(
                    utc_yesterday_midnight,
                    entry_pb.date_statistics,
                ))
                while not job.ready():
                    op_log.debug("waiting for query to complete")
                    gevent.sleep(3)

                job.get()
                self._db.save_notifications_statistics_entry(entry_pb)
            except UNEXPECTED_EXCEPTIONS:
                op_log.exception('Failed to process notification statistics for %s', yesterday_str)
            else:
                op_log.info('Successfully processed notifications statistics for %s', yesterday_str)
        else:
            op_log.info("Notifications statistics for %s has already been processed", yesterday_str)

    def run(self):
        while 1:
            try:
                with self.iteration_timer:
                    self._run()
            except UNEXPECTED_EXCEPTIONS:
                self._log.exception('Unexpected exception')
                raise
            else:
                self._log.debug('Sleeping for {} seconds...'.format(self.INTERVAL))
                time.sleep(self.INTERVAL)


class DummyCron(object):
    """
    A dummy cron implementation for development purposes.
    Can be run locally to copy balancer aspects set from production awacs.
    """
    _log = logging.getLogger(__name__)
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage

    INTERVAL = 60
    TOKEN = 'AQAD-XXX'

    @staticmethod
    def jsondict_to_pb(d, message):
        p = google.protobuf.json_format._Parser(ignore_unknown_fields=True)
        p.ConvertMessage(d, message)

    def _dummy_process_balancer_aspects_set(self, namespace_id, balancer_id, updated_content_pb):
        """
        Copies aspects set from production awacs.

        :param str namespace_id:
        :param str balancer_id:
        :param model_pb2.BalancerAspectsSetContent updated_content_pb:
        """
        url = 'https://dev-awacs.n.yandex-team.ru/api/GetBalancerAspectsSet/?id={}&namespaceId={}'.format(
            balancer_id, namespace_id)
        resp = requests.get(url, headers={'Authorization': 'OAuth {}'.format(self.TOKEN)})
        if resp.status_code == 404:
            return
        else:
            resp.raise_for_status()
        updated_content_pb.Clear()
        data = resp.json()
        content = data['aspectsSet'].get('content', {})
        self.jsondict_to_pb(content, updated_content_pb)

    def _dummy_process_namespace_aspects_set(self, namespace_id, updated_content_pb):
        """
        Copies aspects set from production awacs.

        :param str namespace_id:
        :param model_pb2.NamespaceAspectsSetContent updated_content_pb:
        """
        url = 'https://dev-awacs.n.yandex-team.ru/api/GetNamespaceAspectsSet/?id={}'.format(
            namespace_id)
        resp = requests.get(url, headers={'Authorization': 'OAuth {}'.format(self.TOKEN)})
        if resp.status_code == 404:
            return
        else:
            resp.raise_for_status()
        updated_content_pb.Clear()
        data = resp.json()
        content = data['aspectsSet'].get('content', {})
        self.jsondict_to_pb(content, updated_content_pb)

    def _run(self):
        op_id = rndstr()
        op_log = OpLoggerAdapter(log=self._log, op_id=op_id)
        op_log.info('Processing all balancers...')
        for balancer_pb in self._cache.list_all_balancers():
            if balancer_pb.spec.incomplete:
                continue
            namespace_id = balancer_pb.meta.namespace_id
            balancer_id = balancer_pb.meta.id
            aspects_set_pb = self._cache.must_get_balancer_aspects_set(namespace_id, balancer_id)
            op_log.info('Processing balancer "{}/{}"'.format(namespace_id, balancer_id))

            updated_content_pb = clone_pb(aspects_set_pb.content)
            updated_content_pb.Clear()
            self._dummy_process_balancer_aspects_set(namespace_id, balancer_id, updated_content_pb)

            if aspects_set_pb.content != updated_content_pb:
                for pb in self._zk.update_balancer_aspects_set(namespace_id, balancer_id):
                    pb.content.CopyFrom(updated_content_pb)
            op_log.info('Processed balancer "{}/{}"'.format(namespace_id, balancer_id))

        op_log.info('Processing all namespaces...')
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            aspects_set_pb = self._cache.get_namespace_aspects_set(namespace_id)
            if aspects_set_pb is None:
                aspects_set_pb = model_pb2.NamespaceAspectsSet()
                aspects_set_pb.meta.namespace_id = namespace_id
                aspects_set_pb.meta.ctime.GetCurrentTime()
                self._zk.create_namespace_aspects_set(namespace_id, aspects_set_pb)
            op_log.info('Processing namespace "{}"'.format(namespace_id))
            updated_content_pb = clone_pb(aspects_set_pb.content)
            updated_content_pb.Clear()
            self._dummy_process_namespace_aspects_set(namespace_id, updated_content_pb)
            if aspects_set_pb.content != updated_content_pb:
                for pb in self._zk.update_namespace_aspects_set(namespace_id):
                    pb.content.CopyFrom(updated_content_pb)
            op_log.info('Processed namespace "{}"'.format(namespace_id))

    def run(self):
        while 1:
            try:
                self._run()
            except UNEXPECTED_EXCEPTIONS:
                self._log.exception('Unexpected exception')
                raise
            else:
                self._log.debug('Sleeping for {} seconds...'.format(self.INTERVAL))
                time.sleep(self.INTERVAL)


class CronExclusiveService(exclusiveservice2.ExclusiveService):
    def __init__(self, coord, metrics_registry):
        super(CronExclusiveService, self).__init__(coord, 'cron', Cron(metrics_registry=metrics_registry),
                                                   restart_policy=exclusiveservice2.RestartPolicy.RESTART_ON_EXCEPTION)
