import inject

from awacs.lib import staffclient
import itertools
import yaml
from sepelib.core import config

from awacs.model import cache, dao, util


class AlertingCleaner(object):
    def __init__(self, ns_pb):
        self.ns_pb = ns_pb
        self.logins = set()
        self.alerting_settings = ns_pb.spec.alerting
        self._grab_logins()
        self.cleaned_downtimers = set()
        self.cleaned_notified = set()

    @property
    def ns_id(self):
        return self.ns_pb.meta.id

    def _get_logins(self, template_kwargs, template_name):
        if template_name == "on_status_change":
            return template_kwargs["login"]
        elif template_name == "phone_escalation":
            return template_kwargs["logins"]
        return None

    def _grab_logins(self):
        if self.alerting_settings.WhichOneof("notify_rules") == "juggler_raw_notify_rules":
            notify_rules = list(itertools.chain(
                self.alerting_settings.juggler_raw_notify_rules.balancer,
                self.alerting_settings.juggler_raw_notify_rules.platform,
            ))
            for rule in notify_rules:
                template_kwargs = yaml.safe_load(rule.template_kwargs)
                rule_logins = self._get_logins(template_kwargs, rule.template_name)
                if rule_logins is None:
                    continue
                # beside logins here can be aka telegram channels identifiers
                self.logins.update([l for l in rule_logins if "svc_" not in l])
        if self.alerting_settings.WhichOneof("downtimers") == "juggler_raw_downtimers":
            self.logins.update(self.alerting_settings.juggler_raw_downtimers.staff_logins)

    def get_cleaned_ns(self, dismissed_logins):
        downtimers = self.alerting_settings.juggler_raw_downtimers.staff_logins
        for i, d in enumerate(downtimers):
            if d in dismissed_logins:
                self.cleaned_downtimers.add(d)
                del downtimers[i]

        notify_rules = self.alerting_settings.juggler_raw_notify_rules
        for rule in itertools.chain(notify_rules.balancer, notify_rules.platform):
            template_kwargs = yaml.safe_load(rule.template_kwargs)
            rule_logins = self._get_logins(template_kwargs, rule.template_name)
            if rule_logins is None:
                continue
            for i, l in enumerate(rule_logins):
                if l in dismissed_logins:
                    self.cleaned_notified.add(l)
                    del rule_logins[i]
            if self.cleaned_notified:
                rule.template_kwargs = yaml.safe_dump(template_kwargs)
        return self.ns_pb

    def are_dissmissed_cleaned(self):
        return bool(self.cleaned_downtimers | self.cleaned_notified)


class DismissedCleaner(object):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _staff_client = inject.attr(staffclient.IStaffClient)  # type: staffclient.StaffClient
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    FORBID_SAVE_THRESHOLD = 5

    def __init__(self, metrics_registry, op_log):
        self.op_log = op_log
        self.metrics_registry = metrics_registry.path('dismissed-cleaner')
        self.clean_nses_count = self.metrics_registry.get_counter('clean-nses-count')
        self.forbid_save_threshold = config.get_value("dismissed_cleaner.forbid_save_threshold", self.FORBID_SAVE_THRESHOLD)

    def process(self):
        alerting_cleaners = []
        for ns_pb in self._cache.list_all_namespaces():
            alerting_cleaners.append(AlertingCleaner(ns_pb=util.clone_pb(ns_pb)))

        logins = set()
        for a in alerting_cleaners:
            logins.update(a.logins)
        dismissed_logins = {p["login"] for p in self._staff_client.filter_dismissed(list(logins))}

        cleaned_nses = []
        for a in alerting_cleaners:
            cleaned_ns_pb = a.get_cleaned_ns(dismissed_logins)
            ns_id = a.ns_id
            for d in a.cleaned_downtimers:
                self.op_log.info("{} was removed from downtimers in {}".format(d, ns_id))
            for n in a.cleaned_notified:
                self.op_log.info("{} was removed from notification rules in {}".format(n, ns_id))
            if a.are_dissmissed_cleaned():
                cleaned_nses.append(cleaned_ns_pb)
        self.clean_nses_count.inc(len(cleaned_nses))
        if len(cleaned_nses) > self.forbid_save_threshold:
            self.op_log.info(
                "Attempt of DismissedCleaner to change ns_count_to_clean={} nses has been rejected. "
                "ns_count_to_clean exceeds forbid_save_threshold={}".format(len(cleaned_nses), self.forbid_save_threshold))
            return
        for ns in cleaned_nses:
            self._dao.update_namespace(
                namespace_id=ns.meta.id,
                version=ns.meta.version,
                login=util.NANNY_ROBOT_LOGIN,
                comment="Clean dismissed workers from alerting",
                updated_spec_pb=ns.spec
            )
