# coding: utf-8
import inject
import blinovmatcher
from collections import defaultdict
import copy
import re

from awacs.lib import itsclient
from awacs.lib.gutils import gevent_idle_iter
from awacs.model.util import clone_pb, yp_cluster_to_balancer_location, gencfg_dc_to_balancer_location
from infra.awacs.proto import model_pb2
from .base import BalancerAspectsUpdater, AspectsUpdaterError
import six


I = lambda name: blinovmatcher.Tag(blinovmatcher.TagType.INSTANCE, name)


class RequireTagStorage(object):
    def __init__(self):
        self._require_tag_indexes = defaultdict(set)
        self._dont_require_tag_indexes = set()

    def add(self, value, index):
        if value is not None:
            self._require_tag_indexes[value].add(index)
        else:
            self._dont_require_tag_indexes.add(index)

    def list_relevant_indexes(self, values):
        rv = copy.copy(self._dont_require_tag_indexes)
        for value in values:
            rv |= self._require_tag_indexes[value]
        return rv


class LocationsStorage(object):
    """
    There are two most common types of filters we have:
    1. ~40% of filters now - By service id(s)
    'f@service_id' OR 'f@service_id1 f@service_id2 f@service_id3'
    2. ~59% of filters now - Сonjunctions
    'I@a_geo_man . I@a_itype_flagsprovider . I@a_prj_shared . [ I@a_ctype_prestable I@a_ctype_prod ]'
    In example tags I@a_geo_man, I@a_itype_flagsprovider and I@a_prj_shared are required for apply.
    ~97% filters of type 2 requires itype, 93% - prj, 92% - ctype, 83% - geo.

    Lets just build some indexes for faster filter matching
    """

    TAG_RE = re.compile('^a_([^_S]+)_(\S+)$')

    def __init__(self):
        """
        We divide filters into three disjoint groups:
        1. By service id - indexes_by_service_id
        2. Conjunctions (which are usually require some tags) - indexes_by_tags_requirements
        3. Other filters - indexes_without_clear_requirements
        """
        self._list = []  # type: list[(str, blinovmatcher.Filter)]
        self._indexes_by_tags_requirements = {tag: RequireTagStorage() for tag in ('itype', 'ctype', 'prj', 'geo')}
        self._indexes_by_service_id = defaultdict(set)
        self._indexes_without_clear_requirements = set()

    @staticmethod
    def _is_federated_tag(expr):
        return isinstance(expr, blinovmatcher.Tag) and expr.type == blinovmatcher.TagType.FEDERATED

    @classmethod
    def _is_federated_tag_list(cls, arr):
        return all(cls._is_federated_tag(expr) for expr in arr)

    def add(self, path, filter_expr):
        parsed = blinovmatcher.parse(filter_expr)
        compiled_filter = blinovmatcher.Filter(parsed)
        self._list.append((path, compiled_filter))

        index = len(self._list) - 1
        if self._is_federated_tag(parsed):
            # 'f@service_id'
            self._indexes_by_service_id[parsed.name].add(index)
        elif isinstance(parsed, blinovmatcher.Or) and self._is_federated_tag_list(parsed.args):
            # 'f@service_id1 f@service_id2 f@service_id3'
            for arg in parsed.args:
                self._indexes_by_service_id[arg.name].add(index)
        elif isinstance(parsed, blinovmatcher.And):
            requirements = {}
            for arg in parsed.args:
                if isinstance(arg, blinovmatcher.Tag) and arg.type == blinovmatcher.TagType.INSTANCE:
                    m = self.TAG_RE.match(arg.name)
                    if m is None:
                        continue
                    name, value = m.groups()
                    requirements[name] = value
            for tag, storage in six.iteritems(self._indexes_by_tags_requirements):
                storage.add(requirements.get(tag), index)
        else:
            self._indexes_without_clear_requirements.add(index)

    def list_relevant_locations(self, service_id, itypes, ctypes, prjs, geos):
        for i in self._indexes_by_service_id[service_id]:
            yield self._list[i]
        for i in self._indexes_without_clear_requirements:
            yield self._list[i]

        for i in (self._indexes_by_tags_requirements['itype'].list_relevant_indexes(itypes) &
                  self._indexes_by_tags_requirements['ctype'].list_relevant_indexes(ctypes) &
                  self._indexes_by_tags_requirements['prj'].list_relevant_indexes(prjs) &
                  self._indexes_by_tags_requirements['geo'].list_relevant_indexes(geos)):
            yield self._list[i]


class ItsAspectsUpdater(BalancerAspectsUpdater):
    _its_client = inject.attr(itsclient.IItsClient)  # type: itsclient.ItsClient

    def __init__(self):
        self.locations = LocationsStorage()

    @classmethod
    def iter_its_config_locations(cls, cfg):
        for name, group in six.iteritems(cfg['groups']):
            if 'groups' in group:
                for path, filter_expr in cls.iter_its_config_locations(group):
                    yield name + '/' + path, filter_expr
            else:
                yield name, group['filter']

    def prepare(self):
        self.locations = LocationsStorage()
        its_config = self._its_client.get_config()
        for path, filter_expr in gevent_idle_iter(self.iter_its_config_locations(its_config['locations'])):
            self.locations.add(path, filter_expr)

    def get_aspects_name(self):
        return 'its'

    @classmethod
    def _list_tags(cls, cluster_aspects_content):
        """
        :type cluster_aspects_content: model_pb2.ClusterAspectsContent
        :rtype: set[str]
        """
        tags = cluster_aspects_content.tags
        rv = set()
        for itype in tags.itype:
            rv.add('a_itype_{}'.format(itype))
        for ctype in tags.ctype:
            rv.add('a_ctype_{}'.format(ctype))
        for prj in tags.prj:
            rv.add('a_prj_{}'.format(prj))
        for metaprj in tags.metaprj:
            rv.add('a_metaprj_{}'.format(metaprj))
        for location in cluster_aspects_content.locations:
            rv.add('a_geo_{}'.format(location))
        return rv

    def update(self, balancer_pb, aspects_set_content_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
        """
        if not aspects_set_content_pb.HasField('cluster'):
            raise AspectsUpdaterError('Balancer does not have "cluster" aspects')
        if aspects_set_content_pb.cluster.status.last_attempt.succeeded.status != 'True':
            raise AspectsUpdaterError('Balancer "cluster" aspects last resolving attempt has not been successful')
        if not aspects_set_content_pb.cluster.content.HasField('tags'):
            raise AspectsUpdaterError('Balancer "cluster" aspects do not contain tags')
        if aspects_set_content_pb.cluster.status.invalidated:
            raise AspectsUpdaterError('Balancer "cluster" aspects are invalidated')

        tags = {I(tag) for tag in self._list_tags(aspects_set_content_pb.cluster.content)}
        if balancer_pb.meta.location.type == balancer_pb.meta.location.YP_CLUSTER:
            dc = yp_cluster_to_balancer_location(balancer_pb.meta.location.yp_cluster.upper())
        elif balancer_pb.meta.location.type == balancer_pb.meta.location.GENCFG_DC:
            dc = gencfg_dc_to_balancer_location(balancer_pb.meta.location.gencfg_dc.upper(), default=None)
        else:
            raise AspectsUpdaterError('Location type is not supported')
        if dc:
            tags.add(I('a_dc_{}'.format(dc.lower())))
        service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        if service_id:
            tags.add(blinovmatcher.Tag(blinovmatcher.TagType.FEDERATED, service_id))

        tags_pb = aspects_set_content_pb.cluster.content.tags
        locations = aspects_set_content_pb.cluster.content.locations

        filtered_location_paths = set()
        for location_path, location_filter in self.locations.list_relevant_locations(service_id, tags_pb.itype,
                                                                                     tags_pb.ctype, tags_pb.prj, locations):
            if location_filter.apply(tags):
                filtered_location_paths.add(location_path)

        content_pb = aspects_set_content_pb.its.content
        prev_content_pb = clone_pb(content_pb)
        content_pb.Clear()

        content_pb.location_paths.extend(sorted(filtered_location_paths))

        return prev_content_pb != content_pb
