# coding: utf-8
import collections
import logging
import time

import inject

from infra.swatlib.logutil import rndstr
from infra.swatlib.gevent import exclusiveservice2
from awacs.lib.context import OpLoggerAdapter
from awacs.wrappers import main
from infra.awacs.proto import model_pb2
from awacs.lib import itsclient
from awacs.model import cache, zk, dao, util
import six


class ItsKnobsWatcher(object):
    _log = logging.getLogger(__name__)

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao
    _its_client = inject.attr(itsclient.IItsClient)  # type: itsclient.ItsClient

    INTERVAL = 3600

    @classmethod
    def get_location(cls, its_cfg, path):
        head = path[0]
        rest = path[1:]
        if head not in its_cfg['groups']:
            return
        else:
            group = its_cfg['groups'][head]

            if rest and 'groups' in group:
                return cls.get_location(group, rest)
            elif not rest and 'groups' not in group:
                return group
            else:
                return None

    @staticmethod
    def _is_location_managed_by_sandbox_task(location_config):
        # not a very reliable check, but will do for now
        return location_config['responsible'] == ['robot-searchmon']

    @staticmethod
    def _normalize_filename(filename):
        if filename.startswith('./'):
            return filename[2:]
        else:
            return filename

    def _create_or_update_its_knob(self, namespace_id, knob_id, type, its_watched_state, shared=False):
        """
        :type namespace_id: str
        :type knob_id: str
        :type type: model_pb2.KnobSpec.Type
        :type its_watched_state: model_pb2.ItsWatchedState
        :type shared: bool
        :rtype: model_pb2.Knob
        """
        knob_meta_pb = model_pb2.KnobMeta(namespace_id=namespace_id, id=knob_id)
        knob_meta_pb.auth.type = model_pb2.Auth.STAFF
        knob_spec_pb = model_pb2.KnobSpec(
            mode=model_pb2.KnobSpec.WATCHED,
            type=type,
            shared=shared)
        knob_spec_pb.its_watched_state.CopyFrom(its_watched_state)

        existing_knob_pb = self._cache.get_knob(namespace_id, knob_id)
        if existing_knob_pb:
            if existing_knob_pb.spec.mode == model_pb2.KnobSpec.MANAGED or existing_knob_pb.spec == knob_spec_pb:
                return existing_knob_pb
            else:
                knob_version = existing_knob_pb.meta.version
                knob_pb = self._dao.update_knob(namespace_id=namespace_id,
                                                knob_id=knob_id,
                                                version=knob_version,
                                                comment='Sync changes from ITS',
                                                login=util.NANNY_ROBOT_LOGIN,
                                                updated_spec_pb=knob_spec_pb)
        else:
            knob_pb = self._dao.create_knob(meta_pb=knob_meta_pb,
                                            spec_pb=knob_spec_pb,
                                            login=util.NANNY_ROBOT_LOGIN,
                                            comment='Initial sync from ITS')

        return knob_pb

    def _list_all_namespace_ids(self):
        namespace_pbs = self._cache.list_all_namespaces()
        return [namespace_pb.meta.id for namespace_pb in namespace_pbs
                if namespace_pb.meta.its_knobs_sync_enabled]

    def _run(self):
        op_id = rndstr()
        op_log = OpLoggerAdapter(log=self._log, op_id=op_id)
        op_log.info('Processing all namespaces...')

        op_log.info('Retrieving ITS config...')
        its_config = self._its_client.get_config()
        op_log.info('Retrieved ITS config')

        ruchka_configs = {}
        for ruchka_config in its_config['ruchkas']:
            ruchka_id = ruchka_config['id']
            ruchka_configs[ruchka_id] = ruchka_config

        def suggest_type(ruchka_id, ruchka_config):
            rv = model_pb2.KnobSpec.ANY
            if ruchka_id in main.DEFAULT_KNOB_TYPES:
                rv = main.DEFAULT_KNOB_TYPES[ruchka_id]
            else:
                if ruchka_config.get('formatter') == 'balancer_weights_formatter':
                    rv = model_pb2.KnobSpec.YB_BACKEND_WEIGHTS
            return rv

        for namespace_id in self._list_all_namespace_ids():
            op_log.debug('Processing namespace %s', namespace_id)
            namespace_balancer_ids = set()

            location_configs = {}
            ruchka_paths_by_balancer_id = collections.defaultdict(set)

            balancer_aspects_set_pbs = self._cache.list_all_balancer_aspects_sets(namespace_id=namespace_id)
            for aspects_set_pb in balancer_aspects_set_pbs:
                balancer_id = aspects_set_pb.meta.balancer_id
                namespace_balancer_ids.add(balancer_id)
                for location_path in aspects_set_pb.content.its.content.location_paths:
                    location_config = self.get_location(its_config['locations'], location_path.split('/'))
                    if not self._is_location_managed_by_sandbox_task(location_config):
                        self._log.warn('Matched location appears to be not managed by Sandbox task: %s',
                                       location_path)
                        continue
                    location_configs[location_path] = location_config
                    for ruchka_id in location_config['ruchkas']:
                        ruchka_paths_by_balancer_id[balancer_id].add(location_path + '/' + ruchka_id)
            op_log.debug('ruchka_paths_by_balancer_id: %s', ruchka_paths_by_balancer_id)

            it = six.itervalues(ruchka_paths_by_balancer_id)
            shared_ruchka_paths = set(next(it, set()))
            for balancer_location_paths in it:
                shared_ruchka_paths &= balancer_location_paths
            op_log.debug('shared_ruchka_paths: %s', shared_ruchka_paths)

            for balancer_location_paths in six.itervalues(ruchka_paths_by_balancer_id):
                balancer_location_paths -= shared_ruchka_paths
            op_log.debug('ruchka_paths_by_balancer_id - shared_ruchka_paths: %s', ruchka_paths_by_balancer_id)

            balancer_ids_by_ruchka_id = collections.defaultdict(set)
            for balancer_id, ruchka_paths in six.iteritems(ruchka_paths_by_balancer_id):
                for ruchka_path in ruchka_paths:
                    location_path, ruchka_id = ruchka_path.rsplit('/', 1)
                    balancer_ids_by_ruchka_id[ruchka_id].add((location_path, balancer_id))
            op_log.debug('balancer_ids_by_ruchka_id: %s', balancer_ids_by_ruchka_id)

            for ruchka_path in shared_ruchka_paths:
                location_path, ruchka_id = ruchka_path.rsplit('/', 1)
                ruchka_config = ruchka_configs[ruchka_id]
                its_watched_state = model_pb2.ItsWatchedState()
                its_watched_state.ruchka_id = ruchka_id
                its_watched_state.filename = self._normalize_filename(ruchka_config['path'])
                for balancer_id in sorted(namespace_balancer_ids):
                    its_watched_state.its_location_paths[balancer_id] = location_path
                type = suggest_type(ruchka_id, ruchka_config)
                knob_pb = self._create_or_update_its_knob(namespace_id=namespace_id,
                                                          knob_id=ruchka_id,
                                                          type=type,
                                                          shared=True,
                                                          its_watched_state=its_watched_state)

            for ruchka_id, balancer_ids in six.iteritems(balancer_ids_by_ruchka_id):
                ruchka_config = ruchka_configs[ruchka_id]
                its_watched_state = model_pb2.ItsWatchedState()
                its_watched_state.ruchka_id = ruchka_id
                its_watched_state.filename = self._normalize_filename(ruchka_config['path'])
                for location_path, balancer_id in sorted(balancer_ids):
                    its_watched_state.its_location_paths[balancer_id] = location_path
                type = suggest_type(ruchka_id, ruchka_config)
                knob_pb = self._create_or_update_its_knob(namespace_id=namespace_id,
                                                          knob_id=ruchka_id,
                                                          type=type,
                                                          its_watched_state=its_watched_state)

            op_log.info('Processed namespace "{}"'.format(namespace_id))
        op_log.info('Processing all namespaces')

    def run(self):
        while 1:
            self._run()
            self._log.debug('Sleeping for {} seconds...'.format(self.INTERVAL))
            time.sleep(self.INTERVAL)


class ItsKnobsWatcherExclusiveService(exclusiveservice2.ExclusiveService):
    def __init__(self, coord):
        super(ItsKnobsWatcherExclusiveService, self).__init__(
            coord, 'knobswatcher', ItsKnobsWatcher(),
            restart_policy=exclusiveservice2.RestartPolicy.RESTART_ON_EXCEPTION)
