# coding: utf-8
import datetime
import random

import inject

from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.lib.nannyclient import INannyClient, NannyClient
from awacs.model import cache
from awacs.model.util import clone_pb
from infra.awacs.proto import model_pb2, modules_pb2
from awacs.wrappers.base import Holder
from awacs.wrappers.main import Balancer2
from .base import NamespaceAspectsUpdater


class Balancer2Counts(object):
    def __init__(self, total, total_gpb, with_dynamic, without_arl, without_arl_gpb, without_arl_and_watermark_gpb):
        self.total = total
        self.total_gpb = total_gpb
        self.with_dynamic = with_dynamic
        self.without_arl = without_arl
        self.without_arl_gpb = without_arl_gpb
        self.without_arl_and_watermark_gpb = without_arl_and_watermark_gpb


class NamespaceStatisticsAspectsUpdater(NamespaceAspectsUpdater):
    _nanny_client = inject.attr(INannyClient)  # type: NannyClient
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    MAX_UPSTREAMS_NUMBER = 1000

    def get_update_interval(self):
        return datetime.timedelta(seconds=60 * 30 + random.randint(0, 15 * 30))

    def get_aspects_name(self):
        return 'statistics'

    def get_platform(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: model_pb2.NamespaceStatisticsAspectsContent.BalancerStatisticsAspects.Platform.*
        """
        assert balancer_pb.spec.config_transport.type == model_pb2.NANNY_STATIC_FILE
        service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        resp_data = self._nanny_client.get_service_runtime_attrs(service_id)
        engine_type = resp_data['content']['engines']['engine_type']
        if engine_type == 'YP_LITE':
            return model_pb2.NamespaceStatisticsAspectsContent.BalancerStatisticsAspects.YP_LITE
        else:
            assert 'ISS' in engine_type
            return model_pb2.NamespaceStatisticsAspectsContent.BalancerStatisticsAspects.GENCFG

    def get_instances_count(self, balancer_pb):
        """
        :type balancer_pb: model_pb2.Balancer
        :rtype: int
        """
        assert balancer_pb.spec.config_transport.type == model_pb2.NANNY_STATIC_FILE
        service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
        resp_data = self._nanny_client.list_current_instances(service_id)
        return len(resp_data['result'])

    @staticmethod
    def list_set_pb_fields(pb, prefix=''):
        return [prefix + desc.name for desc, _ in pb.ListFields()]

    @classmethod
    def list_compat_l7_macro_options(cls, pb):
        """
        :type pb: modules_pb2.L7Macro
        """
        rv = set()
        rv.update(cls.list_set_pb_fields(pb.compat))
        rv.update(cls.list_set_pb_fields(pb.http.compat, prefix='http.'))
        rv.update(cls.list_set_pb_fields(pb.https.compat, prefix='https.'))
        return sorted(rv)

    @classmethod
    def list_compat_l7_upstream_macro_options(cls, pb):
        """
        :type pb: modules_pb2.L7UpstreamMacro
        """
        rv = set()
        rv.update(cls.list_set_pb_fields(pb.compat))
        rv.update(cls.list_set_pb_fields(pb.flat_scheme.compat, prefix='flat_scheme.'))
        rv.update(cls.list_set_pb_fields(pb.flat_scheme.balancer.compat, prefix='flat_scheme.balancer.'))
        rv.update(cls.list_set_pb_fields(pb.flat_scheme.balancer.health_check.compat,
                                         prefix='flat_scheme.balancer.health_check.'))
        rv.update(cls.list_set_pb_fields(pb.by_dc_scheme.compat, prefix='by_dc_scheme.'))
        rv.update(cls.list_set_pb_fields(pb.by_dc_scheme.balancer.compat, prefix='by_dc_scheme.balancer.'))
        rv.update(cls.list_set_pb_fields(pb.by_dc_scheme.balancer.health_check.compat,
                                         prefix='by_dc_scheme.balancer.health_check.'))
        rv.update(cls.list_set_pb_fields(pb.by_dc_scheme.dc_balancer.compat, prefix='by_dc_scheme.dc_balancer.'))
        for i, dc_pb in enumerate(pb.by_dc_scheme.dcs):
            rv.update(cls.list_set_pb_fields(dc_pb.compat, prefix='by_dc_scheme.dcs[{}].'.format(i)))
        return sorted(rv)

    @classmethod
    def count_balancer2_modules(cls, pb):
        """
        :type pb: modules_pb2.Holder
        :rtype: Balancer2Counts
        """
        total = 0
        total_gpb = 0
        without_arl = 0
        without_arl_gpb = 0
        without_arl_and_watermark_gpb = 0
        with_dynamic = 0
        holder = Holder(pb)
        for module in holder.walk_chain(visit_branches=True):
            if not isinstance(module, Balancer2):
                continue
            total += 1
            if module.generated_proxy_backends:
                total_gpb += 1
            if module.dynamic:
                with_dynamic += 1
            if (not module.attempts_rate_limiter and
                    not module.pb.disable_attempts_rate_limiter.value and
                    module.pb.attempts != 1):
                without_arl += 1
                if module.generated_proxy_backends:
                    without_arl_gpb += 1
                    if not (module.balancing_policy and
                            'watermark_policy' in module.balancing_policy.list_policy_kinds()):
                        without_arl_and_watermark_gpb += 1
        return Balancer2Counts(total=total,
                               total_gpb=total_gpb,
                               with_dynamic=with_dynamic,
                               without_arl=without_arl,
                               without_arl_gpb=without_arl_gpb,
                               without_arl_and_watermark_gpb=without_arl_and_watermark_gpb)

    def update(self, namespace_pb, aspects_set_content_pb):
        """
        :type namespace_pb: model_pb2.Namespace
        :type aspects_set_content_pb: model_pb2.NamespaceAspectsSetContent
        """
        namespace_id = namespace_pb.meta.id

        content_pb = aspects_set_content_pb.statistics.content
        prev_content_pb = clone_pb(content_pb)
        content_pb.Clear()
        namespace_includes_domains = False

        for balancer_pb in self._cache.list_all_balancers(namespace_id=namespace_id):
            if balancer_pb.spec.incomplete:
                continue
            balancer_id = balancer_pb.meta.id

            prev_balancer_statistics_pb = None
            for balancer_aspects_set_pb in prev_content_pb.balancers:
                if balancer_aspects_set_pb.id == balancer_id:
                    prev_balancer_statistics_pb = balancer_aspects_set_pb

            balancer_aspects_set_pb = self._cache.must_get_balancer_aspects_set(namespace_id, balancer_id)
            balancer_statistics_pb = content_pb.balancers.add()  # type: model_pb2.NamespaceStatisticsAspectsContent.BalancerStatisticsAspects
            balancer_statistics_pb.id = balancer_id
            if (
                    not prev_balancer_statistics_pb or
                    prev_balancer_statistics_pb.active_conf_id != balancer_aspects_set_pb.content.cluster.content.active_conf_id
            ):
                balancer_statistics_pb.active_conf_id = balancer_aspects_set_pb.content.cluster.content.active_conf_id
                balancer_statistics_pb.platform = self.get_platform(balancer_pb)
                balancer_statistics_pb.instances_count = self.get_instances_count(balancer_pb)
            else:
                balancer_statistics_pb.active_conf_id = prev_balancer_statistics_pb.active_conf_id
                balancer_statistics_pb.platform = prev_balancer_statistics_pb.platform
                balancer_statistics_pb.instances_count = prev_balancer_statistics_pb.instances_count

            balancer_state_pb = self._cache.must_get_balancer_state(namespace_id, balancer_id)
            h = L7BalancerStateHandler(balancer_state_pb)
            for full_backend_id, _ in h.iter_backend_items():
                backend_pb = self._cache.must_get_backend(*full_backend_id)
                if backend_pb.spec.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD:
                    balancer_statistics_pb.sd_backends_count += 1
                elif backend_pb.spec.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS:
                    balancer_statistics_pb.yp_endpoint_sets_backends_count += 1
                elif backend_pb.spec.selector.type == model_pb2.BackendSelector.NANNY_SNAPSHOTS:
                    balancer_statistics_pb.nanny_snapshots_backends_count += 1
                elif backend_pb.spec.selector.type == model_pb2.BackendSelector.GENCFG_GROUPS:
                    balancer_statistics_pb.gencfg_groups_backends_count += 1
                elif backend_pb.spec.selector.type == model_pb2.BackendSelector.MANUAL:
                    balancer_statistics_pb.manual_backends_count += 1

            config_pb = balancer_pb.spec.yandex_balancer.config  # type: modules_pb2.Holder

            balancer2_counts = self.count_balancer2_modules(config_pb)
            balancer_statistics_pb.balancer2_modules_total = balancer2_counts.total
            balancer_statistics_pb.balancer2_modules_total_gpb = balancer2_counts.total_gpb
            balancer_statistics_pb.balancer2_modules_with_dynamic_count = balancer2_counts.with_dynamic
            balancer_statistics_pb.balancer2_modules_without_arl_count = balancer2_counts.without_arl
            balancer_statistics_pb.balancer2_modules_without_arl_gpb_count = balancer2_counts.without_arl_gpb
            balancer_statistics_pb.balancer2_modules_without_arl_and_watermark_gpb_count = balancer2_counts.without_arl_and_watermark_gpb

            if config_pb.HasField('main'):
                balancer_statistics_pb.is_unistat_enabled = config_pb.main.HasField('unistat')
                balancer_statistics_pb.is_thread_mode_enabled = config_pb.main.thread_mode
            elif config_pb.HasField('instance_macro'):
                balancer_statistics_pb.is_unistat_enabled = config_pb.instance_macro.HasField('unistat')
                balancer_statistics_pb.is_thread_mode_enabled = config_pb.instance_macro.thread_mode
            elif config_pb.HasField('l7_macro'):
                l7_macro_pb = config_pb.l7_macro
                balancer_statistics_pb.is_unistat_enabled = not (l7_macro_pb.HasField('compat') and
                                                                 l7_macro_pb.compat.disable_unistat)
                balancer_statistics_pb.is_easy_mode_enabled = True
                balancer_statistics_pb.l7_macro_version = l7_macro_pb.version
                balancer_statistics_pb.compat_options.extend(self.list_compat_l7_macro_options(l7_macro_pb))
                if l7_macro_pb.include_domains:
                    balancer_statistics_pb.includes_domains = True
                    namespace_includes_domains = True
            else:
                raise AssertionError('top-level module is not main, instance_macro or l7_macro')

            balancer_statistics_pb.config_lines_count = balancer_pb.spec.yandex_balancer.yaml.strip().count('\n')

        for upstream_pb in self._cache.list_all_upstreams(namespace_id=namespace_id):
            upstream_statistics_pb = content_pb.upstreams.add()  # type: model_pb2.NamespaceStatisticsAspectsContent.UpstreamStatisticsAspects
            upstream_statistics_pb.id = upstream_pb.meta.id
            upstream_statistics_pb.config_lines_count = upstream_pb.spec.yandex_balancer.yaml.strip().count('\n')
            config_pb = upstream_pb.spec.yandex_balancer.config  # type: modules_pb2.Holder

            balancer2_counts = self.count_balancer2_modules(config_pb)
            upstream_statistics_pb.balancer2_modules_total = balancer2_counts.total
            upstream_statistics_pb.balancer2_modules_total_gpb = balancer2_counts.total_gpb
            upstream_statistics_pb.balancer2_modules_without_arl_count = balancer2_counts.without_arl
            upstream_statistics_pb.balancer2_modules_without_arl_gpb_count = balancer2_counts.without_arl_gpb
            upstream_statistics_pb.balancer2_modules_without_arl_and_watermark_gpb_count = balancer2_counts.without_arl_and_watermark_gpb
            upstream_statistics_pb.balancer2_modules_with_dynamic_count = balancer2_counts.with_dynamic

            if config_pb.HasField('l7_upstream_macro'):
                l7_upstream_macro_pb = config_pb.l7_upstream_macro
                upstream_statistics_pb.is_easy_mode_enabled = True
                upstream_statistics_pb.l7_upstream_macro_version = l7_upstream_macro_pb.version
                upstream_statistics_pb.compat_options.extend(
                    self.list_compat_l7_upstream_macro_options(config_pb.l7_upstream_macro))
                balancer_pb = None
                if l7_upstream_macro_pb.HasField('flat_scheme'):
                    balancer_pb = config_pb.l7_upstream_macro.flat_scheme.balancer
                    balancer2_count = 1
                elif l7_upstream_macro_pb.HasField('by_dc_scheme'):
                    balancer_pb = config_pb.l7_upstream_macro.by_dc_scheme.balancer
                    balancer2_count = len(config_pb.l7_upstream_macro.by_dc_scheme.dcs)
                elif l7_upstream_macro_pb.HasField('static_response'):
                    balancer2_count = 0
                else:
                    raise AssertionError(
                        u'l7_upstream_macro does not have flat_scheme, by_dc_scheme, or static_response')
                upstream_statistics_pb.balancer2_modules_total = balancer2_count
                if balancer_pb and balancer_pb.compat.method == balancer_pb.compat.METHOD_NONE:
                    # everything is dynamic
                    upstream_statistics_pb.balancer2_modules_with_dynamic_count = balancer2_count
                else:
                    # everything is not dynamic
                    upstream_statistics_pb.balancer2_modules_with_dynamic_count = 0
            else:
                upstream_statistics_pb.is_easy_mode_enabled = False

        for backend_pb in self._cache.list_all_backends(namespace_id=namespace_id):
            content_pb.backends.add(id=backend_pb.meta.id,
                                    type=backend_pb.spec.selector.type)

        for cert_pb in self._cache.list_all_certs(namespace_id=namespace_id):
            if cert_pb.spec.incomplete:
                continue
            content_pb.certs.add(id=cert_pb.meta.id,
                                 source=cert_pb.spec.source)

        for domain_pb in self._cache.list_all_domains(namespace_id=namespace_id):  # type: model_pb2.Domain
            if domain_pb.spec.incomplete:
                continue
            content_pb.domains.add(id=domain_pb.meta.id,
                                   protocol=domain_pb.spec.yandex_balancer.config.protocol,
                                   is_included=namespace_includes_domains)

        return prev_content_pb != content_pb
