# encoding: utf-8
from __future__ import unicode_literals

import re
import logging
import inject
import enum

from sepelib.core import config as appconfig
from infra.swatlib import randomutil
from infra.swatlib import orly_client

from awacs.model import cache
from awacs.lib import idmclient
import six


logging.getLogger('chardet.charsetprober').setLevel(logging.INFO)

WEBAUTH_SYSTEM = 'webauth-awacs'
REMOVAL_THRESHOLD = 0.3


class ParseIdmJsonToRoleError(Exception):
    pass


class Node(object):
    def __init__(self, slug_path, op_id=None):
        """
        :type slug: six.text_type
        :type op_id: six.text_type
        """
        self.slug_path = slug_path
        self.op_id = op_id
        self.children = {}  # type: dict[six.text_type, Node]

    def __repr__(self):
        # For debug only
        rv = 'RoleNode slug_path={}'.format(self.slug_path)
        if self.op_id is not None:
            rv += ', op_id={}'.format(self.op_id)
        if self.children:
            rv += ', children='
            for key, rolenode in six.iteritems(self.children):
                rv += '\n{}{}: {}'.format('  ' * self.slug_path.count('/'), key, rolenode)
        return rv

    def list_roles(self):
        rv = set()
        for child in six.itervalues(self.children):
            if isinstance(child, RoleNode):
                rv.add(child.role)
            else:
                rv |= child.list_roles()
        return rv

    def ensure_by_path(self, slugs, op_id=None):
        """
        :type slugs: list[six.text_type]
        :type op_id: six.text_type
        """
        if not slugs:
            return self

        slug = slugs[0]
        if slug not in self.children:
            slug_path = self.slug_path + slug + '/'
            child = Node(slug_path=slug_path, op_id=op_id)
            self.children[slug] = child
        else:
            child = self.children[slug]
        return child.ensure_by_path(slugs[1:], op_id=op_id)

    def add(self, role, op_id=None):
        """
        :type role: WebauthIdmRole
        :type op_id: six.text_type
        :rtype: list[six.text_type] - list of slugs, which need to be added to IDM
        """
        slugs = role.role_slug_path.strip('/').split('/')

        parent = self.ensure_by_path(slugs[:-1], op_id=op_id)
        slug = slugs[-1]
        slug_path = parent.slug_path + slug + '/'
        parent.children[slug] = RoleNode(slug_path=slug_path, role=role, op_id=op_id)

    def list_slug_paths_by_op_id(self, op_id):
        """
        :type op_id: six.text_type
        :rtype: set[six.text_type]
        """
        rv = set()
        for slug, child in six.iteritems(self.children):
            if child.op_id == op_id:
                rv.add(child.slug_path)
            rv |= child.list_slug_paths_by_op_id(op_id)
        return rv

    def revert(self, op_id):
        for slug, child in list(six.iteritems(self.children)):
            if child.op_id == op_id:
                del self.children[slug]
            else:
                child.revert(op_id)


class RoleNode(Node):
    def __init__(self, slug_path, role, op_id=None):
        """
        :type slug: six.text_type
        :type role: WebauthIdmRole
        :type op_id: six.text_type
        """
        self.role = role
        super(RoleNode, self).__init__(slug_path, op_id)


class WebauthIdmRole(object):
    _idm_client = inject.attr(idmclient.IIDMClient)  # type: idmclient.IDMClient

    NODE_NAMES = {
        'installation': ('Installation', u'Инсталляция'),
        'namespace': ('Namespace', u'Неймспейс'),
        'domains': ('Domains', u'Домены'),
        'domain': ('Domain', u'Домен'),
        'role_type': ('Type', u'Тип'),
        'user': ('User', u'Пользователь'),
    }

    NAMESPACE_SLUG_PATH = '/installation/{installation}/namespace/{namespace_id}/'
    DOMAIN_SLUG_PATH = '/installation/{installation}/namespace/{namespace_id}/role_type/domains/domain/{domain_id}/'
    ROLE_SUFFIX = 'role_type/user/'

    NAMESPACE_ROLE_SLUG_PATH_RE = re.compile('^/installation/(?P<installation>[^/]*)/namespace/(?P<namespace_id>[^/]*)/role_type/user/$')
    DOMAIN_ROLE_SLUG_PATH_RE = re.compile('^/installation/(?P<installation>[^/]*)/namespace/(?P<namespace_id>[^/]*)/role_type/domains/domain/(?P<domain_id>[^/]*)/role_type/user/$')

    def __init__(self, op_log, installation, namespace_id, domain_id=None, responsible=None):
        self.op_log = op_log
        self.installation = installation
        self.namespace_id = namespace_id
        self.domain_id = domain_id
        self.responsible = responsible or set()
        if self.domain_id:
            self.slug_path = self.DOMAIN_SLUG_PATH.format(installation=self.installation,
                                                          namespace_id=self.namespace_id,
                                                          domain_id=self.domain_id)
        else:
            self.slug_path = self.NAMESPACE_SLUG_PATH.format(installation=self.installation,
                                                             namespace_id=self.namespace_id)
        self.role_slug_path = self.slug_path + self.ROLE_SUFFIX

    def key(self):
        return (self.installation, self.namespace_id, self.domain_id)

    @classmethod
    def from_idm_json(cls, op_log, idm_json):
        responsible = set(r['username'] for r in idm_json.get('responsibilities', []))
        slug_path = idm_json['slug_path']

        m = cls.NAMESPACE_ROLE_SLUG_PATH_RE.match(slug_path)
        if m is None:
            m = cls.DOMAIN_ROLE_SLUG_PATH_RE.match(slug_path)
            if m is None:
                raise ParseIdmJsonToRoleError()

        d = m.groupdict()
        return cls(op_log, d['installation'], d['namespace_id'], domain_id=d.get('domain_id'), responsible=responsible)

    def __eq__(self, other):
        """
        :type other: WebauthIdmRole or None
        """
        if other is None:
            return False
        return (self.installation == other.installation and
                self.namespace_id == other.namespace_id and
                self.domain_id == other.domain_id and
                self.responsible == other.responsible)

    def __hash__(self):
        return hash(self.key())

    def __ne__(self, other):
        return not self.__eq__(other)

    def __repr__(self):
        # For debug only
        return self.role_slug_path

    def get_responsible(self):
        return [{'username': login, 'notify': True} for login in self.responsible]

    def nodes(self):
        rv = []
        slugs = self.role_slug_path.strip('/').split('/')
        parent = '/'
        for node_slug in slugs:
            ru, en = self.NODE_NAMES.get(node_slug, (node_slug, node_slug))
            rv.append((parent, node_slug, ru, en))
            parent += node_slug + '/'
        return rv

    def add(self, slug_paths_to_be_add):
        """
        :type slug_paths_to_be_add: set[six.text_type]
        """
        subrequests = []
        for node in self.nodes():
            parent, slug, en_name, ru_name = node
            if parent + slug + '/' not in slug_paths_to_be_add:
                continue
            body = {
                'system': WEBAUTH_SYSTEM,
                'parent': parent,
                'slug': slug,
                'name': {'en': en_name, 'ru': ru_name}
            }
            if slug == 'user':
                body['responsibilities'] = self.get_responsible()
            subrequest = {'method': 'POST',
                          'path': 'rolenodes',
                          'body': body}
            subrequests.append(subrequest)
        if not subrequests:
            return

        try:
            self._idm_client.batch(subrequests)
        except idmclient.IDMError as e:
            self.op_log.error('Failed to add role to IDM: {}'.format(getattr(e.resp, 'text', '')))
            return False
        return True

    def update(self):
        try:
            self._idm_client.update_role(WEBAUTH_SYSTEM, self.role_slug_path, responsibilities=self.get_responsible())
        except idmclient.IDMError as e:
            self.op_log.error('Failed to update role in IDM: {}'.format(getattr(e.resp, 'text', '')))
            return False
        return True

    def remove(self):
        try:
            self._idm_client.remove_role(WEBAUTH_SYSTEM, self.slug_path)
        except idmclient.IDMError as e:
            if getattr(e.resp, 'status_code', 0) in (204, 404):
                return True
            self.op_log.error('Failed to remove role from IDM: {}'.format(getattr(e.resp, 'text', '')))
            return False
        return True


class WebauthRolesIdmSyncer(object):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _idm_client = inject.attr(idmclient.IIDMClient)  # type: idmclient.IDMClient

    WEBAUTH_SYSTEM = 'webauth-awacs'
    ENABLE_DOMAINS_SYNCING = False

    def __init__(self, metrics_registry, op_log):
        """
        :type metrics_registry: swatlib.metrics.Registry
        """
        self.metrics_registry = metrics_registry.path('webauth-syncer')
        self._orly_brake = orly_client.OrlyBrake(
            rule='awacs-webauth-idm-roles-removal',
            metrics_registry=self.metrics_registry
        )

        self.update_role_counter = self.metrics_registry.get_counter('update-role')
        self.remove_role_counter = self.metrics_registry.get_counter('remove-role')
        self.add_role_counter = self.metrics_registry.get_counter('add-role')
        self.update_role_errors_counter = self.metrics_registry.get_counter('update-role-errors')
        self.remove_role_errors_counter = self.metrics_registry.get_counter('remove-role-errors')
        self.add_role_errors_counter = self.metrics_registry.get_counter('add-role-errors')

        self.iteration_counter = self.metrics_registry.get_counter('iterations-count')
        self.success_iteration_counter = self.metrics_registry.get_counter('success-iterations-count')
        self.op_log = op_log

    def get_roles_from_awacs(self, installation_name):
        roles = []
        for namespace_pb in self._cache.list_all_namespaces():
            responsible = set(namespace_pb.meta.webauth.responsible.logins)
            if not responsible:
                continue
            roles.append(WebauthIdmRole(self.op_log, installation_name, namespace_pb.meta.id, responsible=responsible))
            if not self.ENABLE_DOMAINS_SYNCING:
                continue
            for domain_pb in self._cache.list_all_domains(namespace_pb.meta.id):
                roles.append(WebauthIdmRole(self.op_log, installation_name, namespace_pb.meta.id, domain_id=domain_pb.meta.id,
                                            responsible=responsible))
        return set(roles)

    def process(self):
        self.iteration_counter.inc()
        root = Node(slug_path='/')
        for node in self._idm_client.iterate_system_roles(WEBAUTH_SYSTEM):
            slugs = node['slug_path'].strip('/').split('/') if node['slug_path'].strip('/') else []
            try:
                role = WebauthIdmRole.from_idm_json(self.op_log, node)
            except ParseIdmJsonToRoleError:
                root.ensure_by_path(slugs=slugs)
            else:
                root.add(role)

        idm_roles = root.list_roles()

        installation_name = appconfig.get_value('run.installation')
        if installation_name == 'awacs' and appconfig.get_value('run.debug', False):
            # Extra check to avoid production IDM roles changes from local/dev awacs installation
            raise AssertionError('"run.installation" can not be set to "awacs" if "run.debug" is True')
        awacs_roles = self.get_roles_from_awacs(installation_name)
        awacs_roles_dict = {role.key(): role for role in awacs_roles}
        idm_roles_dict = {role.key(): role for role in idm_roles}

        to_remove = []
        idm_roles_count_in_installation = 0
        for role in idm_roles:
            if role.installation != installation_name:
                continue
            idm_roles_count_in_installation += 1
            if role.key() not in awacs_roles_dict:
                to_remove.append(role)
        if len(to_remove) >= idm_roles_count_in_installation * REMOVAL_THRESHOLD:
            # One more check to avoid removal too much roles from IDM
            self.op_log.exception('An attempt to remove more than {}% roles from IDM - will not remove anything'
                             .format(int(REMOVAL_THRESHOLD * 100)))
            to_remove = []

        to_add = []
        to_update = []
        for role in awacs_roles:
            if role.domain_id and not self.ENABLE_DOMAINS_SYNCING:
                continue
            if role.key() not in idm_roles_dict:
                to_add.append(role)
            elif role != idm_roles_dict[role.key()]:
                to_update.append(role)

        for role in to_update:
            self.op_log.info('Updating {}'.format(role))
            ok = role.update()
            if not ok:
                self.update_role_errors_counter.inc()
            self.update_role_counter.inc()
        for role in to_add:
            self.op_log.info('Adding {}'.format(role))
            op_id = randomutil.gen_random_str(bits=96)
            root.add(role, op_id=op_id)
            ok = role.add(slug_paths_to_be_add=root.list_slug_paths_by_op_id(op_id))
            if not ok:
                self.add_role_errors_counter.inc()
                root.revert(op_id)
            self.add_role_counter.inc()
        for role in to_remove:
            self.op_log.info('Removing {}'.format(role))
            try:
                self._orly_brake.maybe_apply(op_id=role.slug_path, op_log=self.op_log)
            except:
                continue
            ok = role.remove()
            if not ok:
                self.remove_role_errors_counter.inc()
            self.remove_role_counter.inc()

        self.success_iteration_counter.inc()
