# coding: utf-8
import uuid

import inject
import six
from datetime import datetime
from sepelib.core import config as appconfig

from awacs.lib.order_processor.model import is_order_in_progress
from awacs.model import errors, util, objects
from infra.awacs.proto import model_pb2


INFRA_NAMESPACE_ID = 'infra'
DEFAULT_NAME_SERVERS = {
    'in.yandex.net': (model_pb2.NameServerSpec.YP_DNS, model_pb2.NameServerSpec.ANY),
    'in.yandex-team.ru': (model_pb2.NameServerSpec.YP_DNS, model_pb2.NameServerSpec.ANY),
    'on.yandex.net': (model_pb2.NameServerSpec.YP_DNS, model_pb2.NameServerSpec.L3_BALANCERS_ONLY),
    'on.yandex-team.ru': (model_pb2.NameServerSpec.YP_DNS, model_pb2.NameServerSpec.L3_BALANCERS_ONLY),
    'yandex-team.ru': (model_pb2.NameServerSpec.DNS_MANAGER, model_pb2.NameServerSpec.ANY),
    'rtc.yandex.net': (model_pb2.NameServerSpec.AWACS_MANAGED, model_pb2.NameServerSpec.L3_BALANCERS_ONLY),
    'rtc.yandex-team.ru': (model_pb2.NameServerSpec.AWACS_MANAGED, model_pb2.NameServerSpec.L3_BALANCERS_ONLY),
}


class IDao(object):
    @classmethod
    def instance(cls):
        """
        :rtype: Dao
        """
        return inject.instance(cls)


class Dao(object):
    def __init__(self, zk, db, cache):
        """
        :type zk: awacs.model.zk.ZkStorage
        :type db: awacs.model.db.MongoStorage
        :type cache: awacs.model.cache.AwacsCache
        """
        self.zk = zk
        self.db = db
        self.cache = cache

    @classmethod
    def generate_version_id(cls):
        return six.text_type(uuid.uuid4())

    def draft_component(self, component_type, version, spec_pb, login, message='', startrek_issue_key=None):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :type spec_pb: model_pb2.Component.Spec
        :type login: six.text_type
        :type message: six.text_type
        :type startrek_issue_key: Optional[six.text_type]
        :rtype: model_pb2.Component
        """
        component_pb = model_pb2.Component(spec=spec_pb)
        component_pb.meta.type = component_type
        component_pb.meta.version = version
        component_pb.status.drafted.author = login
        component_pb.status.drafted.at.GetCurrentTime()
        component_pb.status.drafted.message = message
        if startrek_issue_key:
            component_pb.status.drafted.startrek_issue_key = startrek_issue_key

        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        try:
            self.zk.draft_component(component_type_str, version, component_pb)
        except errors.ConflictError:
            raise errors.ConflictError(u'Component "{}" of version "{}" already exists'.format(component_type_str,
                                                                                               version))
        return component_pb

    def publish_component(self, component_type, version, login, message='', startrek_issue_key=None):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :type startrek_issue_key: Optional[six.text_type]
        :type login: six.text_type
        :type message: six.text_type
        :rtype: model_pb2.Component
        """
        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        component_pb = None
        utcnow = datetime.utcnow()
        for component_pb in self.zk.update_component(component_type_str, version):
            component_pb.status.status = component_pb.status.PUBLISHED
            component_pb.status.published.author = login
            component_pb.status.published.at.FromDatetime(utcnow)
            component_pb.status.published.message = message
            if startrek_issue_key:
                component_pb.status.published.startrek_issue_key = startrek_issue_key
        return component_pb

    def retire_component(self, component_type, version, superseded_by, login, message=''):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :type superseded_by: six.text_type
        :type login: six.text_type
        :type message: six.text_type
        :rtype: model_pb2.Component
        """
        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        component_pb = None
        utcnow = datetime.utcnow()
        for component_pb in self.zk.update_component(component_type_str, version):
            component_pb.status.status = component_pb.status.RETIRED
            component_pb.status.retired.author = login
            component_pb.status.retired.at.FromDatetime(utcnow)
            component_pb.status.retired.superseded_by = superseded_by
            component_pb.status.retired.message = message
        return component_pb

    def _unset_component_version_as_default(self, component_type, version, cluster):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :type cluster: six.text_type
        """
        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        for component_pb in self.zk.update_component(component_type_str, version):
            component_pb.status.published.marked_as_default.pop(cluster)

    def set_component_version_as_default(self, component_type, version, cluster, login, message=''):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :type cluster: six.text_type
        :type login: six.text_type
        :type message: six.text_type
        """
        default_versions = self.cache.list_component_default_versions(component_type)
        other_versions = [other_version for other_version in default_versions[cluster] if other_version != version]
        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        utcnow = datetime.utcnow()
        for component_pb in self.zk.update_component(component_type_str, version):
            assert component_pb.status.status == component_pb.status.PUBLISHED
            component_pb.status.published.marked_as_default[cluster].author = login
            component_pb.status.published.marked_as_default[cluster].at.FromDatetime(utcnow)
            component_pb.status.published.marked_as_default[cluster].message = message
        for other_version in other_versions:
            self._unset_component_version_as_default(component_type, other_version, cluster)

    def delete_component(self, component_type, version):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        """
        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        self.zk.remove_component(component_type_str, version)

    def create_namespace(self, meta_pb, login, spec_pb=None, order_content_pb=None, project='unknown', dry_run=False):
        """
        :type meta_pb: model_pb2.NamespaceMeta
        :type spec_pb: model_pb2.NamespaceSpec | None
        :type order_content_pb: model_pb2.NamespaceOrder.Content | None
        :type login: six.text_type
        :type project: Optional[six.text_type]
        :type dry_run: bool
        :rtype: model_pb2.Namespace
        """
        namespace_id = meta_pb.id

        utcnow = datetime.utcnow()
        namespace_pb = model_pb2.Namespace(meta=meta_pb)
        namespace_pb.meta.mtime.FromDatetime(utcnow)
        namespace_pb.meta.ctime.FromDatetime(utcnow)
        namespace_pb.meta.author = login
        version = namespace_pb.meta.version = self.generate_version_id()

        owners = namespace_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)

        namespace_pb.meta.annotations['project'] = project
        namespace_pb.meta.annotations['creator'] = login

        if order_content_pb and spec_pb:
            raise AssertionError(u'both order_pb and spec_pb specified')

        preset = model_pb2.NamespaceSpec.PR_UNKNOWN

        if order_content_pb is not None:
            namespace_pb.order.content.CopyFrom(order_content_pb)
            namespace_pb.order.status.status = u'CREATED'
            namespace_pb.order.status.message = u'Order is created and going to be processed soon.'
            namespace_pb.order.status.last_transition_time.FromDatetime(utcnow)
            namespace_pb.spec.incomplete = True
            namespace_pb.spec.env_type = namespace_pb.order.content.env_type
        elif spec_pb is not None:
            namespace_pb.spec.CopyFrom(spec_pb)
            preset = spec_pb.preset

        if preset == model_pb2.NamespaceSpec.PR_UNKNOWN:
            preset_map = appconfig.get_value(u'alerting.project_presets', {})
            preset = model_pb2.NamespaceSpec.NamespaceSettingsPreset.Value(preset_map.get(project, 'PR_DEFAULT'))

        namespace_pb.spec.preset = preset
        namespace_pb.spec.balancer_constraints.instance_tags.prj = namespace_pb.order.content.instance_tags.prj or namespace_id

        if dry_run:
            return namespace_pb
        rev_meta_pb = model_pb2.NamespaceRevisionMeta(
            id=version,
            namespace_id=namespace_id,
            author=login,
            comment=namespace_pb.meta.comment
        )
        rev_meta_pb.ctime.FromDatetime(utcnow)
        rev_pb = model_pb2.NamespaceRevision(meta=rev_meta_pb, spec=namespace_pb.spec)
        self.db.save_namespace_rev(rev_pb)

        try:
            self.zk.create_namespace(namespace_id, namespace_pb)
        except errors.ConflictError as e:
            self.db.remove_namespace_rev(namespace_pb.meta.version)
            raise errors.ConflictError(u'Namespace "{}" already exists'.format(namespace_id))

        return namespace_pb

    def update_namespace(self, namespace_id,
                         version=None,
                         comment=None,
                         login=None,
                         updated_auth_pb=None,
                         updated_category=None,
                         updated_webauth_pb=None,
                         updated_observers_pb=None,
                         updated_abc_service_id=None,
                         updated_its_knobs_sync_enabled=None,
                         updated_spec_pb=None,
                         updated_alerting_sync_status_pb=None,
                         updated_its_sync_status_pb=None,
                         updated_annotations=None):
        """
        :type namespace_id: six.text_type
        :type version: six.text_type | None
        :type comment: six.text_type | None
        :type login: six.text_type | None
        :type updated_auth_pb: model_pb2.Auth
        :type updated_category: six.text_type
        :type updated_webauth_pb: model_pb2.WebauthSettings | None
        :type updated_observers_pb: model_pb2.Observers | None
        :type updated_abc_service_id: int
        :type updated_its_knobs_sync_enabled: bool
        :type updated_spec_pb: model_pb2.NamespaceSpec
        :type updated_alerting_sync_status_pb: model_pb2.SyncStatus
        :type updated_its_sync_status_pb: model_pb2.SyncStatus
        :type updated_annotations: Optional[Mapping[Text, Text]]
        :rtype: model_pb2.Namespace
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert (updated_auth_pb
                or updated_category
                or updated_webauth_pb
                or updated_observers_pb
                or updated_abc_service_id
                or updated_its_knobs_sync_enabled
                or updated_spec_pb
                or updated_alerting_sync_status_pb
                or updated_its_sync_status_pb
                or updated_annotations is not None
                )
        assert not updated_spec_pb or (version is not None and
                                       comment is not None and
                                       login is not None)
        utcnow = datetime.utcnow()
        new_version = self.generate_version_id()

        if updated_spec_pb:
            rev_meta_pb = model_pb2.NamespaceRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.NamespaceRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_namespace_rev(rev_pb)

        namespace_pb = None
        for namespace_pb in self.zk.update_namespace(namespace_id):
            if updated_spec_pb is not None and namespace_pb.spec != updated_spec_pb:
                if namespace_pb.meta.version != version:
                    self.db.remove_namespace_rev(new_version)
                    raise errors.ConflictError(
                        u'Namespace modification conflict: assumed version="{}", current="{}"'.format(
                            version, namespace_pb.meta.version))

                if namespace_pb.spec.deleted:
                    raise errors.ConflictError('Namespace with "spec.deleted" flag can not be updated')

                namespace_pb.spec.CopyFrom(updated_spec_pb)
                namespace_pb.meta.version = new_version
                namespace_pb.meta.comment = comment
                namespace_pb.meta.author = login
                namespace_pb.meta.mtime.FromDatetime(utcnow)

            if updated_auth_pb is not None:
                namespace_pb.meta.auth.CopyFrom(updated_auth_pb)
            if updated_category is not None:
                namespace_pb.meta.category = updated_category
            if updated_webauth_pb is not None:
                namespace_pb.meta.webauth.CopyFrom(updated_webauth_pb)
            if updated_observers_pb is not None:
                namespace_pb.meta.observers.CopyFrom(updated_observers_pb)
            if updated_abc_service_id:
                namespace_pb.meta.abc_service_id = updated_abc_service_id
            if updated_its_knobs_sync_enabled:
                namespace_pb.meta.its_knobs_sync_enabled = updated_its_knobs_sync_enabled
            if updated_alerting_sync_status_pb is not None:
                namespace_pb.alerting_sync_status.CopyFrom(updated_alerting_sync_status_pb)
            if updated_its_sync_status_pb is not None:
                namespace_pb.its_sync_status.CopyFrom(updated_its_sync_status_pb)
            if updated_annotations is not None:
                namespace_pb.meta.annotations.clear()
                namespace_pb.meta.annotations.update(updated_annotations)

        return namespace_pb

    def can_delete_namespace(self, namespace_pb):
        """

        :type namespace_pb: model_pb2.Namespace
        :rtype: tuple[bool, six.text_type | None]
        """
        if is_order_in_progress(namespace_pb):
            return False, u'In-progress namespace with incomplete order can not be removed.'

        self.cache.sync()

        for object_name, count_func in (
                ('balancer', self.cache.count_balancers),
                ('L3 balancer', self.cache.count_l3_balancers),
                ('domain', self.cache.count_domains),
                ('certificate', self.cache.count_certs),
                ('DNS record', self.cache.count_dns_records),
                ('L7Heavy config', objects.L7HeavyConfig.cache.count),
        ):
            singletons_count = count_func(namespace_pb.meta.id)
            if singletons_count > 0:
                return (False, u'Deleting this namespace is not possible: it has {0} {1}(s). '
                               u'Please remove all {1}s before removing the namespace.'.format(singletons_count,
                                                                                               object_name))

        backend_pbs = self.cache.list_all_backends(namespace_pb.meta.id)
        for backend_pb in backend_pbs:
            full_balancer_ids = self.cache.list_full_balancer_ids_for_backend(backend_pb.meta.namespace_id,
                                                                              backend_pb.meta.id)
            other_namespace_full_balancer_ids = {full_balancer_id for full_balancer_id in full_balancer_ids
                                                 if full_balancer_id[0] != namespace_pb.meta.id}
            if other_namespace_full_balancer_ids:
                msg_list = ['/'.join(full_balancer_id) for full_balancer_id in other_namespace_full_balancer_ids]
                msg = (
                    u'Deleting this namespace is not possible. '
                    u'Its backend "{}:{}" is used in other namespaces by the following balancers: "{}".'
                ).format(backend_pb.meta.namespace_id, backend_pb.meta.id, '", "'.join(msg_list))
                return False, msg
        return True, None

    def delete_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        """
        # DNS records, certs & renewals, L3 balancers, L7 balancers, and domains are manually removed by user

        # upstreams
        self.zk.remove_namespace_upstreams(namespace_id)
        self.db.remove_upstream_revs_by_namespace_id(namespace_id)

        # backends
        self.zk.remove_namespace_backends(namespace_id)
        self.db.remove_backend_revs_by_namespace_id(namespace_id)

        # endpoint sets
        self.zk.remove_namespace_endpoint_sets(namespace_id)
        self.db.remove_endpoint_set_revs_by_namespace_id(namespace_id)

        # knobs
        self.zk.remove_namespace_knobs(namespace_id)

        # namespace aux
        self.zk.remove_namespace_aspects_set(namespace_id)
        self.db.remove_namespace_revs_by_namespace_id(namespace_id)
        objects.NamespaceOperation.zk.remove_all(namespace_id)
        objects.WeightSection.zk.remove_all(namespace_id)
        objects.WeightSection.mongo.remove_revs_by_namespace_id(namespace_id)

        # namespace itself - needs to be the last, so we can be sure we removed everything else
        self.zk.remove_namespace(namespace_id)

    def _validate_balancer_id(self, balancer_id):
        """

        :type balancer_id: six.text_type
        :raises: errors.IntegrityError
        """
        # We want our balancers to have unique identifiers across all namespaces.
        # See https://st.yandex-team.ru/SWAT-6195 for details.
        balancer_pbs = self.cache.list_all_balancers(query={
            self.cache.BalancersQueryTarget.CANONICAL_ID_IN: [balancer_id.replace('_', '-')]
        })
        for balancer_pb in balancer_pbs:
            if balancer_pb.meta.id == balancer_id:
                raise errors.ConflictError(u'Balancer "{}" already exists in different namespace "{}".'.format(
                    balancer_pb.meta.id, balancer_pb.meta.namespace_id))
            raise errors.ConflictError(u'New balancer and balancer "{}" from namespace "{}" would have the same '
                                       u'identifiers after replacing "_" with "-", which is forbidden.'.format(
                                           balancer_pb.meta.id, balancer_pb.meta.namespace_id))

    def _validate_balancer_nanny_service_id(self, namespace_id, balancer_id, spec_pb):
        """
        :type namespace_id: six.text_type
        :type balancer_id: six.text_type
        :type spec_pb: model_pb2.BalancerSpec
        :raises: errors.IntegrityError
        """
        if spec_pb.config_transport.type != model_pb2.NANNY_STATIC_FILE:
            return
        service_id = spec_pb.config_transport.nanny_static_file.service_id
        balancer_pbs = self.cache.list_all_balancers(query={
            self.cache.BalancersQueryTarget.NANNY_SERVICE_ID_IN: [service_id],
        })
        for balancer_pb in balancer_pbs:
            if balancer_pb.meta.namespace_id != namespace_id or balancer_pb.meta.id != balancer_id:
                raise errors.IntegrityError(u'Nanny service "{}" is already used in balancer "{}:{}"'.format(
                    service_id, balancer_pb.meta.namespace_id, balancer_pb.meta.id))

    def create_name_server(self, meta_pb, spec_pb, login, comment=''):
        name_server_id = meta_pb.id
        namespace_id = meta_pb.namespace_id

        if name_server_id != spec_pb.zone:
            raise ValueError("Name server's id must match its zone")

        utcnow = datetime.utcnow()

        name_server_pb = model_pb2.NameServer(
            meta=meta_pb,
            spec=spec_pb,
        )
        name_server_pb.meta.mtime.FromDatetime(utcnow)
        name_server_pb.meta.ctime.FromDatetime(utcnow)
        name_server_pb.meta.author = login
        name_server_pb.meta.version = self.generate_version_id()

        owners = name_server_pb.meta.auth.staff.owners

        if login not in owners.logins:
            owners.logins.append(login)

        if spec_pb is not None:
            rev_meta_pb = model_pb2.NameServerRevisionMeta(
                id=name_server_pb.meta.version,
                namespace_id=namespace_id,
                name_server_id=name_server_id,
                author=login,
                comment=name_server_pb.meta.comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.NameServerRevision(meta=rev_meta_pb, spec=name_server_pb.spec)
            self.db.save_name_server_rev(rev_pb)

        try:
            self.zk.create_name_server(meta_pb.namespace_id, meta_pb.id, name_server_pb)
        except errors.ConflictError as e:
            self.db.remove_name_server_rev(name_server_pb.meta.version)
            raise errors.ConflictError('Failed to create Name Server "{}:{}": {}'.format(namespace_id,
                                                                                         name_server_id,
                                                                                         e))

        return name_server_pb

    def create_default_name_servers(self):
        self.create_namespace_if_missing(
            meta_pb=model_pb2.NamespaceMeta(
                id=INFRA_NAMESPACE_ID,
                auth=model_pb2.Auth(type=model_pb2.Auth.STAFF),
                abc_service_id=730,  # Nanny
            ),
            login=util.NANNY_ROBOT_LOGIN)
        name_server_pbs = []
        for name_server_id, (name_server_type, name_server_selector) in six.iteritems(DEFAULT_NAME_SERVERS):
            name_server_pb = self.create_name_server_if_missing(
                meta_pb=model_pb2.NameServerMeta(
                    id=name_server_id,
                    namespace_id=INFRA_NAMESPACE_ID,
                    auth=model_pb2.Auth(type=model_pb2.Auth.STAFF)
                ),
                spec_pb=model_pb2.NameServerSpec(
                    zone=name_server_id,
                    max_records=None,
                    type=name_server_type,
                    selector=name_server_selector,
                ),
                login=util.NANNY_ROBOT_LOGIN)
            name_server_pbs.append(name_server_pb)
        return name_server_pbs

    def create_dns_record(self, meta_pb, login, spec_pb=None, order_content_pb=None, comment=None):
        assert spec_pb or order_content_pb
        assert not (spec_pb and order_content_pb)
        dns_record_id = meta_pb.id
        namespace_id = meta_pb.namespace_id
        utcnow = datetime.utcnow()

        dns_record_pb = model_pb2.DnsRecord(
            meta=meta_pb,
            spec=spec_pb,
        )
        if order_content_pb is not None:
            dns_record_pb.order.content.CopyFrom(order_content_pb)
        dns_record_pb.meta.mtime.FromDatetime(utcnow)
        dns_record_pb.meta.ctime.FromDatetime(utcnow)
        dns_record_pb.meta.author = login
        if comment:
            dns_record_pb.meta.comment = comment
        dns_record_pb.meta.version = self.generate_version_id()
        if order_content_pb is not None:
            dns_record_pb.spec.incomplete = True

        owners = dns_record_pb.meta.auth.staff.owners

        if login not in owners.logins:
            owners.logins.append(login)

        if spec_pb is not None:
            rev_meta_pb = model_pb2.DnsRecordRevisionMeta(
                id=dns_record_pb.meta.version,
                namespace_id=namespace_id,
                dns_record_id=dns_record_id,
                author=login,
                comment=dns_record_pb.meta.comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.DnsRecordRevision(meta=rev_meta_pb, spec=dns_record_pb.spec)
            self.db.save_dns_record_rev(rev_pb)

        try:
            self.zk.create_dns_record(meta_pb.namespace_id, meta_pb.id, dns_record_pb)
        except errors.ConflictError as e:
            if spec_pb is not None:
                self.db.remove_dns_record_rev(dns_record_pb.meta.version)
            raise errors.ConflictError('Failed to create DNS record "{}:{}": {}'.format(namespace_id, dns_record_id, e))

        dns_record_state_pb = model_pb2.DnsRecordState(namespace_id=namespace_id, dns_record_id=dns_record_id)
        dns_record_state_pb.ctime.FromDatetime(utcnow)
        try:
            self.zk.create_dns_record_state(namespace_id, dns_record_id, dns_record_state_pb)
        except errors.ConflictError as e:
            raise errors.ConflictError('Failed to create DNS record state "{}:{}": {}'.format(
                namespace_id, dns_record_id, e))

        return dns_record_pb, dns_record_state_pb

    def update_dns_record(self, namespace_id, dns_record_id, version, comment, login,
                          updated_auth_pb=None, updated_spec_pb=None):
        """
        :type namespace_id: six.text_type
        :type dns_record_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_auth_pb: model_pb2.Auth
        :type updated_spec_pb: model_pb2.DnsRecordSpec
        :rtype: model_pb2.DnsRecord
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_auth_pb or updated_spec_pb
        utcnow = datetime.utcnow()
        dns_record_pb = None

        if updated_spec_pb is not None:
            new_version = self.generate_version_id()
            rev_meta_pb = model_pb2.DnsRecordRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                dns_record_id=dns_record_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.DnsRecordRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_dns_record_rev(rev_pb)

        for dns_record_pb in self.zk.update_dns_record(namespace_id, dns_record_id):
            if updated_spec_pb is not None and dns_record_pb.spec != updated_spec_pb:
                if dns_record_pb.meta.version != version:
                    self.db.remove_dns_record_rev(new_version)
                    raise errors.ConflictError(
                        'DNS record modification conflict: assumed version="{}", current="{}"'.format(
                            version, dns_record_pb.meta.version))

                dns_record_pb.spec.CopyFrom(updated_spec_pb)
                dns_record_pb.meta.version = new_version
                dns_record_pb.meta.comment = comment
                dns_record_pb.meta.author = login
                dns_record_pb.meta.mtime.FromDatetime(utcnow)
            if updated_auth_pb is not None:
                dns_record_pb.meta.auth.CopyFrom(updated_auth_pb)

        return dns_record_pb

    def delete_dns_record(self, namespace_id, dns_record_id):
        """
        :type namespace_id: six.text_type
        :type dns_record_id: six.text_type
        """
        self.delete_dns_record_operation(namespace_id, dns_record_id)
        self.db.remove_dns_record_revs_by_namespace_and_dns_record_id(namespace_id, dns_record_id)
        self.zk.remove_dns_record_state(namespace_id, dns_record_id)
        self.zk.remove_dns_record(namespace_id, dns_record_id)

    def create_dns_record_operation(self, meta_pb, order_pb, login):
        dns_record_op_id = meta_pb.id
        namespace_id = meta_pb.namespace_id
        utcnow = datetime.utcnow()

        dns_record_op_pb = model_pb2.DnsRecordOperation(
            meta=meta_pb,
            order=order_pb,
        )
        dns_record_op_pb.meta.mtime.FromDatetime(utcnow)
        dns_record_op_pb.meta.ctime.FromDatetime(utcnow)
        dns_record_op_pb.meta.author = login
        dns_record_op_pb.meta.version = self.generate_version_id()
        dns_record_op_pb.spec.incomplete = True

        try:
            self.zk.create_dns_record_operation(meta_pb.namespace_id, meta_pb.id, dns_record_op_pb)
        except errors.ConflictError as e:
            raise errors.ConflictError('Failed to create DNS record operation "{}:{}": {}'.format(
                namespace_id, dns_record_op_id, e))

        return dns_record_op_pb

    def update_dns_record_operation(self, namespace_id, dns_record_op_id, version, comment, login,
                                    updated_spec_pb=None):
        """
        :type namespace_id: six.text_type
        :type dns_record_op_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_spec_pb: model_pb2.DnsRecordOperationSpec
        :rtype: model_pb2.DnsRecordOperation
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_spec_pb
        utcnow = datetime.utcnow()
        dns_record_op_pb = None

        if updated_spec_pb is not None:
            new_version = self.generate_version_id()

        for dns_record_op_pb in self.zk.update_dns_record_operation(namespace_id, dns_record_op_id):
            if updated_spec_pb is not None and dns_record_op_pb.spec != updated_spec_pb:
                if dns_record_op_pb.meta.version != version:
                    raise errors.ConflictError(
                        'DNS record operation modification conflict: assumed version="{}", current="{}"'.format(
                            version, dns_record_op_pb.meta.version))
                dns_record_op_pb.spec.CopyFrom(updated_spec_pb)
                dns_record_op_pb.meta.version = new_version
                dns_record_op_pb.meta.comment = comment
                dns_record_op_pb.meta.author = login
                dns_record_op_pb.meta.mtime.FromDatetime(utcnow)

        return dns_record_op_pb

    def delete_dns_record_operation(self, namespace_id, dns_record_op_id):
        """
        :type namespace_id: six.text_type
        :type dns_record_op_id: six.text_type
        """
        self.zk.remove_dns_record_operation(namespace_id, dns_record_op_id)

    def create_cert(self, meta_pb, login, order_pb=None, spec_pb=None):
        """

        :type meta_pb: model_pb2.CertificateMeta
        :type login: six.text_type
        :type order_pb: model_pb2.CertificateOrder.Content
        :type spec_pb: model_pb2.CertificateSpec
        """
        if order_pb and spec_pb:
            raise ValueError('Only one of "spec_pb" or "order_pb" can be set')
        if not order_pb and not spec_pb:
            raise ValueError('One of "spec_pb" or "order_pb" must be set')

        cert_id = meta_pb.id
        namespace_id = meta_pb.namespace_id

        utcnow = datetime.utcnow()

        cert_pb = model_pb2.Certificate(meta=meta_pb)
        cert_pb.meta.mtime.FromDatetime(utcnow)
        cert_pb.meta.ctime.FromDatetime(utcnow)
        cert_pb.meta.author = login

        if order_pb:
            cert_pb.order.content.CopyFrom(order_pb)
            cert_pb.order.status.status = 'CREATED'
            cert_pb.order.status.message = 'Certificate order is created and will be processed soon.'
            cert_pb.order.status.last_transition_time.FromDatetime(datetime.utcnow())
            cert_pb.spec.incomplete = True
        elif spec_pb:
            cert_pb.spec.CopyFrom(spec_pb)
        cert_pb.meta.version = self.generate_version_id()
        rev_meta_pb = model_pb2.CertificateRevisionMeta(
            id=cert_pb.meta.version,
            namespace_id=namespace_id,
            certificate_id=cert_id,
            author=login,
            comment=cert_pb.meta.comment
        )
        rev_meta_pb.ctime.FromDatetime(utcnow)
        rev_pb = model_pb2.CertificateRevision(meta=rev_meta_pb, spec=cert_pb.spec)
        self.db.save_cert_rev(rev_pb)

        cert_pb.meta.auth.type = cert_pb.meta.auth.STAFF
        owners = cert_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)

        try:
            self.zk.create_cert(meta_pb.namespace_id, meta_pb.id, cert_pb)
        except errors.ConflictError as e:
            self.db.remove_cert_rev(cert_pb.meta.version)
            raise errors.ConflictError('Failed to create certificate "{}:{}": {}'.format(namespace_id, cert_id, e))
        return cert_pb

    @staticmethod
    def _maybe_update_meta_field(meta_pb, updated_meta_pb, field_name, comment, login):
        if not updated_meta_pb.HasField(field_name):
            return
        new_field = getattr(updated_meta_pb, field_name)
        current_field = getattr(meta_pb, field_name)
        if current_field == new_field:
            return
        current_field.CopyFrom(new_field)
        if not current_field.comment:
            current_field.comment = comment
        if not current_field.author:
            current_field.author = login
        current_field.mtime.GetCurrentTime()

    def update_cert(self, namespace_id, cert_id, version, comment, login,
                    updated_meta_pb=None, updated_spec_pb=None, disable_force_renewal=False):
        """
        :type namespace_id: six.text_type
        :type cert_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_meta_pb: model_pb2.CertificateMeta
        :type updated_spec_pb: model_pb2.CertificateSpec
        :type disable_force_renewal: bool
        :rtype: model_pb2.Certificate
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_meta_pb or updated_spec_pb
        utcnow = datetime.utcnow()
        new_version = self.generate_version_id()
        rev_pb = None

        if updated_spec_pb:
            rev_meta_pb = model_pb2.CertificateRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                certificate_id=cert_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.CertificateRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_cert_rev(rev_pb)

        cert_pb = None
        for cert_pb in self.zk.update_cert(namespace_id, cert_id):
            if cert_pb.meta.version != version:
                if rev_pb:
                    self.db.remove_cert_rev(new_version)
                raise errors.ConflictError(
                    'Certificate modification conflict: assumed version="{}", current="{}"'.format(
                        version, cert_pb.meta.version))
            if rev_pb:
                cert_pb.spec.CopyFrom(updated_spec_pb)
                cert_pb.meta.version = new_version
                cert_pb.meta.comment = comment
                cert_pb.meta.author = login
                cert_pb.meta.mtime.FromDatetime(utcnow)

            if updated_meta_pb is not None:
                if updated_meta_pb.HasField('auth'):
                    cert_pb.meta.auth.CopyFrom(updated_meta_pb.auth)
                if updated_meta_pb.HasField('discoverability'):
                    cert_pb.meta.discoverability.CopyFrom(updated_meta_pb.discoverability)
                self._maybe_update_meta_field(meta_pb=cert_pb.meta,
                                              updated_meta_pb=updated_meta_pb,
                                              comment=comment,
                                              login=login,
                                              field_name='unrevokable')
                self._maybe_update_meta_field(meta_pb=cert_pb.meta,
                                              updated_meta_pb=updated_meta_pb,
                                              comment=comment,
                                              login=login,
                                              field_name='is_being_transferred')

            if disable_force_renewal:
                cert_pb.meta.force_renewal.value = False
                cert_pb.meta.force_renewal.comment = ''
                cert_pb.meta.force_renewal.author = ''
                cert_pb.meta.force_renewal.mtime.GetCurrentTime()

        return cert_pb

    def update_cert_renewal(self, namespace_id, cert_renewal_id, version, comment, login,
                            updated_spec_pb=None, updated_paused_pb=None, updated_target_rev=None,
                            updated_target_discoverability=None):
        """
        :type namespace_id: six.text_type
        :type cert_renewal_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_spec_pb: model_pb2.CertificateSpec
        :type updated_paused_pb: model_pb2.PausedCondition
        :type updated_target_rev: six.text_type
        :type updated_target_discoverability: model_pb2.DiscoverabilityCondition
        :rtype: model_pb2.CertificateRenewal
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_spec_pb or updated_paused_pb or updated_target_rev
        utcnow = datetime.utcnow()
        new_version = self.generate_version_id()

        cert_renewal_pb = None
        for cert_renewal_pb in self.zk.update_cert_renewal(namespace_id, cert_renewal_id):
            if cert_renewal_pb.meta.version != version:
                raise errors.ConflictError(
                    'CertificateRenewal modification conflict: assumed version="{}", current="{}"'.format(
                        version, cert_renewal_pb.meta.version))
            if updated_spec_pb:
                cert_renewal_pb.spec.CopyFrom(updated_spec_pb)
            if updated_target_rev:
                cert_renewal_pb.meta.target_rev = updated_target_rev
            if updated_paused_pb:
                cert_renewal_pb.meta.paused.CopyFrom(updated_paused_pb)
            if updated_target_discoverability is not None:
                cert_renewal_pb.meta.target_discoverability.CopyFrom(updated_target_discoverability)
            cert_renewal_pb.meta.version = new_version
            cert_renewal_pb.meta.comment = comment
            cert_renewal_pb.meta.author = login
            cert_renewal_pb.meta.mtime.FromDatetime(utcnow)

        return cert_renewal_pb

    def delete_cert(self, namespace_id, cert_id):
        """
        :type namespace_id: six.text_type
        :type cert_id: six.text_type
        """
        self.db.remove_cert_revs_by_namespace_and_cert_id(namespace_id, cert_id)
        self.zk.remove_cert(namespace_id, cert_id)

    def create_knob(self, meta_pb, spec_pb, login, comment=''):
        """
        :type meta_pb: model_pb2.KnobMeta
        :type spec_pb: model_pb2.KnobSpec | None
        :type login: six.text_type
        :type comment: six.text_type | six.text_type
        :rtype: (model_pb2.Knob
        """
        namespace_id = meta_pb.namespace_id
        knob_id = meta_pb.id

        utcnow = datetime.utcnow()
        knob_pb = model_pb2.Knob(
            meta=meta_pb,
            spec=spec_pb,
        )
        knob_pb.meta.mtime.FromDatetime(utcnow)
        knob_pb.meta.ctime.FromDatetime(utcnow)
        knob_pb.meta.author = login
        if comment:
            knob_pb.meta.comment = comment

        owners = knob_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)

        if spec_pb is not None:
            version = knob_pb.meta.version = self.generate_version_id()

            rev_meta_pb = model_pb2.KnobRevisionMeta(
                id=version,
                namespace_id=namespace_id,
                knob_id=knob_id,
                author=login,
                comment=knob_pb.meta.comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.KnobRevision(meta=rev_meta_pb, spec=knob_pb.spec)
            self.db.save_knob_rev(rev_pb)

        try:
            self.zk.create_knob(meta_pb.namespace_id, meta_pb.id, knob_pb)
        except errors.ConflictError as e:
            self.db.remove_knob_rev(version)
            raise errors.ConflictError('Failed to create knob "{}:{}": {}'.format(
                namespace_id, knob_id, e))

        return knob_pb

    def update_knob(self, namespace_id, knob_id, version, comment, login,
                    updated_auth_pb=None, updated_spec_pb=None):
        """
        :type namespace_id: six.text_type
        :type knob_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_auth_pb: model_pb2.Auth
        :type updated_spec_pb: model_pb2.KnobSpec
        :rtype: model_pb2.Knob
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_auth_pb or updated_spec_pb
        utcnow = datetime.utcnow()

        if updated_spec_pb is not None:
            new_version = self.generate_version_id()
            rev_meta_pb = model_pb2.KnobRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                knob_id=knob_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.KnobRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_knob_rev(rev_pb)

        for knob_pb in self.zk.update_knob(namespace_id, knob_id):
            if updated_spec_pb is not None and knob_pb.spec != updated_spec_pb:
                if knob_pb.meta.version != version:
                    self.db.remove_knob_rev(new_version)
                    raise errors.ConflictError(
                        'Knob modification conflict: assumed version="{}", current="{}"'.format(
                            version, knob_pb.meta.version))

                knob_pb.spec.CopyFrom(updated_spec_pb)
                knob_pb.meta.version = new_version
                knob_pb.meta.comment = comment
                knob_pb.meta.author = login
                knob_pb.meta.mtime.FromDatetime(utcnow)
            if updated_auth_pb is not None:
                knob_pb.meta.auth.CopyFrom(updated_auth_pb)

        return knob_pb

    def delete_knob(self, namespace_id, knob_id):
        """
        :type namespace_id: six.text_type
        :type knob_id: six.text_type
        """
        self.db.remove_knob_revs_by_namespace_and_knob_id(namespace_id, knob_id)
        self.zk.remove_knob(namespace_id, knob_id)

    def create_l3_balancer(self, meta_pb, spec_pb, login, order_content_pb=None):
        """
        :type meta_pb: model_pb2.L3BalancerMeta
        :type spec_pb: model_pb2.L3BalancerSpec | None
        :type order_content_pb: Optional[model_pb2.L3BalancerOrder.Content]
        :type login: six.text_type
        :rtype: (model_pb2.L3Balancer, model_pb2.L3BalancerState)
        """
        namespace_id = meta_pb.namespace_id
        l3_balancer_id = meta_pb.id

        utcnow = datetime.utcnow()
        l3_balancer_pb = model_pb2.L3Balancer(
            meta=meta_pb,
            spec=spec_pb
        )
        l3_balancer_pb.meta.author = login
        owners = l3_balancer_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)
        l3_balancer_pb.meta.mtime.FromDatetime(utcnow)
        l3_balancer_pb.meta.ctime.FromDatetime(utcnow)
        if l3_balancer_pb.meta.HasField('transport_paused'):
            l3_balancer_pb.meta.transport_paused.mtime.FromDatetime(utcnow)
        if order_content_pb:
            l3_balancer_pb.spec.incomplete = True
            l3_balancer_pb.order.content.CopyFrom(order_content_pb)

        if spec_pb is not None:
            l3_balancer_pb.meta.version = self.generate_version_id()
            rev_meta_pb = model_pb2.L3BalancerRevisionMeta(
                id=l3_balancer_pb.meta.version,
                namespace_id=namespace_id,
                l3_balancer_id=l3_balancer_id,
                author=login,
                comment=l3_balancer_pb.meta.comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.L3BalancerRevision(meta=rev_meta_pb, spec=l3_balancer_pb.spec)
            self.db.save_l3_balancer_rev(rev_pb)

        try:
            self.zk.create_l3_balancer(meta_pb.namespace_id, meta_pb.id, l3_balancer_pb)
        except errors.ConflictError as e:
            if spec_pb is not None:
                self.db.remove_l3_balancer_rev(l3_balancer_pb.meta.version)
                raise errors.ConflictError('Failed to create L3 balancer "{}:{}": {}'.format(
                    namespace_id, l3_balancer_id, e))

        l3_balancer_state_pb = model_pb2.L3BalancerState(namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
        l3_balancer_state_pb.ctime.FromDatetime(utcnow)
        try:
            self.zk.create_l3_balancer_state(namespace_id, l3_balancer_id, l3_balancer_state_pb)
        except errors.ConflictError as e:
            raise errors.ConflictError('Failed to create L3 balancer state "{}:{}": {}'.format(
                namespace_id, l3_balancer_id, e))

        return l3_balancer_pb, l3_balancer_state_pb

    def update_l3_balancer(self, namespace_id, l3_balancer_id, version, comment, login,
                           updated_auth_pb=None, updated_transport_paused_pb=None, updated_spec_pb=None):
        """
        :type namespace_id: six.text_type
        :type l3_balancer_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_auth_pb: Optional[model_pb2.Auth]
        :type updated_transport_paused_pb: Optional[model_pb2.PausedCondition]
        :type updated_spec_pb: Optional[model_pb2.L3BalancerSpec]
        :rtype: model_pb2.L3Balancer
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_auth_pb or updated_transport_paused_pb or updated_spec_pb
        utcnow = datetime.utcnow()

        new_version = self.generate_version_id()
        if updated_spec_pb is not None:
            rev_meta_pb = model_pb2.L3BalancerRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                l3_balancer_id=l3_balancer_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.L3BalancerRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_l3_balancer_rev(rev_pb)

        for l3_balancer_pb in self.zk.update_l3_balancer(namespace_id, l3_balancer_id):
            if updated_spec_pb is not None and l3_balancer_pb.spec != updated_spec_pb:
                if l3_balancer_pb.meta.version != version:
                    self.db.remove_l3_balancer_rev(new_version)
                    raise errors.ConflictError(
                        'L3 balancer modification conflict: assumed version="{}", current="{}"'.format(
                            version, l3_balancer_pb.meta.version))

                l3_balancer_pb.spec.CopyFrom(updated_spec_pb)
                l3_balancer_pb.meta.version = new_version
                l3_balancer_pb.meta.comment = comment
                l3_balancer_pb.meta.author = login
                l3_balancer_pb.meta.mtime.FromDatetime(utcnow)
            if updated_auth_pb is not None:
                l3_balancer_pb.meta.auth.CopyFrom(updated_auth_pb)
            if updated_transport_paused_pb is not None:
                util.set_condition(l3_balancer_pb.meta.transport_paused,
                                   updated_condition_pb=updated_transport_paused_pb,
                                   author=login,
                                   utcnow=utcnow)
        return l3_balancer_pb

    def delete_l3_balancer(self, namespace_id, l3_balancer_id):
        """
        :type namespace_id: six.text_type
        :type l3_balancer_id: six.text_type
        """
        self.db.remove_l3_balancer_revs_by_namespace_and_l3_balancer_id(namespace_id, l3_balancer_id)
        self.zk.remove_l3_balancer_state(namespace_id, l3_balancer_id)
        self.zk.remove_l3_balancer(namespace_id, l3_balancer_id)
        for op_pb in list(objects.NamespaceOperation.cache.list(namespace_id)):
            if l3_balancer_id in op_pb.meta.parent_versions.l3_versions:
                objects.NamespaceOperation.remove(op_pb)

    def create_balancer(self, meta_pb, login, spec_pb=None, order_content_pb=None, rev_index_pb=None):
        """
        :type meta_pb: model_pb2.BalancerMeta
        :type login: six.text_type
        :type spec_pb: model_pb2.BalancerSpec
        :type order_content_pb: model_pb2.BalancerOrder.Content | None
        :type rev_index_pb: model_pb2.RevisionGraphIndex | None
        :rtype: (model_pb2.Balancer, model_pb2.BalancerState)
        """
        if order_content_pb and spec_pb:
            raise ValueError('Only one of "spec_pb" or "order_content_pb" can be set')
        if not order_content_pb and not spec_pb:
            raise ValueError('One of "spec_pb" or "order_content_pb" must be set')

        namespace_id = meta_pb.namespace_id
        balancer_id = meta_pb.id
        self._validate_balancer_id(balancer_id)

        utcnow = datetime.utcnow()

        balancer_pb = model_pb2.Balancer(meta=meta_pb)
        balancer_pb.meta.mtime.FromDatetime(utcnow)
        balancer_pb.meta.ctime.FromDatetime(utcnow)
        if balancer_pb.meta.HasField('transport_paused'):
            balancer_pb.meta.transport_paused.mtime.FromDatetime(utcnow)
        balancer_pb.meta.author = login
        version = balancer_pb.meta.version = self.generate_version_id()

        owners = balancer_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)

        if order_content_pb:
            balancer_pb.order.content.CopyFrom(order_content_pb)
            balancer_pb.spec.incomplete = True
        elif spec_pb:
            self._validate_balancer_nanny_service_id(namespace_id, balancer_id, spec_pb)
            balancer_pb.spec.CopyFrom(spec_pb)
            rev_meta_pb = model_pb2.BalancerRevisionMeta(
                id=version,
                namespace_id=namespace_id,
                balancer_id=balancer_id,
                author=login,
                comment=balancer_pb.meta.comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.BalancerRevision(meta=rev_meta_pb, spec=balancer_pb.spec)
            self.db.save_balancer_rev(rev_pb)

        if rev_index_pb is not None:
            self._update_indices(balancer_pb, [rev_index_pb])

        try:
            self.zk.create_balancer(meta_pb.namespace_id, meta_pb.id, balancer_pb)
        except errors.ConflictError as e:
            if spec_pb:
                self.db.remove_balancer_rev(version)
            raise errors.ConflictError('Failed to create a balancer "{}:{}": {}'.format(namespace_id, balancer_id, e))

        balancer_state_pb = model_pb2.BalancerState(namespace_id=namespace_id, balancer_id=balancer_id)
        balancer_state_pb.ctime.FromDatetime(utcnow)
        try:
            self.zk.create_balancer_state(namespace_id, balancer_id, balancer_state_pb)
        except errors.ConflictError as e:
            raise errors.ConflictError('Failed to create a balancer state "{}:{}": {}'.format(
                namespace_id, balancer_id, e))

        balancer_aspects_set_pb = model_pb2.BalancerAspectsSet()
        balancer_aspects_set_pb.meta.ctime.FromDatetime(utcnow)
        balancer_aspects_set_pb.meta.namespace_id = namespace_id
        balancer_aspects_set_pb.meta.balancer_id = balancer_id
        try:
            self.zk.create_balancer_aspects_set(namespace_id, balancer_id, balancer_aspects_set_pb)
        except errors.ConflictError as e:
            raise errors.ConflictError('Failed to create a balancer aspects set "{}:{}": {}'.format(
                namespace_id, balancer_id, e))

        return balancer_pb, balancer_state_pb

    def update_balancer(self, namespace_id, balancer_id, comment, login, version=None,
                        updated_auth_pb=None, updated_transport_paused_pb=None, updated_spec_pb=None,
                        updated_location_pb=None, updated_flags_pb=None, updated_approval_pb=None,
                        rev_index_pbs=None, utcnow=None):
        """
        :type namespace_id: six.text_type
        :type balancer_id: six.text_type
        :type version: Optional[six.text_type]
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_auth_pb: model_pb2.Auth | None
        :type updated_transport_paused_pb: model_pb2.PausedCondition | None
        :type updated_spec_pb: model_pb2.BalancerSpec | None
        :type updated_location_pb: model_pb2.BalancerMeta.Location | None
        :type updated_flags_pb: model_pb2.BalancerMeta.Flags | None
        :type updated_approval_pb: model_pb2.BalancerOrder.Approval | None
        :type rev_index_pbs: Iterable[model_pb2.RevisionGraphIndex] | None
        :type utcnow: datetime
        :rtype: model_pb2.Balancer
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert (
            updated_auth_pb
            or updated_transport_paused_pb
            or updated_spec_pb
            or updated_location_pb
            or updated_flags_pb
            or updated_approval_pb
        )
        if utcnow is None:
            utcnow = datetime.utcnow()

        if updated_spec_pb is not None:
            assert version
            self._validate_balancer_nanny_service_id(namespace_id, balancer_id, updated_spec_pb)

            new_version = self.generate_version_id()
            rev_meta_pb = model_pb2.BalancerRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                balancer_id=balancer_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.BalancerRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_balancer_rev(rev_pb)

        for balancer_pb in self.zk.update_balancer(namespace_id, balancer_id):
            if updated_spec_pb is not None and balancer_pb.spec != updated_spec_pb:
                if balancer_pb.meta.version != version:
                    self.db.remove_balancer_rev(new_version)
                    raise errors.ConflictError(
                        'Balancer modification conflict: assumed version="{}", current="{}"'.format(
                            version, balancer_pb.meta.version))

                balancer_pb.spec.CopyFrom(updated_spec_pb)
                balancer_pb.meta.version = new_version
                balancer_pb.meta.comment = comment
                balancer_pb.meta.author = login
                balancer_pb.meta.mtime.FromDatetime(utcnow)

                if rev_index_pbs is not None:
                    self._update_indices(balancer_pb, rev_index_pbs)

            if updated_auth_pb is not None:
                balancer_pb.meta.auth.CopyFrom(updated_auth_pb)
            if updated_transport_paused_pb is not None:
                util.set_condition(balancer_pb.meta.transport_paused,
                                   updated_condition_pb=updated_transport_paused_pb,
                                   author=login,
                                   utcnow=utcnow)
            if updated_location_pb is not None:
                balancer_pb.meta.location.CopyFrom(updated_location_pb)
            if updated_flags_pb is not None:
                for flag in ('skip_rps_check_during_removal', 'forbid_disabling_webauth', 'downtime_stuck_activation_check'):
                    util.set_condition(getattr(balancer_pb.meta.flags, flag),
                                       updated_condition_pb=getattr(updated_flags_pb, flag),
                                       author=login,
                                       utcnow=utcnow)
            if updated_approval_pb:
                balancer_pb.order.approval.CopyFrom(updated_approval_pb)
        return balancer_pb

    def delete_balancer(self, namespace_id, balancer_id):
        self.zk.remove_balancer_operation(namespace_id, balancer_id)
        self.zk.remove_balancer_state(namespace_id, balancer_id)
        self.zk.remove_balancer_aspects_set(namespace_id, balancer_id)
        self.db.remove_balancer_revs_by_namespace_and_balancer_id(namespace_id, balancer_id)
        self.zk.remove_balancer(namespace_id, balancer_id)

    def delete_upstream(self, namespace_id, upstream_id):
        self.zk.remove_upstream(namespace_id, upstream_id)
        self.db.remove_upstream_revs_by_namespace_and_upstream_id(namespace_id, upstream_id)

    def delete_backend(self, namespace_id, backend_id):
        self.db.remove_endpoint_set_revs_by_namespace_and_endpoint_set_id(namespace_id, backend_id)
        self.db.remove_backend_revs_by_namespace_and_backend_id(namespace_id, backend_id)
        self.zk.remove_endpoint_set(namespace_id, backend_id)
        self.zk.remove_backend(namespace_id, backend_id)

    def create_upstream(self, meta_pb, spec_pb, login, rev_index_pb=None):
        """
        :type meta_pb: model_pb2.UpstreamMeta
        :type spec_pb: model_pb2.UpstreamSpec
        :type login: six.text_type
        :type rev_index_pb: model_pb2.RevisionGraphIndex | None
        :rtype: model_pb2.Upstream
        """
        namespace_id = meta_pb.namespace_id
        upstream_id = meta_pb.id

        utcnow = datetime.utcnow()
        upstream_pb = model_pb2.Upstream(meta=meta_pb, spec=spec_pb)
        upstream_pb.meta.mtime.FromDatetime(utcnow)
        upstream_pb.meta.ctime.FromDatetime(utcnow)
        upstream_pb.meta.author = login
        version = upstream_pb.meta.version = self.generate_version_id()

        owners = upstream_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)

        rev_meta_pb = model_pb2.UpstreamRevisionMeta(
            id=version,
            namespace_id=namespace_id,
            upstream_id=upstream_id,
            author=login,
            comment=upstream_pb.meta.comment
        )
        rev_meta_pb.ctime.FromDatetime(utcnow)
        rev_pb = model_pb2.UpstreamRevision(meta=rev_meta_pb, spec=upstream_pb.spec)
        self.db.save_upstream_rev(rev_pb)
        if rev_index_pb is not None:
            self._update_indices(upstream_pb, [rev_index_pb])

        try:
            self.zk.create_upstream(namespace_id, upstream_id, upstream_pb)
        except errors.ConflictError:
            self.db.remove_upstream_rev(version)
            raise errors.ConflictError(
                'Upstream "{}" already exists in namespace "{}".'.format(upstream_id, namespace_id))

        return upstream_pb

    @staticmethod
    def _update_indices(obj_pb, rev_index_pbs):
        """
        :type obj_pb: model_pb2.Balancer | model_pb2.Upstream
        :type rev_index_pbs: Iterable[model_pb2.RevisionGraphIndex] | None
        """
        assert obj_pb.meta.version
        updated_index_pbs = []
        for i, ind_pb in enumerate(rev_index_pbs):
            if not ind_pb.id:
                assert i == len(rev_index_pbs) - 1
                ind_pb = util.clone_pb(ind_pb)
                ind_pb.id = obj_pb.meta.version
                ind_pb.ctime.CopyFrom(obj_pb.meta.mtime)
            updated_index_pbs.append(ind_pb)
        del obj_pb.meta.indices[:]
        obj_pb.meta.indices.extend(updated_index_pbs)

    def update_upstream(self, namespace_id, upstream_id, version, comment, login,
                        updated_auth_pb=None, updated_spec_pb=None, rev_index_pbs=None):
        """
        :type namespace_id: six.text_type
        :type upstream_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_auth_pb: model_pb2.Auth
        :type updated_spec_pb: model_pb2.UpstreamSpec
        :type rev_index_pbs: Iterable[model_pb2.RevisionGraphIndex] | None
        :rtype: model_pb2.Upstream
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_auth_pb or updated_spec_pb
        utcnow = datetime.utcnow()

        if updated_spec_pb is not None:
            new_version = self.generate_version_id()
            rev_meta_pb = model_pb2.UpstreamRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                upstream_id=upstream_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.UpstreamRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_upstream_rev(rev_pb)

        for upstream_pb in self.zk.update_upstream(namespace_id, upstream_id):
            if updated_spec_pb is not None and upstream_pb.spec != updated_spec_pb:
                if upstream_pb.meta.version != version:
                    self.db.remove_upstream_rev(new_version)
                    raise errors.ConflictError(
                        'Upstream modification conflict: assumed version="{}", current="{}"'.format(
                            version, upstream_pb.meta.version))

                if upstream_pb.spec.deleted:
                    raise errors.ConflictError('Upstream with "spec.deleted" flag can not be updated')

                upstream_pb.spec.CopyFrom(updated_spec_pb)
                upstream_pb.meta.version = new_version
                upstream_pb.meta.comment = comment
                upstream_pb.meta.author = login
                upstream_pb.meta.mtime.FromDatetime(utcnow)

                if rev_index_pbs is not None:
                    self._update_indices(upstream_pb, rev_index_pbs)
            if updated_auth_pb is not None:
                upstream_pb.meta.auth.CopyFrom(updated_auth_pb)

        return upstream_pb

    def create_domain(self, meta_pb, login, order_pb=None, spec_pb=None):
        """
        :type meta_pb: model_pb2.DomainMeta
        :type spec_pb: model_pb2.DomainSpec | None
        :type order_pb: model_pb2.DomainOrder.Content | None
        :type login: six.text_type
        :rtype: model_pb2.Domain
        """
        if order_pb and spec_pb:
            raise ValueError('Only one of "spec_pb" or "order_pb" can be set')
        if not order_pb and not spec_pb:
            raise ValueError('One of "spec_pb" or "order_pb" must be set')

        namespace_id = meta_pb.namespace_id
        domain_id = meta_pb.id

        utcnow = datetime.utcnow()
        domain_pb = model_pb2.Domain(meta=meta_pb, spec=spec_pb)
        domain_pb.meta.ctime.FromDatetime(utcnow)
        domain_pb.meta.mtime.FromDatetime(utcnow)
        domain_pb.meta.author = login
        domain_pb.meta.version = self.generate_version_id()

        if order_pb:
            domain_pb.order.content.CopyFrom(order_pb)
            domain_pb.order.status.status = 'CREATED'
            domain_pb.order.status.message = 'Domain order is created and will be processed soon.'
            domain_pb.order.status.last_transition_time.FromDatetime(utcnow)
            domain_pb.spec.incomplete = True
        elif spec_pb:
            domain_pb.spec.CopyFrom(spec_pb)
        rev_meta_pb = model_pb2.DomainRevisionMeta(
            id=domain_pb.meta.version,
            namespace_id=namespace_id,
            domain_id=domain_id,
            author=login,
            comment=domain_pb.meta.comment,
            ctime=domain_pb.meta.ctime,
        )
        rev_pb = model_pb2.DomainRevision(meta=rev_meta_pb, spec=domain_pb.spec)
        self.db.save_domain_rev(rev_pb)

        try:
            self.zk.create_domain(meta_pb.namespace_id, meta_pb.id, domain_pb)
        except errors.ConflictError as e:
            self.db.remove_domain_rev(domain_pb.meta.version)
            raise errors.ConflictError('Failed to create domain "{}:{}": {}'.format(namespace_id, domain_id, e))
        return domain_pb

    def update_domain(self, namespace_id, domain_id, version, comment, login, updated_spec_pb=None,
                      updated_meta_pb=None):
        """
        :type namespace_id: six.text_type
        :type domain_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_meta_pb: Optional[model_pb2.DomainMeta]
        :type updated_spec_pb: Optional[model_pb2.DomainSpec]
        :rtype: model_pb2.Domain
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_meta_pb or updated_spec_pb
        domain_pb = None
        new_version = self.generate_version_id()
        utcnow = datetime.utcnow()

        if updated_spec_pb:
            rev_meta_pb = model_pb2.DomainRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                domain_id=domain_id,
                author=login,
                comment=comment,
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.DomainRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_domain_rev(rev_pb)

        for domain_pb in self.zk.update_domain(namespace_id, domain_id):
            if domain_pb.meta.version != version:
                if new_version:
                    self.db.remove_domain_rev(new_version)
                raise errors.ConflictError(
                    'Domain modification conflict: assumed version="{}", current="{}"'.format(
                        version, domain_pb.meta.version))
            if updated_spec_pb and domain_pb.spec != updated_spec_pb:
                if domain_pb.spec.deleted:
                    raise errors.ConflictError('Domain with "spec.deleted" flag can not be updated')

                domain_pb.spec.CopyFrom(updated_spec_pb)
                domain_pb.meta.version = new_version
                domain_pb.meta.comment = comment
                domain_pb.meta.author = login
                domain_pb.meta.mtime.FromDatetime(utcnow)
            if updated_meta_pb is not None:
                self._maybe_update_meta_field(meta_pb=domain_pb.meta,
                                              updated_meta_pb=updated_meta_pb,
                                              comment=comment,
                                              login=login,
                                              field_name='is_being_transferred')
        return domain_pb

    def delete_domain(self, namespace_id, domain_id, remove_op=True):
        """
        :type namespace_id: six.text_type
        :type domain_id: six.text_type
        :type remove_op: bool
        """
        if remove_op:
            self.delete_domain_operation(namespace_id, domain_id)
        self.db.remove_domain_revs_by_namespace_and_domain_id(namespace_id, domain_id)
        self.zk.remove_domain(namespace_id, domain_id)

    def create_domain_operation(self, meta_pb, order_pb, login):
        """
        :type meta_pb: model_pb2.DomainMeta
        :type order_pb: DomainOperationOrder
        :type login: six.text_type
        :rtype: model_pb2.DomainOperation
        """
        namespace_id = meta_pb.namespace_id
        domain_id = meta_pb.id

        utcnow = datetime.utcnow()
        domain_operation_pb = model_pb2.DomainOperation(meta=meta_pb)
        domain_operation_pb.order.content.CopyFrom(order_pb)
        domain_operation_pb.meta.ctime.FromDatetime(utcnow)
        domain_operation_pb.meta.mtime.FromDatetime(utcnow)
        domain_operation_pb.meta.author = login
        domain_operation_pb.spec.incomplete = True
        self.zk.create_domain_operation(namespace_id, domain_id, domain_operation_pb)
        return domain_operation_pb

    def update_domain_operation(self, namespace_id, domain_id, version, comment, login, updated_spec_pb):
        """
        :type namespace_id: six.text_type
        :type domain_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_spec_pb: model_pb2.DomainOperationSpec
        :rtype: model_pb2.Domain
        :raises: errors.ConflictError, errors.NotFoundError
        """
        domain_op_pb = None
        utcnow = datetime.utcnow()
        new_version = self.generate_version_id()

        for domain_op_pb in self.zk.update_domain_operation(namespace_id, domain_id):
            if domain_op_pb.spec != updated_spec_pb:
                if domain_op_pb.meta.version != version:
                    raise errors.ConflictError(
                        'Domain operation modification conflict: assumed version="{}", current="{}"'.format(
                            version, domain_op_pb.meta.version))

                domain_op_pb.spec.CopyFrom(updated_spec_pb)
                domain_op_pb.meta.version = new_version
                domain_op_pb.meta.comment = comment
                domain_op_pb.meta.author = login
                domain_op_pb.meta.mtime.FromDatetime(utcnow)

        return domain_op_pb

    def delete_domain_operation(self, namespace_id, domain_id):
        """
        :type namespace_id: six.text_type
        :type domain_id: six.text_type
        """
        self.zk.remove_domain_operation(namespace_id, domain_id)

    def create_balancer_operation(self, meta_pb, order_pb, login):
        """
        :type meta_pb: model_pb2.BalancerOperationMeta
        :type order_pb: BalancerOperationOrder
        :type login: six.text_type
        :rtype: model_pb2.BalancerOperation
        """
        namespace_id = meta_pb.namespace_id
        balancer_id = meta_pb.id

        utcnow = datetime.utcnow()
        balancer_operation_pb = model_pb2.BalancerOperation(meta=meta_pb)
        balancer_operation_pb.order.content.CopyFrom(order_pb)
        balancer_operation_pb.meta.ctime.FromDatetime(utcnow)
        balancer_operation_pb.meta.mtime.FromDatetime(utcnow)
        balancer_operation_pb.meta.author = login
        balancer_operation_pb.spec.incomplete = True
        self.zk.create_balancer_operation(namespace_id, balancer_id, balancer_operation_pb)
        return balancer_operation_pb

    def update_balancer_operation(self, namespace_id, balancer_id, version, comment, login, updated_spec_pb):
        """
        :type namespace_id: six.text_type
        :type balancer_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_spec_pb: model_pb2.BalancerOperationSpec
        :rtype: model_pb2.BalancerOperation
        :raises: errors.ConflictError, errors.NotFoundError
        """
        balancer_op_pb = None
        utcnow = datetime.utcnow()
        new_version = self.generate_version_id()

        for balancer_op_pb in self.zk.update_balancer_operation(namespace_id, balancer_id):
            if balancer_op_pb.spec != updated_spec_pb:
                if balancer_op_pb.meta.version != version:
                    raise errors.ConflictError(
                        'Balancer operation modification conflict: assumed version="{}", current="{}"'.format(
                            version, balancer_op_pb.meta.version))

                balancer_op_pb.spec.CopyFrom(updated_spec_pb)
                balancer_op_pb.meta.version = new_version
                balancer_op_pb.meta.comment = comment
                balancer_op_pb.meta.author = login
                balancer_op_pb.meta.mtime.FromDatetime(utcnow)

        return balancer_op_pb

    def delete_balancer_operation(self, namespace_id, balancer_id):
        """
        :type namespace_id: six.text_type
        :type balancer_id: six.text_type
        """
        self.zk.remove_balancer_operation(namespace_id, balancer_id)

    def create_backend(self, meta_pb, spec_pb, login):
        """
        :type meta_pb: model_pb2.BackendMeta
        :type spec_pb: model_pb2.BackendSpec
        :type login: six.text_type
        :rtype: model_pb2.Backend
        """
        namespace_id = meta_pb.namespace_id
        backend_id = meta_pb.id

        utcnow = datetime.utcnow()
        backend_pb = model_pb2.Backend(meta=meta_pb, spec=spec_pb)
        backend_pb.meta.mtime.FromDatetime(utcnow)
        backend_pb.meta.ctime.FromDatetime(utcnow)
        backend_pb.meta.author = login
        version = backend_pb.meta.version = self.generate_version_id()

        owners = backend_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)

        rev_meta_pb = model_pb2.BackendRevisionMeta(
            id=version,
            namespace_id=namespace_id,
            backend_id=backend_id,
            author=login,
            comment=backend_pb.meta.comment
        )
        rev_meta_pb.ctime.FromDatetime(utcnow)
        rev_pb = model_pb2.BackendRevision(meta=rev_meta_pb, spec=backend_pb.spec)

        self.db.save_backend_rev(rev_pb)

        try:
            self.zk.create_backend(namespace_id, backend_id, backend_pb)
        except errors.ConflictError:
            self.db.remove_backend_rev(version)
            raise errors.ConflictError(
                'Backend "{}" already exists in namespace "{}".'.format(backend_id, namespace_id))

        return backend_pb

    def update_backend(self, namespace_id, backend_id,
                       comment=None, login=None, version=None,
                       updated_auth_pb=None, updated_spec_pb=None, updated_resolver_status_pb=None,
                       updated_is_system_pb=None, allow_restoration=False):
        """
        :type namespace_id: six.text_type
        :type backend_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_auth_pb: model_pb2.Auth
        :type updated_is_system_pb: model_pb2.BoolCondition
        :type updated_spec_pb: model_pb2.BackendSpec
        :type updated_resolver_status_pb: model_pb2.BackendResolverStatus
        :type allow_restoration: bool
        :rtype: model_pb2.Backend
        :raises: errors.ConflictError, errors.NotFoundError
        """
        assert updated_auth_pb or updated_spec_pb or updated_resolver_status_pb or updated_is_system_pb
        assert not updated_spec_pb or (version is not None and
                                       comment is not None and
                                       login is not None)
        utcnow = datetime.utcnow()
        new_version = backend_pb = None

        if updated_spec_pb is not None:
            new_version = self.generate_version_id()
            rev_meta_pb = model_pb2.BackendRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                backend_id=backend_id,
                author=login,
                comment=comment
            )
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.BackendRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_backend_rev(rev_pb)

        for backend_pb in self.zk.update_backend(namespace_id, backend_id):
            if updated_spec_pb is not None and backend_pb.spec != updated_spec_pb:
                if backend_pb.meta.version != version:
                    self.db.remove_backend_rev(new_version)
                    raise errors.ConflictError(
                        'Backend modification conflict: assumed version="{}", current="{}"'.format(
                            version, backend_pb.meta.version))

                if backend_pb.spec.deleted and not allow_restoration:
                    raise errors.ConflictError('Backend with "spec.deleted" flag can not be updated')

                is_global_has_changed = backend_pb.spec.is_global.value != updated_spec_pb.is_global.value
                backend_pb.spec.CopyFrom(updated_spec_pb)
                if is_global_has_changed:
                    backend_pb.spec.is_global.author = login
                    backend_pb.spec.is_global.mtime.GetCurrentTime()
                backend_pb.meta.version = new_version
                backend_pb.meta.comment = comment
                backend_pb.meta.author = login
                backend_pb.meta.mtime.FromDatetime(utcnow)
            if updated_auth_pb is not None:
                backend_pb.meta.auth.CopyFrom(updated_auth_pb)
            if updated_is_system_pb is not None:
                backend_pb.meta.is_system.CopyFrom(updated_is_system_pb)
                backend_pb.meta.is_system.author = login
                backend_pb.meta.is_system.mtime.GetCurrentTime()
            if updated_resolver_status_pb is not None:
                backend_pb.resolver_status.CopyFrom(updated_resolver_status_pb)

        return backend_pb

    def create_endpoint_set(self, meta_pb, spec_pb, login):
        """
        :type meta_pb: model_pb2.EndpointSetMeta
        :type spec_pb: model_pb2.EndpointSetSpec
        :type login: six.text_type
        :rtype: model_pb2.EndpointSet
        """
        namespace_id = meta_pb.namespace_id
        endpoint_set_id = meta_pb.id

        utcnow = datetime.utcnow()
        endpoint_set_pb = model_pb2.EndpointSet(meta=meta_pb, spec=spec_pb)
        endpoint_set_pb.meta.mtime.FromDatetime(utcnow)
        endpoint_set_pb.meta.ctime.FromDatetime(utcnow)
        endpoint_set_pb.meta.author = login
        version = endpoint_set_pb.meta.version = self.generate_version_id()

        owners = endpoint_set_pb.meta.auth.staff.owners
        if login not in owners.logins:
            owners.logins.append(login)

        rev_meta_pb = model_pb2.EndpointSetRevisionMeta(
            id=version,
            namespace_id=namespace_id,
            endpoint_set_id=endpoint_set_id,
            author=login,
            comment=endpoint_set_pb.meta.comment,
            backend_versions=meta_pb.backend_versions,
        )
        rev_meta_pb.resolved_from.CopyFrom(meta_pb.resolved_from)
        rev_meta_pb.ctime.FromDatetime(utcnow)
        rev_pb = model_pb2.EndpointSetRevision(meta=rev_meta_pb, spec=endpoint_set_pb.spec)

        self.db.save_endpoint_set_rev(rev_pb)

        try:
            self.zk.create_endpoint_set(namespace_id, endpoint_set_id, endpoint_set_pb)
        except errors.ConflictError:
            self.db.remove_endpoint_set_rev(version)
            raise errors.ConflictError(
                'Endpoint set "{}" already exists in namespace "{}".'.format(endpoint_set_id, namespace_id))

        return endpoint_set_pb

    def update_endpoint_set(self, namespace_id, endpoint_set_id, version, comment, login,
                            backend_version, backend_selector_pb,
                            updated_spec_pb=None, updated_auth_pb=None,
                            updated_is_system_pb=None):
        """
        :type namespace_id: six.text_type
        :type endpoint_set_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type backend_version: six.text_type
        :type backend_selector_pb: model_pb2.BackendSelector
        :type updated_spec_pb: model_pb2.EndpointSetSpec | None
        :type updated_auth_pb: model_pb2.Auth | None
        :type updated_is_system_pb: model_pb2.BoolCondition | None
        :rtype: model_pb2.EndpointSet
        :raises: errors.ConflictError, errors.NotFoundError
        """
        utcnow = datetime.utcnow()
        new_version = self.generate_version_id()
        endpoint_set_pb = None

        if updated_spec_pb is not None:
            rev_meta_pb = model_pb2.EndpointSetRevisionMeta(
                id=new_version,
                namespace_id=namespace_id,
                endpoint_set_id=endpoint_set_id,
                author=login,
                comment=comment,
                backend_versions=[backend_version]
            )
            rev_meta_pb.resolved_from.CopyFrom(backend_selector_pb)
            rev_meta_pb.ctime.FromDatetime(utcnow)
            rev_pb = model_pb2.EndpointSetRevision(meta=rev_meta_pb, spec=updated_spec_pb)
            self.db.save_endpoint_set_rev(rev_pb)

        for endpoint_set_pb in self.zk.update_endpoint_set(namespace_id, endpoint_set_id):
            if updated_spec_pb is not None and endpoint_set_pb.spec != updated_spec_pb:
                if endpoint_set_pb.meta.version != version:
                    self.db.remove_endpoint_set_rev(new_version)
                    raise errors.ConflictError(
                        'Endpoint set modification conflict: assumed version="{}", current="{}"'.format(
                            version, endpoint_set_pb.meta.version))

                if endpoint_set_pb.spec.deleted:
                    raise errors.ConflictError('Endpoint set with "spec.deleted" flag can not be updated')

                endpoint_set_pb.spec.CopyFrom(updated_spec_pb)
                endpoint_set_pb.meta.version = new_version
                endpoint_set_pb.meta.comment = comment
                endpoint_set_pb.meta.author = login
                endpoint_set_pb.meta.mtime.FromDatetime(utcnow)

                del endpoint_set_pb.meta.backend_versions[:]
                endpoint_set_pb.meta.backend_versions.append(backend_version)
                endpoint_set_pb.meta.resolved_from.CopyFrom(backend_selector_pb)
            if updated_auth_pb is not None:
                endpoint_set_pb.meta.auth.CopyFrom(updated_auth_pb)
            if updated_is_system_pb is not None:
                endpoint_set_pb.meta.is_system.CopyFrom(updated_is_system_pb)
                endpoint_set_pb.meta.is_system.author = login
                endpoint_set_pb.meta.is_system.mtime.GetCurrentTime()

        return endpoint_set_pb

    def update_resolved_from_and_add_backend_version_to_endpoint_set_rev(
            self, namespace_id, endpoint_set_id, endpoint_set_version, backend_version_to_add, resolved_from_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type endpoint_set_id:
        :param six.text_type endpoint_set_version:
        :param six.text_type backend_version_to_add:
        :param awacs.proto.model_pb2.BackendSelector resolved_from_pb:
        :rtype: bool
        """
        changed = False
        for endpoint_set_pb in self.zk.update_endpoint_set(namespace_id, endpoint_set_id):
            if endpoint_set_pb.meta.version == endpoint_set_version:
                if backend_version_to_add not in endpoint_set_pb.meta.backend_versions:
                    endpoint_set_pb.meta.backend_versions.append(backend_version_to_add)
                    changed = True
                if endpoint_set_pb.meta.resolved_from != resolved_from_pb:
                    endpoint_set_pb.meta.resolved_from.CopyFrom(resolved_from_pb)
                    changed = True
            if not changed:
                break

        if changed:
            self.db.add_backend_version_to_endpoint_set_rev(endpoint_set_version, backend_version_to_add)
            self.db.update_resolved_from_of_endpoint_set_rev(endpoint_set_version, resolved_from_pb)

        return changed

    def create_backend_if_missing(self, meta_pb, spec_pb, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.BackendMeta
        :type spec_pb: model_pb2.BackendSpec
        :type login: six.text_type
        :rtype: model_pb2.Backend
        """
        existing_backend_pb = self.cache.get_backend(meta_pb.namespace_id, meta_pb.id)
        if existing_backend_pb is None:
            try:
                return self.create_backend(meta_pb=meta_pb, spec_pb=spec_pb, login=login)
            except errors.ConflictError:
                existing_backend_pb = self.zk.must_get_backend(meta_pb.namespace_id, meta_pb.id)
        if existing_backend_pb.spec != spec_pb:
            raise errors.ConflictError(u'Backend "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_backend_pb

    def create_upstream_if_missing(self, meta_pb, spec_pb, rev_index_pb, login=util.NANNY_ROBOT_LOGIN,
                                   compare_with_existing=True):
        """
        :type meta_pb: model_pb2.UpstreamMeta
        :type spec_pb: model_pb2.UpstreamSpec
        :type rev_index_pb: model_pb2.RevisionGraphIndex
        :type login: six.text_type
        :param bool compare_with_existing: raise ConflictError if existing upstream doesn't match the new one
        :rtype: model_pb2.Upstream
        :raises errors.ConflictError
        """
        existing_upstream_pb = self.cache.get_upstream(meta_pb.namespace_id, meta_pb.id)
        if existing_upstream_pb is None:
            try:
                return self.create_upstream(meta_pb=meta_pb, spec_pb=spec_pb, rev_index_pb=rev_index_pb, login=login)
            except errors.ConflictError:
                existing_upstream_pb = self.zk.must_get_upstream(meta_pb.namespace_id, meta_pb.id)
        if compare_with_existing and existing_upstream_pb.spec != spec_pb:
            raise errors.ConflictError(u'Upstream "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_upstream_pb

    def create_domain_if_missing(self, meta_pb, order_content_pb=None, spec_pb=None, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.DomainMeta
        :type order_content_pb: Optional[model_pb2.DomainOrder.Content]
        :type spec_pb: Optional[model_pb2.DomainSpec]
        :type login: six.text_type
        :rtype: model_pb2.Upstream
        """
        assert order_content_pb or spec_pb
        assert not (order_content_pb and spec_pb)
        existing_domain_pb = self.cache.get_domain(meta_pb.namespace_id, meta_pb.id)
        if existing_domain_pb is None:
            try:
                return self.create_domain(order_pb=order_content_pb, spec_pb=spec_pb, meta_pb=meta_pb, login=login)
            except errors.ConflictError:
                existing_domain_pb = self.zk.must_get_domain(meta_pb.namespace_id, meta_pb.id)
        if ((order_content_pb and existing_domain_pb.order.content != order_content_pb)
                or (spec_pb and existing_domain_pb.spec != spec_pb)):
            raise errors.ConflictError(u'Domain "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_domain_pb

    def create_balancer_if_missing(self, meta_pb, order_content_pb, rev_index_pb, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.BalancerMeta
        :type order_content_pb: model_pb2.BalancerOrder.Content
        :type rev_index_pb: model_pb2.RevisionGraphIndex | None
        :type login: six.text_type
        :rtype: model_pb2.Balancer
        """
        existing_balancer_pb = self.cache.get_balancer(meta_pb.namespace_id, meta_pb.id)
        if existing_balancer_pb is None:
            try:
                balancer_pb, _ = self.create_balancer(meta_pb=meta_pb, order_content_pb=order_content_pb,
                                                      rev_index_pb=rev_index_pb, login=login)
                return balancer_pb
            except errors.ConflictError:
                existing_balancer_pb = self.zk.must_get_balancer(meta_pb.namespace_id, meta_pb.id)
        if existing_balancer_pb.order.content != order_content_pb:
            raise errors.ConflictError(u'Balancer "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_balancer_pb

    def create_endpoint_set_if_missing(self, meta_pb, spec_pb, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.EndpointSetMeta
        :type spec_pb: model_pb2.EndpointSetSpec
        :type login: six.text_type
        :rtype: model_pb2.EndpointSet
        """
        existing_es_pb = self.cache.get_endpoint_set(meta_pb.namespace_id, meta_pb.id)
        if existing_es_pb is None:
            try:
                return self.create_endpoint_set(meta_pb=meta_pb, spec_pb=spec_pb, login=login)
            except errors.ConflictError:
                existing_es_pb = self.zk.must_get_endpoint_set(meta_pb.namespace_id, meta_pb.id)
        if existing_es_pb.spec != spec_pb:
            raise errors.ConflictError(
                u'Endpoint set "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_es_pb

    def create_cert_if_missing(self, meta_pb, order_content_pb=None, spec_pb=None, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.CertificateMeta
        :type order_content_pb: Optional[model_pb2.CertificateOrder.Content]
        :type spec_pb: Optional[model_pb2.CertificateSpec]
        :type login: six.text_type
        :rtype: model_pb2.Certificate
        """
        assert order_content_pb or spec_pb
        assert not (order_content_pb and spec_pb)
        existing_cert_pb = self.cache.get_cert(meta_pb.namespace_id, meta_pb.id)
        if existing_cert_pb is None:
            try:
                return self.create_cert(meta_pb=meta_pb, order_pb=order_content_pb, spec_pb=spec_pb, login=login)
            except errors.ConflictError:
                existing_cert_pb = self.zk.must_get_cert(meta_pb.namespace_id, meta_pb.id)
        if ((order_content_pb and existing_cert_pb.order.content != order_content_pb)
                or (spec_pb and existing_cert_pb.spec != spec_pb)):
            raise errors.ConflictError(
                u'Certificate "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_cert_pb

    def create_dns_record_if_missing(self, meta_pb, spec_pb, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.DnsRecordMeta
        :type spec_pb: model_pb2.DnsRecordSpec
        :type login: six.text_type
        :rtype: model_pb2.DnsRecord
        """
        existing_dns_record_pb = self.cache.get_dns_record(meta_pb.namespace_id, meta_pb.id)
        if existing_dns_record_pb is None:
            try:
                return self.create_dns_record(meta_pb=meta_pb, spec_pb=spec_pb, login=login)
            except errors.ConflictError:
                existing_dns_record_pb = self.zk.must_get_dns_record(meta_pb.namespace_id, meta_pb.id)
        if existing_dns_record_pb.spec != spec_pb:
            raise errors.ConflictError(u'DNS record "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_dns_record_pb

    def create_dns_record_operation_if_missing(self, meta_pb, order_pb, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.DnsRecordOperationMeta
        :type order_pb: model_pb2.DnsRecordOperationOrder
        :type login: six.text_type
        :rtype: model_pb2.DnsRecordOperation
        """
        existing_dns_record_op_pb = self.cache.get_dns_record_operation(meta_pb.namespace_id, meta_pb.id)
        if existing_dns_record_op_pb is None:
            try:
                return self.create_dns_record_operation(meta_pb=meta_pb, order_pb=order_pb, login=login)
            except errors.ConflictError:
                existing_dns_record_op_pb = self.zk.must_get_dns_record_operation(meta_pb.namespace_id, meta_pb.id)
        if existing_dns_record_op_pb.order != order_pb:
            raise errors.ConflictError(u'DNS record operation "{}" already exists and has unexpected settings'.format(
                meta_pb.id))
        return existing_dns_record_op_pb

    def create_name_server_if_missing(self, meta_pb, spec_pb, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.NameServerMeta
        :type spec_pb: model_pb2.NameServerSpec
        :type login: six.text_type
        :rtype: model_pb2.NameServer
        """
        existing_name_server_pb = self.cache.get_name_server(meta_pb.namespace_id, meta_pb.id)
        if existing_name_server_pb is None:
            try:
                return self.create_name_server(meta_pb=meta_pb, spec_pb=spec_pb, login=login)
            except errors.ConflictError:
                existing_name_server_pb = self.zk.must_get_name_server(meta_pb.namespace_id, meta_pb.id)
        if existing_name_server_pb.spec != spec_pb:
            raise errors.ConflictError(
                u'Name server "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_name_server_pb

    def create_namespace_if_missing(self, meta_pb, order_content_pb=None, spec_pb=None, login=util.NANNY_ROBOT_LOGIN):
        """
        :type meta_pb: model_pb2.NamespaceMeta
        :type order_content_pb: model_pb2.NamespaceOrder.Content
        :type spec_pb: model_pb2.NamespaceSpec
        :type login: six.text_type
        :rtype: model_pb2.NameServer
        """
        assert not (order_content_pb and spec_pb)
        existing_namespace_pb = self.cache.get_namespace(meta_pb.id)
        if existing_namespace_pb is None:
            try:
                return self.create_namespace(meta_pb=meta_pb,
                                             order_content_pb=order_content_pb,
                                             spec_pb=spec_pb,
                                             login=login)
            except errors.ConflictError:
                existing_namespace_pb = self.zk.must_get_namespace(meta_pb.id)
        if ((order_content_pb and existing_namespace_pb.order.content != order_content_pb)
                or (spec_pb and existing_namespace_pb.spec != spec_pb)):
            raise errors.ConflictError(
                u'Namespace "{}" already exists and has unexpected settings'.format(meta_pb.id))
        return existing_namespace_pb

    # DO NOT USE IN PRODUCTION! For initial initialization only, special for AWACS-1326
    def update_indices(self, object_name, namespace_id, id, version, rev_index_pbs):
        """
        :type namespace_id: six.text_type
        :type id: six.text_type
        :type version: six.text_type
        :type rev_index_pbs: Iterable[model_pb2.RevisionGraphIndex]
        :raises: errors.ConflictError, errors.NotFoundError
        """
        new_version = self.generate_version_id()

        assert object_name in ('upstream', 'balancer')
        update = self.zk.update_upstream if object_name == 'upstream' else self.zk.update_balancer

        for object_pb in update(namespace_id, id):
            if object_pb.meta.version != version:
                raise errors.ConflictError(
                    'Upstream indices modification conflict: assumed version="{}", current="{}"'.format(
                        version, object_pb.meta.version))

            if object_pb.spec.deleted:
                raise errors.ConflictError('Upstream with "spec.deleted" flag can not be updated')

            del object_pb.meta.indices[:]
            updated_index_pbs = []
            for ind_pb in rev_index_pbs:
                if not ind_pb.id:
                    ind_pb = util.clone_pb(ind_pb)
                    ind_pb.id = object_pb.meta.version
                    ind_pb.ctime.CopyFrom(object_pb.meta.ctime)
                updated_index_pbs.append(ind_pb)
            object_pb.meta.indices.extend(updated_index_pbs)
