# coding: utf-8
import datetime

import bson
import inject
import pymongo
from awacs.lib.vectors.version import Version
from google.protobuf.message import DecodeError

from awacs.lib.mongo import get_db
from awacs.lib.pagination import SliceResult
from awacs.model import errors, cache, objects
from infra.awacs.proto import model_pb2


class IMongoStorage(object):
    @classmethod
    def instance(cls):
        """
        :rtype: MongoStorage
        """
        return inject.instance(cls)


class MongoStorage(IMongoStorage):
    DEFAULT_LIMIT = 30

    @property
    def _db(self):
        return get_db()

    def ensure_indexes(self):
        l3_balancer_revisions = self._db.l3_balancer_revisions
        l3_balancer_revisions.ensure_index('namespace', name='namespace_index')
        l3_balancer_revisions.ensure_index('l3_balancer_id', name='l3_balancer_id_index')
        l3_balancer_revisions.ensure_index('author', name='author_index')
        l3_balancer_revisions.ensure_index('ctime', name='ctime_index')
        knob_revisions = self._db.knob_revisions
        knob_revisions.ensure_index('namespace', name='namespace_index')
        knob_revisions.ensure_index('knob_id', name='knob_id_index')
        knob_revisions.ensure_index('author', name='author_index')
        knob_revisions.ensure_index('ctime', name='ctime_index')
        balancer_revisions = self._db.balancer_revisions
        balancer_revisions.ensure_index('namespace', name='namespace_index')
        balancer_revisions.ensure_index('balancer_id', name='balancer_id_index')
        balancer_revisions.ensure_index('author', name='author_index')
        balancer_revisions.ensure_index('ctime', name='ctime_index')
        upstream_revisions = self._db.upstream_revisions
        upstream_revisions.ensure_index('balancer_id', name='balancer_id_index')
        upstream_revisions.ensure_index('namespace', name='namespace_index')
        upstream_revisions.ensure_index('upstream_id', name='upstream_id_index')
        upstream_revisions.ensure_index('author', name='author_index')
        upstream_revisions.ensure_index('ctime', name='ctime_index')
        domain_revisions = self._db.domain_revisions
        domain_revisions.ensure_index('namespace', name='namespace_index')
        domain_revisions.ensure_index('domain_id', name='domain_id_index')
        domain_revisions.ensure_index('author', name='author_index')
        domain_revisions.ensure_index('ctime', name='ctime_index')
        backend_revisions = self._db.backend_revisions
        backend_revisions.ensure_index('balancer_id', name='balancer_id_index')
        backend_revisions.ensure_index('namespace', name='namespace_index')
        backend_revisions.ensure_index('backend_id', name='backend_id_index')
        backend_revisions.ensure_index('author', name='author_index')
        backend_revisions.ensure_index('ctime', name='ctime_index')
        endpoint_set_revisions = self._db.endpoint_set_revisions
        endpoint_set_revisions.ensure_index('balancer_id', name='balancer_id_index')
        endpoint_set_revisions.ensure_index('namespace', name='namespace_index')
        endpoint_set_revisions.ensure_index('endpoint_set_id', name='endpoint_set_id_index')
        endpoint_set_revisions.ensure_index('author', name='author_index')
        endpoint_set_revisions.ensure_index('ctime', name='ctime_index')
        dns_record_revisions = self._db.dns_record_revisions
        dns_record_revisions.ensure_index('namespace', name='namespace_index')
        dns_record_revisions.ensure_index('dns_record_id', name='dns_record_id_index')
        dns_record_revisions.ensure_index('author', name='author_index')
        dns_record_revisions.ensure_index('ctime', name='ctime_index')
        name_server_revisions = self._db.name_server_revisions
        name_server_revisions.ensure_index('namespace', name='namespace_index')
        name_server_revisions.ensure_index('name_server_id', name='name_server_id_index')
        name_server_revisions.ensure_index('author', name='author_index')
        name_server_revisions.ensure_index('ctime', name='ctime_index')
        cert_revisions = self._db.cert_revisions
        cert_revisions.ensure_index('namespace', name='namespace_index')
        cert_revisions.ensure_index('cert_id', name='cert_id_index')
        cert_revisions.ensure_index('author', name='author_index')
        cert_revisions.ensure_index('ctime', name='ctime_index')
        namespace_revisions = self._db.namespace_revisions
        namespace_revisions.ensure_index('namespace', name='namespace_index')
        namespace_revisions.ensure_index('author', name='author_index')
        namespace_revisions.ensure_index('ctime', name='ctime_index')
        objects.WeightSection.mongo.ensure_index('namespace')
        objects.WeightSection.mongo.ensure_index(objects.WeightSection.desc.canonical_id)
        objects.WeightSection.mongo.ensure_index('author')
        objects.WeightSection.mongo.ensure_index('ctime')
        objects.L7HeavyConfig.mongo.ensure_index('namespace')
        objects.L7HeavyConfig.mongo.ensure_index(objects.L7HeavyConfig.desc.canonical_id)
        objects.L7HeavyConfig.mongo.ensure_index('author')
        objects.L7HeavyConfig.mongo.ensure_index('ctime')

        notifications_statistics_entries = self._db.notifications_statistics_entries
        notifications_statistics_entries.ensure_index([
            ('span', pymongo.ASCENDING),
            ('start', pymongo.ASCENDING),
        ], unique=True)
        usage_statistics_entries = self._db.usage_statistics_entries
        usage_statistics_entries.ensure_index([
            ('span', pymongo.ASCENDING),
            ('start', pymongo.ASCENDING),
        ], unique=True)

    def save_namespace_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.NamespaceRevision
        """
        content = rev_pb.SerializeToString()
        self._db.namespace_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,  # not `namespace_id` for consistency's sake
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def remove_namespace_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.namespace_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def get_namespace_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.NamespaceRevision or None
        """
        data = self._db.namespace_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.NamespaceRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_namespace_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.NamespaceRevision
        """
        rev_pb = self.get_namespace_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Namespace revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_namespace_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.namespace_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_namespace_revs_by_namespace_id(self, namespace_id):
        return self.remove_namespace_revs({
            'namespace': namespace_id,
        })

    def list_namespace_revs(self, namespace_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace id
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
        }
        total = self._db.namespace_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.namespace_revisions.find(spec,
                                                skip=skip or 0,
                                                limit=limit or self.DEFAULT_LIMIT,
                                                sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.NamespaceRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def save_knob_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.KnobRevision
        """
        content = rev_pb.SerializeToString()
        self._db.knob_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'knob_id': rev_pb.meta.knob_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def get_knob_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.KnobRevision or None
        """
        data = self._db.knob_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.KnobRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_knob_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.KnobRevision
        """
        rev_pb = self.get_knob_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Knob revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_knob_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.knob_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_knob_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.knob_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_knob_revs_by_namespace_id(self, namespace_id):
        return self.remove_knob_revs({
            'namespace': namespace_id,
        })

    def remove_knob_revs_by_namespace_and_knob_id(self, namespace_id, knob_id):
        return self.remove_knob_revs({
            'namespace': namespace_id,
            'knob_id': knob_id,
        })

    def list_knob_revs(self, namespace_id, knob_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the knob
        :param six.text_type knob_id: An identifier of the knob
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'knob_id': knob_id,
        }
        total = self._db.knob_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.knob_revisions.find(spec,
                                           skip=skip or 0,
                                           limit=limit or self.DEFAULT_LIMIT,
                                           sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.KnobRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_knob_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.knob_revisions.find(spec,
                                            projection=fields,
                                            skip=skip,
                                            limit=limit,
                                            sort=sort)

    def save_l3_balancer_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.L3BalancerRevision
        """
        content = rev_pb.SerializeToString()
        self._db.l3_balancer_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'l3_balancer_id': rev_pb.meta.l3_balancer_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def get_l3_balancer_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.L3BalancerRevision or None
        """
        data = self._db.l3_balancer_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.L3BalancerRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_l3_balancer_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.L3BalancerRevision
        """
        rev_pb = self.get_l3_balancer_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('L3 balancer revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_l3_balancer_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.l3_balancer_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_l3_balancer_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.l3_balancer_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_l3_balancer_revs_by_namespace_id(self, namespace_id):
        return self.remove_l3_balancer_revs({
            'namespace': namespace_id,
        })

    def remove_l3_balancer_revs_by_namespace_and_l3_balancer_id(self, namespace_id, l3_balancer_id):
        return self.remove_l3_balancer_revs({
            'namespace': namespace_id,
            'l3_balancer_id': l3_balancer_id,
        })

    def list_l3_balancer_revs(self, namespace_id, l3_balancer_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the l3 balancer
        :param six.text_type l3_balancer_id: An identifier of the l3 balancer
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'l3_balancer_id': l3_balancer_id,
        }
        total = self._db.l3_balancer_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.l3_balancer_revisions.find(spec,
                                                  skip=skip or 0,
                                                  limit=limit or self.DEFAULT_LIMIT,
                                                  sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.L3BalancerRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_l3_balancer_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.l3_balancer_revisions.find(spec,
                                                   projection=fields,
                                                   skip=skip,
                                                   limit=limit,
                                                   sort=sort)

    def save_balancer_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.BalancerRevision
        """
        content = rev_pb.SerializeToString()
        self._db.balancer_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'balancer_id': rev_pb.meta.balancer_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def get_balancer_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.BalancerRevision or None
        """
        data = self._db.balancer_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.BalancerRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_balancer_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.BalancerRevision
        """
        rev_pb = self.get_balancer_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Balancer revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_balancer_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.balancer_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_balancer_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.balancer_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_balancer_revs_by_namespace_id(self, namespace_id):
        return self.remove_balancer_revs({
            'namespace': namespace_id,
        })

    def remove_balancer_revs_by_namespace_and_balancer_id(self, namespace_id, balancer_id):
        return self.remove_balancer_revs({
            'namespace': namespace_id,
            'balancer_id': balancer_id,
        })

    def list_balancer_revs(self, namespace_id, balancer_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the balancer
        :param six.text_type balancer_id: An identifier of the balancer
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'balancer_id': balancer_id,
        }
        total = self._db.balancer_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.balancer_revisions.find(spec,
                                               skip=skip or 0,
                                               limit=limit or self.DEFAULT_LIMIT,
                                               sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.BalancerRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_balancer_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.balancer_revisions.find(spec,
                                                projection=fields,
                                                skip=skip,
                                                limit=limit,
                                                sort=sort)

    def save_upstream_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.UpstreamRevision
        """
        content = rev_pb.SerializeToString()
        self._db.upstream_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'upstream_id': rev_pb.meta.upstream_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def get_upstream_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.UpstreamRevision or None
        """
        data = self._db.upstream_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.UpstreamRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_upstream_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.UpstreamRevision
        """
        rev_pb = self.get_upstream_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Upstream revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_upstream_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.upstream_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_upstream_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.upstream_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_upstream_revs_by_namespace_id(self, namespace_id):
        return self.remove_upstream_revs({
            'namespace': namespace_id,
        })

    def remove_upstream_revs_by_namespace_and_upstream_id(self, namespace_id, upstream_id):
        return self.remove_upstream_revs({
            'namespace': namespace_id,
            'upstream_id': upstream_id,
        })

    def list_upstream_revs(self, namespace_id, upstream_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the upstream
        :param six.text_type upstream_id: An identifier of the upstream
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'upstream_id': upstream_id,
        }
        total = self._db.upstream_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.upstream_revisions.find(spec,
                                               skip=skip or 0,
                                               limit=limit or self.DEFAULT_LIMIT,
                                               sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.UpstreamRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_upstream_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.upstream_revisions.find(spec,
                                                projection=fields,
                                                skip=skip,
                                                limit=limit,
                                                sort=sort)

    def save_domain_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.DomainRevision
        """
        content = rev_pb.SerializeToString()
        self._db.domain_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'domain_id': rev_pb.meta.domain_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def get_domain_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.DomainRevision or None
        """
        data = self._db.domain_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.DomainRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_domain_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.DomainRevision
        """
        rev_pb = self.get_domain_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Domain revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_domain_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.domain_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_domain_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.domain_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_domain_revs_by_namespace_id(self, namespace_id):
        return self.remove_domain_revs({
            'namespace': namespace_id,
        })

    def remove_domain_revs_by_namespace_and_domain_id(self, namespace_id, domain_id):
        return self.remove_domain_revs({
            'namespace': namespace_id,
            'domain_id': domain_id,
        })

    def list_domain_revs(self, namespace_id, domain_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the domain
        :param six.text_type domain_id: An identifier of the domain
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'domain_id': domain_id,
        }
        total = self._db.domain_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.domain_revisions.find(spec,
                                             skip=skip or 0,
                                             limit=limit or self.DEFAULT_LIMIT,
                                             sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.DomainRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_domain_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.domain_revisions.find(spec,
                                              projection=fields,
                                              skip=skip,
                                              limit=limit,
                                              sort=sort)

    def save_backend_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.BackendRevision
        """
        content = rev_pb.SerializeToString()
        self._db.backend_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'backend_id': rev_pb.meta.backend_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def list_backend_revs(self, namespace_id, backend_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the backend
        :param six.text_type backend_id: An identifier of the backend
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'backend_id': backend_id,
        }
        total = self._db.backend_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.backend_revisions.find(spec,
                                              skip=skip or 0,
                                              limit=limit or self.DEFAULT_LIMIT,
                                              sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.BackendRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_backend_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.backend_revisions.find(spec,
                                               projection=fields,
                                               skip=skip,
                                               limit=limit,
                                               sort=sort)

    def get_backend_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.BackendRevision or None
        """
        data = self._db.backend_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.BackendRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_backend_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.BackendRevision
        """
        rev_pb = self.get_backend_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Backend revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_backend_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.backend_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_backend_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.backend_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_backend_revs_by_namespace_id(self, namespace_id):
        return self.remove_backend_revs({
            'namespace': namespace_id,
        })

    def remove_backend_revs_by_namespace_and_backend_id(self, namespace_id, backend_id):
        return self.remove_backend_revs({
            'namespace': namespace_id,
            'backend_id': backend_id,
        })

    def save_endpoint_set_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.EndpointSetRevision
        """
        content = rev_pb.SerializeToString()
        meta_pb = rev_pb.meta
        self._db.endpoint_set_revisions.insert({
            '_id': meta_pb.id,
            'namespace': meta_pb.namespace_id,
            'endpoint_set_id': meta_pb.endpoint_set_id,
            'backend_versions': list(meta_pb.backend_versions),
            'ctime': meta_pb.ctime.ToMilliseconds(),
            'author': meta_pb.author,
            'comment': meta_pb.comment,
            'content': bson.Binary(content),
        })

    def add_backend_version_to_endpoint_set_rev(self, rev_id, backend_rev_id):
        """
        Marks endpoint set revision `rev_id` as compatible with `backend_rev_id`.

        :param six.text_type rev_id: An identifier of the endpoint set revision to update
        :param six.text_type backend_rev_id: An identifier of the backend revision to add to endpoint set revision
        :rtype: bool
        """
        r = self._db.endpoint_set_revisions.update({
            '_id': rev_id,
        }, {
            '$addToSet': {
                'backend_versions': backend_rev_id,
            },
        }, multi=False)
        return bool(r.get('n'))

    def update_resolved_from_of_endpoint_set_rev(self, rev_id, backend_selector_pb):
        """
        :param six.text_type rev_id: An identifier of the endpoint set revision to update
        :param awacs.proto.model_pb2.BackendSelector backend_selector_pb:
            Backend selector to be written in the "resolved_from" field of the endpoint set revision
        :rtype: bool
        """
        content = backend_selector_pb.SerializeToString()
        r = self._db.endpoint_set_revisions.update({
            '_id': rev_id,
        }, {
            '$set': {
                'resolved_from': bson.Binary(content),
            },
        })
        return bool(r.get('n'))

    def list_endpoint_set_revs(self, namespace_id, endpoint_set_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the backend
        :param six.text_type endpoint_set_id: An identifier of the backend
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'endpoint_set_id': endpoint_set_id,
        }
        total = self._db.endpoint_set_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.endpoint_set_revisions.find(spec,
                                                   skip=skip or 0,
                                                   limit=limit or self.DEFAULT_LIMIT,
                                                   sort=[('ctime', -1)])
        endpoint_set_revs = []
        for item in cur:
            endpoint_set_rev = model_pb2.EndpointSetRevision()
            try:
                endpoint_set_rev.MergeFromString(item['content'])
            except DecodeError:
                continue
            endpoint_set_revs.append(endpoint_set_rev)
        return SliceResult(endpoint_set_revs, total)

    def get_endpoint_set_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.EndpointSetRevision or None
        """
        data = self._db.endpoint_set_revisions.find_one(
            rev_id, projection=('content', 'backend_versions', 'resolved_from'))
        if not data:
            return None
        rev_pb = model_pb2.EndpointSetRevision()
        rev_pb.MergeFromString(data['content'])

        del rev_pb.meta.backend_versions[:]
        rev_pb.meta.backend_versions.extend(data['backend_versions'])

        if 'resolved_from' in data:
            rev_pb.meta.ClearField('resolved_from')
            rev_pb.meta.resolved_from.MergeFromString(data['resolved_from'])

        return rev_pb

    def must_get_endpoint_set_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.EndpointSetRevision
        """
        rev_pb = self.get_endpoint_set_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Endpoint set revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_endpoint_set_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.endpoint_set_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_endpoint_set_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.endpoint_set_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_endpoint_set_revs_by_namespace_id(self, namespace_id):
        return self.remove_endpoint_set_revs({
            'namespace': namespace_id,
        })

    def remove_endpoint_set_revs_by_namespace_and_endpoint_set_id(self, namespace_id, endpoint_set_id):
        return self.remove_endpoint_set_revs({
            'namespace': namespace_id,
            'endpoint_set_id': endpoint_set_id,
        })

    def save_name_server_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.NameServerRevision
        """
        content = rev_pb.SerializeToString()
        self._db.name_server_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'name_server_id': rev_pb.meta.name_server_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def remove_name_server_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.name_server_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def save_dns_record_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.DnsRecordRevision
        """
        content = rev_pb.SerializeToString()
        self._db.dns_record_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'dns_record_id': rev_pb.meta.dns_record_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def get_dns_record_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.DnsRecordRevision or None
        """
        data = self._db.dns_record_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.DnsRecordRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_dns_record_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.DnsRecordRevision
        """
        rev_pb = self.get_dns_record_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('DNS record revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_dns_record_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.dns_record_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_dns_record_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.dns_record_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_dns_record_revs_by_namespace_id(self, namespace_id):
        return self.remove_dns_record_revs({
            'namespace': namespace_id,
        })

    def remove_dns_record_revs_by_namespace_and_dns_record_id(self, namespace_id, dns_record_id):
        return self.remove_dns_record_revs({
            'namespace': namespace_id,
            'dns_record_id': dns_record_id,
        })

    def list_dns_record_revs(self, namespace_id, dns_record_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the dns record
        :param six.text_type dns_record_id: An identifier of the dns record
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'dns_record_id': dns_record_id,
        }
        total = self._db.dns_record_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.dns_record_revisions.find(spec,
                                                 skip=skip or 0,
                                                 limit=limit or self.DEFAULT_LIMIT,
                                                 sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.DnsRecordRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_dns_record_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.dns_record_revisions.find(spec,
                                                  projection=fields,
                                                  skip=skip,
                                                  limit=limit,
                                                  sort=sort)

    def save_cert_rev(self, rev_pb):
        """
        :type rev_pb: awacs.proto.model_pb2.CertificateRevision
        """
        content = rev_pb.SerializeToString()
        self._db.cert_revisions.insert({
            '_id': rev_pb.meta.id,
            'namespace': rev_pb.meta.namespace_id,
            'cert_id': rev_pb.meta.certificate_id,
            'ctime': rev_pb.meta.ctime.ToMilliseconds(),
            'author': rev_pb.meta.author,
            'comment': rev_pb.meta.comment,
            'content': bson.Binary(content),
        })

    def get_cert_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :rtype: awacs.proto.model_pb2.CertificateRevision or None
        """
        data = self._db.cert_revisions.find_one(rev_id, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.CertificateRevision()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_cert_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to return
        :raises: errors.NotFoundError
        :rtype: awacs.proto.model_pb2.CertificateRevision
        """
        rev_pb = self.get_cert_rev(rev_id)
        if not rev_pb:
            raise errors.NotFoundError('Certificate revision "{}" does not exist'.format(rev_id))
        return rev_pb

    def remove_cert_rev(self, rev_id):
        """
        :param six.text_type rev_id: An identifier of the revision to remove
        :returns: Whether the revision was removed
        :rtype: bool
        """
        r = self._db.cert_revisions.remove({'_id': rev_id}, multi=False)
        return bool(r.get('n'))

    def remove_cert_revs(self, spec):
        """
        :param dict spec: pymongo's query dictionary matching revisions to remove
        :returns: A number of removed revisions
        :rtype: int
        """
        r = self._db.cert_revisions.remove(spec, multi=True)
        return r.get('n', 0)

    def remove_cert_revs_by_namespace_id(self, namespace_id):
        return self.remove_cert_revs({
            'namespace': namespace_id,
        })

    def remove_cert_revs_by_namespace_and_cert_id(self, namespace_id, cert_id):
        return self.remove_cert_revs({
            'namespace': namespace_id,
            'cert_id': cert_id,
        })

    def list_cert_revs(self, namespace_id, cert_id, skip=None, limit=None):
        """
        :param six.text_type namespace_id: A namespace of the certificate
        :param six.text_type cert_id: An identifier of the certificate
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'namespace': namespace_id,
            'cert_id': cert_id,
        }
        total = self._db.cert_revisions.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self._db.cert_revisions.find(spec,
                                           skip=skip or 0,
                                           limit=limit or self.DEFAULT_LIMIT,
                                           sort=[('ctime', -1)])
        rev_pbs = []
        for item in cur:
            rev_pb = model_pb2.CertificateRevision()
            try:
                rev_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            rev_pbs.append(rev_pb)
        return SliceResult(rev_pbs, total)

    def raw_list_cert_revs(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.cert_revisions.find(spec,
                                            projection=fields,
                                            skip=skip,
                                            limit=limit,
                                            sort=sort)

    def get_notifications_statistics_entry(self, span, start):
        """
        :param six.text_type span: NotificationsStatisticsEntry.Span.*
        :param datetime.datetime start:
        :rtype: awacs.proto.model_pb2.NotificationsStatisticsEntry or None
        """
        data = self._db.notifications_statistics_entries.find_one({
            'span': span,
            'start': start,
        }, projection=('content',))
        if not data:
            return None
        entry_pb = model_pb2.NotificationsStatisticsEntry()
        entry_pb.MergeFromString(data['content'])
        return entry_pb

    def save_notifications_statistics_entry(self, entry_pb):
        """
        :type entry_pb: awacs.proto.model_pb2.NotificationsStatisticsEntry
        """
        content = entry_pb.SerializeToString()
        self._db.notifications_statistics_entries.insert({
            'span': model_pb2.NotificationsStatisticsEntry.Span.Name(entry_pb.span),
            'start': entry_pb.start.ToDatetime(),
            'content': bson.Binary(content),
        })

    def list_notifications_statistics_entries(self, span='DATE', start=None, end=None, skip=0, limit=0):
        """
        :param six.text_type span: model_pb2.NotificationsStatisticsEntry.Span.*
        :type start: datetime.date
        :type end: datetime.end
        :type skip: int
        :type limit: int
        :rtype: Iterator[model_pb2.NotificationsStatisticsEntry]
        """
        spec = {
            'span': span,
        }
        if start is not None:
            spec.setdefault('start', {})['$gte'] = datetime.datetime(*start.timetuple()[:3])
        if end is not None:
            spec.setdefault('start', {})['$lte'] = datetime.datetime(*end.timetuple()[:3])

        for item in self._db.notifications_statistics_entries.find(
            spec,
            skip=skip or 0,
            limit=limit or self.DEFAULT_LIMIT,
            sort=[('start', 1)],
        ):
            entry_pb = model_pb2.NotificationsStatisticsEntry()
            entry_pb.MergeFromString(item['content'])
            yield entry_pb

    def save_usage_statistics_entry(self, entry_pb):
        """
        :type entry_pb: awacs.proto.model_pb2.UsageStatisticsEntry
        """
        content = entry_pb.SerializeToString()
        self._db.usage_statistics_entries.insert({
            'span': model_pb2.UsageStatisticsEntry.Span.Name(entry_pb.span),
            'start': entry_pb.start.ToDatetime(),
            'content': bson.Binary(content),
        })

    def must_get_usage_statistics_entry(self, span, start):
        """
        :param six.text_type span: model_pb2.UsageStatisticsEntry.Span.*
        :param datetime.datetime start:
        :rtype: awacs.proto.model_pb2.UsageStatisticsEntry
        :raises: errors.NotFoundError
        """
        entry_pb = self.get_usage_statistics_entry(span, start)
        if not entry_pb:
            raise errors.NotFoundError('Usage statistics entry starting {} '
                                       'spanning {} does not exist'.format(start, span))
        return entry_pb

    def get_usage_statistics_entry(self, span, start):
        """
        :param six.text_type span: UsageStatisticsEntry.Span.*
        :param datetime.datetime start:
        :rtype: awacs.proto.model_pb2.UsageStatisticsEntry or None
        """
        data = self._db.usage_statistics_entries.find_one({
            'span': span,
            'start': start,
        }, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.UsageStatisticsEntry()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def list_usage_statistics_entries(self, span='DATE', skip=None, limit=None):
        """
        :param six.text_type span: UsageStatisticsEntry.Span.*
        :param int skip: A number of items to be skipped
        :param int limit: Maximum number of returned revisions
        :rtype: SliceResult
        """
        spec = {
            'span': span,
        }
        total = self._db.usage_statistics_entries.find(spec).count()
        if total == 0:
            return SliceResult([], 0)
        cur = self.raw_list_usage_statistics_entries(spec,
                                                     skip=skip or 0,
                                                     limit=limit or self.DEFAULT_LIMIT,
                                                     sort=[('start', 1)])
        entry_pbs = []
        for item in cur:
            entry_pb = model_pb2.UsageStatisticsEntry()
            try:
                entry_pb.MergeFromString(item['content'])
            except DecodeError:
                continue
            entry_pbs.append(entry_pb)
        return SliceResult(entry_pbs, total)

    def raw_list_usage_statistics_entries(self, spec=None, fields=None, skip=0, limit=0, sort=None):
        return self._db.usage_statistics_entries.find(spec,
                                                      projection=fields,
                                                      skip=skip,
                                                      limit=limit,
                                                      sort=sort)

    def save_load_statistics_entry(self, entry_pb):
        """
        :type entry_pb: awacs.proto.model_pb2.LoadStatisticsEntry
        """
        content = entry_pb.SerializeToString()
        self._db.load_statistics_entries.insert({
            'span': model_pb2.LoadStatisticsEntry.Span.Name(entry_pb.span),
            'start': entry_pb.start.ToDatetime(),
            'content': bson.Binary(content),
        })

    def remove_load_statistics_entry(self, span, start):
        """
        :type span: model_pb2.UsageStatisticsEntry.Span
        :type start: datetime.datetime
        """
        self._db.load_statistics_entries.remove({'span': span, 'start': start})

    def get_load_statistics_entry(self, span, start):
        """
        :param six.text_type span: LoadStatisticsEntry.Span.*
        :param datetime.datetime start:
        :rtype: awacs.proto.model_pb2.LoadStatisticsEntry or None
        """
        data = self._db.load_statistics_entries.find_one({
            'span': span,
            'start': start,
        }, projection=('content',))
        if not data:
            return None
        rev_pb = model_pb2.LoadStatisticsEntry()
        rev_pb.MergeFromString(data['content'])
        return rev_pb

    def must_get_load_statistics_entry(self, span, start):
        """
        :param six.text_type span: model_pb2.LoadStatisticsEntry.Span.*
        :param datetime.datetime start:
        :rtype: awacs.proto.model_pb2.LoadStatisticsEntry
        :raises: errors.NotFoundError
        """
        entry_pb = self.get_load_statistics_entry(span, start)
        if not entry_pb:
            raise errors.NotFoundError('Load statistics entry starting {} '
                                       'spanning {} does not exist'.format(start, span))
        return entry_pb


def find_balancer_revision_spec(version):
    """
    :type version: BalancerVersion
    :rtype: model_pb2.BalancerSpec
    """
    namespace_id, balancer_id = version.balancer_id
    balancer_pb = cache.IAwacsCache.instance().get_balancer(namespace_id, balancer_id)
    if balancer_pb is not None and balancer_pb.meta.version == version.version:
        return balancer_pb.spec
    else:
        return IMongoStorage.instance().must_get_balancer_rev(version.version).spec


def find_upstream_revision_spec(version):
    """
    :type version: UpstreamVersion
    :rtype: model_pb2.UpstreamSpec
    """
    namespace_id, upstream_id = version.upstream_id
    upstream_pb = cache.IAwacsCache.instance().get_upstream(namespace_id, upstream_id)
    if upstream_pb is not None and upstream_pb.meta.version == version.version:
        return upstream_pb.spec
    else:
        return IMongoStorage.instance().must_get_upstream_rev(version.version).spec


def find_domain_revision_spec(version):
    """
    :type version: DomainVersion
    :rtype: model_pb2.DomainSpec
    """
    namespace_id, domain_id = version.domain_id
    domain_pb = cache.IAwacsCache.instance().get_domain(namespace_id, domain_id)
    if domain_pb is not None and domain_pb.meta.version == version.version:
        return domain_pb.spec
    else:
        return IMongoStorage.instance().must_get_domain_rev(version.version).spec


def find_backend_revision_spec(version):
    """
    :type version: BackendVersion
    :rtype: model_pb2.BackendSpec
    """
    namespace_id, backend_id = version.backend_id
    backend_pb = cache.IAwacsCache.instance().get_backend(namespace_id, backend_id)
    if backend_pb is not None and backend_pb.meta.version == version.version:
        return backend_pb.spec
    else:
        return IMongoStorage.instance().must_get_backend_rev(version.version).spec


def find_endpoint_set_revision(version, current_pb=None):
    """
    :type current_pb: model_pb2.EndpointSet
    :param six.text_type version: revision identifier
    :rtype: model_pb2.EndpointSet | model_pb2.EndpointSetRevision
    """
    if current_pb is not None and current_pb.meta.version == version:
        return current_pb
    else:
        return IMongoStorage.instance().must_get_endpoint_set_rev(version)


def find_endpoint_set_revision2(version):
    """
    :type version: EndpointSet
    :rtype: model_pb2.EndpointSet | model_pb2.EndpointSetRevision
    """
    if isinstance(version, Version):
        namespace_id, endpoint_set_id = version.id  # new generic Version
    else:
        namespace_id, endpoint_set_id = version.endpoint_set_id  # old specific ESVersion
    endpoint_set_pb = cache.IAwacsCache.instance().get_endpoint_set(namespace_id, endpoint_set_id)
    if endpoint_set_pb is not None and endpoint_set_pb.meta.version == version.version:
        return endpoint_set_pb
    else:
        return IMongoStorage.instance().must_get_endpoint_set_rev(version.version)


def find_endpoint_set_revision_spec(version):
    """
    :type version: EndpointSet
    :rtype: model_pb2.EndpointSetSpec
    """
    return find_endpoint_set_revision2(version).spec


def find_knob_revision_spec(version):
    """
    :type version: KnobVersion
    :rtype: model_pb2.KnobSpec
    """
    namespace_id, knob_id = version.knob_id
    knob_pb = cache.IAwacsCache.instance().get_knob(namespace_id, knob_id)
    if knob_pb is not None and knob_pb.meta.version == version.version:
        return knob_pb.spec
    else:
        return IMongoStorage.instance().must_get_knob_rev(version.version).spec


def find_cert_revision_spec(version):
    """
    :type version: CertVersion
    :rtype: model_pb2.CertificateSpec
    """
    namespace_id, cert_id = version.cert_id
    cert_pb = cache.IAwacsCache.instance().get_cert(namespace_id, cert_id)
    if cert_pb is not None and cert_pb.meta.version == version.version:
        return cert_pb.spec
    else:
        return IMongoStorage.instance().must_get_cert_rev(version.version).spec


def find_l3_balancer_revision_spec(version, current_pb=None):
    """
    :type current_pb: model_pb2.L3Balancer
    :param six.text_type version: revision identifier
    :rtype: model_pb2.L3BalancerSpec
    """
    if current_pb is not None and current_pb.meta.version == version:
        return current_pb.spec
    else:
        return IMongoStorage.instance().must_get_l3_balancer_rev(version).spec


def find_dns_record_revision_spec(version, current_pb=None):
    """
    :type current_pb: model_pb2.DnsRecord
    :param six.text_type version: revision identifier
    :rtype: model_pb2.DnsRecordSpec
    """
    if current_pb is not None and current_pb.meta.version == version:
        return current_pb.spec
    else:
        return IMongoStorage.instance().must_get_dns_record_rev(version).spec
