# coding: utf-8
import random

import inject
import monotonic

from awacs.lib import ctlmanager
from awacs.model import events, cache
from awacs.model.dns_records import discoverer, validator, dns_record


class DnsRecordCtl(ctlmanager.ContextedCtl):
    PROCESS_INTERVAL = 30
    FORCE_PROCESS_INTERVAL = 60
    FORCE_PROCESS_INTERVAL_JITTER = 10
    EVENTS_QUEUE_GET_TIMEOUT = 5

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    _dns_record_events = (events.DnsRecordUpdate, events.DnsRecordStateUpdate,
                          events.DnsRecordRemove, events.DnsRecordStateRemove)
    _namespace_events = (events.BackendUpdate, events.EndpointSetUpdate,
                         events.BackendRemove, events.EndpointSetRemove)
    _subscribed_events = list(_dns_record_events) + list(_namespace_events)

    def __init__(self, namespace_id, dns_record_id):
        name = 'dns-record-ctl("{}:{}")'.format(namespace_id, dns_record_id)
        super(DnsRecordCtl, self).__init__(name)

        self._namespace_id = namespace_id
        self._dns_record_id = dns_record_id
        self._dns_record_path = '{}/{}'.format(self._namespace_id, self._dns_record_id)
        self._namespace_path_prefix = '{}/'.format(self._namespace_id)

        self._discoverer = None  # type: discoverer.DnsRecordDiscoverer or None
        self._validator = None  # type: validator.DnsRecordValidator or None

        current_time = monotonic.monotonic()
        self._waiting_for_processing_since = None
        self._processed_at = None
        self._processing_deadline = current_time  # to trigger first processing immediately after start
        self._force_processing_deadline = current_time  # to trigger first processing immediately after start

    def _accept_event(self, event):
        """
        :type event: events.*
        :rtype: bool
        """
        if isinstance(event, self._dns_record_events):
            return event.path == self._dns_record_path
        if isinstance(event, self._namespace_events):
            return event.path.startswith(self._namespace_path_prefix)
        return False

    def _init_processors(self):
        self._discoverer = discoverer.DnsRecordDiscoverer(
            namespace_id=self._namespace_id,
            dns_record_id=self._dns_record_id)
        self._validator = validator.DnsRecordValidator(
            namespace_id=self._namespace_id,
            dns_record_id=self._dns_record_id)

    def _start(self, ctx):
        ctx.log.info('Starting...')
        self._init_processors()
        self._cache.bind_on_specific_events(self._callback, self._subscribed_events)
        ctx.log.debug('Started')

    def _stop(self):
        self._log.info('Stopping...')
        self._cache.unbind_from_specific_events(self._callback, self._subscribed_events)
        self._log.info('Stopped')

    def _process_event(self, ctx, event):
        # Don't process every event, it is too costly. Instead, just start a timer.
        if self._waiting_for_processing_since is None:
            ctx.log.debug('Started _processing_deadline timer, processing will begin after %s seconds',
                          self.PROCESS_INTERVAL)
            self._waiting_for_processing_since = monotonic.monotonic()
            self._processing_deadline = self._waiting_for_processing_since + self.PROCESS_INTERVAL

    def _should_process(self, current_time):
        """
        Process changes at most every self.PROCESS_INTERVAL seconds,
        and at least every self.FORCE_PROCESS_INTERVAL seconds
        """
        if self._processing_deadline is not None and current_time >= self._processing_deadline:
            return True
        return current_time >= self._force_processing_deadline

    def _process_empty_queue(self, ctx):
        current_time = monotonic.monotonic()
        should_process = self._should_process(current_time)
        if not should_process:
            return

        dns_record_state_pb = self._cache.must_get_dns_record_state(self._namespace_id, self._dns_record_id)
        state_handler = dns_record.DnsRecordStateHandler(dns_record_state_pb)
        self._do_process(ctx, state_handler)
        self._reset_processing_timers()

    def _do_process(self, ctx, state_handler):
        """
        :rtype: bool
        """
        self._discoverer.discover(ctx, state_handler)
        if state_handler.was_updated:
            ctx.log.debug(u'state was updated after discovery')
            return
        self._validator.validate(ctx, state_handler)
        if state_handler.was_updated:
            ctx.log.debug(u'state was updated after validation')
            return
        self._discoverer.cleanup_state(ctx, state_handler)
        if state_handler.was_updated:
            ctx.log.debug(u'state was updated after cleanup')

    def _reset_processing_timers(self):
        self._waiting_for_processing_since = None
        self._processing_deadline = None
        self._processed_at = monotonic.monotonic()
        self._force_validation = False
        self._force_processing_deadline = self._processed_at + random.randint(
            self.FORCE_PROCESS_INTERVAL - self.FORCE_PROCESS_INTERVAL_JITTER,
            self.FORCE_PROCESS_INTERVAL + self.FORCE_PROCESS_INTERVAL_JITTER)
