from awacs.model import util
from awacs.model.dns_records import dns_record


class DnsRecordDiscoverer(object):
    def __init__(self, namespace_id, dns_record_id):
        self._namespace_id = namespace_id
        self._dns_record_id = dns_record_id
        self._full_dns_record_id = (namespace_id, dns_record_id)
        self._is_large_namespace = util.is_large_namespace(namespace_id)

    def discover(self, ctx, state_handler):
        """
        1) gather all revisions of DNS record, backends, and endpoint sets in namespace
        2) determine which of them are included in current and valid vectors
        3) add them to the DNS record state, so it can be validated

        :type state_handler: dns_record.DnsRecordStateHandler
        :type ctx: context.OpCtx
        """
        ctx = ctx.with_op(op_id=u'discover')

        discovered_vector = dns_record.DiscoveredDnsRecordVector.from_cache(self._namespace_id, self._dns_record_id)
        vectors = state_handler.generate_vectors()
        if not self._is_large_namespace:
            ctx.log.debug(u'discovered vector: %s', discovered_vector)
            ctx.log.debug(u'current vector: %s', vectors.current)

        versions_to_add, versions_to_delete = discovered_vector.find_versions_to_update_in_state(vectors)
        if versions_to_add or versions_to_delete:
            if versions_to_add or versions_to_delete:
                ctx.log.debug(u'versions_to_add: %s, versions_to_delete: %s', versions_to_add, versions_to_delete)
            state_handler.update_versions(
                versions_to_add=versions_to_add,
                versions_to_delete=versions_to_delete)

    def cleanup_state(self, ctx, state_handler):
        """
        Remove from state all entities that are older than the latest valid vector

        :type ctx: context.OpCtx
        :type state_handler: dns_record.DnsRecordStateHandler
        """
        ctx = ctx.with_op(op_id=u'cleanup_state')
        valid_vector = state_handler.generate_vectors().valid
        if valid_vector.dns_record_version is None:
            return
        removed_versions = state_handler.remove_obsolete_versions(valid_vector)
        if removed_versions and not self._is_large_namespace:
            ctx.log.debug(u'removed versions: %s', removed_versions)
