from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import has_actionable_spec, needs_removal, is_order_in_progress
from awacs.model import events
from awacs.model.dns_records.ctl import DnsRecordCtl
from awacs.model.dns_records.operations.ctl import DnsRecordOperationCtl
from awacs.model.dns_records.order.ctl import DnsRecordOrderCtl
from awacs.model.dns_records.removal.ctl import DnsRecordRemovalCtl


class DnsRecordCtlManager(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DnsRecordCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='dns-record',
            subscribed_events=(events.DnsRecordUpdate, events.DnsRecordRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.DnsRecordRemove):
            return False
        return has_actionable_spec(event.pb) and not needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for dns_record_pb in self._cache.list_all_dns_records(namespace_id):
                yield events.DnsRecordUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, dns_record_pb.meta.id),
                    pb=dns_record_pb)

    def _create_ctl(self, ctl_id):
        return DnsRecordCtl(*ctl_id)


class DnsRecordOrderCtlManager(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DnsRecordOrderCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='dns-record-order',
            subscribed_events=(events.DnsRecordUpdate, events.DnsRecordRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.DnsRecordRemove):
            return False
        return is_order_in_progress(event.pb) and not needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for dns_record_pb in self._cache.list_all_dns_records(namespace_id):
                yield events.DnsRecordUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, dns_record_pb.meta.id),
                    pb=dns_record_pb)

    def _create_ctl(self, ctl_id):
        return DnsRecordOrderCtl(*ctl_id)


class DnsRecordRemovalCtlManager(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DnsRecordRemovalCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='dns-record-removal',
            subscribed_events=(events.DnsRecordUpdate, events.DnsRecordRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.DnsRecordRemove):
            return False
        return needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for dns_record_pb in self._cache.list_all_dns_records(namespace_id):
                yield events.DnsRecordUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, dns_record_pb.meta.id),
                    pb=dns_record_pb)

    def _create_ctl(self, ctl_id):
        return DnsRecordRemovalCtl(*ctl_id)


class DnsRecordOperationCtlManager(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DnsRecordOperationCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='dns-record-op',
            subscribed_events=(events.DnsRecordOperationUpdate, events.DnsRecordOperationRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.DnsRecordOperationRemove):
            return False
        return is_order_in_progress(event.pb) or needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for dns_record_op_pb in self._cache.list_all_dns_record_operations(namespace_id):
                yield events.DnsRecordOperationUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, dns_record_op_pb.meta.id),
                    pb=dns_record_op_pb)

    def _create_ctl(self, ctl_id):
        return DnsRecordOperationCtl(*ctl_id)
