import inject
import six

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import needs_removal
from awacs.lib.order_processor.runner import StateRunner
from awacs.lib.strutils import quote_join_sorted
from awacs.model import events, cache, dao
from awacs.model.dns_records.operations.modify_addresses import (
    ModifyAddressesOperation,
    get_modify_address_processors,
)
from infra.awacs.proto import model_pb2


class DnsRecordOperationCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    EVENT_CLASS = events.DnsRecordOperationUpdate

    _runners = {
        'modify_addresses': StateRunner(entity_class=ModifyAddressesOperation,
                                        initial_state=ModifyAddressesOperation.states.STARTED,
                                        final_state=ModifyAddressesOperation.states.FINISHED,
                                        final_cancelled_state=ModifyAddressesOperation.states.CANCELLED,
                                        processors=get_modify_address_processors(),
                                        processing_interval=PROCESSING_INTERVAL)
    }

    def __init__(self, namespace_id, dns_record_op_id):
        name = 'dns-record-op-ctl("{}:{}")'.format(namespace_id, dns_record_op_id)
        super(DnsRecordOperationCtl, self).__init__(name)
        self._dns_record_op_id = dns_record_op_id
        self._namespace_id = namespace_id
        self._pb = None  # type: model_pb2.DnsRecordOperation or None

    def _accept_event(self, event):
        return isinstance(event, self.EVENT_CLASS) and event.pb.meta.id == self._dns_record_op_id

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process DNS record operation on start: %s', e)
        self._cache.bind_on_specific_events(self._callback, (self.EVENT_CLASS,))

    def _stop(self):
        self._cache.unbind_from_specific_events(self._callback, (self.EVENT_CLASS,))

    def _process(self, ctx, event=None):
        """
        :type ctx: context.OpCtx
        :type event: events.DnsRecordOperationUpdate
        """
        cached_pb = self._cache.must_get_dns_record_operation(self._namespace_id, self._dns_record_op_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if needs_removal(self._pb):
            return self._self_delete(ctx)

        if not self._pb.HasField('order'):
            raise RuntimeError('DNS record operation has no "order" field')

        for field, runner in six.iteritems(self._runners):
            if self._pb.order.content.HasField(field):
                return runner.process(ctx, self._pb)
        raise RuntimeError('DNS record operation unknown or not specified. '
                           'Supported operations: "{}"'.format(quote_join_sorted(self._runners)))

    def _self_delete(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        ctx.log.info('started self deletion')
        self._dao.delete_dns_record_operation(self._namespace_id, self._dns_record_op_id)

    def _process_event(self, ctx, event):
        """
        :type ctx: context.OpCtx
        :type event: events.DnsRecordOperationUpdate
        """
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        self._process(ctx)
