import dns
import enum
import inject
from abc import ABCMeta

import six

from awacs.lib import dns_manager_client, dns_resolver
from awacs.lib.dns_manager_client import DnsManagerError
from awacs.lib.order_processor.model import BaseProcessor, WithOrder, FeedbackMessage
from awacs.model import dao, zk, util, cache
from awacs.model.dns_records import dns_record
from infra.awacs.proto import model_pb2


DEFAULT_TTL = 300


class State(enum.Enum):
    STARTED = 1
    SYNC_DNS_RECORDS_IN_AWACS_MANAGED_ZONE = 4
    SENDING_DNS_REQUEST = 2
    POLLING_DNS_REQUEST = 3
    FINISHING = 19
    FINISHED = 20
    CANCELLING = 29
    CANCELLED = 30


def get_modify_address_processors():
    return (
        Started,
        SyncDnsRecordsInAwacsManagedZone,
        SendingDnsRequest,
        PollingDnsRequest,
        Finishing,
        Cancelling,
    )


class ModifyAddressesOperation(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'ModifyAddresses'
    states = State

    def zk_update(self):
        return self.zk.update_dns_record_operation(namespace_id=self.namespace_id,
                                                   dns_record_op_id=self.id)

    def dao_update(self, comment):
        return self.dao.update_dns_record_operation(namespace_id=self.namespace_id,
                                                    dns_record_op_id=self.id,
                                                    version=self.pb.meta.version,
                                                    comment=comment,
                                                    login=util.NANNY_ROBOT_LOGIN,
                                                    updated_spec_pb=self.pb.spec)


class ModifyAddressProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = (u'operation', u'op_content')

    def __init__(self, entity):
        super(ModifyAddressProcessor, self).__init__(entity)
        self.operation = self.entity  # type: ModifyAddressesOperation
        self.op_content = self.operation.pb.order.content.modify_addresses  # type: model_pb2.DnsRecordOperationOrder.Content.ModifyAddresses


class Started(ModifyAddressProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.SENDING_DNS_REQUEST
    next_state_changing_dns_records = State.SYNC_DNS_RECORDS_IN_AWACS_MANAGED_ZONE
    cancelled_state = State.CANCELLING
    _cache = inject.attr(cache.IAwacsCache)

    def process(self, ctx):
        dns_record_pb = self._cache.must_get_dns_record(self.operation.namespace_id,
                                                        self.operation.pb.meta.dns_record_id)
        nameserver_type = dns_record.get_nameserver_type(dns_record_pb)
        if nameserver_type == model_pb2.NameServerSpec.AWACS_MANAGED:
            return self.next_state_changing_dns_records
        elif nameserver_type == model_pb2.NameServerSpec.DNS_MANAGER:
            return self.next_state
        else:
            nameserver_type_name = model_pb2.NameServerSpec.Type.Name(nameserver_type)
            raise ValueError("For nameserver_type={} ModifyAddressesOperation is forbidden".format(nameserver_type_name))


class SyncDnsRecordsInAwacsManagedZone(ModifyAddressProcessor):
    state = State.SYNC_DNS_RECORDS_IN_AWACS_MANAGED_ZONE
    cancelled_state = State.CANCELLING
    next_state = State.FINISHING

    _dns_manager_client = inject.attr(dns_manager_client.IDnsManagerClient)  # type: dns_manager_client.DnsManagerClient
    _dns_resolver = inject.attr(dns_resolver.IDnsResolver)  # type: dns_resolver.DnsResolver
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        dns_record_pb = self._cache.must_get_dns_record(self.operation.namespace_id,
                                                        self.operation.pb.meta.dns_record_id)
        fqdn = dns_record.get_fqdn(dns_record_pb)
        for request_pb in self.op_content.requests:
            record_type = self._dns_resolver.get_record_type_by_address(request_pb.address)
            if request_pb.action == request_pb.CREATE:
                self._dns_manager_client.add_record(fqdn, record_type, DEFAULT_TTL, request_pb.address)
            elif request_pb.action == request_pb.REMOVE:
                self._dns_manager_client.remove_record(fqdn, record_type, request_pb.address)
            else:
                raise ValueError("unknown request.action={}".format(request_pb.action))
        return self.next_state


class SendingDnsRequest(ModifyAddressProcessor):
    __slots__ = ()
    state = State.SENDING_DNS_REQUEST
    next_state = State.POLLING_DNS_REQUEST
    cancelled_state = State.CANCELLING

    _dns_manager_client = inject.attr(dns_manager_client.IDnsManagerClient)  # type: dns_manager_client.DnsManagerClient
    _dns_resolver = inject.attr(dns_resolver.IDnsResolver)  # type: dns_resolver.DnsResolver
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    @staticmethod
    def _is_matching_request(requests_batch, new_requests):
        """
        Try to find an existing request that exactly matches our params. Don't consider partial matches for now.
        """
        if requests_batch.get(u'owner') != u'robot-awacs-dns':
            return False
        requests = requests_batch.get(u'requests', [])
        if len(requests) != len(new_requests):
            return False
        for i, request in enumerate(requests):
            new_request = new_requests[i]
            if new_request[u'operation'] != request.get(u'operation'):
                return False
            for field in (u'fqdn', u'type', u'data'):
                if new_request[u'resource'][field] != request.get(u'resource', {}).get(field):
                    return False
        return True

    def _create_requests_if_missing(self, ctx, new_requests):
        for requests_batch in self._dns_manager_client.list_requests():
            if self._is_matching_request(requests_batch, new_requests):
                ctx.log.info(u'Found existing request in DNS manager, using its uuid')
                return requests_batch[u'uuid']
        ctx.log.info(u'Creating new request in DNS manager')
        return self._dns_manager_client.create_request({u'requests': new_requests,
                                                        u'requester': self.operation.pb.meta.author})[u'uuid']

    def process(self, ctx):
        requests = []
        dns_record_pb = self._cache.must_get_dns_record(self.operation.namespace_id,
                                                        self.operation.pb.meta.dns_record_id)
        fqdn = dns_record.get_fqdn(dns_record_pb)
        resolved_addresses = set()
        for record_type in (dns.rdatatype.A, dns.rdatatype.AAAA):
            resolved_addresses |= self._dns_resolver.get_address_record(domain_name=fqdn, record_type=record_type)
        for request_pb in self.op_content.requests:
            if request_pb.action == request_pb.CREATE and request_pb.address in resolved_addresses:
                continue
            if request_pb.action == request_pb.REMOVE and request_pb.address not in resolved_addresses:
                continue
            action_name = model_pb2.DnsRecordOperationOrder.Content.ModifyAddresses.Request.Action.Name(
                request_pb.action).lower()
            requests.append(
                {
                    u'operation': action_name,
                    u'resource': {
                        u'data': request_pb.address,
                        u'fqdn': fqdn,
                        u'type': self._dns_resolver.get_record_type_by_address(request_pb.address),
                        u'ttl': DEFAULT_TTL
                    }
                }
            )
        if not requests:
            ctx.log.info(u'Nothing to do')
        self.operation.context[u'request_id'] = self._create_requests_if_missing(ctx, requests)
        return self.next_state


class PollingDnsRequest(ModifyAddressProcessor):
    __slots__ = ()
    state = State.POLLING_DNS_REQUEST
    next_state = State.FINISHING
    cancelled_state = State.CANCELLING

    _dns_manager_client = inject.attr(dns_manager_client.IDnsManagerClient)  # type: dns_manager_client.DnsManagerClient

    def process(self, ctx):
        resp = self._dns_manager_client.get_request_status(self.operation.context[u'request_id'])
        meta = resp[u'meta']
        state = meta.get(u'state')
        self.operation.context[u'request_state'] = state
        st_ticket = meta.get(u'ref')
        if st_ticket:
            self.operation.context[u'st_ticket'] = st_ticket
        if state == u'new':
            return self.state
        if state == u'done':
            return self.next_state
        if state == u'error':
            return FeedbackMessage(
                pb_error_type=model_pb2.DnsRecordOperationOrder.OrderFeedback.DNS_MANAGER_REQUEST_ERROR,
                message=meta[u'error'])
        raise RuntimeError(u'Unknown DNS Manager request state "{}" in meta: {}'.format(state, meta))


class Finishing(ModifyAddressProcessor):
    __slots__ = ()
    state = State.FINISHING
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        self.operation.pb.spec.incomplete = False
        self.operation.dao_update(u'Operation finished, spec.incomplete = False')
        return self.next_state


class Cancelling(ModifyAddressProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None
    _cache = inject.attr(cache.IAwacsCache)

    _dns_manager_client = inject.attr(dns_manager_client.IDnsManagerClient)  # type: dns_manager_client.DnsManagerClient

    def _cancel_request(self, request_id):
        try:
            self._dns_manager_client.cancel_request(request_id)
        except DnsManagerError:
            return  # already removed

    def process(self, ctx):
        dns_record_pb = self._cache.must_get_dns_record(self.operation.namespace_id,
                                                        self.operation.pb.meta.dns_record_id)
        if dns_record.get_nameserver_type(dns_record_pb) == model_pb2.NameServerSpec.DNS_MANAGER:
            request_id = self.operation.context.get(u'request_id')
            if request_id:
                self._cancel_request(request_id)
        self.operation.pb.spec.incomplete = False
        self.operation.dao_update(u'Operation cancelled, spec.incomplete = False')
        return self.next_state
