import enum
import inject
from abc import ABCMeta

import six

from awacs.lib import l3mgrclient
from awacs.lib.order_processor.model import BaseProcessor, WithOrder, is_order_in_progress, cancel_order
from awacs.model import dao, zk, cache, util, errors
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    STARTED = 1
    GETTING_L3_IP_ADDRESSES = 2
    CREATING_DNS_RECORD_OPERATION = 3
    WAITING_FOR_DNS_RECORD_OPERATION = 4
    SAVING_SPEC = 19
    FINISHED = 20
    CANCELLING = 29
    CANCELLED = 30


def get_dns_record_order_processors():
    return (
        Started,
        GettingL3IpAddresses,
        CreatingDnsRecordOperation,
        WaitingForDnsRecordOperation,
        SavingSpec,
        Cancelling,
    )


class DnsRecordOrder(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'DnsRecord'
    states = State

    def zk_update(self):
        return self.zk.update_dns_record(namespace_id=self.namespace_id,
                                         dns_record_id=self.id)

    def dao_update(self, updated_spec_pb, comment):
        return self.dao.update_dns_record(
            namespace_id=self.namespace_id,
            dns_record_id=self.id,
            version=self.pb.meta.version,
            login=self.pb.meta.author,
            comment=comment,
            updated_spec_pb=updated_spec_pb,
        )


class DnsRecordOrderProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = (u'order', u'order_content')

    def __init__(self, entity):
        super(DnsRecordOrderProcessor, self).__init__(entity)
        self.order = self.entity  # type: DnsRecordOrder
        self.order_content = self.order.pb.order.content  # type: model_pb2.DnsRecordOrder.Content


class Started(DnsRecordOrderProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.GETTING_L3_IP_ADDRESSES
    next_state_saving_spec = State.SAVING_SPEC
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        ns_pb = self._cache.must_get_name_server(self.order_content.name_server.namespace_id,
                                                 self.order_content.name_server.id)
        if (self.order_content.address.backends.type != model_pb2.DnsBackendsSelector.L3_BALANCERS or
            ns_pb.spec.type == model_pb2.NameServerSpec.YP_DNS):
            return self.next_state_saving_spec
        return self.next_state


class GettingL3IpAddresses(DnsRecordOrderProcessor):
    __slots__ = ()
    state = State.GETTING_L3_IP_ADDRESSES
    next_state = State.CREATING_DNS_RECORD_OPERATION
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient

    def _get_l3_ip_addresses(self):
        ip_addresses = set()
        for l3_balancer in self.order_content.address.backends.l3_balancers:
            l3_pb = self._cache.must_get_l3_balancer(self.order.namespace_id, l3_balancer.id)
            l3_service_id = l3_pb.spec.l3mgr_service_id
            resp = self._l3mgr_client.get_service(l3_service_id, request_timeout=3)
            for vs_id in resp[u'config'][u'vs_id']:
                vs_config = self._l3mgr_client.get_vs(l3_service_id, vs_id, request_timeout=3)
                ip_addresses.add(vs_config[u'ip'])
        return sorted(ip_addresses)

    def process(self, ctx):
        self.order.context[u'l3_ip_addresses'] = self._get_l3_ip_addresses()
        return self.next_state


class CreatingDnsRecordOperation(DnsRecordOrderProcessor):
    __slots__ = ()
    state = State.CREATING_DNS_RECORD_OPERATION
    next_state = State.WAITING_FOR_DNS_RECORD_OPERATION
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    def process(self, ctx):
        op_pb = model_pb2.DnsRecordOperation()
        op_pb.meta.namespace_id = self.order.namespace_id
        op_pb.meta.id = self.order.id
        op_pb.meta.dns_record_id = self.order.id

        for address in self.order.context[u'l3_ip_addresses']:
            op_pb.order.content.modify_addresses.requests.add(
                address=address,
                action=model_pb2.DnsRecordOperationOrder.Content.ModifyAddresses.Request.CREATE)

        self._dao.create_dns_record_operation_if_missing(meta_pb=op_pb.meta,
                                                         order_pb=op_pb.order,
                                                         login=self.order.pb.meta.author)
        return self.next_state


class WaitingForDnsRecordOperation(DnsRecordOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_DNS_RECORD_OPERATION
    next_state = State.SAVING_SPEC
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        op_pb = self._cache.get_dns_record_operation(self.order.namespace_id, self.order.id)
        if op_pb is None or not op_pb.spec.incomplete:
            return self.next_state
        return self.state


class SavingSpec(DnsRecordOrderProcessor):
    __slots__ = ()
    state = State.SAVING_SPEC
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        self.order.pb.spec.address.CopyFrom(self.order_content.address)
        self.order.pb.spec.name_server.CopyFrom(self.order_content.name_server)
        self.order.pb.spec.type = self.order_content.type
        self.order.pb.spec.ctl_version = self.order_content.ctl_version
        self.order.pb.spec.incomplete = False
        self.order.dao_update(self.order.pb.spec, u'Finished order, spec.incomplete = False')
        return self.next_state


class Cancelling(DnsRecordOrderProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def _cancel_dns_record_op(self):
        try:
            for dns_record_op_pb in self.order.zk.update_dns_record_operation(self.order.namespace_id, self.order.id):
                if not is_order_in_progress(dns_record_op_pb):
                    return False
                cancel_order(dns_record_op_pb,
                             author=util.NANNY_ROBOT_LOGIN,
                             comment=u'Cancelled because DNS record order was cancelled')
        except errors.NotFoundError:
            pass
        return True

    def process(self, ctx):
        self._cancel_dns_record_op()
        # TODO: op can be already completed, so we need to create another op to remove records from DNS Manager
        self.order.pb.spec.incomplete = False
        self.order.dao_update(self.order.pb.spec, u'Cancelled order, spec.incomplete = False')
        return self.next_state
