import inject

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import needs_removal
from awacs.lib.order_processor.runner import StateRunner
from awacs.model import events, cache, dao
from awacs.model.dns_records.removal.processors import (
    DnsRecordRemoval,
    get_dns_record_removal_processors,
)
from infra.awacs.proto import model_pb2


class DnsRecordRemovalCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    EVENT_CLASS = events.DnsRecordUpdate

    _runner = StateRunner(entity_class=DnsRecordRemoval,
                          initial_state=DnsRecordRemoval.states.STARTED,
                          final_state=DnsRecordRemoval.states.FINISHED,
                          final_cancelled_state=DnsRecordRemoval.states.CANCELLED,
                          processors=get_dns_record_removal_processors(),
                          processing_interval=PROCESSING_INTERVAL)

    def __init__(self, namespace_id, dns_record_id):
        name = 'dns-record-removal-ctl("{}:{}")'.format(namespace_id, dns_record_id)
        super(DnsRecordRemovalCtl, self).__init__(name)
        self._dns_record_id = dns_record_id
        self._namespace_id = namespace_id
        self._pb = None  # type: model_pb2.DnsRecord or None

    def _accept_event(self, event):
        return isinstance(event, self.EVENT_CLASS) and event.pb.meta.id == self._dns_record_id

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process dns record removal on start: %s', e)
        self._cache.bind_on_specific_events(self._callback, (self.EVENT_CLASS,))

    def _stop(self):
        self._cache.unbind_from_specific_events(self._callback, (self.EVENT_CLASS,))

    def _process(self, ctx, event=None):
        """
        :type event: events.DnsRecordUpdate
        :type ctx: context.OpCtx
        """
        cached_pb = self._cache.must_get_dns_record(self._namespace_id, self._dns_record_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if needs_removal(self._pb):
            return self._runner.process(ctx, self._pb)

    def _process_event(self, ctx, event):
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
