import dns
import enum
import inject
from abc import ABCMeta

import six

from awacs.lib import l3mgrclient, dns_resolver
from awacs.lib.order_processor.model import BaseProcessor, WithRemoval
from awacs.model import dao, zk, cache
from awacs.model.dns_records import dns_record
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    STARTED = 1
    CREATING_DNS_RECORD_OPERATION = 2
    WAITING_FOR_DNS_RECORD_OPERATION = 3
    REMOVING_DNS_RECORD = 19
    FINISHED = 20
    CANCELLED = 30


def get_dns_record_removal_processors():
    return (
        Started,
        CreatingDnsRecordOperation,
        WaitingForDnsRecordOperation,
        RemovingDnsRecord,
    )


class DnsRecordRemoval(WithRemoval):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'DnsRecord'
    states = State

    def zk_update(self):
        return self.zk.update_dns_record(namespace_id=self.namespace_id,
                                         dns_record_id=self.id)

    def self_remove(self):
        return self.dao.delete_dns_record(
            namespace_id=self.namespace_id,
            dns_record_id=self.id,
        )


class DnsRecordRemovalProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = ('dns_record',)

    def __init__(self, entity):
        super(DnsRecordRemovalProcessor, self).__init__(entity)
        self.dns_record = self.entity  # type: DnsRecordRemoval


class Started(DnsRecordRemovalProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.CREATING_DNS_RECORD_OPERATION
    next_state_removing = State.REMOVING_DNS_RECORD
    cancelled_state = None

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        ns_pb = self._cache.must_get_name_server(self.dns_record.pb.spec.name_server.namespace_id or self.dns_record.pb.order.content.name_server.namespace_id,
                                                 self.dns_record.pb.spec.name_server.id or self.dns_record.pb.order.content.name_server.id)
        if (self.dns_record.pb.spec.address.backends.type != model_pb2.DnsBackendsSelector.L3_BALANCERS or
            ns_pb.spec.type == model_pb2.NameServerSpec.YP_DNS):
            return self.next_state_removing
        return self.next_state


class CreatingDnsRecordOperation(DnsRecordRemovalProcessor):
    __slots__ = ()
    state = State.CREATING_DNS_RECORD_OPERATION
    next_state = State.WAITING_FOR_DNS_RECORD_OPERATION
    cancelled_state = None

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _dao = inject.attr(dao.IDao)  # type: dao.Dao
    _dns_resolver = inject.attr(dns_resolver.IDnsResolver)  # type: dns_resolver.DnsResolver
    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient

    def process(self, ctx):
        op_pb = model_pb2.DnsRecordOperation()
        op_pb.meta.namespace_id = self.dns_record.namespace_id
        op_pb.meta.id = self.dns_record.id
        op_pb.meta.dns_record_id = self.dns_record.id
        dns_record_pb = self._cache.must_get_dns_record(self.dns_record.pb.meta.namespace_id,
                                                        self.dns_record.pb.meta.id)
        fqdn = dns_record.get_fqdn(dns_record_pb)
        resolved_addresses = set()
        for record_type in (dns.rdatatype.A, dns.rdatatype.AAAA):
            resolved_addresses |= self._dns_resolver.get_address_record(domain_name=fqdn, record_type=record_type)
            for address in sorted(resolved_addresses):
                op_pb.order.content.modify_addresses.requests.add(
                    address=address,
                    action=model_pb2.DnsRecordOperationOrder.Content.ModifyAddresses.Request.REMOVE)
        self._dao.create_dns_record_operation_if_missing(meta_pb=op_pb.meta,
                                                        order_pb=op_pb.order,
                                                        login=self.dns_record.pb.meta.author)
        return self.next_state


class WaitingForDnsRecordOperation(DnsRecordRemovalProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_DNS_RECORD_OPERATION
    next_state = State.REMOVING_DNS_RECORD
    cancelled_state = None

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        op_pb = self._cache.get_dns_record_operation(self.dns_record.namespace_id, self.dns_record.id)
        if op_pb is None or not op_pb.spec.incomplete:
            return self.next_state
        return self.state
        

class RemovingDnsRecord(DnsRecordRemovalProcessor):
    __slots__ = ()
    state = State.REMOVING_DNS_RECORD
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        if not self.dns_record.pb.spec.deleted:
            raise RuntimeError(u'dns_record.spec.deleted is False for some reason')
        self.dns_record.self_remove()
        return self.next_state
