import inject
import monotonic
import six
import time

from awacs.lib import ctlmanager, context
from awacs.lib.order_processor.model import has_actionable_spec, needs_removal
from awacs.lib.strutils import to_full_id
from awacs.model import events, cache, dao, zk
from infra.awacs.proto import model_pb2


class DomainCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    EVENTS_QUEUE_GET_TIMEOUT = 10
    SELF_ACTIONS_DELAY_INTERVAL = 30
    SELF_DELETION_COOLDOWN_PERIOD = 30

    def __init__(self, namespace_id, domain_id):
        name = 'domain-ctl("{}:{}")'.format(namespace_id, domain_id)
        super(DomainCtl, self).__init__(name)
        self._domain_id = domain_id
        self._namespace_id = namespace_id
        self._self_deletion_check_deadline = monotonic.monotonic()
        self._full_domain_id = (self._namespace_id, self._domain_id)
        self._pb = None  # type: model_pb2.Domain or None

    def _accept_event(self, event):
        return (isinstance(event, events.DomainUpdate) and
                event.pb.meta.namespace_id == self._namespace_id and
                event.pb.meta.id == self._domain_id and
                has_actionable_spec(event.pb))

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process domain on start: %s', e)
        self._cache.bind(self._callback)

    def _stop(self):
        self._cache.unbind(self._callback)

    def _process(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        self._pb = self._cache.must_get_domain(self._namespace_id, self._domain_id)

        if needs_removal(self._pb):
            ctx.log.debug('Maybe deleting domain marked for removal')
            if self._ready_to_delete(ctx):
                self._self_delete(ctx)
            return

    def _is_used(self):
        """
        A bullet-proof and hugely inefficient method to know if domain is referenced by any balancer state --
        avoid caches, query Zookeeper with sync=True.
        Let's call it at least for the time being, until we make sure there are no bugs in our caching code.

        :rtype: bool
        """

        def get_keys_w_statuses(m):
            rv = set()
            for k, v_pb in six.iteritems(m):
                if v_pb.statuses:
                    rv.add(k)
            return rv

        for balancer_pb in self._cache.list_all_balancers(namespace_id=self._namespace_id):
            balancer_state_pb = self._zk.must_get_balancer_state(
                namespace_id=balancer_pb.meta.namespace_id,
                balancer_id=balancer_pb.meta.id,
                sync=True)
            included_full_domain_ids = {
                to_full_id(balancer_state_pb.namespace_id, domain_id)
                for domain_id in get_keys_w_statuses(balancer_state_pb.domains)}
            if self._full_domain_id in included_full_domain_ids:
                return True

        return False

    def _self_delete(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        ctx.log.info('started self deletion')
        try:
            ctx.log.info('starting _is_used()')
            with ctx.with_forced_timeout(60 * 10):
                is_domain_used = self._is_used()
        except context.CtxTimeoutExceeded:
            ctx.log.warn('_is_used() timed out, returning...')
            return
        except context.CtxTimeoutCancelled:
            ctx.log.debug('ctx is cancelled: %s, returning...', ctx.error())
            return
        ctx.log.info('finished _is_used()')
        if is_domain_used:
            raise RuntimeError("Critical error: would delete a referenced domain if it wasn't for this raise")

        should_remove_op = not self._pb.meta.is_being_transferred.value
        if should_remove_op:
            ctx.log.info('removing domain and its operation from db')
            self._dao.delete_domain(*self._full_domain_id, remove_op=True)
        else:
            ctx.log.info('removing domain from db, leaving its operation running')
            self._dao.delete_domain(*self._full_domain_id, remove_op=False)

    def _ready_to_delete(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        current_time = monotonic.monotonic()

        # don't check too often
        if current_time < self._self_deletion_check_deadline:
            ctx.log.info('too little time passed since last deletion check: current_time is %s, next check at %s',
                         current_time, self._self_deletion_check_deadline)
            return False

        # let balancers update their state before proceeding
        self_deletion_elapsed_time = time.time() - self._pb.meta.mtime.ToSeconds()
        if self_deletion_elapsed_time < self.SELF_DELETION_COOLDOWN_PERIOD:
            ctx.log.info('too little time passed since domain modification: %s out of minimum %s',
                         self_deletion_elapsed_time, self.SELF_DELETION_COOLDOWN_PERIOD)
            return False
        self._self_deletion_check_deadline = current_time + self.SELF_ACTIONS_DELAY_INTERVAL

        full_balancer_ids = self._cache.list_full_balancer_ids_for_domain(*self._full_domain_id)
        if full_balancer_ids:
            ctx.log.info("cannot delete domain since it's used in balancers: %s",
                         ', '.join(i for _, i in full_balancer_ids))
            return False
        return True

    def _process_event(self, ctx, event):
        assert isinstance(event, events.DomainUpdate)
        self._process(ctx)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
