from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import is_order_in_progress, has_actionable_spec, needs_removal
from awacs.model import events
from awacs.model.domain.ctl import DomainCtl
from awacs.model.domain.operations.ctl import DomainOperationCtl
from awacs.model.domain.order.ctl import DomainOrderCtl


class DomainCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, member_id, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DomainCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='domain-ctl-manager',
            starting_events=(events.DomainUpdate,),
            stopping_events=(events.DomainRemove,),
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'domain("{}:{}") ctl'.format(*ctl_id)

    def _is_starting_event(self, event):
        return (super(DomainCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.namespace_id) and
                has_actionable_spec(event.pb))

    def _list_all_ctl_ids(self):
        """
        :rtype: (six.text_type, six.text_type)
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if not self._is_allowed_namespace(namespace_id):
                continue
            for domain_pb in self._cache.list_all_domains(namespace_id=namespace_id):
                if has_actionable_spec(domain_pb):
                    yield namespace_id, domain_pb.meta.id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        namespace_id, domain_id = ctl_id
        return DomainCtl(namespace_id, domain_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct domain ctl id'.format(event.path))
        return ctl_id


class DomainOrderCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, member_id, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DomainOrderCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='domain-order-ctl-manager',
            starting_events=(events.DomainUpdate,),
            stopping_events=(events.DomainRemove,),
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'domain-order("{}:{}") ctl'.format(*ctl_id)

    def _is_starting_event(self, event):
        return (super(DomainOrderCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.namespace_id) and
                is_order_in_progress(event.pb))

    def _is_stopping_event(self, event):
        if super(DomainOrderCtlManager, self)._is_stopping_event(event):
            return True
        return super(DomainOrderCtlManager, self)._is_starting_event(event) and has_actionable_spec(event.pb)

    def _list_all_ctl_ids(self):
        """
        :rtype: (six.text_type, six.text_type)
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if not self._is_allowed_namespace(namespace_id):
                continue
            for domain_pb in self._cache.list_all_domains(namespace_id=namespace_id):
                if is_order_in_progress(domain_pb):
                    yield namespace_id, domain_pb.meta.id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        namespace_id, domain_id = ctl_id
        return DomainOrderCtl(namespace_id, domain_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct domain order ctl id'.format(event.path))
        return ctl_id


class DomainOperationCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, member_id, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DomainOperationCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='domain-op-ctl-manager',
            starting_events=(events.DomainOperationUpdate,),
            stopping_events=(events.DomainOperationRemove,),
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        """
        return 'domain-op("{}:{}") ctl'.format(*ctl_id)

    def _is_starting_event(self, event):
        return (super(DomainOperationCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.namespace_id) and
                (is_order_in_progress(event.pb) or needs_removal(event.pb)))

    def _is_stopping_event(self, event):
        if super(DomainOperationCtlManager, self)._is_stopping_event(event):
            return True

    def _list_all_ctl_ids(self):
        """
        :rtype: (six.text_type, six.text_type)
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if not self._is_allowed_namespace(namespace_id):
                continue
            for domain_op_pb in self._cache.list_all_domain_operations(namespace_id=namespace_id):
                if is_order_in_progress(domain_op_pb):
                    yield namespace_id, domain_op_pb.meta.id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        namespace_id, domain_id = ctl_id
        return DomainOperationCtl(namespace_id, domain_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = tuple(event.path.strip('/').split('/'))
        if len(ctl_id) != 2 or not ctl_id[0] or not ctl_id[1]:
            raise AssertionError('"{}" can not be parsed into a correct domain operation ctl id'.format(event.path))
        return ctl_id


class DomainCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DomainCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='domain',
            subscribed_events=(events.DomainUpdate, events.DomainRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.DomainRemove):
            return False
        return has_actionable_spec(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for domain_pb in self._cache.list_all_domains(namespace_id):
                yield events.DomainUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, domain_pb.meta.id),
                    pb=domain_pb)

    def _create_ctl(self, ctl_id):
        return DomainCtl(*ctl_id)


class DomainOrderCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DomainOrderCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='domain-order',
            subscribed_events=(events.DomainUpdate, events.DomainRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.DomainRemove):
            return False
        return is_order_in_progress(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for domain_pb in self._cache.list_all_domains(namespace_id):
                yield events.DomainUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, domain_pb.meta.id),
                    pb=domain_pb)

    def _create_ctl(self, ctl_id):
        return DomainOrderCtl(*ctl_id)


class DomainOperationCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(DomainOperationCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='domain-op',
            subscribed_events=(events.DomainOperationUpdate, events.DomainOperationRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.DomainOperationRemove):
            return False
        return is_order_in_progress(event.pb) or needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for domain_op_pb in self._cache.list_all_domain_operations(namespace_id):
                yield events.DomainOperationUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, domain_op_pb.meta.id),
                    pb=domain_op_pb)

    def _create_ctl(self, ctl_id):
        return DomainOperationCtl(*ctl_id)
