# coding: utf-8

import inject
import six

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import needs_removal
from awacs.lib.order_processor.runner import StateRunner
from awacs.lib.strutils import quote_join_sorted
from awacs.model import events, cache, zk, dao
from awacs.model.domain.operations.set_cert import DomainSetCert, get_set_cert_processors
from awacs.model.domain.operations.set_fqdns import DomainSetFqdns, get_set_fqdns_processors
from awacs.model.domain.operations.set_protocol import DomainSetProtocol, get_set_protocol_processors
from awacs.model.domain.operations.set_upstreams import DomainSetUpstreams, get_set_upstreams_processors
from awacs.model.domain.operations.transfer import DomainTransfer, get_transfer_processors
from infra.awacs.proto import model_pb2


class DomainOperationCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    _runners = {
        u'set_fqdns': StateRunner(entity_class=DomainSetFqdns,
                                  initial_state=DomainSetFqdns.states.STARTED,
                                  final_state=DomainSetFqdns.states.FINISHED,
                                  final_cancelled_state=DomainSetFqdns.states.CANCELLED,
                                  processors=get_set_fqdns_processors(),
                                  processing_interval=PROCESSING_INTERVAL),
        u'set_protocol': StateRunner(entity_class=DomainSetProtocol,
                                     initial_state=DomainSetProtocol.states.STARTED,
                                     final_state=DomainSetProtocol.states.FINISHED,
                                     final_cancelled_state=DomainSetProtocol.states.CANCELLED,
                                     processors=get_set_protocol_processors(),
                                     processing_interval=PROCESSING_INTERVAL),
        u'set_cert': StateRunner(entity_class=DomainSetCert,
                                 initial_state=DomainSetCert.states.STARTED,
                                 final_state=DomainSetCert.states.FINISHED,
                                 final_cancelled_state=DomainSetCert.states.CANCELLED,
                                 processors=get_set_cert_processors(),
                                 processing_interval=PROCESSING_INTERVAL),
        u'set_upstreams': StateRunner(entity_class=DomainSetUpstreams,
                                      initial_state=DomainSetUpstreams.states.STARTED,
                                      final_state=DomainSetUpstreams.states.FINISHED,
                                      final_cancelled_state=DomainSetUpstreams.states.CANCELLED,
                                      processors=get_set_upstreams_processors(),
                                      processing_interval=PROCESSING_INTERVAL),
        u'transfer': StateRunner(entity_class=DomainTransfer,
                                 initial_state=DomainTransfer.states.STARTED,
                                 final_state=DomainTransfer.states.FINISHED,
                                 final_cancelled_state=DomainTransfer.states.CANCELLED,
                                 processors=get_transfer_processors(),
                                 processing_interval=PROCESSING_INTERVAL),

    }

    def __init__(self, namespace_id, domain_id):
        name = u'domain-op-ctl("{}:{}")'.format(namespace_id, domain_id)
        super(DomainOperationCtl, self).__init__(name)
        self._domain_id = domain_id
        self._namespace_id = namespace_id
        self._pb = None  # type: model_pb2.DomainOperation or None

    def _accept_event(self, event):
        return (isinstance(event, events.DomainOperationUpdate) and
                event.pb.meta.id == self._domain_id)

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception(u'failed to process domain operation on start: %s', e)
        self._cache.bind_on_specific_events(self._callback, (events.DomainOperationUpdate,))

    def _stop(self):
        self._cache.unbind_from_specific_events(self._callback, (events.DomainOperationUpdate,))

    def _process(self, ctx, event=None):
        """
        :type event: events.DomainOperationUpdate
        :type ctx: context.OpCtx
        """
        cached_pb = self._cache.must_get_domain_operation(self._namespace_id, self._domain_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if needs_removal(self._pb):
            return self._self_delete(ctx)

        if not self._pb.HasField('order'):
            raise RuntimeError(u'Domain operation has no "order" field')

        for field, runner in six.iteritems(self._runners):
            if self._pb.order.content.HasField(field):
                return runner.process(ctx, self._pb)
        raise RuntimeError(u'Domain operation unknown or not specified. Supported operations: "{}"'.format(
            quote_join_sorted(self._runners)))

    def _self_delete(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        ctx.log.info(u'started self deletion')
        self._dao.delete_domain_operation(self._namespace_id, self._domain_id)

    def _process_event(self, ctx, event):
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
