import abc
import enum
import inject
import six

from awacs.lib.order_processor.model import (
    WithOrder,
    BaseProcessor,
)
from awacs.model import dao, zk, util
from awacs.model.domain.order import processors as order_processors
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    STARTED = 1
    CHECKING_CERT_INFO = 2
    CREATING_CERT_ORDER = 3
    WAITING_FOR_CERT_ORDER = 4
    SAVING_DOMAIN_SPEC = 5
    FINISHED = 20
    CANCELLING = 30
    WAITING_FOR_CERT_CANCEL = 31
    CANCELLED = 50


def get_set_protocol_processors():
    return (
        Started,
        CheckingCertInfo,
        CreatingCertOrders,
        WaitingForCertOrder,
        SavingDomainSpec,
        Cancelling,
        WaitingForCertCancel,
    )


class DomainSetProtocol(WithOrder):
    __slots__ = (u'order',)

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'DomainSetProtocol'
    states = State

    def __init__(self, pb):
        super(DomainSetProtocol, self).__init__(pb)
        self.order = self.pb.order.content.set_protocol  # type: model_pb2.DomainOperationOrder.Content.SetProtocol

    def zk_update(self):
        return self.zk.update_domain_operation(namespace_id=self.namespace_id,
                                               domain_id=self.id)

    def zk_get_domain(self):
        return self.zk.must_get_domain(namespace_id=self.namespace_id,
                                       domain_id=self.id,
                                       sync=True)

    def dao_update(self, comment):
        return self.dao.update_domain_operation(namespace_id=self.namespace_id,
                                                domain_id=self.id,
                                                version=self.pb.meta.version,
                                                comment=comment,
                                                login=util.NANNY_ROBOT_LOGIN,
                                                updated_spec_pb=self.pb.spec)

    def dao_update_domain(self, domain_pb, comment):
        return self.dao.update_domain(namespace_id=self.namespace_id,
                                      domain_id=self.id,
                                      version=domain_pb.meta.version,
                                      comment=comment,
                                      login=self.pb.meta.author,
                                      updated_spec_pb=domain_pb.spec)


class DomainSetProtocolProcessor(six.with_metaclass(abc.ABCMeta, BaseProcessor)):
    __slots__ = (u'order',)

    def __init__(self, entity):
        """
        :type entity: DomainSetProtocol
        """
        super(DomainSetProtocolProcessor, self).__init__(entity)
        self.order = entity.order  # type: model_pb2.DomainOperationOrder.Content.SetProtocol


class Started(order_processors.Started):
    __slots__ = ()
    state = State.STARTED
    next_state = State.CHECKING_CERT_INFO
    cancelled_state = State.CANCELLING


class CheckingCertInfo(order_processors.CheckingCertsInfo):
    __slots__ = ()
    state = State.CHECKING_CERT_INFO
    next_state = State.CREATING_CERT_ORDER
    next_state_saving = State.SAVING_DOMAIN_SPEC
    cancelled_state = State.CANCELLING


class CreatingCertOrders(order_processors.CreatingCertOrders):
    __slots__ = ()
    state = State.CREATING_CERT_ORDER
    next_state = State.WAITING_FOR_CERT_ORDER
    cancelled_state = State.CANCELLING


class WaitingForCertOrder(order_processors.WaitingForCertOrders):
    __slots__ = ()
    state = State.WAITING_FOR_CERT_ORDER
    next_state = State.SAVING_DOMAIN_SPEC
    cancelled_state = State.CANCELLING


class SavingDomainSpec(order_processors.SavingDomainSpec):
    __slots__ = ()
    state = State.SAVING_DOMAIN_SPEC
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        domain_pb = self.entity.zk_get_domain()
        domain_config_pb = domain_pb.spec.yandex_balancer.config
        domain_config_pb.protocol = self.order.protocol
        if self.order.HasField('set_redirect_to_https'):
            if self.order.set_redirect_to_https.HasField('redirect_to_https'):
                domain_config_pb.redirect_to_https.CopyFrom(self.order.set_redirect_to_https.redirect_to_https)
            else:
                domain_config_pb.ClearField('redirect_to_https')
        if self.order.HasField('set_verify_client_cert'):
            if self.order.set_verify_client_cert.HasField('verify_client_cert'):
                domain_config_pb.verify_client_cert.CopyFrom(self.order.set_verify_client_cert.verify_client_cert)
            else:
                domain_config_pb.ClearField('verify_client_cert')
        self._set_cert_ids(domain_config_pb)
        if self.order.remove_secondary_cert:
            domain_config_pb.ClearField('secondary_cert')

        self.entity.dao_update_domain(comment=u'Changed protocol', domain_pb=domain_pb)

        self.entity.pb.spec.incomplete = False
        self.entity.dao_update(comment=u'Finished operation, spec.incomplete = False')
        return self.next_state


class Cancelling(order_processors.Cancelling):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.WAITING_FOR_CERT_CANCEL
    next_state_cancelled = State.CANCELLED
    cancelled_state = None


class WaitingForCertCancel(order_processors.WaitingForCertsCancel):
    __slots__ = ()
    state = State.WAITING_FOR_CERT_CANCEL
    next_state = State.CANCELLED
    cancelled_state = None
