import abc
import enum
import inject
import six

from awacs.lib.order_processor.model import (
    WithOrder,
    BaseProcessor,
)
from awacs.model import dao, zk, util
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    STARTED = 1
    CHECKING_UPSTREAM_IDS = 2
    SAVING_DOMAIN_SPEC = 10
    FINISHED = 11
    CANCELLING = 20
    CANCELLED = 21


def get_set_upstreams_processors():
    return (
        Started,
        CheckingUpstreamIds,
        SavingDomainSpec,
        Cancelling,
    )


class DomainSetUpstreams(WithOrder):
    __slots__ = (u'order',)

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'DomainSetUpstreams'
    states = State

    def __init__(self, pb):
        super(DomainSetUpstreams, self).__init__(pb)
        self.order = self.pb.order.content.set_upstreams  # type: model_pb2.DomainOperationOrder.Content.SetUpstreams

    def zk_update(self):
        return self.zk.update_domain_operation(namespace_id=self.namespace_id,
                                               domain_id=self.id)

    def zk_get_domain(self):
        return self.zk.must_get_domain(namespace_id=self.namespace_id,
                                       domain_id=self.id,
                                       sync=True)

    def dao_update(self, comment):
        return self.dao.update_domain_operation(namespace_id=self.namespace_id,
                                                domain_id=self.id,
                                                version=self.pb.meta.version,
                                                comment=comment,
                                                login=util.NANNY_ROBOT_LOGIN,
                                                updated_spec_pb=self.pb.spec)

    def dao_update_domain(self, domain_pb, comment):
        return self.dao.update_domain(namespace_id=self.namespace_id,
                                      domain_id=self.id,
                                      version=domain_pb.meta.version,
                                      comment=comment,
                                      login=self.pb.meta.author,
                                      updated_spec_pb=domain_pb.spec)


class DomainSetUpstreamsProcessor(six.with_metaclass(abc.ABCMeta, BaseProcessor)):
    __slots__ = (u'order',)

    def __init__(self, entity):
        """
        :type entity: DomainSetUpstreams
        """
        super(DomainSetUpstreamsProcessor, self).__init__(entity)
        self.order = entity.order  # type: model_pb2.DomainOperationOrder.Content.SetUpstreams


class Started(DomainSetUpstreamsProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.CHECKING_UPSTREAM_IDS
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class CheckingUpstreamIds(DomainSetUpstreamsProcessor):
    __slots__ = ()
    state = State.CHECKING_UPSTREAM_IDS
    next_state = State.SAVING_DOMAIN_SPEC
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class SavingDomainSpec(DomainSetUpstreamsProcessor):
    __slots__ = ()
    state = State.SAVING_DOMAIN_SPEC
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        domain_pb = self.entity.zk_get_domain()
        domain_config = domain_pb.spec.yandex_balancer.config
        domain_config.include_upstreams.CopyFrom(self.order.include_upstreams)

        self.entity.dao_update_domain(comment='Changed upstreams', domain_pb=domain_pb)

        self.entity.pb.spec.incomplete = False
        self.entity.dao_update(comment='Saved domain operation spec')
        return self.next_state


class Cancelling(DomainSetUpstreamsProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    def process(self, ctx):
        self.entity.pb.spec.incomplete = False
        self.entity.dao_update(comment='Saved domain operation spec')
        return self.next_state
