import base64
import logging
import uuid

import abc
import enum
import inject
import six

from awacs.lib import ya_vault, certs
from awacs.lib.order_processor.model import (
    WithOrder,
    BaseProcessor,
    FeedbackMessage,
)
from awacs.model import dao, zk, util, cache
from awacs.model.errors import NotFoundError
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    STARTED = 1
    MARKING_OLD_DOMAIN = 2
    MARKING_OLD_CERT = 3
    CREATING_NEW_CERT_SECRET = 4
    CREATING_NEW_CERT = 5
    CREATING_NEW_DOMAIN = 6
    WAITING_FOR_REMOVAL_APPROVAL = 10
    REMOVING_OLD_DOMAIN = 11
    WAITING_FOR_OLD_DOMAIN_REMOVAL = 12
    REMOVING_OLD_CERT = 13
    WAITING_FOR_OLD_CERT_REMOVAL = 14
    UNMARKING_NEW_CERT = 15
    UNMARKING_NEW_DOMAIN = 16
    FINISHING = 20
    FINISHED = 25
    REMOVING_NEW_DOMAIN = 30
    WAITING_FOR_NEW_DOMAIN_REMOVAL = 31
    REMOVING_NEW_CERT = 32
    WAITING_FOR_NEW_CERT_REMOVAL = 33
    UNMARKING_OLD_CERT = 34
    UNMARKING_OLD_DOMAIN = 35
    CANCELLING = 40
    CANCELLED = 41


def get_transfer_processors():
    return (
        Started,
        MarkingOldDomain,
        MarkingOldCerts,
        CreatingNewCertSecrets,
        CreatingNewCerts,
        CreatingNewDomain,
        WaitingForRemovalApproval,
        RemovingOldDomain,
        WaitingForOldDomainRemoval,
        RemovingOldCerts,
        WaitingForOldCertsRemoval,
        UnmarkingNewCerts,
        UnmarkingNewDomain,
        Finishing,
        RemovingNewDomain,
        WaitingForNewDomainRemoval,
        RemovingNewCerts,
        WaitingForNewCertsRemoval,
        UnmarkingOldCerts,
        UnmarkingOldDomain,
        Cancelling,
    )


class DomainTransfer(WithOrder):
    __slots__ = (u'order',)

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'DomainTransfer'
    states = State

    def __init__(self, pb):
        super(DomainTransfer, self).__init__(pb)
        self.order = self.pb.order.content.transfer  # type: model_pb2.DomainOperationOrder.Content.Transfer

    def zk_update(self):
        return self.zk.update_domain_operation(self.namespace_id, self.id)

    def zk_get_domain(self):
        return self.zk.must_get_domain(self.namespace_id, self.id)

    def zk_get_new_domain(self):
        return self.zk.must_get_domain(self.order.target_namespace_id, self.id)

    def zk_get_cert(self):
        return self.zk.must_get_cert(self.namespace_id, self.context[u'cert_id'])

    def zk_get_secondary_cert(self):
        return self.zk.must_get_cert(self.namespace_id, self.context[u'secondary_cert_id'])

    def zk_get_new_cert(self):
        return self.zk.must_get_cert(self.order.target_namespace_id, self.context[u'cert_id'])

    def zk_get_new_secondary_cert(self):
        return self.zk.must_get_cert(self.order.target_namespace_id, self.context['secondary_cert_id'])

    def dao_update(self, comment):
        return self.dao.update_domain_operation(
            namespace_id=self.namespace_id,
            domain_id=self.id,
            version=self.pb.meta.version,
            comment=comment,
            login=util.NANNY_ROBOT_LOGIN,
            updated_spec_pb=self.pb.spec,
        )

    def dao_update_domain(self, domain_pb, comment):
        return self.dao.update_domain(
            namespace_id=domain_pb.meta.namespace_id,
            domain_id=domain_pb.meta.id,
            version=domain_pb.meta.version,
            login=self.pb.meta.author,
            comment=comment,
            updated_meta_pb=domain_pb.meta,
            updated_spec_pb=domain_pb.spec,
        )

    def dao_update_cert(self, comment, meta_pb, spec_pb=None):
        return self.dao.update_cert(
            namespace_id=meta_pb.namespace_id,
            cert_id=meta_pb.id,
            version=meta_pb.version,
            login=self.pb.meta.author,
            comment=comment,
            updated_meta_pb=meta_pb,
            updated_spec_pb=spec_pb,
        )


class DomainTransferProcessor(six.with_metaclass(abc.ABCMeta, BaseProcessor)):
    __slots__ = (u'domain_op',)

    def __init__(self, entity):
        """
        :type entity: DomainTransfer
        """
        super(DomainTransferProcessor, self).__init__(entity)
        self.domain_op = entity


class Started(DomainTransferProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.MARKING_OLD_DOMAIN
    cancelled_state = State.REMOVING_NEW_DOMAIN

    def process(self, ctx):
        domain_pb = self.domain_op.zk_get_domain()
        cert_id = domain_pb.spec.yandex_balancer.config.cert.id
        if cert_id:
            self.domain_op.context[u'cert_id'] = cert_id
        secondary_cert_id = domain_pb.spec.yandex_balancer.config.secondary_cert.id
        if secondary_cert_id:
            self.domain_op.context[u'secondary_cert_id'] = secondary_cert_id
        return self.next_state


class MarkingOldDomain(DomainTransferProcessor):
    __slots__ = ()
    state = State.MARKING_OLD_DOMAIN
    next_state = State.MARKING_OLD_CERT
    next_state_without_cert = State.CREATING_NEW_DOMAIN
    cancelled_state = State.REMOVING_NEW_DOMAIN

    def process(self, ctx):
        domain_pb = self.domain_op.zk_get_domain()
        domain_pb.meta.is_being_transferred.Clear()
        domain_pb.meta.is_being_transferred.value = True
        self.domain_op.dao_update_domain(
            domain_pb=domain_pb,
            comment=u'Transferring domain to namespace "{}"'.format(self.domain_op.order.target_namespace_id))
        if self.domain_op.context.get(u'cert_id'):
            return self.next_state
        else:
            return self.next_state_without_cert


class MarkingOldCerts(DomainTransferProcessor):
    __slots__ = ()
    state = State.MARKING_OLD_CERT
    next_state = State.CREATING_NEW_CERT_SECRET
    cancelled_state = State.REMOVING_NEW_DOMAIN

    def _update_cert(self, cert_pb):
        cert_pb.meta.unrevokable.Clear()
        cert_pb.meta.unrevokable.value = True
        cert_pb.meta.is_being_transferred.Clear()
        cert_pb.meta.is_being_transferred.value = True
        self.domain_op.dao_update_cert(
            meta_pb=cert_pb.meta,
            comment=u'Used in domain "{}", which is being transferred to namespace "{}"'.format(
                self.domain_op.id, self.domain_op.order.target_namespace_id)
        )

    def process(self, ctx):
        if self.domain_op.context.get(u'cert_id'):
            self._update_cert(self.domain_op.zk_get_cert())
        if self.domain_op.context.get(u'secondary_cert_id'):
            self._update_cert(self.domain_op.zk_get_secondary_cert())
        return self.next_state


class CreatingNewCertSecrets(DomainTransferProcessor):
    __slots__ = ()
    state = State.CREATING_NEW_CERT_SECRET
    next_state = State.CREATING_NEW_CERT
    cancelled_state = State.REMOVING_NEW_DOMAIN

    _yav_client = inject.attr(ya_vault.IYaVaultClient)  # type: ya_vault.YaVaultClient

    def _create_secret_copy(self, cert_pb, is_secondary=False):
        cert_secret = self._yav_client.get_version(version=cert_pb.spec.storage.ya_vault_secret.secret_id)
        secret_ver, cert_secret = cert_secret[u'version'], cert_secret[u'value']
        cert_sn = certs.decimal_to_padded_hexadecimal(cert_pb.spec.fields.serial_number)
        public_key, private_key, cert_tarball = certs.extract_certs_from_yav_secret(
            log=logging.getLogger(u'domain_transfer'),
            flat_cert_id='{}/{}'.format(cert_pb.meta.namespace_id, cert_pb.meta.id),
            serial_number=cert_sn,
            cert_secret=cert_secret)
        new_secret_name = u'certificate_{}_private_key_copy_{}'.format(cert_sn, uuid.uuid4())
        new_secret = [{
            u'key': u'{}_certificate'.format(cert_sn),
            u'value': public_key.decode('ascii'),
        }, {
            u'key': u'{}_private_key'.format(cert_sn),
            u'value': private_key.decode('ascii'),
        }]
        if cert_tarball:
            new_secret.append({
                u'key': u'secrets.tgz',
                u'value': base64.b64encode(cert_tarball).decode('ascii'),
                u'encoding': u'base64'
            })
        secret_uuid = self._yav_client.create_secret(new_secret_name)
        if is_secondary:
            context_uuid_key = u'secondary_cert_secret_uuid'
            context_ver_key = u'secondary_cert_secret_version'
        else:
            context_uuid_key = u'cert_secret_uuid'
            context_ver_key = u'cert_secret_version'
        self.domain_op.context[context_uuid_key] = secret_uuid
        self.domain_op.context[context_ver_key] = self._yav_client.create_secret_version(secret_uuid, new_secret)

    def process(self, ctx):
        if self.domain_op.context.get(u'cert_id'):
            self._create_secret_copy(self.domain_op.zk_get_cert())
        if self.domain_op.context.get(u'secondary_cert_id'):
            self._create_secret_copy(self.domain_op.zk_get_secondary_cert(), is_secondary=True)
        return self.next_state


class CreatingNewCerts(DomainTransferProcessor):
    __slots__ = ()
    state = State.CREATING_NEW_CERT
    next_state = State.CREATING_NEW_DOMAIN
    cancelled_state = State.REMOVING_NEW_DOMAIN

    def _create_new_cert(self, cert_pb, comment, is_secondary=False):
        cert_pb.meta.namespace_id = self.domain_op.order.target_namespace_id
        cert_pb.meta.is_being_transferred.comment = comment
        cert_pb.meta.is_being_transferred.mtime.GetCurrentTime()
        cert_pb.meta.unrevokable.comment = comment
        cert_pb.meta.unrevokable.mtime.GetCurrentTime()
        cert_pb.spec.state = cert_pb.spec.PRESENT
        if is_secondary:
            context_uuid_key = u'secondary_cert_secret_uuid'
            context_ver_key = u'secondary_cert_secret_version'
        else:
            context_uuid_key = u'cert_secret_uuid'
            context_ver_key = u'cert_secret_version'
        cert_pb.spec.storage.ya_vault_secret.secret_id = self.domain_op.context[context_uuid_key]
        cert_pb.spec.storage.ya_vault_secret.secret_ver = self.domain_op.context[context_ver_key]
        self.domain_op.dao.create_cert_if_missing(
            meta_pb=cert_pb.meta,
            spec_pb=cert_pb.spec,
            login=self.domain_op.pb.meta.author,
        )

    def process(self, ctx):
        comment = u'Used in domain "{}", which is being transferred from namespace "{}"'.format(
            self.domain_op.id, self.domain_op.namespace_id)
        if self.domain_op.context.get(u'cert_id'):
            self._create_new_cert(self.domain_op.zk_get_cert(), comment)
        if self.domain_op.context.get(u'secondary_cert_id'):
            self._create_new_cert(self.domain_op.zk_get_secondary_cert(), comment, is_secondary=True)
        return self.next_state


class CreatingNewDomain(DomainTransferProcessor):
    __slots__ = ()
    state = State.CREATING_NEW_DOMAIN
    next_state = State.WAITING_FOR_REMOVAL_APPROVAL
    cancelled_state = State.REMOVING_NEW_DOMAIN

    def process(self, ctx):
        domain_pb = self.domain_op.zk_get_domain()
        domain_pb.meta.namespace_id = self.domain_op.order.target_namespace_id
        domain_pb.meta.is_being_transferred.comment = u'Transferring from namespace "{}"'.format(
            self.domain_op.namespace_id)
        domain_pb.spec.yandex_balancer.config.include_upstreams.CopyFrom(self.domain_op.order.include_upstreams)
        domain_pb.meta.is_being_transferred.mtime.GetCurrentTime()
        self.domain_op.dao.create_domain_if_missing(
            meta_pb=domain_pb.meta,
            spec_pb=domain_pb.spec,
            login=self.domain_op.pb.meta.author,
        )
        return self.next_state


class WaitingForRemovalApproval(DomainTransferProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_REMOVAL_APPROVAL
    next_state = State.REMOVING_OLD_DOMAIN
    cancelled_state = State.REMOVING_NEW_DOMAIN

    def process(self, ctx):
        if self.domain_op.pb.order.approval.before_removal:
            return self.next_state
        f = model_pb2.DomainOperationOrder.OrderFeedback
        return FeedbackMessage(
            pb_error_type=f.WAITING_FOR_REMOVAL_APPROVAL,
            message=u'Please confirm old domain removal',
            content=None)


class RemovingOldDomain(DomainTransferProcessor):
    __slots__ = ()
    state = State.REMOVING_OLD_DOMAIN
    next_state = State.WAITING_FOR_OLD_DOMAIN_REMOVAL
    cancelled_state = None

    def process(self, ctx):
        try:
            domain_pb = self.domain_op.zk_get_domain()
            domain_pb.spec.deleted = True
            self.domain_op.dao_update_domain(
                domain_pb=domain_pb,
                comment=u'Removing old domain',
            )
        except NotFoundError:
            pass
        return self.next_state


class WaitingForOldDomainRemoval(DomainTransferProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_OLD_DOMAIN_REMOVAL
    next_state = State.REMOVING_OLD_CERT
    next_state_without_cert = State.UNMARKING_NEW_DOMAIN
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        if self.cache.get_domain(self.domain_op.namespace_id, self.domain_op.id):
            return self.state

        if self.domain_op.context.get(u'cert_id'):
            return self.next_state
        else:
            return self.next_state_without_cert


class RemovingOldCerts(DomainTransferProcessor):
    __slots__ = ()
    state = State.REMOVING_OLD_CERT
    next_state = State.WAITING_FOR_OLD_CERT_REMOVAL
    cancelled_state = None

    def _remove_cert(self, get_cert):
        try:
            cert_pb = get_cert()
            cert_pb.spec.state = cert_pb.spec.REMOVED_FROM_AWACS_AND_STORAGE
            self.domain_op.dao_update_cert(
                meta_pb=cert_pb.meta,
                spec_pb=cert_pb.spec,
                comment=u'Removing old cert to transfer domain "{}"'.format(self.domain_op.id),
            )
        except NotFoundError:
            return

    def process(self, ctx):
        if self.domain_op.context.get(u'cert_id'):
            self._remove_cert(self.domain_op.zk_get_cert)
        if self.domain_op.context.get(u'secondary_cert_id'):
            self._remove_cert(self.domain_op.zk_get_secondary_cert)
        return self.next_state


class WaitingForOldCertsRemoval(DomainTransferProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_OLD_CERT_REMOVAL
    next_state = State.UNMARKING_NEW_CERT
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        cert_id = self.domain_op.context.get(u'cert_id')
        if cert_id and self.cache.get_cert(self.domain_op.namespace_id, cert_id):
            return self.state
        secondary_cert_id = self.domain_op.context.get(u'secondary_cert_id')
        if secondary_cert_id and self.cache.get_cert(self.domain_op.namespace_id, secondary_cert_id):
            return self.state
        return self.next_state


class UnmarkingNewCerts(DomainTransferProcessor):
    __slots__ = ()
    state = State.UNMARKING_NEW_CERT
    next_state = State.UNMARKING_NEW_DOMAIN
    cancelled_state = None

    def _unmark_cert(self, cert_pb):
        cert_pb.meta.unrevokable.Clear()
        cert_pb.meta.is_being_transferred.Clear()
        self.domain_op.dao_update_cert(
            meta_pb=cert_pb.meta,
            comment=u'Transfer of domain "{}" from namespace "{}" is complete'.format(
                self.domain_op.id, self.domain_op.namespace_id
            ))

    def process(self, ctx):
        if self.domain_op.context.get(u'cert_id'):
            self._unmark_cert(self.domain_op.zk_get_new_cert())
        if self.domain_op.context.get(u'secondary_cert_id'):
            self._unmark_cert(self.domain_op.zk_get_new_secondary_cert())
        return self.next_state


class UnmarkingNewDomain(DomainTransferProcessor):
    __slots__ = ()
    state = State.UNMARKING_NEW_DOMAIN
    next_state = State.FINISHING
    cancelled_state = None

    def process(self, ctx):
        new_domain_pb = self.domain_op.zk_get_new_domain()
        new_domain_pb.meta.is_being_transferred.Clear()
        self.domain_op.dao_update_domain(new_domain_pb,
                                         comment=u'Domain transfer from namespace "{}" is complete'.format(
                                             self.domain_op.namespace_id
                                         ))
        return self.next_state


class Finishing(DomainTransferProcessor):
    __slots__ = ()
    state = State.FINISHING
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        self.domain_op.pb.spec.incomplete = False
        self.domain_op.dao_update(comment=u'Finished domain transfer')
        return self.next_state


class RemovingNewDomain(DomainTransferProcessor):
    __slots__ = ()
    state = State.REMOVING_NEW_DOMAIN
    next_state = State.WAITING_FOR_NEW_DOMAIN_REMOVAL
    cancelled_state = None

    def process(self, ctx):
        try:
            new_domain_pb = self.domain_op.zk_get_new_domain()
            new_domain_pb.spec.deleted = True
            self.domain_op.dao_update_domain(
                domain_pb=new_domain_pb,
                comment=u'Removing new domain to cancel transfer',
            )
        except NotFoundError:
            pass
        return self.next_state


class WaitingForNewDomainRemoval(DomainTransferProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_NEW_DOMAIN_REMOVAL
    next_state = State.REMOVING_NEW_CERT
    next_state_without_cert = State.UNMARKING_OLD_DOMAIN
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        if self.cache.get_domain(self.domain_op.order.target_namespace_id, self.domain_op.id):
            return self.state

        if self.domain_op.context.get(u'cert_id'):
            return self.next_state
        else:
            return self.next_state_without_cert


class RemovingNewCerts(DomainTransferProcessor):
    __slots__ = ()
    state = State.REMOVING_NEW_CERT
    next_state = State.WAITING_FOR_NEW_CERT_REMOVAL
    cancelled_state = None

    def _remove_cert(self, get_cert):
        try:
            cert_pb = get_cert()
            cert_pb.spec.state = cert_pb.spec.REMOVED_FROM_AWACS_AND_STORAGE
            self.domain_op.dao_update_cert(
                meta_pb=cert_pb.meta,
                spec_pb=cert_pb.spec,
                comment=u'Removing new cert to cancel transfer of domain "{}"'.format(self.domain_op.id),
            )
        except NotFoundError:
            return

    def process(self, ctx):
        if self.domain_op.context.get(u'cert_id'):
            self._remove_cert(self.domain_op.zk_get_new_cert)
        if self.domain_op.context.get(u'secondary_cert_id'):
            self._remove_cert(self.domain_op.zk_get_new_secondary_cert)
        return self.next_state


class WaitingForNewCertsRemoval(DomainTransferProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_NEW_CERT_REMOVAL
    next_state = State.UNMARKING_OLD_CERT
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        cert_id = self.domain_op.context.get(u'cert_id')
        if cert_id and self.cache.get_cert(self.domain_op.order.target_namespace_id, cert_id):
            return self.state
        secondary_cert_id = self.domain_op.context.get(u'secondary_cert_id')
        if secondary_cert_id and self.cache.get_cert(self.domain_op.order.target_namespace_id, secondary_cert_id):
            return self.state
        return self.next_state


class UnmarkingOldCerts(DomainTransferProcessor):
    __slots__ = ()
    state = State.UNMARKING_OLD_CERT
    next_state = State.UNMARKING_OLD_DOMAIN
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def _unmark_cert(self, cert_pb):
        cert_pb.meta.unrevokable.Clear()
        cert_pb.meta.is_being_transferred.Clear()
        self.domain_op.dao_update_cert(
            meta_pb=cert_pb.meta,
            comment=u'Cancelling transfer of domain "{}"'.format(self.domain_op.id)
            )

    def process(self, ctx):
        if self.domain_op.context.get(u'cert_id'):
            self._unmark_cert(self.domain_op.zk_get_cert())
        if self.domain_op.context.get(u'secondary_cert_id'):
            self._unmark_cert(self.domain_op.zk_get_secondary_cert())
        return self.next_state


class UnmarkingOldDomain(DomainTransferProcessor):
    __slots__ = ()
    state = State.UNMARKING_OLD_DOMAIN
    next_state = State.CANCELLING
    cancelled_state = None

    def process(self, ctx):
        domain_pb = self.domain_op.zk_get_domain()
        domain_pb.meta.is_being_transferred.Clear()
        self.domain_op.dao_update_domain(
            domain_pb=domain_pb,
            comment=u'Cancelling transfer of domain "{}"'.format(self.domain_op.id))
        return self.next_state


class Cancelling(DomainTransferProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    def process(self, ctx):
        self.domain_op.pb.spec.incomplete = False
        self.domain_op.dao_update(comment=u'Cancelled domain transfer')
        return self.next_state
